/* ----------------------------------------------------
 *  Filename:           sortlog.c
 *  Summary:            Polyphase Merge Sort log.
 *  Author:             T.W. Nelson
 *  Compiler:           BCC++ 4.0
 *  Compile options:
 *  Start date:         06-Oct-1995
 *  Last update:        06-Oct-1995
 *  Version:            1.00
 *  Notes:
 *
 *  Source code Copyright (c) 1995 T.W. Nelson.
 *  This source (including derivations thereof) may
 *  be used in any manner provided this copyright
 *  notice is displayed appropriately.
 * ------------------------------------------------- */

#include <iostream.h>
#include <iomanip.h>
#include "sortlog.h"

static ulong items_input;   //#items read from input
static int runs_input;
static int runs_produced;
static long xfer_total;      //total transfer count
static long xfer_count;      //number transfers
void display_runs( MergeList *, MergeFile * );

ulong d_ItemsIn() { return items_input;  }
void d_SetItemsIn( ulong n ) { items_input = n;  }

int  d_RunsIn() { return runs_input; }
void d_SetRunsIn( int n )  { runs_input = n; }

int  d_RunsOut() { return runs_produced; }
void d_SetRunsOut( int n ) { runs_produced = n; }

long d_XferTotal() { return xfer_total; }
void d_SetXferTotal( long n ) { xfer_total = n;  }

long d_XferCount() { return xfer_count; }
void d_SetXferCount( long n) { xfer_count = n; }

void d_ResetLog()
{
    items_input   = 0;
    runs_input    = 0;
    runs_produced = 0;
    xfer_total    = 0;
    xfer_count    = 0;
}

void d_RunDistributionSummary( int levels,
                               MergeList *mp,
                               MergeFile *targ )
{
    size_t runtot = 0;

    for( mp->Restart(); mp->Test(); mp->Next() )
    {   MergeFile *p = mp->Current();
        runtot += ( p->Actual() - p->Null() );
    }

    cout << "Run Distribution Summary:" << endl << endl;
    display_runs( mp, targ );
    cout << endl;
    cout << "Levels        = " << levels << endl;
    cout << "Merge files   = " << (mp->MergeFileCount() + 1) << endl;
    cout << "Buffer size   = " << mp->MergeBufferSize() << endl;
    cout << "Runs input    = " << d_RunsIn() << endl;
    cout << "Runs produced = " << d_RunsOut() << endl;

    if( runtot != d_RunsOut() )
        cout << "*** Run total inconsistent!\n";

    cout << endl << endl;
    cout << "Beginning merge ......" << endl << endl;
}

void d_DisplayRunsAtBegLevel( int level,
                              MergeList *mp,
                              MergeFile *targ )
{
    cout << "-------------------------" << endl;
    cout << "Level = " << level << endl << endl;
    display_runs( mp, targ );
}

void d_DisplayRunsAtEndPass( MergeList *mp, MergeFile *targ )
{
    cout << endl;
    display_runs( mp, targ );
    cout << "Number transfers = " << d_XferCount() << endl;
    cout << endl;
}

void d_DisplayXferTotals()
{
    cout << "Total #items read   = " << d_ItemsIn() << endl;
    cout << "Total #items sorted = " << d_XferCount() << endl;
    cout << "Total #transfers    = " << d_XferTotal() << endl;
}

void d_VerifySort( const char *outfile,
                   InternalSort *insort,
                   MergeList *mp,
                   MergeFile *targ )
{
    /* Detect if file is actually sorted */
    PCDATA i1, i2;
    MergeFile *Dest;
    int ok = 1;
    ulong cnt = 0;
    Dest = mp->PeekHead();
    targ->ResetIOMode( MergeFile::in );
    Dest->ResetIOMode( MergeFile::out );
    while( (i1 = targ->Get()) != 0 )
    {
        ++cnt;
        Dest->Put( i1 );
        i1 = Dest->Lastp();
        if( (i2 = targ->Nextg()) == 0 )
            break;
        if( insort->Compare( &i1, &i2) > 0 )
            ok = 0;
    }

    if( ok && (cnt == items_input) )    {
        cout << "File sorted\n";
        cout << "Sorted output is in \"" << outfile << "\"" << endl;
    }
    else
        cout << "**** File not sorted ****\n";

    cout << "=======================" << endl << endl;
}

static void display_runs( MergeList *mp, MergeFile *targ )
{
    MergeFile *p;
    cout.setf(ios::left,ios::adjustfield);
    cout << setw(10) << "File";

    for( mp->Restart(); mp->Test(); mp->Next() )
    {   p = mp->Current();
        cout << setw(10) << p->Fname();
    }
    cout << endl << setw(10) << "Actual";

    for( mp->Restart(); mp->Test(); mp->Next() )
    {   p = mp->Current();
        cout << setw(10) << p->Actual();
    }
    cout << endl << setw(10) << "Null";

    for( mp->Restart(); mp->Test(); mp->Next() )
    {   p = mp->Current();
        cout << setw(10) << p->Null();
    }
    cout << endl << setw(10) << "Target";
    cout << "File: " << setw(10) << targ->Fname()
         << "Actual = " << setw(10) << targ->Actual()
         << "Null = " << targ->Null() << endl;

    cout.setf(ios::right,ios::adjustfield);  //reset default
}

/* ----- EOF --------------------------------------- */
