//=============================================================================
// File    : XModem.cpp
// Author  : Eric Woodruff,  CIS ID: 72134,1150
// Updated : Mon 04/07/97 21:27:53
// Note    : Copyright 1996-97, Eric Woodruff, All rights reserved
// Compiler: Borland C++ 5.xx, Visual C++ 4.xx
//
// This file contains the XMODEM file transfer functions.
//
//=============================================================================

#include "stdafx.h"
#include "Resource.h"
#include "GUITermDoc.h"
#include "ProgDlg.h"

#include "SerialPort.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

//=============================================================================

// XMODEM definitions
#define BLOCK_SIZE      128
#define RETRIES         10

struct XModemMessage
{
    char    chStart;
    char    chBlockNbr;
    char    chInvBlockNbr;
    char    byData[BLOCK_SIZE];
    char    chChecksum;
};

//=============================================================================

BOOL CGUITermDoc::XModemSend(const char *szFilename)
{
    DCB CurrDCB;
    COMMTIMEOUTS CurrTimeOuts;

	XModemMessage XMsg;

    HANDLE hUpload;
    BOOL  bReadOkay, bDone = FALSE, bFinished = FALSE, bRetVal = FALSE;
    DWORD dwBytesRead, dwSize;
    long lFileOffset = 0L;
    int  nSteps, nIdx, nRetries = 0, nBlockNbr = 1;
    char chResponse, szMsg[80], byChecksum;

    hUpload = CreateFile(szFilename, GENERIC_READ, FILE_SHARE_READ, NULL,
        OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL | FILE_FLAG_SEQUENTIAL_SCAN,
        NULL);
    if(hUpload == INVALID_HANDLE_VALUE)
    {
        MessageBox(NULL, "Error opening file for read!", "File Error",
            MB_ICONEXCLAMATION | MB_OK);
        return FALSE;
    }

    // XMODEM requires N81 communication, and 1 second interval timeouts.
    // Preserve the existing settings and switch to the XMODEM configuration.
    COMPort->GetCommState(&CurrDCB);
    COMPort->GetCommTimeouts(&CurrTimeOuts);

    // Use default N81.  Timeout is 1 second between characters,
    // 10 seconds total timeout.
    COMPort->SetParityDataStop();
    COMPort->SetReadTimeouts(1000L, 0L, 10000L);

    CProgressDlg progDlg;
    progDlg.Create(NULL);
    progDlg.SetWindowText("XMODEM Send File");
    progDlg.SetStatus("Waiting for start...");

    // Progress bar is limited to 65535, so base the
    // step count on the number of read operations.
    dwSize = GetFileSize(hUpload, NULL);

    nSteps = int(dwSize / BLOCK_SIZE);
    if(nSteps != 1)
        nSteps++;

    progDlg.SetRange(0, nSteps);
    progDlg.SetStep(1);
    progDlg.SetPos(0);

    // Wait for initial character
    for(nIdx = 0; nIdx < RETRIES; nIdx++)
    {
        if(COMPort->ReadCommBlock(&chResponse, 1, TRUE) != 1)
            chResponse = ASCII_CAN;
        else
            switch(chResponse)
            {
                case ASCII_NAK:
                case ASCII_CAN:
                    nIdx = RETRIES;
                    break;

                default:
                    break;
            }

        if(progDlg.CheckCancelButton())
        {
            chResponse = ASCII_CAN;
            nIdx = RETRIES;
        }
    }

    // Cancelled or failed?
    if(chResponse != ASCII_NAK)
        bDone = TRUE;

    while(!bDone)
    {
        memset(&XMsg, ' ', sizeof(XModemMessage));

        // Increment file offset and block count if last send
        // was successful
        if(chResponse == ASCII_ACK)
        {
            progDlg.StepIt();
            lFileOffset += BLOCK_SIZE;
            nBlockNbr++;
        }

        // Read next block
        SetFilePointer(hUpload, lFileOffset, 0, FILE_BEGIN);
        bReadOkay = ReadFile(hUpload, &XMsg.byData, BLOCK_SIZE, &dwBytesRead,
            NULL);

        // End of file or error?
        if(dwBytesRead != BLOCK_SIZE)
            if(!bReadOkay)
            {
                // Error reading the file, give up.
                chResponse = ASCII_CAN;
                COMPort->WriteCommBlock(&chResponse, 1);
                COMPort->WriteCommBlock(&chResponse, 1);

                bDone = TRUE;
                dwBytesRead = 0;
            }
            else
                bFinished = TRUE;   // All done, may be some to send

        if(dwBytesRead > 0L)
        {
            // Notify user that a block is being sent
            sprintf(szMsg, "Sending block #%d", nBlockNbr);
            progDlg.SetStatus(szMsg);

            // Send the block
            XMsg.chStart = ASCII_SOH;
            XMsg.chBlockNbr = nBlockNbr % 256;
            XMsg.chInvBlockNbr = ~(nBlockNbr % 256);
            for(byChecksum = nIdx = 0; nIdx < BLOCK_SIZE; nIdx++)
                byChecksum += XMsg.byData[nIdx];

            XMsg.chChecksum = byChecksum;
            COMPort->WriteCommBlock((char *)&XMsg, sizeof(XModemMessage));

            // Wait for a response
            if(COMPort->ReadCommBlock(&chResponse, 1, TRUE) != 1)
                chResponse = ASCII_NAK;
        }
        else
            chResponse = ASCII_NAK;     // Nothing to send

        switch(chResponse)
        {
            case ASCII_ACK:
                nRetries = 0;
                break;

            case ASCII_NAK:
                if(!bFinished)
                {
                    progDlg.SetStatus("Receive error, retrying...");

                    nRetries++;
                    if(nRetries > RETRIES)
                    {
                        // Too many retries, give up.
                        chResponse = ASCII_CAN;
                        COMPort->WriteCommBlock(&chResponse, 1);
                        COMPort->WriteCommBlock(&chResponse, 1);
                        bDone = TRUE;
                    }
                }
                break;

            case ASCII_CAN:
                progDlg.SetStatus("Send cancelled");

                // Wait for another cancel to confirm, else ignore it
                if(COMPort->ReadCommBlock(&chResponse, 1, TRUE) == 1 &&
                  chResponse == ASCII_CAN)
                {
                    // Send two cancels back
                    COMPort->WriteCommBlock(&chResponse, 1);
                    COMPort->WriteCommBlock(&chResponse, 1);
                    bDone = TRUE;
                }
                break;

            default:        // Garbage, retry.
                nRetries++;
                if(nRetries > RETRIES)
                {
                    // Too many retries, give up.
                    chResponse = ASCII_CAN;
                    COMPort->WriteCommBlock(&chResponse, 1);
                    COMPort->WriteCommBlock(&chResponse, 1);
                    bDone = TRUE;
                }
                break;
        }

        // When finished, send EOT and wait for acknowledgement
        if(bFinished)
        {
            progDlg.SetStatus("Transfer complete.  Waiting for EOT flag...");

            nIdx = 0;
            do
            {
                chResponse = ASCII_EOT;
                COMPort->WriteCommBlock(&chResponse, 1);
                if(COMPort->ReadCommBlock(&chResponse, 1, TRUE) != 1)
                    chResponse = ASCII_NAK;

            } while(nIdx++ < RETRIES && chResponse != ASCII_ACK);

            if(chResponse == ASCII_ACK)
                bRetVal = TRUE;         // Receive ack on EOT, success

            bDone = TRUE;
        }

        // User cancelled?
        if(!bDone && progDlg.CheckCancelButton())
        {
            // Send two cancels to the receiver
            progDlg.SetStatus("Cancelling send at user's request");
            chResponse = ASCII_CAN;
            COMPort->WriteCommBlock(&chResponse, 1);
            COMPort->WriteCommBlock(&chResponse, 1);
            bDone = TRUE;
        }
    }

    CloseHandle(hUpload);

    // Restore previous settings
    COMPort->SetCommState(&CurrDCB);
    COMPort->SetCommTimeouts(&CurrTimeOuts);

    return bRetVal;
}

//=============================================================================

BOOL CGUITermDoc::XModemReceive(const char *szFilename)
{
    DCB CurrDCB;
    COMMTIMEOUTS CurrTimeOuts;

	XModemMessage XMsg;

    HANDLE hDownload;
    BOOL   bDone = FALSE, bFinished = FALSE;
    DWORD  dwBytesWritten;
    long lFileOffset;
    int  nIdx, nRetries = 0, nBlockNbr = 1;
    char chResponse, byChecksum, szMsg[80];

    hDownload = CreateFile(szFilename, GENERIC_WRITE, 0, NULL,
        OPEN_ALWAYS, FILE_ATTRIBUTE_NORMAL, NULL);
    if(hDownload == INVALID_HANDLE_VALUE)
    {
        int nErr = GetLastError();
        MessageBox(NULL, "Error opening file for write!", "File Error",
            MB_ICONEXCLAMATION | MB_OK);
        return FALSE;
    }

    // XMODEM requires N81 communication, and 1 second interval timeouts.
    // Preserve the existing settings and switch to the XMODEM configuration.
    COMPort->GetCommState(&CurrDCB);
    COMPort->GetCommTimeouts(&CurrTimeOuts);

    // Use default N81.
    COMPort->SetParityDataStop();

    CProgressDlg progDlg;
    progDlg.Create(NULL);
    progDlg.SetWindowText("XMODEM Receive File");
    progDlg.SetStatus("Waiting for start...");

    while(!bDone)
    {
        memset(&XMsg, ' ', sizeof(XModemMessage));

        // Timeout is 1 second between characters, 10 seconds total timeout
        // while waiting for the first character
        COMPort->SetReadTimeouts(1000L, 0L, 10000L);

        if(COMPort->ReadCommBlock(&chResponse, 1, TRUE) != 1)
            chResponse = ASCII_NAK;

        // Use a one second interval and total timeout after the first
        // character is received
        COMPort->SetReadTimeouts(1000L, 0L, 1000L);

        switch(chResponse)
        {
            case ASCII_EOT:     // All done
                bDone = TRUE;
                bFinished = TRUE;
                chResponse = ASCII_ACK;
                progDlg.SetStatus("Receive complete");
                break;

            case ASCII_SOH:     // Start of block
                chResponse = ASCII_ACK;

                // Read the remainder of the block and check for errors
                if(COMPort->ReadCommBlock((char *)&XMsg + 1,
                  sizeof(XModemMessage) - 1, TRUE) !=
                  sizeof(XModemMessage) - 1)
                    chResponse = ASCII_NAK;     // Bad read, retry
                else
                    if(~(XMsg.chBlockNbr % 256) != XMsg.chInvBlockNbr)
                        chResponse = ASCII_NAK;     // Bad block, retry
                    else
                        if(XMsg.chBlockNbr > nBlockNbr)
                            chResponse = ASCII_CAN;     // Lost data, give up
                        else
                        {
                            for(byChecksum = nIdx = 0; nIdx < BLOCK_SIZE;
                              nIdx++)
                                byChecksum += XMsg.byData[nIdx];

                            if(XMsg.chChecksum != byChecksum)
                                chResponse = ASCII_NAK; // Bad checksum, retry
                        }

                if(chResponse == ASCII_ACK)
                {
                    sprintf(szMsg, "Received block #%d", nBlockNbr);
                    progDlg.SetStatus(szMsg);
                    nRetries = 0;

                    // Save data block to file
                    lFileOffset = BLOCK_SIZE * (XMsg.chBlockNbr - 1);
                    SetFilePointer(hDownload, lFileOffset, 0, FILE_BEGIN);

                    // Give up if write fails
                    if(!WriteFile(hDownload, &XMsg.byData, BLOCK_SIZE,
                      &dwBytesWritten, NULL) || dwBytesWritten != BLOCK_SIZE)
                        chResponse = ASCII_CAN;

                    nBlockNbr = XMsg.chBlockNbr + 1;
                }
                break;

            default:            // Garbage or no response
                nRetries++;

                chResponse = ASCII_NAK;

                if(nRetries > RETRIES)
                    chResponse = ASCII_CAN;     // Too many retries, give up.
                break;
        }

        // Give up?
        if(chResponse == ASCII_CAN)
        {
            progDlg.SetStatus("Fatal Error - Cancelling receive");
            COMPort->WriteCommBlock(&chResponse, 1);
            COMPort->WriteCommBlock(&chResponse, 1);
            bDone = TRUE;
        }
        else
        {
            // Purge remaining data if NAK
            if(chResponse == ASCII_NAK)
                while(COMPort->ReadCommBlock(&chResponse, 1, FALSE) == 1)
                    ;

            COMPort->WriteCommBlock(&chResponse, 1);    // General response
        }

        // User cancelled?
        if(!bDone && progDlg.CheckCancelButton())
        {
            // Send two cancels to the sender
            progDlg.SetStatus("Cancelling receive at user's request");
            chResponse = ASCII_CAN;
            COMPort->WriteCommBlock(&chResponse, 1);
            COMPort->WriteCommBlock(&chResponse, 1);
            bDone = TRUE;
        }
    }

    CloseHandle(hDownload);

    // Restore previous settings
    COMPort->SetCommState(&CurrDCB);
    COMPort->SetCommTimeouts(&CurrTimeOuts);

    return bFinished;
}
