// GUITermDoc.cpp : implementation of the CGUITermDoc class
//

#include "stdafx.h"
#include "GUITerm.h"

#include "SerialPort.h"
#include "GUITermDoc.h"
#include "ComSettings.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

/////////////////////////////////////////////////////////////////////////////
// CGUITermDoc

IMPLEMENT_DYNCREATE(CGUITermDoc, CDocument)

BEGIN_MESSAGE_MAP(CGUITermDoc, CDocument)
	//{{AFX_MSG_MAP(CGUITermDoc)
	ON_COMMAND(ID_FILE_SETTINGS, OnFileSettings)
	ON_UPDATE_COMMAND_UI(ID_FILE_CONNECT, OnUpdateFileConnect)
	ON_UPDATE_COMMAND_UI(ID_FILE_DISCONNECT, OnUpdateFileDisconnect)
	ON_COMMAND(ID_FILE_CONNECT, OnFileConnect)
	ON_COMMAND(ID_FILE_DISCONNECT, OnFileDisconnect)
	ON_COMMAND(ID_FILE_XRECEIVE, OnFileXreceive)
	ON_COMMAND(ID_FILE_XSEND, OnFileXsend)
	ON_UPDATE_COMMAND_UI(ID_FILE_XRECEIVE, OnUpdateFileXreceive)
	ON_UPDATE_COMMAND_UI(ID_FILE_XSEND, OnUpdateFileXsend)
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CGUITermDoc construction/destruction

CGUITermDoc::CGUITermDoc()
{
    COMPort = NULL;

    m_csPort = "COM1";
    m_csBaudRate = "19200";
    m_csParity = "None";
    m_csDataBits = "8";
    m_csStopBits = "1";
    m_csFlowControl = "Hardware";
    m_bLocalEcho = FALSE;
    m_bSendCRLF = FALSE;
}

CGUITermDoc::~CGUITermDoc()
{
    if(COMPort)
        delete COMPort;
}

BOOL CGUITermDoc::OnNewDocument()
{
	if (!CDocument::OnNewDocument())
		return FALSE;

    SetTitle("Not connected");

	return TRUE;
}

/////////////////////////////////////////////////////////////////////////////
// CGUITermDoc serialization

void CGUITermDoc::Serialize(CArchive& ar)
{
	if (ar.IsStoring())
	{
		// TODO: add storing code here
	}
	else
	{
		// TODO: add loading code here
	}
}

/////////////////////////////////////////////////////////////////////////////
// CGUITermDoc diagnostics

#ifdef _DEBUG
void CGUITermDoc::AssertValid() const
{
	CDocument::AssertValid();
}

void CGUITermDoc::Dump(CDumpContext& dc) const
{
	CDocument::Dump(dc);
}
#endif //_DEBUG

/////////////////////////////////////////////////////////////////////////////
// CGUITermDoc communication functions

// This is the thread function that handles incoming data.  For this
// application, the caller of CSerialPort::StartCommThread() passes in
// a pointer to a GUITermData structure which contains a pointer to
// the port object and a pointer to the document object.
DWORD WINAPI CGUITermDoc::CommReader(void *pvData)
{
    GUITermData *pTermData = (GUITermData *)pvData;
    CSerialPort *COMPort = pTermData->COMPort;
    CGUITermDoc *pDoc = pTermData->TermDoc;

    DWORD dwEventMask;
    int   nSize;
    char  byBuffer[MAX_SIZE];

    // Set the event mask
    COMPort->SetCommMask(EV_RXCHAR);

    // Tell the port to notify us when a read event occurs
    if(!COMPort->WaitCommEvent())
        return 0xFFFFFFFF;      // Error

    // Wait for an empty Comm event that signals the COM port object
    // is shutting down.
    do
    {
        // Since this is a separate thread, we can pass TRUE so
        // that the function doesn't return until an event occurs.
        dwEventMask = COMPort->CheckForCommEvent(TRUE);
        if(dwEventMask & EV_RXCHAR)
        {
            // Read as much as possible and display it on the screen
            nSize = COMPort->ReadCommBlock(byBuffer, MAX_SIZE);
            if(nSize)
                pDoc->ReceiveChars(byBuffer, nSize);

            // Set it up to wait for the next event
            COMPort->WaitCommEvent();
        }

    } while(dwEventMask);

    return 0L;
}

// Send a character out to the port and optionally echo it back
// to the screen.
BOOL CGUITermDoc::SendChar(char chChar)
{
    char *byData = &chChar;
    int nSize = 1;

    // Send CR/LF?
    if(chChar == '\r' && m_bSendCRLF)
    {
        byData = "\r\n";
        nSize = 2;
    }

    if(!COMPort || COMPort->WriteCommBlock(byData, nSize) != nSize)
    {
        MessageBeep(0xFFFFFFFF);
        return FALSE;
    }

    // Echo locally?
    if(m_bLocalEcho)
    {
        POSITION pos = GetFirstViewPosition();
        while(pos != NULL)
        {
            CView *pView = GetNextView(pos);
            pView->SendMessage(WM_COMMDATA, nSize, (DWORD)byData);
        }
    }

    return TRUE;
}

// Send a block of data out to the port and optionally echo it back
// to the screen.
BOOL CGUITermDoc::SendBlock(const char *byData, int nSize)
{
    if(!COMPort || COMPort->WriteCommBlock((LPSTR)byData, nSize) != nSize)
    {
        MessageBeep(0xFFFFFFFF);
        return FALSE;
    }

    // Echo locally?
    if(m_bLocalEcho)
    {
        POSITION pos = GetFirstViewPosition();
        while(pos != NULL)
        {
            CView *pView = GetNextView(pos);
            pView->SendMessage(WM_COMMDATA, nSize, (DWORD)byData);
        }
    }

    return TRUE;
}

// This is called from the comm thread to display received data
void CGUITermDoc::ReceiveChars(const char *byBuffer, int nSize)
{
    POSITION pos = GetFirstViewPosition();
    while(pos != NULL)
    {
        CView *pView = GetNextView(pos);
        pView->SendMessage(WM_COMMDATA, nSize, (DWORD)byBuffer);
    }
}

/////////////////////////////////////////////////////////////////////////////
// CGUITermDoc commands

void CGUITermDoc::OnFileSettings()
{
	CComSettings comDlg;
    BOOL bChange = TRUE, bReconnect = FALSE;

    comDlg.m_csPort = m_csPort;
    comDlg.m_csBaudRate = m_csBaudRate;
    comDlg.m_csParity = m_csParity;
    comDlg.m_csDataBits = m_csDataBits;
    comDlg.m_csStopBits = m_csStopBits;
    comDlg.m_csFlowControl = m_csFlowControl;
    comDlg.m_bLocalEcho = m_bLocalEcho;
    comDlg.m_bSendCRLF = m_bSendCRLF;

    if(comDlg.DoModal() == IDOK)
    {
        if(m_csPort != comDlg.m_csPort && COMPort)
            if(MessageBox(NULL,"Already connected to a different port.  "
              "Close and reconnect using new port?", "Change Port",
              MB_ICONINFORMATION | MB_YESNO | MB_DEFBUTTON2) == IDYES)
            {
                OnFileDisconnect();
                bReconnect = TRUE;
            }
            else
                bChange = FALSE;

        if(bChange)
        {
            // Change settings and possible the port too
            m_csPort = comDlg.m_csPort;
            m_csBaudRate = comDlg.m_csBaudRate;
            m_csParity = comDlg.m_csParity;
            m_csDataBits = comDlg.m_csDataBits;
            m_csStopBits = comDlg.m_csStopBits;
            m_csFlowControl = comDlg.m_csFlowControl;
            m_bLocalEcho = comDlg.m_bLocalEcho;
            m_bSendCRLF = comDlg.m_bSendCRLF;

            if(COMPort || bReconnect)
                OnFileConnect();
        }
        else
            MessageBox(NULL, "Port settings not changed", "No Change",
                MB_ICONEXCLAMATION | MB_OK);
    }
}

// Disable Connect if already connected
void CGUITermDoc::OnUpdateFileConnect(CCmdUI* pCmdUI)
{
    pCmdUI->Enable(COMPort ? FALSE : TRUE);
}

// Disable Disconnect if already disconnected
void CGUITermDoc::OnUpdateFileDisconnect(CCmdUI* pCmdUI)
{
    pCmdUI->Enable(COMPort ? TRUE : FALSE);
}

void CGUITermDoc::OnFileConnect()
{
    BOOL bStartThread = FALSE, bOkay = TRUE;
    char szText[256];
    int nBaud, nParity = NOPARITY, nData, nStop = ONESTOPBIT,
        nFlow = PCF_DTRDSR | PCF_RTSCTS;

    // If not already done, create the comm object
    if(!COMPort)
    {
        // Delete current content from all views
        POSITION pos = GetFirstViewPosition();
        while(pos != NULL)
        {
            // This is for the demo, so we'll assume
            // it really is a CEditView.
            CEditView *pView = (CEditView *)GetNextView(pos);

            CEdit &eCtrl = pView->GetEditCtrl();
            eCtrl.SetSel(0, -1, FALSE);
            eCtrl.Clear();
        }

        // Open the named port
        COMPort = new CSerialPort(m_csPort);
        if(!COMPort || !COMPort->IsValid())
            bOkay = FALSE;
        else
            bStartThread = TRUE;
    }

    if(bOkay)
    {
        // Change the port settings
        nBaud = atoi(m_csBaudRate);
        nData = atoi(m_csDataBits);
        if(m_csStopBits == "2")
            nStop = TWOSTOPBITS;

        if(m_csParity == "Even")
            nParity = EVENPARITY;
        else
            if(m_csParity == "Odd")
                nParity = ODDPARITY;

        if(m_csFlowControl == "Software")
            nFlow = PCF_XONXOFF;
        else
            if(m_csFlowControl == "None")
                nFlow = PCF_NOFLOWCTL;

        COMPort->SetParityDataStop(nParity, nData, nStop);
        COMPort->SetFlowControl(nFlow);

        // Start the terminal thread
        TermData.COMPort = COMPort;
        TermData.TermDoc = this;

        if(bStartThread && !COMPort->StartCommThread(CommReader, &TermData))
        {
            MessageBox(NULL, "Error starting comm thread!", "Error",
                MB_ICONEXCLAMATION | MB_OK);

            delete COMPort;
            COMPort = NULL;
        }
        else
            SetTitle(m_csPort);
    }
    else
    {
        // Get error message
        if(!COMPort)
            strcpy(szText, "Error creating port object!");
        else
            FormatMessage(FORMAT_MESSAGE_FROM_SYSTEM |
                FORMAT_MESSAGE_IGNORE_INSERTS, NULL, COMPort->GetLastError(),
                MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT),
                (LPTSTR)szText, sizeof(szText), NULL);

        MessageBox(NULL, szText, "Error", MB_ICONEXCLAMATION | MB_OK);

        if(COMPort)
        {
            delete COMPort;
            COMPort = NULL;
        }
    }
}

void CGUITermDoc::OnFileDisconnect()
{
    delete COMPort;
    COMPort = NULL;
    SetTitle("Not connected");
}

void CGUITermDoc::OnFileXreceive()
{
    CFileDialog fileDlg(FALSE, NULL, "*.*");

    if(fileDlg.DoModal() == IDOK)
    {
        // Stop the background receiver thread temporarily
        COMPort->StopCommThread();

        if(!XModemReceive(fileDlg.GetPathName()))
            MessageBox(NULL, "Error receiving file!", "Receive Error",
                MB_ICONEXCLAMATION | MB_OK);
        else
            MessageBox(NULL, "Receive completed successfully", "Receive Okay",
                MB_ICONINFORMATION | MB_OK);

        // Restart the background receiver thread with the last used options
        if(!COMPort->StartCommThread(CommReader, &TermData))
        {
            MessageBox(NULL, "Error starting comm thread!", "Error",
                MB_ICONEXCLAMATION | MB_OK);

            // Disconnect on failure to restart
            OnFileDisconnect();
        }
    }
}

void CGUITermDoc::OnFileXsend()
{
    CFileDialog fileDlg(TRUE, NULL, "*.*", OFN_FILEMUSTEXIST |
        OFN_HIDEREADONLY);

    if(fileDlg.DoModal() == IDOK)
    {
        // Stop the background receiver thread temporarily
        COMPort->StopCommThread();

        if(!XModemSend(fileDlg.GetPathName()))
            MessageBox(NULL, "Error sending file!", "Send Error",
                MB_ICONEXCLAMATION | MB_OK);
        else
            MessageBox(NULL, "Send completed successfully", "Send Okay",
                MB_ICONINFORMATION | MB_OK);

        // Restart the background receiver thread with the last used options
        if(!COMPort->StartCommThread(CommReader, &TermData))
        {
            MessageBox(NULL, "Error starting comm thread!", "Error",
                MB_ICONEXCLAMATION | MB_OK);

            // Disconnect on failure to restart
            OnFileDisconnect();
        }
    }
}

void CGUITermDoc::OnUpdateFileXreceive(CCmdUI* pCmdUI)
{
    pCmdUI->Enable(COMPort ? TRUE : FALSE);
}

void CGUITermDoc::OnUpdateFileXsend(CCmdUI* pCmdUI)
{
    pCmdUI->Enable(COMPort ? TRUE : FALSE);
}
