// File    : TermPoll.cpp
// Author  : Eric Woodruff,  CIS ID: 72134,1150
// Updated : Mon 04/07/97 21:28:54
// Note    : Copyright 1996-97, Eric Woodruff, All rights reserved
// Compiler: Borland C++ 5.xx, Visual C++ 4.xx
//
// Compile with:
//      bcc32 -DBUILD_SPCOMM_NODLL termpoll.cpp serialport.cpp
//   or
//      cl -DBUILD_SPCOMM_NODLL termpoll.cpp serialport.cpp
//
// A less efficient implementation of Terminal.cpp.  This polls for
// keyboard input and incoming serial data in the same thread.
//

#include <windows.h>
#include <ctype.h>
#include <stdio.h>

#include "SerialPort.h"

BOOL InitConsole(void);
void DumbTerminal(const char *pszPortName, const char *pszOptions);

#define MAX_SIZE    1024

HANDLE hStdIn, hStdOut;

void main(int argc, char *argv[])
{
    if(argc < 3)
    {
        printf(" Syntax: Terminal portname parity_data_stop_localecho\n"
               "Example: Terminal COM1 7E1\n"
               "Example: Terminal COM1 N81L  (Add L for local echo)");
        exit(1);
    }

    if(!InitConsole())
        printf("Could not initialize console handles\n");
    else
        DumbTerminal(argv[1], argv[2]);

    exit(0);
}

BOOL InitConsole(void)
{
    // Get handles for standard input and output
    hStdIn = GetStdHandle(STD_INPUT_HANDLE);
    if(hStdIn == INVALID_HANDLE_VALUE)
        return FALSE;

    hStdOut = GetStdHandle(STD_OUTPUT_HANDLE);
    if(hStdOut == INVALID_HANDLE_VALUE)
        return FALSE;

    // Turn off all input mode processing
    SetConsoleMode(hStdIn, 0);

    return TRUE;
}

void DumbTerminal(const char *pszPortName, const char *pszOptions)
{
    INPUT_RECORD irInput;
    BOOL bContinue = TRUE;
    DWORD dwRecords, dwEventCount, dwEventMask, dwBytesWritten;

    char  chChar, byBuffer[MAX_SIZE];
    int   nSize;

    // Assumed defaults are N81
    BOOL bLocalEcho = FALSE;
    int  nParity = NOPARITY, nDataBits = EIGHTDATABITS,
         nStopBits = ONESTOPBIT;

    // This is just a demo, so there is no real editing
    chChar = toupper(pszOptions[0]);
    if(chChar == 'E')
        nParity = EVENPARITY;
    else
        if(chChar == 'O')
            nParity = ODDPARITY;

    if(pszOptions[1] == '7')
        nDataBits = SEVENDATABITS;

    if(pszOptions[2] == '2')
        nDataBits = TWOSTOPBITS;

    if(toupper(pszOptions[3]) == 'L')
        bLocalEcho = TRUE;

    // Open the port and use the given settings
    CSerialPort COMPort(pszPortName);
    COMPort.SetParityDataStop(nParity, nDataBits, nStopBits);

    // If valid, start watching for events
    if(!COMPort.IsValid() || !COMPort.WaitCommEvent())
        printf("Error %ld initializing COM port!",
            COMPort.GetLastError());
    else
    {
        printf("Connected on %s using options %s\nESC to quit\n",
            pszPortName, pszOptions);

        // Poll for keyboard and serial input until ESC is hit
        do
        {
            // Just check for waiting events, don't block.
            dwEventMask = COMPort.CheckForCommEvent(FALSE);
            if(dwEventMask & EV_RXCHAR)
            {
                // Read as much as possible and display it on the screen
                nSize = COMPort.ReadCommBlock(byBuffer, MAX_SIZE);
                if(nSize)
                    WriteConsole(hStdOut, byBuffer, nSize,
                        &dwBytesWritten, NULL);

                // Set it up to wait for the next event
                COMPort.WaitCommEvent();
            }

            GetNumberOfConsoleInputEvents(hStdIn, &dwEventCount);
            while(dwEventCount--)
            {
                if(ReadConsoleInput(hStdIn, &irInput, 1, &dwRecords))
                {
                    if(irInput.EventType == KEY_EVENT &&
                      irInput.Event.KeyEvent.bKeyDown)
                    {
                        chChar = irInput.Event.KeyEvent.uChar.AsciiChar;

                        // If ESC, stop.  If not zero, write it to the
                        // port.
                        if(chChar == 27)
                            bContinue = FALSE;
                        else
                            if(chChar)
                            {
                                COMPort.WriteCommBlock(&chChar, 1);

                                // For non-modem connections
                                if(bLocalEcho)
                                    WriteConsole(hStdOut, &chChar, 1,
                                        &dwRecords, NULL);
                            }
                    }
                }
            }

            // Without a slight delay here, the CPU usage runs at
            // about 100% all the time.  A short delay levels out
            // the usage but may hurt overall performance.
            Sleep(1L);

        } while(bContinue);
    }
}
