unit FileInfo;

{ ==============================================================
  FileInfo 1.0.0
{ ==============================================================

  SUMMARY

  Provides commonly used properties and version information of
  a file

  Author:       1997, Andy Schmidt
  Email:       Andy_Schmidt@CompuServe.com
  Compiler:    Delphi 2.01
  Runtime:     Win32


{ --------------------------------------------------------------


  USAGE

  1. include FileInfo in your "USES" clause

  2. declare an object variable to hold the object reference, e.g.

     var ObjFileInfo: TFileInfo;

  3. anywhere inside your implementation, create the object, e.g.

     ObjFileInfo := TFileInfo.Create;   // Get info for current executable
     -- or --
     ObjFileInfo := TFileInfo.CreateFromName('c:\test.exe');

  4. access the properties and methods:

     print ObjFileInfo.QueryVersionString('FileDescription');

{ --------------------------------------------------------------


  PROPERTIES  (Read-only, unless indicated)

  NamePath         Name and path to the file.
                   (Defaults to Application.ExeName)

  Name             Filename (without extension or path)

  Date             Timestamp of the file (creation date/time)

  ProductVersion
  FileVersion      Objects created if this is a program and was
                   linked with VersionInfo resources.
                   Subproperties:
                   vv            = Version Number
                   rr            = Release Number
                   mm            = Modification Level/Build
                   pp            = Patch/Hotfix Number

  Lang_Charset     [Read/Write] For multi-language resource files
                   selects the proper VersionInfo.
                   (Defaults to U.S. English '040904E4')

  FixedInfo        Pointer to the VSFixedFileInfo (see Windows unit),
                   which contains version numbers and release status
                   flags.


  METHODS

  QueryVersionString
                   Returns a string of a VersionInfo resource
                   such as LegalCopyright, CompanyName, FileDescription.


  EVENTS

  - none -


  FUNCTIONS

  ExtractFileNameOnly
                   Strips path and extension information off a filename
                   string.


{ --------------------------------------------------------------

  LICENSE

  The Author hereby grants to you a nonexclusive license to use
  this software and the accompanying Instructions, only as
  authorized in this License.

  You agree that you will not assign, sublicense, transfer,
  pledge, lease, rent, or share your rights under this License
  in return for compensation of any kind. Before you use this
  software for commercial purposes, you are required to
  pay a license fee of $20.00 to the Author.

  You acknowledge and agree that the Software and the
  accompanying Instructions are intellectual property of
  the Author, protected under U.S. copyright law. You further
  acknowledge and agree that all right, title and interest in
  and to the Software, are and shall remain with the Author.
  This License does not convey to you an interest in or to the
  Software, but only a limited and revocable right of use.

  THIS SOFTWARE IS LICENSED "AS IS," AND LICENSOR DISCLAIMS ANY
  AND ALL WARRANTIES, WHETHER EXPRESS OR IMPLIED, INCLUDING,
  WITHOUT LIMITATION, ANY IMPLIED WARRANTIES OF MERCHANTABILITY
  OR FITNESS FOR A PARTICULAR PURPOSE.

  Author's cumulative liability to you or any other party for
  any loss or damages resulting from any claims, demands, or
  actions arising out of or relating to this License shall not
  exceed the license fee paid (if any) to Author for the use of
  the Software. In no event shall Author be liable for any
  indirect, incidental, consequential, special, or exemplary
  damages or lost profits, even if Author has been advised of
  the possibility of such damages.

  This software and accompanying instructions are provided with
  restricted rights. Use, duplication or disclosure by the
  Government is subject to restrictions as set forth in
  subparagraph (c)(1)(ii) of The Rights in Technical Data and
  Computer Software clause at DFARS 252.227-7013 or
  subparagraphs (c)(1) and (2) of the Commercial Computer
  Software - Restricted Rights 48 CFR 52.227-19, as applicable.

{ --------------------------------------------------------------

  CHANGE HISTORY

  1.0.0 19-Jan-97 (AS)  Initial Development
  1.0.1 28-Jan-97 (AS)  For Date property use an access method
                        with FileAge() instead of FileGetDate()
                        in the Create method.
        28-Jan-97 (AS)  ReadFileInfo protected method became part of
                        the secondary constructor, which is called
                        by the Create method.
        28-Jan-97 (AS)  added AssignTo method for TProjVersion

  -------------------------------------------------------------- }

interface

uses
  Classes, SysUtils, Forms, Windows,
  UCommon;

type

  // Project Version in format vv.rr.mm.pp
  TProjVersion = class(TPersistent)       // Must descend from TPersistent to allow subproperties
  private
    Fvv, Frr, Fmm, Fpp: byte;
  protected
    procedure AssignTo(Dest: TPersistent); override;
  published
    property vv: byte read Fvv write Fvv default 0;
    property rr: byte read Frr write Frr default 0;
    property mm: byte read Fmm write Fmm default 0;
    property pp: byte read Fpp write Fpp default 0;
  end;

  // File Info (Main object)
  TFileInfo = class(TObject)
  private
  { Private declarations: visible only in this unit }
    // Fields to store property values
    FNamePath: string;
    FFixedInfo: PVSFixedFileInfo;
    FLang_CharSet: string;
    FFileVersion: TProjVersion;
    FProductVersion: TProjVersion;
    // Other instance variables
    VersionInfoBuffer: PWideChar;
    VersionInfoSize: integer;
    LastRC: LongBool;
    // Property access methods
    function GetFileName: string;
    function GetDate: TDateTime;

  public
  { Public declarations: visible only at run-time }
    property Date: TDateTime read GetDate;
    property NamePath: string read FNamePath;
    property Name: string read GetFileName;
    property FixedInfo: PVSFixedFileInfo read FFixedInfo;
    property FileVersion: TProjVersion read FFileVersion;
    property ProductVersion: TProjVersion read FProductVersion;
    property Lang_Charset: string read FLang_Charset write FLang_Charset;
    constructor Create;
    constructor CreateFromName(const FileNamePath: string);
    destructor Destroy; override;
    function QueryVersionString(const Stringname: string): string; virtual;

  end;

function ExtractFileNameOnly(const FileNamePath: string): string;


implementation

const
    { Default Values }
    DLang_Charset = '040904E4';


{ Initialize the object properties }
constructor TFileInfo.Create;

begin
    self.CreateFromName(Application.ExeName);
end;

constructor TFileInfo.CreateFromName(const FileNamePath: string);

var
    TempVerHandle: integer;
    TempVerFieldLength: integer;

begin
    FNamePath := FileNamePath;              // Save file path

    { Access version information of an executable }
    VersionInfoSize := GetFileVersionInfoSize( PChar(FileNamePath), TempVerHandle );
    if VersionInfoSize > 0 then
        begin
        GetMem(VersionInfoBuffer, VersionInfoSize);
        FLang_Charset := DLang_Charset;
        LastRC := GetFileVersionInfo( PChar(FileNamePath), TempVerHandle, VersionInfoSize, VersionInfoBuffer);
        if LastRC then
            { Initialize version objects from the fixed-format version info }
            begin
            LastRC := VerQueryValue( VersionInfoBuffer, '\', Pointer(FFixedInfo), TempVerFieldLength);
            if LastRC then
                begin
                FFileVersion := TProjVersion.Create;
                with FFileVersion do
                    begin
                    vv := PDWORD(@FFixedInfo.dwFileVersionMS).ms;
                    rr := PDWORD(@FFixedInfo.dwFileVersionMS).ls;
                    mm := PDWORD(@FFixedInfo.dwFileVersionLS).ms;
                    pp := PDWORD(@FFixedInfo.dwFileVersionLS).ls;
                    end;
                FProductVersion := TProjVersion.Create;
                with FProductVersion do
                    begin
                    vv := PDWORD(@FFixedInfo.dwProductVersionMS).ms;
                    rr := PDWORD(@FFixedInfo.dwProductVersionMS).ls;
                    mm := PDWORD(@FFixedInfo.dwProductVersionLS).ms;
                    pp := PDWORD(@FFixedInfo.dwProductVersionLS).ls;
                    end;{with}
                end;{if}
            end; {if}
        end; {if}
end;

destructor TFileInfo.Destroy;
begin
    FFileVersion.Free;                      // Free the version object
    FProductVersion.Free;
    if VersionInfoBuffer <> nil then FreeMem(VersionInfoBuffer, VersionInfoSize);
    inherited destroy;                      // Call the inherited destructor last
end;


{ Property Access Methods }
function TFileInfo.GetFilename: string;
begin
    Result := ExtractFileNameOnly(FNamePath);
end;

function TFileInfo.GetDate: TDateTime;
begin
    Result := FileDateToDateTime(FileAge(FNamePath));
end;

procedure TProjVersion.AssignTo(Dest: TPersistent);
begin
    if Dest.ClassName = ClassName then
        with TProjVersion(Dest) do
        begin
            vv := Self.Fvv;
            rr := Self.Frr;
            mm := Self.Fmm;
            pp := Self.Fpp;
        end
    else inherited;
end;

{ Extract a version information string }
function TFileInfo.QueryVersionString(const Stringname: string): string;

var
    PVerField: PChar;
    TempVerFieldLength: integer;
    PTempVersionString: PChar;

begin
    if (VersionInfoBuffer = nil) then exit;

    LastRC := VerQueryValue( VersionInfoBuffer,
                             PChar('\StringFileInfo\' + FLang_Charset + '\' + Stringname),
                             Pointer(PVerField),
                             TempVerFieldLength
                             );
    if LastRC then
        { Use temporary space to extract version information and pass as string}
        begin
        PTempVersionString := StrAlloc(TempVerFieldLength + 1);
        Result := StrLCopy(PTempVersionString, PVerField, TempVerFieldLength);
        StrDispose(PTempVersionString);
        end
    else Result := '';
end;


{ Extract the filename without extension }
function ExtractFileNameOnly(const FileNamePath: string): string;

var
    TempFileID: string;
    LengthOfFileExt: byte;

begin
    // Extract the filename and extension
    TempFileID := ExtractFileName(FileNamePath);
    // Determine length of extension
    LengthOfFileExt := Length(ExtractFileExt(TempFileID));
    // Extract the filename without the extension
    Result := Copy(TempFileID, 1, Length(TempFileID) - LengthOfFileExt);
end;


end.
