{**************************************}
{       TRFM 2.5 for Delphi 16/32      }
{ Copyright 1997  RealSoft Development }
{                - - -                 }
{       RealForm Image Components      }
{**************************************}

{$DEFINE _FLY}

unit Rfm;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls, Buttons,
  Forms, Dialogs, StdCtrls, ExtCtrls, Printers, DsgnIntf, Menus, FileCtrl,
  DynSlim, RfmRec{$IFDEF FLY}, FlyCtrl{$ENDIF};

const
  INCH = 96;

type
  TRFM = class(TPersistent)
    private
      FOnChange: TNotifyEvent;
      FDesigning: boolean;
      procedure DrawObj(MyCanvas : TCanvas; MyRect : TRect; Num : smallint);
      function GetEmpty : boolean;
      function GetWidth : smallint;
      function GetHeight : smallint;
      function GetCanvas : TCanvas;
      function MakeRect : TRect;
    protected
      Bitmap   : TBitmap;
      DynArray : TDynArray;
      Header   : RFormHdr;
      StrTable : TStrings;
      procedure DefineProperties(Filer : TFiler); override;
    public
      constructor Create;
      destructor Destroy; override;
      procedure Clear;
      procedure ClearBMP;
      procedure Print;
      procedure Draw;
      procedure ScaleDraw(MyRect : TRect);
      procedure DrawTo(MyCanvas : TCanvas; MyRect : TRect);
      procedure Assign(Source: TRFM);
      procedure LoadFromfile(filename : String);
      procedure SaveToFile(filename : String);
      procedure LoadFromStream(Stream : TStream);
      procedure SaveToStream(Stream : TStream);
      procedure SaveAsBitmap(filename : String);
      procedure SaveAsMetafile(filename : String);
      function GetString( Index : smallint ) : String;
      function ByteToColor (AByte : byte) : TColor;

      property Width: smallint           read GetWidth;
      property Height: smallint          read GetHeight;
      property Empty: boolean            read GetEmpty;
      property Canvas: TCanvas           read GetCanvas;
      property Designing: boolean        read FDesigning    write FDesigning;
      property OnChange: TNotifyEvent    read FOnChange     write FOnChange;
    end;

  TTextFormat = ( tfGeneral, tfInteger, tfReal, tfDollar, tfPhone, tfDate, tfCheck,
                  tfTime, tfCustom1, tfCustom2,tfCustom3,tfCustom4,tfCustom5,
                  tfCustom6,tfCustom7, tfCustom8,tfCustom9,tfCustom10,tfCustom11,
                  tfCustom12,tfCustom13, tfCustom14,tfCustom15 );
  TOnPrintField = procedure ( Sender: TObject; FieldName: String; Format: TTextFormat;
                              var Text : String ) of object;

  TRFMImage = class(TGraphicControl)
  private
    FRFM : TRFM;
    FAutoSize: Boolean;
    FStretch: Boolean;
    FCenter: Boolean;
    FAspect: Boolean;
    FAutoFill: Boolean;
    FAutoLogo: Boolean;
    FAutoFields: Boolean;
    FFillList: TStrings;
    FLogoList: TStrings;
    FOnChange: TNotifyEvent;
    FOnPrintField: TOnPrintField;
    procedure SetAspect(AValue: Boolean);
    procedure SetAutoSize(AValue: Boolean);
    procedure SetCenter(AValue: Boolean);
    procedure SetStretch(AValue: Boolean);
    procedure SetRFM(AValue: TRFM);
    procedure SetFillList(AValue: TStrings);
    procedure SetLogoList(AValue: TStrings);
    function  GetCanvas : TCanvas;
    procedure RFMChanged(Sender : TObject);
    procedure UpdateRFM;
    function  CalcRect : TRect;
  protected
    procedure Paint; override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function  RFMRect : TRect;
    procedure SetPaperSize;
    procedure Clear;
    procedure Print;
    procedure PrintTo(ACanvas : TCanvas; MyRect : TRect);
    procedure FillTo(ACanvas : TCanvas; MyRect : TRect);
    procedure LogoTo(ACanvas : TCanvas; MyRect : TRect);
    procedure FieldsTo(ACanvas : TCanvas; MyRect : TRect);
    property Canvas: TCanvas     read GetCanvas;
  published
    property AutoSize: Boolean   read FAutoSize   write SetAutoSize default False;
    property AutoFill: Boolean   read FAutoFill   write FAutoFill   default False;
    property AutoLogo: Boolean   read FAutoLogo   write FAutoLogo   default False;
    property AutoFields: Boolean read FAutoFields write FAutoFields default False;
    property Center: Boolean     read FCenter     write SetCenter   default False;
    property Stretch: Boolean    read FStretch    write SetStretch  default False;
    property Aspect: Boolean     read FAspect     write SetAspect   default True;
    property RFM: TRFM           read FRFM        write SetRFM;
    property FillList: TStrings  read FFillList   write SetFillList;
    property LogoList: TStrings  read FLogoList   write SetLogoList;
    property Align;
    property DragCursor;
    property DragMode;
    property Enabled;
    property ParentShowHint;
    property PopupMenu;
    property ShowHint;
    property Visible;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnChange: TNotifyEvent       read FOnChange      write FOnChange;
    property OnPrintField : TOnPrintField read FOnPrintField  write FOnPrintField;
  end;

  TRFMProperty = class( TPropertyEditor )
  private
    Dialog : TForm;
    Label1 : TLabel;
    DirectoryListBox1 : TDirectoryListBox;
    FileListBox1 : TFileListBox;
    DriveComboBox1 : TDriveComboBox;
    Edit1 : TEdit;
    btnCancel : TBitBtn;
    btnOK : TBitBtn;
    btnClear : TBitBtn;
    ScrollBox1 : TScrollBox;
    RFMImage1 : TRFMImage;
    ckScale : TCheckBox;
    btnSaveBMP : TBitBtn;
    btnSaveWMF : TBitBtn;
    btnSaveRFM : TBitBtn;
    SaveDialog1 : TSaveDialog;
    SaveDialog2 : TSaveDialog;
    SaveDialog3 : TSaveDialog;
    procedure ckScaleMouseUp(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure ButtonClick(Sender : TObject);
    procedure ButtonDblClick(Sender : TObject);
    procedure CreateForm;
    procedure Edit; override;
    function GetAttributes : TPropertyAttributes; override;
    function GetValue : string; override;
  end;

  TRFMEditor = class(TDefaultEditor)
  public
    procedure EditProperty(PropertyEditor: TPropertyEditor; var Continue, FreeEditor: Boolean); override;
  end;

procedure Register;

implementation
{$IFDEF WIN32}
{$R *.R32}
{$ELSE}
{$R *.R16}
{$ENDIF}

procedure Register;
begin
  RegisterComponents('RSD', [TRFMImage]);
  RegisterPropertyEditor(TypeInfo(TRFM), nil, '', TRFMProperty);
  RegisterComponentEditor(TRFMImage, TRFMEditor);
end;

{*****************************}
{     TRFM Public Methods     }
{*****************************}
constructor TRFM.Create;
begin
  inherited Create;
  Dynarray  := TDynArray.Create(sizeof(RFormRec));
  StrTable  := TStringList.Create;
  Bitmap    := TBitmap.Create;
  Designing := false;
  FillChar(Header, sizeof(RFormHdr), 0);
end;

destructor TRFM.Destroy;
begin
  Bitmap.Free;
  Dynarray.Free;
  StrTable.Free;
  inherited Destroy;
end;

procedure TRFM.Clear;
begin
  StrTable.Clear;
  Dynarray.Clear;
  FillChar(Header, sizeof(RFormHdr), 0);
  ClearBMP;
  if Assigned(FOnChange) then FOnChange(Self); {notify OnChange}
end;

procedure TRFM.Assign(Source: TRFM);
begin
  Clear;
  if not Source.Empty then begin
    Dynarray.Assign(Source.Dynarray);
    StrTable.Assign(Source.StrTable);
    Header := Source.Header;
    end;
  if Assigned(FOnChange) then FOnChange(Self); {notify OnChange}
end;

procedure TRFM.Print;
var R : TRect;
begin
  Printer.BeginDoc;
  R:= Rect(0,0,Printer.PageWidth, Printer.PageHeight);
  DrawTo(Printer.Canvas,R);
  Printer.EndDoc;
end;

procedure TRFM.Loadfromfile(filename : String);
var
  MS : TMemoryStream;
  tmprec : RFormRec;
  x : smallint;
begin
  try
    MS:= TMemoryStream.Create;
  if fileexists(filename) then begin
    MS.loadfromfile(filename);
    MS.Seek(0,0);
    if MS.Read(Header, sizeof(RFormHdr)) = sizeof(RFormHdr) then begin
      if Header.VersionMajor <= 2 then {check version}
        if Header.VersionMinor < 1 then begin
        case Header.PageSize of
          psLetter: begin Header.Width:= 800;  Header.Height:= 1035; end;
          psLegal:  begin Header.Width:= 800;  Header.Height:= 1320; end;
          psA3:     begin Header.Width:= 1100; Header.Height:= 1556; end;
          psA4:     begin Header.Width:= 778;  Header.Height:= 1095; end;
          psA5:     begin Header.Width:= 551;  Header.Height:= 778;  end;
          else      begin Header.Width:= 800;  Header.Height:= 1035; end;
          end;
        Header.VersionMajor := 2;
        Header.VersionMinor := 5;
        Header.Orientation:= orPortrait;
        end;
      DynArray.Clear;
      for x:= 0 to Header.NumRecs - 1 do
        if MS.Read(tmprec, sizeof (RFormRec)) = sizeof (RFormRec) then
           Dynarray.Add( tmprec );
      StrTable.LoadFromStream(MS);
      end;
    if Assigned(FOnChange) then FOnChange(Self); {notify OnChange}
    end;
  finally
    MS.Free;
  end;
end;

procedure TRFM.SavetoFile(filename : String);
var
  x : smallint;
  MS : TMemoryStream;
begin
  try
    MS:= TMemoryStream.Create;
  if DynArray.Count > 0 then begin
    Header.NumRecs:= DynArray.Count;
    MS.Write(Header, sizeof(RFormHdr));
    for x:= 0 to DynArray.Count - 1 do
      MS.Write( DynArray[x]^, sizeof(RFormRec) );
    StrTable.SavetoStream(MS);
    MS.SavetoFile(filename);
    end;
  finally
    MS.Free;
  end;
end;

procedure TRFM.LoadfromStream(Stream : TStream);
var
  tmprec : RFormRec;
  x : smallint;
begin
  if Stream.Read(Header, sizeof(RFormHdr)) = sizeof(RFormHdr) then begin
    if Header.VersionMajor <= 2 then {check version}
      if Header.VersionMinor < 1 then begin
        case Header.PageSize of
          psLetter: begin Header.Width:= 800;  Header.Height:= 1035; end;
          psLegal:  begin Header.Width:= 800;  Header.Height:= 1320; end;
          psA3:     begin Header.Width:= 1100; Header.Height:= 1556; end;
          psA4:     begin Header.Width:= 778;  Header.Height:= 1095; end;
          psA5:     begin Header.Width:= 551;  Header.Height:= 778;  end;
          else      begin Header.Width:= 800;  Header.Height:= 1035; end;
          end;
        Header.VersionMajor := 2;
        Header.VersionMinor := 5;
        Header.Orientation:= orPortrait;
        end;
    DynArray.Clear;
    for x:= 0 to Header.NumRecs - 1 do
      if Stream.Read(tmprec, sizeof (RFormRec)) = sizeof (RFormRec) then
         Dynarray.Add( tmprec );
    StrTable.LoadFromStream(Stream);
    if Assigned(FOnChange) then FOnChange(Self); {notify OnChange}
    end;
end;

procedure TRFM.SavetoStream(Stream : TStream);
var x : smallint;
begin
  if DynArray.Count > 0 then begin
    Header.NumRecs:= DynArray.Count;
    Stream.Write(Header, sizeof(RFormHdr));
    for x:= 0 to DynArray.Count - 1 do
      Stream.Write( DynArray[x]^, sizeof(RFormRec) );
    StrTable.SavetoStream(Stream);
    end;
end;

procedure TRFM.SaveAsBitmap(filename : String);
var E : boolean;
begin
  E:= Bitmap.Empty;
  if E then ScaleDraw(MakeRect);
  Bitmap.Savetofile(filename);
  if E then ClearBMP;
end;

procedure TRFM.SaveAsMetafile(filename : String);
var
  FMeta   : TMetaFile;
  {$IFDEF WIN32}
  FCanvas : TMetaFileCanvas;
  {$ELSE}
  FCanvas : TCanvas;
  {$ENDIF}
begin
  FMeta:= TMetafile.Create;
{$IFDEF WIN32}
  FCanvas:= TMetafileCanvas.Create(FMeta,0);
{$ELSE}
  FCanvas:= TCanvas.Create;
  FCanvas.Handle:= CreateMetaFile(NIL);
  SetMapMode(FCanvas.Handle, MM_ANISOTROPIC);
  SetWindowOrg(FCanvas.Handle, 0, 0);
  SetWindowExt(FCanvas.Handle,longint(width) * inch div Screen.PixelsPerInch,
               longint(height) * inch div Screen.PixelsPerInch);
  FCanvas.Font.PixelsPerInch:= inch;
{$ENDIF}

  DrawTo(FCanvas, Rect(0,0,Header.Width, Header.Height));

{$IFDEF WIN32}
  FCanvas.Free;
{$ELSE}
  FMeta.Handle := CloseMetafile(FCanvas.Handle);
{$ENDIF}
  FMeta.Height:= Header.Height;
  FMeta.Width:= Header.Width;
  FMeta.Savetofile(filename);
end;

procedure TRFM.Draw;
var R : TRect;
begin
  R:= MakeRect;
  ClearBMP;
  Bitmap.Free;
  Bitmap   := TBitmap.Create;
  Bitmap.Width:= R.Right - R.Left;
  Bitmap.Height:= R.Bottom - R.Top;
  DrawTo(Bitmap.Canvas, R);
end;

procedure TRFM.ScaleDraw(MyRect : TRect);
var R : TRect;
begin
  ClearBMP;
  Bitmap.Free;
  Bitmap   := TBitmap.Create;
  Bitmap.Width:= MyRect.Right - MyRect.Left;
  Bitmap.Height:= MyRect.Bottom - MyRect.Top;
  R:= Rect(0,0,Bitmap.Width, Bitmap.Height);
  DrawTo(Bitmap.Canvas, R);
end;

procedure TRFM.DrawTo(MyCanvas : TCanvas; MyRect : TRect);
var
  x: smallint;
  VScale, HScale   : real;
begin
  if Empty then Exit;

  HScale:= (myRect.Right - myRect.Left) / Header.Width;
  VScale:= (myRect.Bottom - myRect.Top) / Header.Height;

  with MyCanvas do begin
    Font.Name:= 'Arial';
    Font.Size:= 10;
    Font.Style:= [];
    Font.Color:= clBlack;
    Font.PixelsPerInch:= round(INCH * HScale);
    Brush.Style := bsSolid;
    Brush.Color := clWhite;
    Pen.Color:= clBlack;
    Pen.Style:= psClear;
    Pen.Width:= round(Header.Thick * HScale);
    if Header.PStyle = psSolid then Pen.style:=psSolid;
    if Header.PStyle = psDash  then Pen.style:=psDash;
    if Header.PStyle = psDot   then Pen.style:=psDot;
    if Header.PStyle = psClear then Pen.style:=psClear;
    {draw border}
    FillRect( MyRect );
    end;

  for x:= 0 to Dynarray.Count -1 do DrawObj(MyCanvas, MyRect, x);
end;

{*****************************}
{    TRFM Private Methods     }
{*****************************}
procedure TRFM.DefineProperties(Filer : TFiler);
begin
  inherited DefineProperties(Filer);
  Filer.DefineBinaryProperty('RFMData', LoadFromStream, SaveToStream, True);
end;

function TRFM.GetEmpty : boolean;
begin
  Result:= Dynarray.Count = 0;
end;

function TRFM.GetWidth : smallint;
begin
  Result:= Header.Width;
end;

function TRFM.GetHeight : smallint;
begin
  Result:= Header.Height;
end;

function TRFM.GetCanvas : TCanvas;
begin
  Result:= Bitmap.Canvas;
end;

procedure TRFM.ClearBMP;
begin
  Bitmap.Free;
  Bitmap:= TBitmap.Create;
end;

function TRFM.MakeRect : TRect;
begin
  result:= Rect(0,0,Header.Width, Header.Height);
end;

procedure TRFM.DrawObj(MyCanvas : TCanvas; MyRect : TRect; Num : smallint);
var
  FormRec          : RFormRec;
  LogFont          : TLogFont;
  VScale, HScale   : real;
  TX, FLeft, FTop,
  FRight, FBottom,
  FHCent, FVCent   : smallint;
  tmpb             : byte;
begin
  FormRec:= RFormRec(DynArray[Num]^);

  HScale:= (myRect.Right - myRect.Left) / Header.Width;
  VScale:= (myRect.Bottom - myRect.Top) / Header.Height;

  FLeft:=   round(FormRec.Left * HScale) + myRect.Left;
  FTop :=   round(FormRec.Top * VScale) + myRect.Top;
  FRight:=  round((FormRec.Width + FormRec.Left) * HScale) + myRect.Left;
  FBottom:= round((FormRec.Height + FormRec.Top) * VScale) + myRect.Top;
  FVCent:=  round((FRight - FLeft) / 2)+ FLeft;
  FHCent:=  round((FBottom - FTop) / 2)+ FTop;

  with MyCanvas do begin
    Brush.Color := clWhite;
    Brush.Style := bsClear;
    Pen.Color   := clBlack;
    Pen.Width   := 1;
    case FormRec.DrawType of
      rfText :  {Text}
        begin
        Font.Name:= GetString(RFormText( FormRec ).FontIdx);
        Font.Size:= RFormText( FormRec ).FontSize;
        Font.Style:= RFormText( FormRec ).FontStyle;
        Font.Color:= BytetoColor( RFormText( FormRec ).Color );
      {check orientation}
        if RFormText( FormRec ).Vert then begin
          TX:= RFormText( FormRec ).Width;
          GetObject(Font.Handle, sizeof(TLogFont), @LogFont);
          LogFont.lfEscapement := 2700;  {270 degrees}
          Font.Handle:= CreateFontIndirect(LogFont);
          end
        else begin
          TX:= 0;
          GetObject(Font.Handle, sizeof(TLogFont), @LogFont);
          LogFont.lfEscapement := 0;     {0 degrees}
          Font.Handle:= CreateFontIndirect(LogFont);
          end;
        TextOut( FLeft, FTop, GetString(RFormText( FormRec ).StrIdx) );
        end;

      rfField :  {Field Label}
        begin
        if Designing then begin
          Font.Name:= GetString(RFormText( FormRec ).FontIdx);
          Font.Size:= RFormText( FormRec ).FontSize;
          Font.Style:= RFormText( FormRec ).FontStyle;
          Font.Color:= BytetoColor( RFormText( FormRec ).Color );
          TextOut( FLeft, FTop, GetString(RFormText( FormRec ).StrIdx) );
          end;
        end;

      rfFrame:  {Frame}
        begin
        Brush.color      := BytetoColor( RFormFrame(FormRec).Color  );
        Pen.color        := BytetoColor( RFormFrame(FormRec).LColor );
        tmpb:= round(RFormFrame(FormRec).Thick * HScale);
        Pen.width := tmpb;
        case RFormFrame(FormRec).PStyle of
          psSolid:    Pen.style:=psSolid;
          psDash:     Pen.style:=psDash;
          psDot:      Pen.style:=psDot;
          psClear:    Pen.style:=psClear;
          end;
        case RFormFrame(FormRec).FStyle of
          fsClear:    Brush.style:= bsClear;
          fsSolid:    Brush.style:= bsSolid;
          fsDiagonal: Brush.style:= bsFDiagonal;
          fsHash:     Brush.style:= bsDiagCross;
          end;
        Rectangle(FLeft, FTop, FRight, FBottom);
        end;

      rfLine  :  {Line}
        begin
        case RFormLine(FormRec).PStyle of
          psSolid:    Pen.style:=psSolid;
          psDash:     Pen.style:=psDash;
          psDot:      Pen.style:=psDot;
          psClear:    Pen.style:=psClear;
          end;
        Pen.color := BytetoColor( RFormFrame(FormRec).Color );
        tmpb:= round(RFormLine(FormRec).Thick * HScale);
        Pen.width := tmpb;
        case RFormLine( FormRec ).LType of
          ltHorz: begin
            MoveTo(FLeft, FHCent);
            LineTo(FRight,FHCent);
            end;
          ltVert: begin
            MoveTo(FVCent, FTop);
            LineTo(FVCent, FBottom);
            end;
          ltDiagL: begin
            MoveTo(FLeft, FTop);
            LineTo(FRight, FBottom);
            end;
          ltDiagR: begin
            MoveTo(FRight, FTop);
            LineTo(FLeft, FBottom);
            end;
          end;
        end;

      rfCkBox :  {Checkbox}
        begin
        Pen.style   := psSolid;
        Pen.color   := BytetoColor( RFormCkBox(FormRec).Color );
        Pen.width   := round(RFormCkBox(FormRec).Thick * HScale);
        Rectangle(FLeft, FTop, FRight, FBottom);
        end;
      end; {case}
    end; {with canvas}
end;

function TRFM.GetString( Index : smallint ) : String;
begin
  if (Index < StrTable.Count) and (Index > -1) then Result:= StrTable[Index]
  else Result:= 'ERROR '+inttostr(Index);
end;

function TRFM.ByteToColor (AByte : byte) : TColor;
begin
  case Abyte of
    0 : result:= clBlack;
    1 : result:= clMaroon;
    2 : result:= clGreen;
    3 : result:= clOlive;
    4 : result:= clNavy;
    5 : result:= clPurple;
    6 : result:= clTeal;
    7 : result:= clGray;
    8 : result:= clSilver;
    9 : result:= clRed;
   10 : result:= clLime;
   11 : result:= clYellow;
   12 : result:= clBlue;
   13 : result:= clFuchsia;
   14 : result:= clAqua;
   15 : result:= clWhite;
   16 : result:= $00AAFFFF; {pale yellow}
   else result:= clBlack; {default}
   end;
end;

{*****************************}
{     TRFMImage Component     }
{*****************************}
constructor TRFMImage.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FFillList:= TStringList.Create;
  FFillList.Add('TLabel');
  FFillList.Add('TDBLabel');
  FFillList.Add('TRFormLab');
  FFillList.Add('TRFormChk');
  FLogoList:= TStringList.Create;
  FLogoList.Add('TImage');
  FLogoList.Add('TDBImage');
  FLogoList.Add('TRFLogo');
  FRFM:= TRFM.Create;
  Height:= 105;
  Width:= 105;
  Aspect:= true;
  AutoFill:= False;
  AutoLogo:= False;
  AutoFields:= false;
  AutoSize:= false;
  Center:= false;
  Stretch:= false;
  if (csDesigning in ComponentState) then RFM.Designing:= true
  else RFM.Designing:= false;
  RFM.OnChange:= RFMChanged;
end;

destructor TRFMImage.Destroy;
begin
  FRFM.Free;
  FFillList.Free;
  FLogoList.Free;
  inherited Destroy;
end;

function TRFMImage.CalcRect : TRect;
var  H,W : smallint;
begin
  {default}
  Result := ClientRect;
  {stretch}
  if Stretch then Result := ClientRect
  {center}
  else if Center then
    Result := Bounds((Width - RFM.Width) div 2, (Height - RFM.Height) div 2, RFM.Width, RFM.Height)
    else result := Rect(0, 0, RFM.Width, RFM.Height);
  {Aspect Ratio}
  if Aspect and Stretch then begin
    H:= Height;
    W:= round((H / RFM.Height) * RFM.Width);
    if not Center then result:= Rect(0, 0, W, H)
    else Result:= Bounds((Width - W) div 2, 0, W, H);
    end;
end;

function TRFMImage.RFMRect : TRect;
begin
  Result:= Rect(0,0,RFM.Header.Width, RFM.Header.Height);
end;

procedure TRFMImage.Print;
var R : TRect;
    x: smallint;
    VScale, HScale: real;
begin
  SetPaperSize;
  Printer.BeginDoc;
  if not FRFM.Empty then begin
    HScale:= Printer.PageWidth / FRFM.Header.Width;
    VScale:= Printer.PageHeight / FRFM.Header.Height;
    with Printer.Canvas do begin
      Font.Name:= 'Arial';
      Font.Size:= 10;
      Font.Style:= [];
      Font.Color:= clBlack;
      Font.PixelsPerInch:= round(INCH * HScale);
      end;
    end;
  R:= Rect(0,0,Printer.PageWidth, Printer.PageHeight);
  RFM.DrawTo(Printer.Canvas,R);
  if AutoLogo then LogoTo(Printer.Canvas, R);
  if AutoFill then FillTo(Printer.Canvas, R);
  if AutoFields then FieldsTo(Printer.Canvas, R);
  Printer.EndDoc;
end;

procedure TRFMImage.PrintTo(ACanvas : TCanvas; MyRect : TRect);
var x: smallint;
    VScale, HScale: real;
begin
  RFM.DrawTo(ACanvas, MyRect);
  if AutoFill then FillTo(ACanvas, MyRect);
  if AutoLogo then LogoTo(ACanvas, MyRect);
end;

procedure TRFMImage.SetPaperSize;
var
   aDevice, aDriver, aPort : Array[0..255] of Char;
   aHandle : THandle;
   Paper, Orient : smallint;
begin
  case RFM.Header.PageSize of
    psLetter: Paper:= DMPAPER_LETTER;
    psLegal:  Paper:= DMPAPER_LEGAL;
    psA3:     Paper:= DMPAPER_A3;
    psA4:     Paper:= DMPAPER_A4;
    psA5:     Paper:= DMPAPER_A5;
    else      Paper:= DMPAPER_LETTER;
    end;

   if RFM.Header.Orientation = orLandscape then
     Orient:= DMORIENT_LANDSCAPE
   else Orient:= DMORIENT_PORTRAIT;

{$IFDEF WIN32}
  if RFM.Header.Orientation = orLandscape then
    Printer.Orientation:= poLandscape
  else Printer.Orientation:= poPortrait;
{$ELSE}
  Printer.GetPrinter (aDevice, aDriver, aPort, aHandle );
  Printer.PrinterIndex := Printer.PrinterIndex;
  if aHandle <> 0 then
    with pDevMode( Ptr(aHandle,0) )^ do begin
      dmPaperSize:= Paper;
      dmOrientation:= Orient;
      dmFields:= dmFields or DM_PAPERSIZE or DM_ORIENTATION;
      end;
{$ENDIF}
end;

procedure TRFMImage.LogoTo(ACanvas : TCanvas; MyRect : TRect);
var
  VScale, HScale   : real;
  PForm    : TForm;
  myimage  : TImage;
  i,j      : smallint;
  X,Y,W,H  : Longint;
  sx,sy    : smallint;
begin
  HScale:= (myRect.Right - myRect.Left) / RFM.Header.Width;
  VScale:= (myRect.Bottom - myRect.Top) / RFM.Header.Height;

  PForm:= (Owner as TForm);
  sx:= GetScrollPos(Parent.Handle, SB_HORZ);
  sy:= GetScrollPos(Parent.Handle, SB_VERT);
  for i:= 0 to PForm.ComponentCount - 1 do
    for j:= 0 to FLogoList.Count-1 do
      if CompareText(PForm.components[i].ClassName, FLogoList[j]) = 0 then begin
        myimage:= (PForm.components[i] as TImage);
        W:= trunc((myImage.width)*HScale);
        H:= trunc((myImage.height)*VScale);
        X:= trunc((myImage.left+sx)*HScale);
        Y:= trunc((myImage.top+sy)*VScale);
        if (X > MyRect.Right) or (Y > MyRect.Bottom) then continue;
        if not myimage.visible then continue;
        if not MyImage.Picture.Bitmap.Empty then
          ACanvas.StretchDraw(Rect(X, Y, W+X, H+Y), MyImage.Picture.Bitmap)
        else if not MyImage.Picture.Metafile.Empty then
          ACanvas.StretchDraw(Rect(X, Y, W+X, H+Y), MyImage.Picture.Metafile)
        end;
end;

procedure TRFMImage.FillTo(ACanvas : TCanvas; MyRect : TRect);
var
  VScale, HScale   : real;
  PForm    : TForm;
  mylabel  : TLabel;
  i,j      : smallint;
  X,Y      : Longint;
  sx,sy    : smallint;
begin
  HScale:= (myRect.Right - myRect.Left) / RFM.Header.Width;
  VScale:= (myRect.Bottom - myRect.Top) / RFM.Header.Height;

  ACanvas.Font.PixelsPerInch:= round(INCH * HScale);
  ACanvas.Brush.Style := bsClear;
  ACanvas.Pen.Color:= clBlack;
  ACanvas.Pen.Style:= psSolid;
  ACanvas.Pen.Width:= 1;

  PForm:= (Owner as TForm);
  sx:= GetScrollPos(Parent.Handle, SB_HORZ);
  sy:= GetScrollPos(Parent.Handle, SB_VERT);
  for i:= 0 to PForm.ComponentCount - 1 do
    for j:= 0 to FFillList.Count-1 do begin
      {$IFDEF FLY}
      if (PForm.components[i] is TFlyControl) then begin
        if (PForm.components[i] as TFlyControl).DisplayType = dtText then begin
          ACanvas.Font.Assign( (PForm.components[i] as TFlyControl).Font );
          ACanvas.Font.PixelsPerInch:= 96;
          X:= trunc( (PForm.components[i] as TFlyControl).left );
          Y:= trunc( (PForm.components[i] as TFlyControl).top );
          ACanvas.TextOut( X, Y, (PForm.components[i] as TFlyControl).text);
          end;
        end;
      {$ENDIF}
      if CompareText(PForm.components[i].ClassName, FFillList[j]) = 0 then begin
        mylabel:= (PForm.components[i] as TLabel);
        X:= trunc((mylabel.left+1+sx)*HScale);
        Y:= trunc((mylabel.top+2+sy)*VScale);
        if (X > MyRect.Right) or (Y > MyRect.Bottom) then continue;
        if not mylabel.visible then continue;
        ACanvas.Font.Assign(mylabel.Font);
        ACanvas.TextOut(X, Y, mylabel.caption);
        end;
      end;
end;

procedure TRFMImage.FieldsTo(ACanvas : TCanvas; MyRect : TRect);
var
  VScale, HScale   : real;
  i        : smallint;
  X,Y      : Longint;
  tmpstr   : string;
  tmprec   : RFormField;
begin
  HScale:= (myRect.Right - myRect.Left) / RFM.Header.Width;
  VScale:= (myRect.Bottom - myRect.Top) / RFM.Header.Height;

  ACanvas.Font.PixelsPerInch:= round(INCH * HScale);
  ACanvas.Brush.Style := bsClear;
  ACanvas.Pen.Color:= clBlack;
  ACanvas.Pen.Style:= psSolid;
  ACanvas.Pen.Width:= 1;

  for i:= 0 to FRFM.Dynarray.Count - 1 do begin
    if RFormRec(FRFM.Dynarray[i]^).DrawType <> rfField then continue;
    tmprec:= RFormField(FRFM.Dynarray[i]^);
    X:= trunc(tmprec.Left * HScale);
    Y:= trunc(tmprec.Top  * VScale);
    tmpstr:= '';
    if Assigned(FOnPrintField) then
      FOnPrintField( nil, FRFM.GetString(tmprec.StrIdx), TTextFormat(tmprec.Format), tmpstr);
    ACanvas.Font.Name:= FRFM.GetString(tmprec.FontIdx);
    ACanvas.Font.Style:= tmprec.fontstyle;
    ACanvas.Font.Size:= tmprec.fontsize;
    ACanvas.Font.Color:= tmprec.color;
    ACanvas.TextOut(X, Y, tmpstr);
    end;

end;

procedure TRFMImage.Paint;
begin
  {Autosize}
  if AutoSize and ((Align = alNone) or (Align = alLeft) or (Align = alTop)) then
    if (RFM.Width > 0) and (RFM.Height > 0) then begin
      Width:= RFM.Width;
      Height:= RFM.Height;
      end;
  {draw dashes at design time}
  if csDesigning in ComponentState then
    with inherited Canvas do begin
      Pen.Style := psDash;
      Brush.Style := bsClear;
      Rectangle(0, 0, Width, Height);
      end;
  {copy RFM bitmap to image canvas}
  if not RFM.Bitmap.Empty then
    inherited Canvas.Draw(CalcRect.Left, CalcRect.Top, RFM.Bitmap);
end;

procedure TRFMImage.Clear;
begin
  FRFM.Clear;
end;

procedure TRFMImage.UpdateRFM;
begin
  if FRFM.Empty then Exit;
  FRFM.ScaleDraw(CalcRect);
  Invalidate;
end;

procedure TRFMImage.SetAutoSize(AValue: Boolean);
begin
  if FAutoSize = AValue then Exit;
  FAutoSize := AValue;
  UpdateRFM;
end;

procedure TRFMImage.SetCenter(AValue: Boolean);
begin
  if FCenter = AValue then Exit;
  FCenter := AValue;
  UpdateRFM;
end;

procedure TRFMImage.SetStretch(AValue: Boolean);
begin
  if FStretch = AValue then Exit;
  FStretch := AValue;
  UpdateRFM;
end;

procedure TRFMImage.SetAspect(AValue: Boolean);
begin
  if FAspect = AValue then Exit;
  FAspect := AValue;
  UpdateRFM;
end;

procedure TRFMImage.SetRFM(AValue: TRFM);
begin
  FRFM.Assign(AValue);
  UpdateRFM;
end;

procedure TRFMImage.SetFillList(AValue: TStrings);
begin
  FFillList.Assign(AValue);
end;

procedure TRFMImage.SetLogoList(AValue: TStrings);
begin
  FLogoList.Assign(AValue);
end;

function TRFMImage.GetCanvas : TCanvas;
begin
  Result:= Canvas;
end;

procedure TRFMImage.RFMChanged(Sender : TObject);
begin
  if Assigned(FOnChange) then FOnChange(Self);
  UpdateRFM;
end;

{************************}
{  Double Click Editor   }
{************************}
procedure TRFMEditor.EditProperty(PropertyEditor: TPropertyEditor; var Continue, FreeEditor: Boolean);
var PropName: string;
begin
  PropName := PropertyEditor.GetName;
  if CompareText(PropName, 'RFM') <> 0 then Exit;
  PropertyEditor.Edit;
  Continue := False;
end;

{**************************}
{   TRFM Property Editor   }
{**************************}
function TRFMProperty.GetAttributes : TPropertyAttributes;
begin
  Result := [ paDialog ];
end;

function TRFMProperty.GetValue : string;
begin
  Result := '(RealForm Image)';
end;

procedure TRFMProperty.Edit;
var myRFM : TRFM;
begin
  myRFM := TRFM(GetOrdValue);
  try
    CreateForm;
    RFMImage1.RFM.Assign(myRFM);
    Dialog.ShowModal;
    if Dialog.Modalresult = mrOk then begin
      myRFM.Assign(RFMImage1.RFM);
      Designer.Modified;
      end;
  finally
    Dialog.Free;
  end;
end;

procedure TRFMProperty.ButtonClick(Sender: TObject);
begin
  if Sender = btnClear then RFMImage1.RFM.Clear;
  if Sender = btnSaveBMP then
    if SaveDialog1.Execute then RFMImage1.RFM.SaveAsBitmap(SaveDialog1.Filename);
  if Sender = btnSaveWMF then
    if SaveDialog2.Execute then RFMImage1.RFM.SaveAsMetafile(SaveDialog2.Filename);
  if Sender = btnSaveRFM then
    if SaveDialog3.Execute then RFMImage1.RFM.SaveToFile(SaveDialog3.Filename);
  if Sender = FileListBox1 then
    if FileExists(FileListBox1.FileName) then
      RFMImage1.RFM.Loadfromfile(FileListBox1.FileName);
end;

procedure TRFMProperty.ButtonDblClick(Sender: TObject);
begin
  if Sender = FileListBox1 then
    if FileExists(FileListBox1.FileName) then btnOk.Click;
end;

procedure TRFMProperty.ckScaleMouseUp(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
begin
  if ckScale.checked then begin
    RFMImage1.Stretch:= true;
    RFMImage1.AutoSize:= false;
    RFMImage1.Width:= 175;
    RFMImage1.Height:= 147;
    RFMImage1.Center := True;
    end
  else begin
    RFMImage1.Stretch:= false;
    RFMImage1.AutoSize:= true;
    end;
  RFMImage1.UpdateRFM;
end;

procedure TRFMProperty.CreateForm;
begin
  Dialog := TForm.CreateNew(Application);
  Label1 := TLabel.create(Dialog);
  DirectoryListBox1 := TDirectoryListBox.create(Dialog);
  FileListBox1 := TFileListBox.create(Dialog);
  DriveComboBox1 := TDriveComboBox.create(Dialog);
  Edit1 := TEdit.create(Dialog);
  btnCancel := TBitBtn.create(Dialog);
  btnOK := TBitBtn.create(Dialog);
  btnClear := TBitBtn.create(Dialog);
  ScrollBox1 := TScrollBox.create(Dialog);
  RFMImage1 := TRFMImage.create(Dialog);
  ckScale := TCheckBox.create(Dialog);
  btnSaveBMP := TBitBtn.create(Dialog);
  btnSaveWMF := TBitBtn.create(Dialog);
  btnSaveRFM := TBitBtn.create(Dialog);
  SaveDialog1 := TSaveDialog.create(Dialog);
  SaveDialog2 := TSaveDialog.create(Dialog);
  SaveDialog3 := TSaveDialog.create(Dialog);
  with Dialog do begin
    Left := 314;
    Top := 133;
    BorderIcons := [biSystemMenu];
    BorderStyle := bsDialog;
    Caption := 'Open RFM';
    ClientHeight := 291;
    ClientWidth := 307;
    Font.Color := clBlack;
    Font.Height := -11;
    Font.Name := 'MS Sans Serif';
    Font.Style := [];
    PixelsPerInch := 96;
    Position := poScreenCenter;
    end;
  with Label1 do begin
    Parent:= Dialog;
    Name:= 'Label1';
    Left := 94;
    Top := 274;
    Width := 181;
    Height := 13;
    Caption := 'Copyright 1997 RealSoft Development';
    Font.Color := clNavy;
    Font.Height := -11;
    Font.Name := 'MS Sans Serif';
    Font.Style := [];
    ParentFont := False;
    end;
  with Edit1 do begin
    Parent:= Dialog;
    Name:= 'Edit1';
    Left := 160;
    Top := 8;
    Width := 145;
    Height := 20;
    TabOrder := 3;
    Text := '*.RFM';
    end;
  with FileListBox1 do begin
    Parent:= Dialog;
    Name:= 'FileListBox1';
    Left := 160;
    Top := 32;
    Width := 145;
    Height := 81;
    FileEdit := Edit1;
    ItemHeight := 13;
    Mask := '*.RFM';
    TabOrder := 1;
    OnClick := ButtonClick;
    OnDblClick := ButtonDblClick;
    end;
  with DirectoryListBox1 do begin
    Parent:= Dialog;
    Name:= 'DirectoryListBox1';
    Left := 8;
    Top := 32;
    Width := 145;
    Height := 81;
    FileList := FileListBox1;
    ItemHeight := 16;
    TabOrder := 0;
    end;
  with DriveComboBox1 do begin
    Parent:= Dialog;
    Name:= 'DriveComboBox1';
    Left := 8;
    Top := 8;
    Width := 145;
    Height := 19;
    DirList := DirectoryListBox1;
    TabOrder := 2;
    end;
  with btnCancel do begin
    Parent:= Dialog;
    Name:= 'btnCancel';
    Caption:= 'Cancel';
    Left := 201;
    Top := 244;
    Width := 95;
    Height := 25;
    TabOrder := 4;
    Kind := bkCustom;
    ModalResult:= mrCancel;
    end;
  with btnOK do begin
    Parent:= Dialog;
    Name:= 'btnOK';
    Caption:= 'OK';
    Left := 201;
    Top := 220;
    Width := 95;
    Height := 25;
    TabOrder := 5;
    Kind := bkCustom;
    ModalResult:= mrOK;
    end;
  with btnClear do begin
    Parent:= Dialog;
    Name:= 'btnClear';
    Left := 201;
    Top := 119;
    Width := 95;
    Height := 25;
    Caption := 'Clear Image';
    TabOrder := 6;
    OnClick := ButtonClick;
    end;
  with ScrollBox1 do begin
    Parent:= Dialog;
    Name:= 'ScrollBox1';
    Left := 8;
    Top := 120;
    Width := 177;
    Height := 149;
    TabOrder := 7;
    end;
  with RFMImage1 do begin
    Parent:= ScrollBox1;
    Name:= 'RFMImage1';
    Left := 0;
    Top := 0;
    Width := 175;
    Height := 147;
    Center := True;
    Stretch := True;
    end;
  with ckScale do begin
    Parent:= Dialog;
    Name:= 'ckScale';
    Left := 8;
    Top := 273;
    Width := 76;
    Height := 17;
    Caption := 'Auto Scale';
    State := cbChecked;
    TabOrder := 8;
    OnMouseUp := ckScaleMouseUp;
    end;
  with btnSaveBMP do begin
    Parent:= Dialog;
    Name:= 'btnSaveBMP';
    Left := 201;
    Top := 143;
    Width := 95;
    Height := 25;
    Caption := 'Save as Bitmap';
    TabOrder := 9;
    OnClick := ButtonClick;
    end;
  with btnSaveWMF do begin
    Parent:= Dialog;
    Name:= 'btnSaveWMF';
    Left := 201;
    Top := 167;
    Width := 95;
    Height := 25;
    Caption := 'Save as Metafile';
    TabOrder := 10;
    OnClick := ButtonClick;
    end;
  with btnSaveRFM do begin
    Parent:= Dialog;
    Name:= 'btnSaveRFM';
    Left := 201;
    Top := 191;
    Width := 95;
    Height := 25;
    Caption := 'Save as RFM';
    TabOrder := 11;
    OnClick := ButtonClick;
    end;
  with SaveDialog1 do begin
    Name:= 'SaveDialog1';
    DefaultExt := 'BMP';
    Filter := 'Bitmap Files|*.BMP';
    end;
  with SaveDialog2 do begin
    Name:= 'SaveDialog2';
    Filter := 'Metafiles|*.WMF';
    end;
  with SaveDialog3 do begin
    Name:= 'SaveDialog3';
    Filter := 'Realform Images|*.RFM';
    end;
end;


end.
