/*
 * This file is part of PB-Lib v3.0 C++ Programming Library
 *
 * Copyright (c) 1995, 1997 by Branislav L. Slantchev
 * A fine product of Silicon Creations, Inc. (gargoyle)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the License which accompanies this
 * software. This library is distributed in the hope that it will
 * be useful, but without any warranty; without even the implied
 * warranty of merchantability or fitness for a particular purpose.
 *
 * You should have received a copy of the License along with this
 * library, in the file LICENSE.DOC; if not, write to the address
 * below to receive a copy via electronic mail.
 *
 * You can reach Branislav L. Slantchev (Silicon Creations, Inc.)
 * at bslantch@cs.angelo.edu. The file SUPPORT.DOC has the current
 * telephone numbers and the postal address for contacts.
*/

#include "parsevar.h"
#include "parsedef.h"
#include "stdmac.h"
#include "terminal.h"
#include "utils.h"
#include "pblsdk.h"

#ifndef PB_SDK
	#include <conio.h>
	#include <dos.h>
	#include <string.h>
	#include <stdlib.h>
	#include "emulator.h"
	#include "compiler.h"
#endif

/*
 * this table contains pre-calculated hash values for the macro text strings
 * (upper-case). they are arranged in sorted order by hash value so bsearch()
 * can be used. don't forget to strupr() the text before calculating the hash
*/
static int nTableSize = 49;
struct HashMacroValues
{
	ulong hash;
	int   pdef;
}
	HashedTokens[] =
{
	{ 0x000004D4UL, PDEF_USR_ID         }, // ID
//	{ 0x00004813UL, PDEF_ACTION_CLRSCR  }, // CLS
	{ 0x00046694UL, PDEF_SYS_BAUD       }, // BAUD
//	{ 0x000469A0UL, PDEF_ACTION_BEEP    }, // BEEP
	{ 0x00047E99UL, PDEF_USR_CITY       }, // CITY
	{ 0x00048685UL, PDEF_SYS_DATE       }, // DATE
	{ 0x00052615UL, PDEF_USR_NAME       }, // NAME
	{ 0x00053385UL, PDEF_SYS_NODE       }, // NODE
	{ 0x00055474UL, PDEF_SYS_PORT       }, // PORT
	{ 0x00058E15UL, PDEF_SYS_TIME       }, // TIME
//	{ 0x0005B5E4UL, PDEF_ACTION_WAIT    }, // WAIT
	{ 0x0050AA9CUL, PDEF_USR_LEVEL      }, // LEVEL
	{ 0x005308E2UL, PDEF_SYS_NODELIST   }, // NLDIR
//	{ 0x00546A75UL, PDEF_ACTION_PAUSE   }, // PAUSE
	{ 0x005A48E2UL, PDEF_SYS_UPLOAD     }, // UPDIR
	{ 0x0161E463UL, PDEF_USR_FGROUPNO   }, // CURFILEGROUP#
	{ 0x024309B5UL, PDEF_USR_TONLINE    }, // TMONLINE
	{ 0x027FECE1UL, PDEF_USR_MAREA      }, // CURMSGAREA
	{ 0x04161E40UL, PDEF_USR_FGROUP     }, // CURFILEGROUP
	{ 0x041C03D3UL, PDEF_USR_FAREANO    }, // CURFILEAREA#
//	{ 0x04816A3CUL, PDEF_ACTION_CLREOL  }, // CLREOL
	{ 0x048617E7UL, PDEF_MSG_HMBTOTAL   }, // TOTALMSG
	{ 0x04C62905UL, PDEF_USR_ALIAS      }, // HANDLE
//	{ 0x04C62CA0UL, PDEF_ACTION_HANGUP  }, // HANGUP
	{ 0x0514C277UL, PDEF_MSG_HMBLOW     }, // LOWMSG
	{ 0x052398E2UL, PDEF_SYS_MENUDIR    }, // MNUDIR
	{ 0x0527B8E2UL, PDEF_MSG_HMBDIR     }, // MSGDIR
	{ 0x053A2277UL, PDEF_MSG_TOTAL      }, // NUMMSG
	{ 0x054AC8E2UL, PDEF_SYS_PEX        }, // PEXDIR
	{ 0x055B88E2UL, PDEF_SYS_PVTUP      }, // PVTDIR
	{ 0x058E78E2UL, PDEF_SYS_SYSTEM     }, // SYSDIR
	{ 0x059209B4UL, PDEF_USR_TLEFT      }, // TMLEFT
	{ 0x059D88E2UL, PDEF_SYS_TEXTFILES  }, // TXTDIR
	{ 0x0602D593UL, PDEF_SYS_TOTALCALLS }, // TOTALCALLS
	{ 0x062C90B5UL, PDEF_USR_LANGUAGE   }, // LANGUAGE
	{ 0x06888385UL, PDEF_USR_LASTDATE   }, // LASTDATE
	{ 0x0688C124UL, PDEF_USR_PASSWD     }, // PASSWORD
	{ 0x06898915UL, PDEF_USR_LASTTIME   }, // LASTTIME
	{ 0x078970F5UL, PDEF_USR_FIRST      }, // FIRSTNAME
	{ 0x07A181F5UL, PDEF_USR_VOICE      }, // VOICEPHONE
	{ 0x07F4D380UL, PDEF_USR_MGROUP     }, // CURMSGGROUP
	{ 0x07FECE13UL, PDEF_USR_MAREANO    }, // CURMSGAREA#
	{ 0x08457EF5UL, PDEF_SYS_SYSOP      }, // SYSOPNAME
	{ 0x084A3939UL, PDEF_USR_COUNTRY    }, // COUNTRY
	{ 0x08649C45UL, PDEF_USR_DATA       }, // DATAPHONE
	{ 0x08678D62UL, PDEF_SYS_STARTUP    }, // STARTDIR
	{ 0x08A71A75UL, PDEF_SYS_MENU       }, // CURMENU
	{ 0x0A2A7F43UL, PDEF_SYS_USERS      }, // NUMUSERS
	{ 0x0A2DA623UL, PDEF_SYS_PAGES      }, // NUMYELLS
	{ 0x0A7A76C3UL, PDEF_USR_RECNO      }, // USERREC
	{ 0x0AA77E6EUL, PDEF_SYS_VERSION    }, // VERSION
	{ 0x0C03BCF2UL, PDEF_USR_FAREADIR   }, // CURFILEAREADIR
	{ 0x0CDBD237UL, PDEF_MSG_LAST       }, // HIGHMSG
	{ 0x0E41C031UL, PDEF_USR_FAREA      }, // CURFILEAREA
	{ 0x0F4D3853UL, PDEF_USR_MGROUPNO   }, // CURMSGGROUP#
};

/*
 * t h e   m a c r o   p a r s i n g   r o u t i n e
 * 
 * this assumes that the macro text is only present, and that the
 * bounding "@<" and ">@" have been stripped, i.e. "MACRO:-35" is ok
 * generally, 'result' should be large enough to accomodate the strings
 * since we have path specification macros, 80 is a good choice (255 - best)
 * this will transform the macro code into a string (or perform the action
 * requested). returns 0 - error, 1 - string, 2 - action
*/
int
parse_macro( char *result, char *source )
{
	int   justify;
	int   token = PDEF_ERROR;
	int   maxLen = 0;
	char *p, buf[256];

#ifdef PB_SDK /* local terminal object */
	zTerminal terminal, *Terminal = &terminal;
	ansi_interp     ansi;
	avatar_interp   avatar;
	proboard_interp proboard;

	terminal.RegisterHandler(&avatar);
	terminal.RegisterHandler(&proboard);
	terminal.RegisterHandler(&ansi);
#endif

	*result = EOS;

	// check for arguments and justification characters
	if( 0 != (p = strchr(source, ':')) )
	{
		switch( p[1] )
		{
			case '-': justify = PDEF_JUST_RIGHT;  maxLen = atoi(p+2); break;
			case '%': justify = PDEF_JUST_CENTER; maxLen = atoi(p+2); break;
			default : justify = PDEF_JUST_LEFT;   maxLen = atoi(p+1);
		}
		*p = EOS; // now cut off the last formatting chars
		// this works because we have cut off the >@ already
	}

	// now parse the source text itself, first check for special
	// characters, also split search alphabetically (a lot faster)
	p = strupr(strcpy(buf,source));
	switch( *p )
	{
		uchar  fore, back;
		char  *envptr;

		case '!': parse_action(PDEF_ACTION_DISPLAY, &p[1]); return 2;
		case '~': parse_action(PDEF_ACTION_PEXEC, &p[1]);   return 2;
		case '$': parse_action(PDEF_ACTION_EXEC, &p[1]);    return 2;
		case ']': parse_action(PDEF_ACTION_PROMPT, &p[1]);  return 2;

		case '#': // colors
			fore = (uchar)((EOS != p[2]) ? p[2] : p[1]);
			back = (uchar)((EOS != p[2]) ? p[1] : '0');
			Terminal->setColor(hexColor(back, fore));
			return True;

		case '%': // you cannot format environment variables
			envptr = getenv(&p[1]);
			if( envptr )
			{
				strcpy(result, envptr);
				token = PDEF_DEAD;
				return 1;
			}
			return 0; // no environment variable found

		case 'B':
			if( 0 == strcmp("BEEP", p) )
			{
				putch(7);
				return 2;
			}
			break;

		case 'P':
			if( 0 == strcmp("PAUSE", p) )
			{
				WaitKeys("\r");
				return 2;
			}
			break;

		case 'W':
			if( 0 == strcmp("WAIT", p) )
			{
				delay(100);
				return 2;
			}
			break;

		case 'H':
			if( 0 == strcmp("HANGUP", p) )
			{
				parse_action(PDEF_ACTION_HANGUP, 0);
				return 2;
			}
			break;

		case 'C':
			if( 0 == strcmp("CLS", p) )
			{
				parse_action(PDEF_ACTION_CLRSCR, 0);
				return 2;
			}
			else if( 0 == strcmp("CLREOL", p) )
			{
				parse_action(PDEF_ACTION_CLREOL, 0);
				return 2;
			}
			break;
	}

	// if we got here, then we need to look at the table
	ulong hashedValue = hash(buf);
	for( int i = 0; i < nTableSize; ++i )
	{
		if( HashedTokens[i].hash == hashedValue )
		{
			token = HashedTokens[i].pdef;
			break;
		}
	}

	// see if we found the source code for the string
	// (note that all action sequences return immediately
	// and never get here). either token is not error or
	// we could not match the requested text with anything
	switch( token )
	{
		case PDEF_ERROR: return 0;
		case PDEF_DEAD : return 2;
		default: if( parse_token(result, token, justify, maxLen) ) return 1;
	}

	return 0; // some kind of error!
}
