/*
 * This file is part of PB-Lib v3.0 C++ Programming Library
 *
 * Copyright (c) 1995, 1997 by Branislav L. Slantchev
 * A fine product of Silicon Creations, Inc. (gargoyle)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the License which accompanies this
 * software. This library is distributed in the hope that it will
 * be useful, but without any warranty; without even the implied
 * warranty of merchantability or fitness for a particular purpose.
 *
 * You should have received a copy of the License along with this
 * library, in the file LICENSE.DOC; if not, write to the address
 * below to receive a copy via electronic mail.
 *
 * You can reach Branislav L. Slantchev (Silicon Creations, Inc.)
 * at bslantch@cs.angelo.edu. The file SUPPORT.DOC has the current
 * telephone numbers and the postal address for contacts.
*/

#include "dlist.h"

zDoubleList::zDoubleList(Boolean shouldPurge)
{
	root = focus = 0;
	count = cpos = 0L;
	err = 0;
	purge = shouldPurge;
}

// destroy the list
zDoubleList::~zDoubleList()
{
	if( count )
	{
		zDLNode *node = root;
		while( count )
		{
			zDLNode *temp = node->next;
			if( purge ) freeData(node->data);
			delete node;
			node = temp;
			count--;
		}
	}
}

// Find the first element that matches the test from the test function
ushort
zDoubleList::firstThat(fpTestFunc test, void *arg)
{
	if( count )
	{
		begin();
		do
		{
			if( True == test(focus->data, arg) ) return 1;
		}while( next() );
	}
	return 0;
}

// Execute the action function for each element in the list
void
zDoubleList::forEach(fpAppFunc action, void *arg)
{
	if( count )
	{
    	zDLNode *current = focus;
        long     cur_pos = cpos;

		begin();
	    do
	    {
			action(focus->data, arg);
		}while( next() );
        focus = current;
        cpos  = cur_pos;
    }
}

// add a node at the end of the list
ushort
zDoubleList::append(void *data)
{
	end();
	return link(data);
}

// move the focal point to the beginning of the list
ushort
zDoubleList::begin()
{
	if( count )
	{
		focus = root;
		cpos = 1L;
		return 1;
	}
	return 0;
}

// move the focal point to the end of the list
ushort
zDoubleList::end()
{
	if( count )
	{
		focus = root->prev;
		cpos = count;
		return 1;
	}
	return 0;
}

// Find the last element that matches the test from the test function
ushort
zDoubleList::lastThat(fpTestFunc test, void *arg)
{
	if( count )
	{
		end();
		do{
			if( True == test(focus->data, arg) ) return 1;
		}while( prev() );
	}
	return 0;
}


// This is a circular list implementation, so we need only one pointer
// to keep track of the head and the tail. This is the 'root' pointer
// with 'root' being the head and 'root->prev' being the tail.
//
// Also note that the list is just a container (it doesn't copy the data
// but only the pointer to it). If your data has been dynamically allocated,
// you will need to set shouldPurge to True or your objects will remain in
// memore after being unlinked from the list.
ushort
zDoubleList::link(void *data)
{
	zDLNode *node = new zDLNode(data);
	if( !node )
	{
		err = 1;
		return 0;
	}

	if( !root )
	{ // first item in the list
		root = node;
		root->next = root;
		root->prev = root;
		focus = root;
	}
	else
	{
		node->next = focus->next;
		node->prev = focus;
		focus->next->prev = node;
		focus->next = node;
		focus = node;
	}

	count++;
	cpos++;
	err = 0;
	return 1;
}

// move the focal point to the next available node (if any)
ushort
zDoubleList::next()
{
	if( cpos < count )
	{
		focus = focus->next;
		cpos++;
		return 1;
	}
	return 0;
}

// Advance the focal point nelem elements (or move back if nelem is < 0).
// Note that there is no wrap around.
ushort
zDoubleList::operator+=(short nelem)
{
	Boolean fwd = True;
	ushort  rv;

	if( nelem < 0 )
	{
		fwd = False;
		nelem = -nelem;
	}

	for(rv = 1; rv && nelem; --nelem)
	{
		if( fwd ) rv = next();
		else rv = prev();
	}

	return rv;
}

// Move the focal point back nelem (or advance if nelem is negative).
// Note that there is no wrap-around.
ushort
zDoubleList::operator-=(short nelem)
{
	Boolean bck = True;
	ushort  rv;

	if( nelem < 0 )
	{
		bck = False;
		nelem = -nelem;
	}

	for(rv = 1; rv && nelem; --nelem)
	{
		if( bck ) rv = prev();
		else rv = next();
	}

	return rv;
}

// move the focal point to the previous available node (if any)
ushort
zDoubleList::prev()
{
	if( cpos > 1L )
	{
		focus = focus->prev;
		cpos--;
		return 1;
	}
	return 0;
}

// add a node at the beginning of the list
ushort
zDoubleList::prepend(void *data)
{
	if( append(data) )
	{
		root = root->prev;
		return 1;
	}
	return 0;
}

// remove a node from the list
ushort
zDoubleList::unlink(void* &pData)
{
	if( count )
	{
		focus->prev->next = focus->next;
		focus->next->prev = focus->prev;
		pData = focus->data;
#if 0
//		if( purge ) freeData(focus->data);
#endif
		zDLNode *p = focus->next;
		delete focus;
		focus = p;
		if( cpos == count ) cpos--;
		count--;
		if( 0 == count )
		{
			root = 0;
			focus = 0;
		}
	}
	return 0;
}

ushort
zDoubleList::unlink()
{
	void   *data;
	ushort  retval = unlink(data);

	if( purge ) freeData(data);
	return retval;
}


// update a node with new data
ushort
zDoubleList::update(void *data)
{
	if( focus )
	{
    	focus->data = data;
    	return 1;
	}
    return 0;
}

