/*
 * This file is part of PB-Lib v3.0 C++ Programming Library
 *
 * Copyright (c) 1995, 1997 by Branislav L. Slantchev
 * A fine product of Silicon Creations, Inc. (gargoyle)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the License which accompanies this
 * software. This library is distributed in the hope that it will
 * be useful, but without any warranty; without even the implied
 * warranty of merchantability or fitness for a particular purpose.
 *
 * You should have received a copy of the License along with this
 * library, in the file LICENSE.DOC; if not, write to the address
 * below to receive a copy via electronic mail.
 *
 * You can reach Branislav L. Slantchev (Silicon Creations, Inc.)
 * at bslantch@cs.angelo.edu. The file SUPPORT.DOC has the current
 * telephone numbers and the postal address for contacts.
*/

#include "pblibc.h"
#include "compiler.h"
#include "stdmac.h"

#ifndef PB_SDK
	#include "typedef.h"
	#include <io.h>
	#include <dos.h>
	#include <string.h>
	#include <dir.h>
	#include <time.h>
	#include <ctype.h>
	#include <sys/stat.h>
#endif

static time_t DosToUnix(short dosdate, short dostime);

/* return status for file from a handle for an open file (no S_IWRITE) */
int
fstat(int handle, struct stat *statbuf)
{
	union REGS r;

	r.x.ax = 0x4400;  /* IOCTL status on device */
	r.x.bx = handle;
	int86(0x21, &r, &r);

	if( r.x.cflag ) return -1;  /* file not open, can't get status */

	if( r.x.dx & 0x80 )
	{  /* handle refers to a device */
		statbuf->st_mode = S_IFCHR | S_IREAD | S_IWRITE;
		statbuf->st_dev = statbuf->st_rdev = handle;
		statbuf->st_size = 0L;
		statbuf->st_mtime = statbuf->st_atime = statbuf->st_ctime = 0L;
	}
	else
	{  /* handle refers to a disk file */
		long cfpos;
		/* get the drive information and set the access mode */
		statbuf->st_dev = statbuf->st_rdev = r.x.dx & 0x3f;
		statbuf->st_mode = S_IFREG | S_IREAD;
		/* get the file size on disk */
		cfpos = lseek(handle, 0L, SEEK_CUR);
		statbuf->st_size = lseek(handle, 0L, SEEK_END);
		lseek(handle, cfpos, SEEK_SET);
		/* get the file date and size */
		r.x.ax = 0x5700;
		r.x.bx = handle;
		int86(0x21, &r, &r);
		statbuf->st_mtime = DosToUnix(r.x.dx, r.x.cx);
		statbuf->st_atime = statbuf->st_ctime = statbuf->st_mtime;
	}

	statbuf->st_nlink = 1; /* always set since MS-DOS does not have links */
	statbuf->st_ino = 0;   /* this and the rest are not used under MS-DOS */
	statbuf->st_uid = statbuf->st_gid = 0;

	return 0;
}

/* return status info about a file or device from name */
int
stat(char *path, struct stat *statbuf)
{
	zDosFile ffblk;
	int      mode;

	if( strpbrk(path, "*?") ) return -1; /* cannot have wildcards   */
	/* init the structure to 0 (for all fields we won't be needing) */
	memset(statbuf, 0, sizeof(struct stat));
	/* if we couldn't find anything, it could be the root directory */
	if( 0 != FindFirst(path, FA_DIREC|FA_SYSTEM|FA_HIDDEN, &ffblk) )
	{
		char cwd[MAXDIR+MAXDRIVE];

		if( !getcwd(cwd, sizeof(cwd)) ) return -1;
		if( -1 == chdir(path) ) return -1;
		chdir(cwd);
		mode = FA_DIREC;
	}
	/* we got something (a file probably) */
	else
	{
		statbuf->st_size = ffblk.size;
		statbuf->st_mtime = DosToUnix(ffblk.date, ffblk.time);
		statbuf->st_atime = statbuf->st_ctime = statbuf->st_mtime;
		mode = ffblk.attrib;
	}
	/* now determine the st_mode bits */
	if( mode & FA_DIREC ) statbuf->st_mode = S_IFDIR | S_IEXEC | S_IREAD;
	else
	{
		char *ext;
		/* set the access mode and check if it's an executable */
		statbuf->st_mode = S_IFREG | S_IREAD;
		if( 0 != (ext = strrchr(path, '.')) )
		{
			if( 0 != stricmp(ext, ".exe") ||
				0 != stricmp(ext, ".com") ||
				0 != stricmp(ext, ".bat") ||
				0 != stricmp(ext, ".cmd") )
			{
				statbuf->st_mode |= S_IEXEC;
			}
		}
		/* if not read-only, the file is writable */
		if( !(mode & FA_RDONLY) ) statbuf->st_mode |= S_IWRITE;
		/* figure out the drive letter either from the pathspec */
		if( ':' == path[1] ) statbuf->st_dev = toupper(path[0]) - 'A';
		else
		{ /* or from MS-DOS, get the default disk */
			union REGS r;
			r.x.ax = 0x1900;
			statbuf->st_dev = intdos(&r, &r);
		}
		statbuf->st_rdev = statbuf->st_dev; /* set to the same field */
	}
	/* these don't have any meaning under MS-DOS */
	statbuf->st_nlink = 1;

	return 0;
}

time_t
DosToUnix(short dosdate, short dostime)
{
	stamp_t   dos;
	struct tm tm_unix, tm_dos;

	dos.st_ssdate = dosdate; dos.st_sstime = dostime;

	/* create a fake tm structure with the file date/time info */
	tm_dos.tm_hour   = dos.st_time.hour;
	tm_dos.tm_min    = dos.st_time.min;
	tm_dos.tm_sec    = dos.st_time.sec * 2;
	tm_dos.tm_year   = dos.st_date.year + 80;
	tm_dos.tm_mon    = dos.st_date.mon - 1;
	tm_dos.tm_mday   = dos.st_date.day;
	tm_dos.tm_isdst  = -1;

	/* create a fake tm structure with the Jan 1, 1970 info    */
	tm_unix.tm_hour  = 0;
	tm_unix.tm_min   = 0;
	tm_unix.tm_sec   = 0;
	tm_unix.tm_year  = 0;
	tm_unix.tm_mon   = 0;
	tm_unix.tm_mday  = 1;
	tm_unix.tm_isdst = -1;
	/* now, convert those structures to local time */
	time_t dos_secs  = mktime(&tm_dos);
	time_t unix_secs = mktime(&tm_unix);

	return dos_secs - unix_secs;
}

