/*
 * This file is part of PB-Lib v3.0 C++ Programming Library
 *
 * Copyright (c) 1995, 1997 by Branislav L. Slantchev
 * A fine product of Silicon Creations, Inc. (gargoyle)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the License which accompanies this
 * software. This library is distributed in the hope that it will
 * be useful, but without any warranty; without even the implied
 * warranty of merchantability or fitness for a particular purpose.
 *
 * You should have received a copy of the License along with this
 * library, in the file LICENSE.DOC; if not, write to the address
 * below to receive a copy via electronic mail.
 *
 * You can reach Branislav L. Slantchev (Silicon Creations, Inc.)
 * at bslantch@cs.angelo.edu. The file SUPPORT.DOC has the current
 * telephone numbers and the postal address for contacts.
*/

#include "keyboard.h"
#include "pblsdk.h"
#include "utils.h"

#ifndef PB_SDK
	#include <string.h>
    #include <dos.h>
	#define  ResetInactivity()
	static Boolean ExternalInput(){ return False; }
#else
	#include "pblibc.h"
#endif

/*
 ****************************************************************************
 * the private (and simplistic) queue class
*/
class private_queue
{
public:
	private_queue(int nelem);
	~private_queue();

	Boolean isEmpty() const;
	Boolean put(ushort elem);
	ushort  get();
	ushort  peek();
	void    flush();

private:
	ushort *m_queue;
	ushort  m_nelem;
	ushort  m_used;
};

private_queue::private_queue(int nelem)
{
	m_queue = new ushort [nelem];
	m_nelem = nelem;
	m_used  = 0;
}

private_queue::~private_queue()
{
	if( m_queue ) delete[] m_queue;
}

Boolean
private_queue::isEmpty() const
{
	return Boolean( 0 == m_used );
}

Boolean
private_queue::put(ushort elem)
{
	Boolean retval = False;

	if( m_used < m_nelem )
	{
		m_queue[m_used++] = elem;
		retval = True;
	}

	return retval;
}

ushort
private_queue::get()
{
	ushort retval = 0;

	if( m_used > 0 )
	{
		retval = m_queue[0];
		if( --m_used > 0 )
		{
			memmove(&m_queue[0], &m_queue[1], sizeof(ushort) * m_used);
		}
	}

	return retval;
}

ushort
private_queue::peek()
{
	if( m_used > 0 ) return m_queue[0];
	return 0;
}

void
private_queue::flush()
{
	m_used = 0;
}

/*
 ****************************************************************************
 * the keyboard handling class
*/
int            zKeyboard::s_inited = 0;
private_queue* zKeyboard::s_local  = 0;
private_queue* zKeyboard::s_remote = 0;
Boolean        zKeyboard::paranoid = False;

zKeyboard::zKeyboard()
{
	// if this is the first object, install the handler
	if( !s_inited )
	{
		s_local  = new private_queue(16);
		s_remote = new private_queue(16);
		InstallHandler(HANDLER_SYSOPKEY, handler);
		_dos_detect_os();
	}
	s_inited++;
	m_isLocal = True;
}

zKeyboard::~zKeyboard()
{
	if( 0 == --s_inited )
	{	// if this is the last object, uninstall the local handler
		RemoveHandler(HANDLER_SYSOPKEY, handler);
		if( s_local ) delete s_local;
		if( s_remote ) delete s_remote;
	}
}

Boolean
zKeyboard::put(ushort aCode, Boolean isLocal)
{
	if( isLocal ) return s_local->put(aCode);
	else return s_remote->put(aCode);
}

void
zKeyboard::flush()
{
	s_local->flush();
	s_remote->flush();
}

Boolean
zKeyboard::isEmpty()
{
	Boolean retval = False;

	if( s_local->isEmpty() && s_remote->isEmpty() )
	{	// make sure next call will get the key if any
		pending();
		retval = True;
	}

	return retval;
}

Boolean
zKeyboard::isLocal() const
{
	return m_isLocal;
}

ushort
zKeyboard::peek()
{
	ushort retval = s_remote->peek();

	m_isLocal = False;
	if( 0 == retval )
	{
		retval = s_local->peek();
		m_isLocal = True;
	}

	if( 0 == retval ) pending();

	return retval;
}

ushort
zKeyboard::get()
{
	ushort retval = 0;

	while( 0 == retval )
	{
		m_isLocal = False;
		retval = s_remote->get();
		if( 0 == retval )
		{
			m_isLocal = True;
			retval = s_local->get();
		}
		// check for pending, let ProBoard update status
		pending();
		ResetInactivity();
		TimeLeft();
		// no key yet, release time slice
		if( 0 == retval ) _dos_idle();
	}

	return retval;
}

// checks both local and remote for incoming keys
void
zKeyboard::pending()
{
	ushort retval = scan(0);

	if( 0 != retval )
	{	// remote key found
		if( 0x1b == retval )
		{	// we have an ESC (sequence??)
			retval = scan(9);
			if( '[' == retval )
			{
				retval = scan(9);
				switch( retval )
				{
					case 0x41: s_remote->put(KEY_UP);   break;
					case 0x42: s_remote->put(KEY_DN);   break;
					case 0x43: s_remote->put(KEY_RT);   break;
					case 0x44: s_remote->put(KEY_LT);   break;
					case 0x48: s_remote->put(KEY_HOME); break;
					case 0x4b: s_remote->put(KEY_END);  break;
					default  :
						// ah, not a sequence
						s_remote->put(0x1b);
						s_remote->put('[');
						s_remote->put(retval);
				}
			}
			else
			{	// not a '[', not a sequence
				s_remote->put(0x1b);
				s_remote->put(retval);
			}
		}
		else
		{	// not an ESC, store it
			s_remote->put(retval);
		}
	}
}

// scan for a key (waits nTicks), local keys are NOT returned by this
// routine, instead, they are placed in the local queue and 0 is returned
ushort
zKeyboard::scan(int nTicks)
{
	ushort retval = PeekChar();

	if( 0 != retval && !ExternalInput() )
	{	// this was a local key
		s_local->put(retval);
		retval = 0;
	}

	if( nTicks > 0 )
	{
		volatile long far *timer = (long far *)MK_FP(0x40, 0x6C);

		for( long begin = *timer; !retval && (*timer - begin) < nTicks; )
		{
			retval = PeekChar();

			if( retval && !ExternalInput() )
			{
				s_local->put(retval);
				retval = 0;
			}
		}
	}

	return retval;
}

int
zKeyboard::handler(ushort aCode)
{
	int retval = NOT_HANDLED;

	switch( aCode )
	{	// generic stuff, cursor keys only, nothing else
		case KEY_UP  : case KEY_DN   : case KEY_LT  :
		case KEY_RT  : case KEY_HOME : case KEY_END :
		case KEY_DEL : case KEY_INS  :
			s_local->put(aCode);
			retval = HANDLED;
			break;
	}

	if( NOT_HANDLED == retval && paranoid )
	{	// get some other keys here?
		switch( aCode )
		{
			case KEY_AF1  : case KEY_AF2  : case KEY_AF3  :
			case KEY_AF4  : case KEY_AF5  : case KEY_AF6  :
			case KEY_AF7  : case KEY_AF8  : case KEY_AF9  :
			case KEY_AF10 : case KEY_PGUP : case KEY_PGDN :
			case KEY_ALTB : case KEY_ALTK : case KEY_ALTP :
			case 0x7300   : case 0x7400   : // Ctrl+Left, Ctrl+Right
				s_local->put(aCode);
				retval = HANDLED;
				break;
		}
	}

	return retval;
}
