/*
 * This file is part of PB-Lib v3.0 C++ Programming Library
 *
 * Copyright (c) 1995, 1997 by Branislav L. Slantchev
 * A fine product of Silicon Creations, Inc. (gargoyle)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the License which accompanies this
 * software. This library is distributed in the hope that it will
 * be useful, but without any warranty; without even the implied
 * warranty of merchantability or fitness for a particular purpose.
 *
 * You should have received a copy of the License along with this
 * library, in the file LICENSE.DOC; if not, write to the address
 * below to receive a copy via electronic mail.
 *
 * You can reach Branislav L. Slantchev (Silicon Creations, Inc.)
 * at bslantch@cs.angelo.edu. The file SUPPORT.DOC has the current
 * telephone numbers and the postal address for contacts.
*/

#include "pblang.h"
#include "proboard.h"
#include "pblsdk.h"
#include "pbnames.h"
#include "file.h"
#include "terminal.h"
#include "utils.h"
#include "str.h"

// comparison function that looks at the prompt numbers
static Boolean
compare(const void *data, void *key)
{
	return Boolean( ((zPrompt *)data)->num == *(short *)key );
}

////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////
zPrompt::zPrompt(PBL_ITEM &aData, short aNum):
	num(aNum),
	text(0),
	keys(0)
{
	memcpy(&data, &aData, sizeof(data));
}

////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////
// destroy the list of prompts and free memory
zPromptList::~zPromptList()
{
	begin();
	if( count )
	{
		do
		{
			zPrompt *p = (zPrompt *)get();
			if( p->text ) delete[] p->text;
			if( p->keys ) delete[] p->keys;
		}
		while( next() );
	}
}

////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////
zProLang::zProLang( zTerminal * const pTerminal )
	:indexed(False)
	,err(False)
	,terminal(pTerminal)
{
	sprintf(path, "%s%s%s", pb_getdir(PBDIR_SYS), CurUser->language, fnLANGEXT);
	file_chext(index, path, ".PLI");

    FILE *fp = fopen(path, "rb");
	if( !fp )
	{
    	err = True;
        return;
	}

    fread(&text, sizeof(text), 1, fp);
    fclose(fp);

	if( file_exist(index) && 0 >= file_cmpdate(index, path) ) indexed = True;
	if( 0 == (list = new zPromptList) ) err = True;

	if( !terminal ) err = True;
}

zProLang::~zProLang()
{
	delete list;
}

// loads a prompt in the list, if not loaded already
Boolean
zProLang::Load( short aNum )
{
	FILE     *fp;
	PBL_ITEM  rec;
	zPrompt  *node;
	char      buf[0x1000]; // more than enough!

	// see if it's in the list already
	if( 0 != list->firstThat(compare, &aNum) ) return True;
	// try to find it in the language file
	if( NULL == (fp = find(aNum)) ) return False;

	// read the header record and allocate a new node
	fread(&rec, sizeof(rec), 1, fp);
	node = new zPrompt(rec, aNum);
	if( !node )
	{
		err = True;
		fclose(fp);
		return False;
	}
	// store the text in the prompt list
	fread(buf, rec.len, sizeof(char), fp);
	buf[rec.len] = EOS;
	node->text = newStr(buf);
	// and the hotkeys, if any
	if( rec.numHotkeys )
	{
		fread(buf, rec.numHotkeys, sizeof(char), fp);
		buf[rec.numHotkeys] = EOS;
		node->keys = newStr(buf);
	}
	// cleanup (close file and link new node)
	fclose(fp);
	return Boolean( list->link(node) );
}

// this is a special function. it is called by Exec() when '@' is found
// in the prompt line. it will attempt to parse it and (on success) execute
// the pex or display the file (depending on the type). it will return the
// new position in the string where the caller must resume scanning (to
// skip the macro). note that pexen are not supported in the exe version!
char*
zProLang::atCode(char *s)
{
	char    buf[80];
	size_t  len;
	char   *p = strchr(s + 2, '@'); // find terminating '@'
	// at entry, *s should be pointing to a '@'
	// and we MUST have the ending '@' too
	if( '@' != *s || 0 == p )
	{
		terminal->handle(*s);
		return s;
	}
	// parse macro according to the next character
	// first, get the text into a local buffer (w/o leading chars)
	len = size_t(p - s) - 2;
	memcpy(buf, &s[2], len); buf[len] = EOS; // make it a string
	switch( s[1] )
	{ // the codes ARE case-sensitive
		case 'a':
			if( 9 < len ) goto normal; // invalid format
			pb_showfile(buf, 0, pb_getdir(PBDIR_TEXT), terminal);
			return s + len + 2; // return points to '@'
		case 'p':  // pex exec, can't check for length, assume ok
			#ifdef PB_SDK
			MenuFunction(MENU_RUN_SDKFILE, buf);
			#endif
			return s + len + 2;
		default : // argh, what's that?
		normal:
			terminal->handle(*s); // handle this character
			return s; // same position, character handled
	}
}

// displays a prompt (will load it if not loaded already), if this is
// a prompt with hotkeys, will also poll for hotkeys and return the
// number of the selection (1..lastkey) - this can be disabled if you
// pass False as the second argument. in that case, only display and return
// returns: -1 on error, 0 - normal display, >0 - hotkey pressed
int
zProLang::Exec( short aNum, Boolean honorType )
{
	Boolean highColor = False;
	int     retval    = 0;

	if( 0 == list->firstThat(compare, &aNum) )
	{
		if( False == Load(aNum) ) return -1;
		else list->firstThat(compare, &aNum);
	}
	// here we have a focused prompt entry
	zPrompt *p = (zPrompt *)list->get();

	// set the first color to prompt's normal color
	terminal->setColor(p->data.color);
	// display the string (colors are done by Terminal)
	for( char *s = p->text; EOS != *s; ++s )
	{
		switch( *s )
		{
			case '\\': // only check for \^ and \\, else -> Terminal
				if( '^' == s[1] ) terminal->handle(*++s);
				else if( '\\' == s[1] )
				{
					terminal->handle('\b');
					s++;
				}
				else terminal->handle(*s);
				break;

			case '^':  // toggle highlight color
				if( terminal->busy() ) terminal->handle(*s);
				else
				{
					highColor = Boolean( !highColor );
					if( highColor ) terminal->setColor(p->data.highCol);
					else terminal->setColor(p->data.color);
				}
				break;

			case '@':
				s = atCode(s);
				break;

			case '\n':
				terminal->handle("\r\n");
				break;

			default:
				terminal->handle(*s);
		}
	}

	// is this a prompt and we have to honor it
	if( honorType && p->keys )
	{
		char  ch  = WaitKeys(p->keys);        // get the key from user
		char *ptr = strichr(p->keys, ch);     // find the offset in string
		retval    = 1 + (int)(ptr - p->keys); // calculate return value
	}
	// default return value is 0 if no keys/honor
	return retval;
}

// tries to find a prompt in the language file and returns the
// file with read pointer positioned at the beginning of the prompt
FILE*
zProLang::find(short aNum)
{
	FILE     *fp = NULL;
	PBL_ITEM  rec;
	long      offset;

	if( 0 >= aNum || text.numItems < aNum ) return NULL; // invalid number
	if( indexed )
	{	// language file has been indexed!
		if( NULL == (fp = fopen(index, "rb")) ) goto normal;
		fseek(fp, l_mul(aNum - 1L, sizeof(ulong)), SEEK_SET);
		fread(&offset, sizeof(offset), 1, fp);
		fclose(fp);
		if( NULL == (fp = fopen(path, "rb")) ) return NULL;
		fseek(fp, offset, SEEK_SET);
	}
	else
	{
	normal:
		if( NULL == (fp = fopen(path, "rb")) ) return NULL;
		fseek(fp, (long)sizeof(PBL_HEADER), SEEK_SET);
		while( --aNum && !feof(fp) )
		{
			fread(&rec, sizeof(rec), 1, fp);
			fseek(fp, long(rec.len + rec.numHotkeys), SEEK_CUR);
		}
	}
	if( NULL == fp ) err = True;
	return fp;	// with the pointer positioned in the stream at the prompt
}

// returns the hotkeys associated with a prompt item
// the item must have been loaded already (unlike Exec)
const char*
zProLang::GetItemKeys(short item)
{
	if( 0 != list->firstThat(compare, &item) )
	{
		return (const char *)((zPrompt *)list->get())->keys;
	}
	return 0;
}

// returns the text associated with a prompt item
// the item must have been loaded already (unlike Exec)
const char*
zProLang::GetItemText(short item)
{
	if( 0 != list->firstThat(compare, &item) )
	{
		return (const char *)((zPrompt *)list->get())->text;
	}
	return 0;
}
