// File: SGIWriter.C
//
// (c) 1996 Mayur Patel


// *******************************************************************

#include <SGIWriter.h>
#include <assert.h>
#include <string.h> // memset()

SGIWriter::SGIWriter( const char *cpOutFilename, SampledImage *pInput )
{
   if ( cpOutFilename && pInput )
      _iOK = writeSGIFile( 
            cpOutFilename, 
            (InputImage*) pInput, 
            pInput->getArea(), 
            pInput->getType() );
            
   else
      _iOK = 0;
   return;
}



SGIWriter::SGIWriter( 
   const char *cpOutFilename, 
   SampledImage *pInput, 
   Type setType )
{
   if ( cpOutFilename && pInput )
      _iOK = writeSGIFile( 
         cpOutFilename, 
         (InputImage*) pInput, 
         pInput->getArea(), 
         setType );
   else
      _iOK = 0;
   return;
}



int
SGIWriter::writeSGIFile( 
   const char *cpFile, 
   InputImage *pInput, 
   const ImageArea& area, 
   const Type setType )
{
   int		iOK = -1;
   Type	fileType;
   ofstream	out( cpFile );
   
   if ( cpFile && pInput && ( !out.fail() ) ) {
      
      //
      // set data type:
      //
      fileType = UShort;
      if ( setType == UChar )
         fileType = UChar;
         
      //
      // open & fill header:
      //
      iOK &= validateFile( out, area, fileType );
      
      //
      // fill rows:
      //
      if ( iOK )
         iOK &= writeRows( out, pInput, area, fileType );
 
   } else
      iOK = 0;
   
   return ( iOK );
}


int
SGIWriter::validateFile( 
   ofstream& out, 
   const ImageArea& area, 
   const Type fileType )
{
   int 			iOK = -1;
   unsigned char	cp[512];
   
   assert(( fileType == UChar )||( fileType == UShort));

   //
   // init to zeros
   //    RLE == off ( 0 )
   //
   memset( cp, 0, 512 );
   
   //  magic cookie:
   castFromShort( cp, 474 );
   
   // bytes per pix
   if ( fileType == UChar )
      cp[3] = 1;
   else
      cp[3] = 2;
      
   // x, y + c number of channels:
   castFromShort( cp + 4, 3 );
   
   // x, y, c
   castFromShort( cp + 6, area.width );
   castFromShort( cp + 8, area.height );
   castFromShort( cp + 10, area.c );
   
   // min & max (unused)
   castFromLong( cp + 12, 0 );
   if ( fileType == UChar )
      castFromLong( cp + 16, 0x00FF );
   else
      castFromLong( cp + 16, 0x0FFFF );
   
   //
   // commit:
   //
   out.write( cp, 512 );

   if ( out.fail() )
      iOK = 0;
   
   return ( iOK );
}


int
SGIWriter::writeRows( 
   ofstream& outfile, 
   InputImage *pInput, 
   const ImageArea& area, 
   const Type setType )
{
   int			iOK = -1;
   int			iChannels;
   signed		iLoop;
   ImageArea		reqArea; 
   ImageTile 		*pTile; 
   
   assert( (setType == UShort )|| (setType == UChar) );
   
   reqArea = area;
   reqArea.height = 1;
   reqArea.z = 0;
   for( iChannels=0; iChannels<area.c; iChannels++ )
   {
      reqArea.c = iChannels;
      
      //
      // compiler requires explicit typecast to
      // (signed) in the following for-compare:
      //
      for( iLoop=area.y; iLoop<(signed)(area.y+area.height); iLoop++)
      {
         reqArea.y = iLoop;
            
	 pTile = pInput->newTile( reqArea );
	 if ( pTile )
	 {
	    pTile->typecast( setType );
	    
	    _commit( 
	       outfile, 
	       pTile->getBuffer(), 
	       reqArea.width, 
	       setType );
	       
            pTile->deleteTile(); 
         }
	 else
	 {
            iOK = 0;
            break;
	 }
      }
   }
   return( iOK );
}


void
SGIWriter::_commit( 
   ofstream& outfile,
   void *vpBuffer, 
   unsigned uPix, 
   Type setType )
{
   unsigned char	*cpBuffer;
   unsigned short	*spBuffer;   
   unsigned		uLoop;
   
   cpBuffer = (unsigned char *) vpBuffer;
   spBuffer = (unsigned short *) vpBuffer;
   
   if ( setType == UChar )
      outfile.write( cpBuffer, uPix );
   else
   {
      assert( setType == UShort );
      for( uLoop=0; uLoop<uPix; uLoop++ )
      {
         //
         // endian safe:
         //
         castFromShort( cpBuffer, *spBuffer );
         cpBuffer += sizeof( unsigned short );
         spBuffer++;
      }
      outfile.write( 
         (unsigned char *) vpBuffer, 
         uPix * sizeof(unsigned short) );
   }
   return;
}
