// renderer.h
// 1996 Mark Carolan
// abstract base class
// Responsibility of derived class to draw triangles

#ifndef RENDERER_H
#define RENDERER_H

#include "WinDOS.h"

#define ALERTBOX 128
#define MAX_MESHES 100
#define MAX_PANELS 100

#if defined(__INTEL__)
	extern char szAppName[];
#endif

#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "Scanner.h"
#include "Vector3D.h"
#include "Tri3D.h"
#include "Color16.h"
#include "Texture.h"
#include "MCutils.h"
#include "Panel.h"

#if defined(macintosh)
	#define FALSE false
	#define TRUE true
	#define BOOL bool
#elif defined(__INTEL__)
	#include <windows.h>
	#include <ddraw.h>
	extern char szAppName[];
	typedef void *GDHandle;
#endif

#define X_ROT 0.01
#define Y_ROT 0.01
#define Z_ROT 0.01
#define X_MOVE 1.0
#define Y_MOVE 1.0
#define Z_MOVE 1.0

struct Movement {
	float rx, ry, rz;
	float mx, my, mz;

	void Set(float ratio) 
	{	
		rx = X_ROT * ratio;
		ry = Y_ROT * ratio;
		rz = Z_ROT * ratio;
		mx = X_MOVE * ratio;
		my = Y_MOVE * ratio;
		mz = Z_MOVE * ratio;
	}
};

struct  Mesh {
	Tri3D 			*start_tri;
	int 			cellsX, cellsY;	
};

typedef enum { OBJ_MESH, OBJ_QUAD, OBJ_PANEL } ObjectType;
typedef struct {
	ObjectType Obj;
	union {
		Mesh &mesh;
		struct Quad &quad;
		class Panel &panel;
	} ObjType;
} RenderObject;
		 
class Renderer
{	
	inline int		near2(int x, int y) { return (x >= (y - 2) && x <= (y + 2)); }
	inline int		near4(int x, int y) { return (x >= (y - 4) && x <= (y + 4)); }	

	inline void		TranslateVertex(int v, float tx, float ty, float tz, long t);
	inline void		RotateVertex(int v, float rx, float ry, float rz, long t);			
	inline void		PositionVertex(int v, float tx, float ty, float tz, float rx, float ry, float rz);	

protected:
	friend class	Vector3D;
	friend class	Tri3D;
	friend class	Panel;

	void			TranslatePanel(Panel &p);
	void			RotatePanel(Panel &p);
	
	int				buffer_width, buffer_height, buffer_middle, buffer_size;
	int				shift_1, shift_2;
	int				clipping, using_DD;
	int				player_x, player_y, player_view_angle;        
	float			*tan_table, *inv_tan_table;
	float			*cos_table, *inv_cos_table;
	float			*sin_table, *inv_sin_table;
	int				field_of_view, half_fov;
	long			*distances; // for "manual" z-buffering
	int				SPACE_HIT, keyStatus, STEP_LENGTH;
	Rgb				frames[256];
	int				new_surface, vListSize;
	int				num_vertices, num_lights;
	Movement		mNormal, mHi, mLow, *mCurrent;
	Vertex3D		*vertex_list, world_offset;
	Tri3D			*tri_list;
	Quad3D			*quad_list;
	Light3D			lightList[MAX_LIGHTS];	

#if defined(__INTEL__)
	LPDIRECTDRAWSURFACE	primarySurface;
	LPDIRECTDRAW		directDraw;
#endif	
	
#if defined(macintosh)
	GDHandle		gDevice;
	RgnHandle		clipRegion;
	Rect			windowRect;
	int				oldScreenDepth;
#else
	HRGN			clipRegion;
	HDC				gDevice;
	RECT			windowRect;
#endif	
	
	void			TransformWorld(float tx, float ty, float tz, float rx, float ry, float rz);
	void			TransformVertex(int v, float tx, float ty, float tz, float rx, float ry, float rz);
	void			PositionWorld(float tx, float ty, float tz, float rx, float ry, float rz);
	Tri3D 			*AddTri(int vi1, int vi2, int vi3, Uv uv1, Uv uv2, Uv uv3, Rgb &p_rgb, int tmp, int type = TRI_FIXED, long triData = 0);
	struct Quad		AddQuad(int vi1, int vi2, int vi3, int vi4, Rgb &p_rgb, int tmp, int type = TRI_FIXED, long data = 0);
	int				AddVertex(float x, float y, float z, float r, float g, float b, long vData = 0);
	void			InitVertexList();
	void			DeleteTriList();
	void			DeletePanelList();
	void			ClearWindow();
	void			SetClipRegion(int inset_x, int inset_y);
	void 			Error(const char *message);	

#if defined(__INTEL__)
	void 			Shrink(RECT *rect, int w, int h);
#else
	void 			Shrink(Rect *rect, int w, int h);
#endif
					Renderer(int vw, int vh, void *inst, int vls);
public:
#if defined(macintosh)
	GDHandle		device;
	WindowPtr		window;
	void			*appInstance;
#elif defined(__INTEL__)
	HWND			window;
	HINSTANCE		appInstance;
#endif


	int				viewport_width, viewport_height, viewport_middle;
	int 			CreateTransformPanel(
						float width, float height, int mesh_width, int mesh_height,
						float rcol, float gcol, float bcol,	float rx, float ry, float rz,
						float tx, float ty, float tz, int tex_index, int vData = 0);
int 				CreateTransformQuad(	
						float width, float height, float rcol, float gcol, float bcol,	
						float tx, float ty, float tz, float rx, float ry, float rz,
						int tex_index, int vData);						
	void			ClearLights();
	void			SetLightSource(int i, float lx, float ly, float lz) { 
						if (i >= MAX_LIGHTS) return;
						if (!lightList[i].active) num_lights++; // new one
						lightList[i].active = 1;
						lightList[i].x = lx;
						lightList[i].y = ly;
						lightList[i].z = lz; }
	void			RemoveLightSource(int i) {
						if (i >= MAX_LIGHTS) return;
						if (lightList[i].active) num_lights--;
						lightList[i].active = 0; }
					
	virtual			~Renderer();

//	Object transforms					
	void			MoveWorld(long keys);
	void			TranslateObject(RenderObject &obj, float tx, float ty, float tz, long t);
	void			RotateObject(RenderObject &obj, float rx, float ry, float rz, long t);
		
// 	Object creators	
	int 			LoadDXFTris(
						char *fileName, int tex_index, float scale, 
						float r, float g, float b, 
						float tx, float ty, float tz,
						float rx, float ry, float rz);	
	virtual int		Valid() { return valid && (!mem_alloc_error); }	
	int 			AddPoint(float x, float y);
	int 			CreateMesh(	
						float width, float height, 
						int mesh_width, int mesh_height,
						float rcol, float gcol, float bcol,	
						float tx, float ty, float tz,
						float rx, float ry, float rz,
						int tex_index, int vData);	
	void 			ColorMeshCell(
						int theMeshix, int posx, int posy, 
						float r,  float g, float b, int texix);
	void 			ColorPanelCell(
						int panelID, int posx, int posy,
						float red, float green, float blue, int tmapIndex = 0, int dat = 0);						
	
	Mesh			meshList[MAX_MESHES];
	Panel			*panel_list;
	Panel			*panel_listMoving;
	int				last_mesh_index, last_panel_index;	
	int				valid;
	int				mem_alloc_error;	
};

#endif
