// wdj.c

#include "ntddk.h"
#include "ntdddisk.h"
#include "wdj.h"

typedef struct _DEVICE_EXTENSION {
    PKEVENT     Event;
    PMDL        Mdl;
    ULONG       BufferSize;
} DEVICE_EXTENSION, *PDEVICE_EXTENSION;

#define WDJ_INFO        "DATA FROM WDJ.SYS"
#define WDJ_INFO_SIZE   18

NTSTATUS WdjDrvDispatch(IN PDEVICE_OBJECT, IN PIRP);
VOID     WdjDrvUnload(IN PDRIVER_OBJECT);


NTSTATUS DriverEntry(IN PDRIVER_OBJECT  DriverObject,
                     IN PUNICODE_STRING RegistryPath)
{
    NTSTATUS            status = STATUS_SUCCESS;
    PDEVICE_OBJECT      deviceObject = NULL;
    UNICODE_STRING      deviceName, dosName;

    //
    // Create an non-EXCLUSIVE device object.
    //
    RtlInitUnicodeString(&deviceName, L"\\Device\\WdjDrv");

    status = IoCreateDevice(DriverObject, sizeof(DEVICE_EXTENSION),
                            &deviceName, FILE_DEVICE_WDJDRV,
                            0, FALSE, &deviceObject);

    if (!NT_SUCCESS(status)) {
        return STATUS_UNSUCCESSFUL;
    }

    //
    // Create a symbolic link that Win32 apps can specify to gain
    // access to this driver/device
    //
    RtlInitUnicodeString(&dosName, L"\\DosDevices\\WDJDRV");

    status = IoCreateSymbolicLink(&dosName, &deviceName);

    if (!NT_SUCCESS(status)) {
        DbgPrint("WDJ.SYS: create symbolic link failed\n");
        IoDeleteDevice(deviceObject);
        return STATUS_UNSUCCESSFUL;
    }

    //
    // Create dispatch points for device control, create, close.
    //
    DriverObject->MajorFunction[IRP_MJ_CREATE]         =
    DriverObject->MajorFunction[IRP_MJ_CLOSE]          =
    DriverObject->MajorFunction[IRP_MJ_DEVICE_CONTROL] = WdjDrvDispatch;
    DriverObject->DriverUnload                         = WdjDrvUnload;

    return status;

} // DriverEntry


NTSTATUS WdjDrvDispatch(IN PDEVICE_OBJECT DeviceObject,
                        IN PIRP           Irp)
{
    NTSTATUS            status;
    PIO_STACK_LOCATION  irpStack;
    PDEVICE_EXTENSION   deviceExt;
    PVOID               ioBuffer;
    ULONG               inLength;
    ULONG               outLength;
    ULONG               ioctl;

    irpStack = IoGetCurrentIrpStackLocation(Irp);
    deviceExt = DeviceObject->DeviceExtension;

    Irp->IoStatus.Status      = STATUS_SUCCESS;
    Irp->IoStatus.Information = 0;

    //
    // Get the pointer to the input/output buffer and it's length
    //
    ioBuffer  = Irp->AssociatedIrp.SystemBuffer;
    inLength  = irpStack->Parameters.DeviceIoControl.InputBufferLength;
    outLength = irpStack->Parameters.DeviceIoControl.OutputBufferLength;

    switch (irpStack->MajorFunction) {

        case IRP_MJ_CREATE:
            DbgPrint("WDJ.SYS: IRP_MJ_CREATE\n");
            break;

        case IRP_MJ_CLOSE:
            DbgPrint("WDJ.SYS: IRP_MJ_CLOSE\n");
            break;

        case IRP_MJ_DEVICE_CONTROL:
            DbgPrint("WDJ.SYS: IRP_MJ_DEVICE_CONTROL\n");
            ioctl = irpStack->Parameters.DeviceIoControl.IoControlCode;

            switch (ioctl) {

                case IOCTL_WDJ_REQUEST: {

                    PWDJ_DATA Data = (PWDJ_DATA)ioBuffer;

                    DbgPrint(("*** WDJ: IOCTL_WDJ_REQUEST\n"));

                    if (inLength < sizeof(WDJ_DATA) ||
                        Data->BufferSize < WDJ_INFO_SIZE) {

                        Irp->IoStatus.Status = STATUS_BUFFER_TOO_SMALL;
                        break;
                    }

                    ObReferenceObjectByHandle(Data->Event, 0x0002,
                                       NULL, UserMode,
                                       (PVOID *)(&(deviceExt->Event)),
                                       NULL);

                    KeClearEvent(deviceExt->Event);

                    deviceExt->Mdl = IoAllocateMdl(Data->Buffer,
                                                   Data->BufferSize,
                                                   FALSE, FALSE, NULL);

                    if (deviceExt->Mdl == NULL) {
                        ExRaiseStatus(STATUS_INSUFFICIENT_RESOURCES);
                    }

                    MmProbeAndLockPages(deviceExt->Mdl, UserMode,
                                        IoWriteAccess);

                    deviceExt->BufferSize = Data->BufferSize;
                    break;
                }

                case IOCTL_WDJ_TRIGGER:

                    if (deviceExt->Mdl) {

                        PVOID buffer;
                        PWDJ_DATA Data;

                        buffer = MmGetSystemAddressForMdl(deviceExt->Mdl);
                        Data = (PWDJ_DATA)buffer;

                        RtlFillMemory(buffer, deviceExt->BufferSize, 0);
                        RtlCopyMemory(buffer, WDJ_INFO, WDJ_INFO_SIZE);

                        MmUnlockPages(deviceExt->Mdl);
                        IoFreeMdl(deviceExt->Mdl);
                        deviceExt->Mdl = NULL;

                        KeSetEvent(deviceExt->Event, +2, FALSE);
                    }
                    break;

                default:
                    DbgPrint("WDJ.SYS: unknown IOCTL\n");
                    Irp->IoStatus.Status = STATUS_INVALID_PARAMETER;
                    break;
            }
            break;
    }

    status = Irp->IoStatus.Status;  // save before releasing irp
    IoCompleteRequest(Irp, IO_NO_INCREMENT);
    return status;                  // no pending ops, return status

} // WdjDrvDispatch


VOID WdjDrvUnload(IN PDRIVER_OBJECT DriverObject)
{
    UNICODE_STRING unicodeString;

    //
    // Delete the symbolic link and the device object.
    //
    RtlInitUnicodeString(&unicodeString, L"\\DosDevices\\WDJDRV");
    IoDeleteSymbolicLink(&unicodeString);
    IoDeleteDevice(DriverObject->DeviceObject);

} // WdjUnload
