unit qiFormat;
{
Copyright 1996 by Justin Turberville
EMail: JustinT @ cyberjoe.co.za

TQIFormatter is designed to propduce both SQL-compliant and user-orintated
versions of the query action data stored in A TQueryInfo component.
Sub-type this component to customise how the data is interpreted (for
various SQL and QBE standards, and database types and drivers, or just to
introduce improved  routines) and for international use (other languages
for user version routines).
A bright spark can produce a version and a fancy query builder that uses
more than one QueryInfo component for the production of nested queries!
See DempU.pas to see how to ensure that TQIFormatter includes required
bridging joins in multi-table queries (application-specific logic).
}
interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  qiField, qiQuery;

type
  TSQLClauseEvent = procedure(Sender: TObject; var Clause: string) of object;

  TQIFormatter = class(TComponent)
  private
    FAbout: TQIAboutProperty;
    FPrefix: Boolean;
    FQualify: Boolean;
    FQueryInfo: TQueryInfo;
    FDefaultSelect: Boolean;
    FOnSelectClause: TSQLClauseEvent;
    FOnFromClause:   TSQLClauseEvent;
    FOnJoinClause:   TSQLClauseEvent;
    FOnFilterClause: TSQLClauseEvent;
    FOnGroupClause:  TSQLClauseEvent;
    FOnOrderClause:  TSQLClauseEvent;
    procedure SetQueryInfo(Value: TQueryInfo);
  protected
    procedure CheckQueryInfo;
  public
    constructor Create(AOwner: TComponent); override;
    procedure Notification(AComponent: TComponent; Operation: TOperation); override;
    function SQLSelectClause: string;
    function SQLFromClause: string;
    function SQLJoinClause: string;
    function SQLFilterClause: string;
    function SQLGroupClause: string;
    function SQLOrderClause: string;
    function UserSelectClause: string;
    function UserFromClause: string;
    function UserFilterClause: string;
    function UserGroupClause: string;
    function UserOrderClause: string;
    procedure BuildSQLQuery(Strings: TStrings);
    procedure BuildUserQuery(Strings: TStrings);
  published
    property About: TQIAboutProperty read FAbout write FAbout stored False;
    property DefaultSelect: Boolean read FDefaultSelect write FDefaultSelect default True;
    property Prefix: Boolean read FPrefix write FPrefix default True;
    property Qualify: Boolean read FQualify write FQualify default True;
    property QueryInfo: TQueryInfo read FQueryInfo write SetQueryInfo;
    property OnSelectClause: TSQLClauseEvent read FOnSelectClause write FOnSelectClause;
    property OnFromClause: TSQLClauseEvent read FOnFromClause write FOnFromClause;
    property OnJoinClause: TSQLClauseEvent read FOnJoinClause write FOnJoinClause;
    property OnFilterClause: TSQLClauseEvent read FOnFilterClause write FOnFilterClause;
    property OnGroupClause: TSQLClauseEvent read FOnOrderClause write FOnGroupClause;
    property OnOrderClause: TSQLClauseEvent read FOnOrderClause write FOnOrderClause;
  end;

implementation

constructor TQIFormatter.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FDefaultSelect := True;
  FPrefix := True;
  FQualify := True;
end;

procedure TQIFormatter.Notification(AComponent: TComponent; Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if (Operation = opRemove) and (AComponent = FQueryInfo) then
    FQueryInfo := nil;
end;

procedure TQIFormatter.SetQueryInfo(Value: TQueryInfo);
begin
  FQueryInfo := Value;
  if Value <> nil then
    Value.FreeNotification(Value);
end;

procedure TQIFormatter.CheckQueryInfo;
begin
  if FQueryInfo = nil then
    raise EComponentError.Create('No QueryInfo assigned');
end;

function TQIFormatter.SQLSelectClause: string;
var
  i: Integer;
begin
  Result := '';
  CheckQueryInfo;
  with FQueryInfo do
    if Selections.Count = 0 then  {if Selections not used}
      if FDefaultSelect then
        for i := 0 to Fields.Count-1 do begin     {then include by default -}
          if (fuSelect in Fields[i].FieldUsage) or FieldRequired(Fields[i]) then
            Result := Result + Fields[i].SQLNameStr(FQualify) + ', ';
        end
      else Exit
    else begin                         {else do it the hard way -}
      for i := 0 to Fields.Count-1 do         {initilize flags -}
        Fields[i].EditFlag := efNew;
      for i := 0 to Selections.Count-1 do
        with Selections[i] do begin
          Result := Result + SQLSelectStr(FQualify) + ', ';
          Field.EditFlag := efOld;     {mark as included}
        end;
      for i := 0 to Fields.Count-1 do   {get any required fields not in -}
        if (Fields[i].EditFlag <> efOld) and FieldRequired(Fields[i]) then
          Result := Result + Fields[i].SQLNameStr(FQualify) + ', ';
    end;
  System.Delete(Result, Length(Result) -1, 2);         {remove last comma}
  if FPrefix then Result := 'SELECT ' + Result;
  if Assigned(FOnSelectClause) then FOnSelectClause(Self, Result);
end;

function TQIFormatter.UserSelectClause: string;
var
  i: Integer;
begin
  Result := '';
  CheckQueryInfo;
  with FQueryInfo do
    if Selections.Count = 0 then      {if Selections not used -}
      if FDefaultSelect then
        Result := 'all fields'        {then presumed to be all defined fields}
      else Exit
    else begin                   {else only add specifically selected fields -}
      for i := 0 to Selections.Count-1 do
        Result := Result + Selections[i].UserSelectStr + ', ';
      System.Delete(Result, Length(Result) -1, 2);
    end;
  if FPrefix then Result := 'Show ' + Result;
end;

function TQIFormatter.SQLFromClause: string;
var
  i: Integer;
begin
  Result := '';
  CheckQueryInfo;
  with FQueryInfo do
    for i := 0 to Tables.Count-1 do
      if TableUsed(Tables[i]) then begin
        Result := Result + StripExt(Tables[i].SQLName);
        if FQualify then                  {give alias -}
          Result := Result + ' ' + TableChar + IntToStr(TableS[i].Index +1);
        Result := Result + ', ';
      end;
  if Result <> '' then begin
    System.Delete(Result, Length(Result)-1, 2);
    if FPrefix then Result := 'FROM ' + Result;
  end;
  if Assigned(FOnFromClause) then FOnFromClause(Self, Result);
end;

function TQIFormatter.UserFromClause: string;
var
  i: Integer;
begin
  Result := '';
  CheckQueryInfo;
  with FQueryInfo do
    for i := 0 to Tables.Count-1 do
      if TableUsed(Tables[i]) then
        Result := Result + Tables[i].UserName + ', ';
  if Result <> '' then begin
    System.Delete(Result, Length(Result)-1, 2);
    if FPrefix then Result := 'From ' + Result;
  end;
end;

function TQIFormatter.SQLJoinClause: string;
var
  i: Integer;
begin
  Result := '';
  CheckQueryInfo;
  with FQueryInfo do        {include all joins defined between used tables -}
    for i := 0 to Joins.Count-1 do
      with Joins[i] do
        if TableUsed(Table1) and TableUsed(Table2) then
           Result := Result + SQLJoinStr(FQualify) + ' AND ';
  if Result <> '' then begin
    System.Delete(Result, Length(Result)-4, 5);        {delete last ' AND '}
    if FPrefix then Result := 'WHERE ' + Result;
  end;
  if Assigned(FOnJoinClause) then FOnJoinClause(Self, Result);
end;

function TQIFormatter.SQLFilterClause: string;
  function WithBkt(F: TFilterItem): string;
  begin
    Result := F.SQLFilterStr(FQualify);
    if F.IsNested then
      if QueryInfo.Filters.NestingType(F) = ntOpening then
        if Result[1] = 'A' then
          Insert(' (', Result, 4) else      {insert after AND}
          Insert(' (', Result, 3)           {insert after OR}
      else Result := Result + ')'           {add to end}
  end;
var
  i: Integer;
begin
  Result := '';
  CheckQueryInfo;
  with FQueryInfo do
    for i := 0 to Filters.Count-1 do
      Result := Result + WithBkt(Filters[i]) + ' ';
  if Result <> '' then begin
    System.Delete(Result, Length(Result), 1);    {delete last space}
    if Result[1] = 'A' then
      System.Delete(Result, 1, 4) else         {delete first AND}
      System.Delete(Result, 1, 3);             {delete first OR}
    Result := '(' + Result + ')';    {precaution for un-nested OR's after join}
    if FPrefix then Result := 'WHERE ' + Result;
  end;
  if Assigned(FOnFilterClause) then FOnFilterClause(Self, Result);
end;

function TQIFormatter.UserFilterClause: string;
  function WithBkt(F: TFilterItem): string;
  begin
    Result := F.UserFilterStr;
    if F.IsNested then
      if QueryInfo.Filters.NestingType(F) = ntOpening then
        if Result[1] = 'a' then
          Insert(' (', Result, 4) else      {insert after "and"}
          Insert(' (', Result, 3)           {insert after "or"}
      else Result := Result + ')'           {add to end}
  end;
var
  i: Integer;
begin
  Result := '';
  CheckQueryInfo;
  with FQueryInfo do
    for i := 0 to Filters.Count-1 do
      Result := Result + WithBkt(Filters[i]) + ' ';
  if Result <> '' then begin
    if Result[1] = 'a' then
      System.Delete(Result, 1, 4) else         {'and '}
      System.Delete(Result, 1, 3);             {'or '}
    System.Delete(Result, Length(Result), 1);  {' '}
    if FPrefix then Result := 'Filter where ' + Result;
  end
  else if FPrefix then Result := {'No filtering'}'Unfiltered';
end;

function TQIFormatter.SQLGroupClause: string;
var
  i: Integer;
begin
  Result := '';
  CheckQueryInfo;
  with FQueryInfo do
    for i := 0 to Grouping.Count-1 do
      Result := Result + Grouping[i].SQLGroupStr(FQualify) + ', ';
  if Result <> '' then begin
    System.Delete(Result, Length(Result) -1, 2);
    if FPrefix then Result := 'GROUP BY ' + Result;
  end;
  if Assigned(FOnGroupClause) then FOnGroupClause(Self, Result);
end;

function TQIFormatter.UserGroupClause: string;
var
  i: Integer;
begin
  Result := '';
  CheckQueryInfo;
  with FQueryInfo do
    for i := 0 to Grouping.Count-1 do
      Result := Result + Grouping[i].UserGroupStr + ', ';
  if Result <> '' then begin
    System.Delete(Result, Length(Result) -1, 2);
    if FPrefix then Result := 'Group by ' + Result;
  end
 {else if FPrefix then Result := 'No grouping'};
end;

function TQIFormatter.SQLOrderClause: string;
var
  i: Integer;
begin
  Result := '';
  CheckQueryInfo;
  with FQueryInfo do
    for i := 0 to Ordering.Count-1 do
      Result := Result + Ordering[i].SQLOrderStr(FQualify) + ', ';
  if Result <> '' then begin
    System.Delete(Result, Length(Result) -1, 2);
    if FPrefix then Result := 'ORDER BY ' + Result;
  end;
  if Assigned(FOnOrderClause) then FOnOrderClause(Self, Result);
end;

function TQIFormatter.UserOrderClause: string;
var
  i: Integer;
begin
  Result := '';
  CheckQueryInfo;
  with FQueryInfo do
    for i := 0 to Ordering.Count-1 do
      Result := Result + Ordering[i].UserOrderStr + ', ';
  if Result <> '' then begin
    System.Delete(Result, Length(Result) -1, 2);
    if FPrefix then Result := 'Order by ' + Result;
  end
  else if FPrefix then Result := 'In default order';
end;

procedure TQIFormatter.BuildSQLQuery(Strings: TStrings);
var
  s: string;
  q, p, b: Boolean;
begin
  CheckQueryInfo;
  q := FQualify;
  p := FPrefix;
  with Strings do
  try
    BeginUpdate;
    Clear;
    FPrefix := True;
    FQualify := FQueryInfo.TablesUsedCount > 1;
    s := SQLSelectClause;
    if s = '' then Exit;
    Add(s);
    Add(SQLFromClause);
    s := SQLJoinClause;
    b := s <> '';
    if b then Add(s);
    s := SQLFilterClause;
    if b and (s <> '') then begin    {join clause included }
      System.Delete(s, 1, 5);        {so make WHERE into AND -}
      s := 'AND' + s;
    end;
    if s <> '' then Add(s);
    s := SQLGroupClause;
    if s <> '' then Add(s);
    s := SQLOrderClause;
    if s <> '' then Add(s);
  finally
    EndUpdate;
    FPrefix := p;
    FQualify := q;
  end;
end;

procedure TQIFormatter.BuildUserQuery(Strings: TStrings);
var
  s: string;
  q, p: Boolean;
begin
  CheckQueryInfo;
  q := FQualify;
  p := FPrefix;
  with Strings do
  try
    BeginUpdate;
    Clear;
    FPrefix := True;
    FQualify := FQueryInfo.TablesUsedCount > 1;
    s := UserSelectClause;
    if s = '' then Exit;
    Add(s);
    Add(UserFromClause);
    Add(UserFilterClause);
    s := UserGroupClause;
    if s <> '' then Add(s);
    Add(UserOrderClause);
  finally
    EndUpDate;
    FPrefix := p;
    FQualify := q;
  end;
end;

end.
