(******************************************************************************
*                                   hierObj                                   *
******************************************************************************)
unit hierObj;

interface

uses
   classes
   ,sysUtils
   ;

type
   TObjectInHierarchy = class;

   EHierarchyException = class(Exception)

      private
         FProblemObject : TObjectInHierarchy;

      protected

      public
         constructor create(const msg : string; TheObject : TObjectInHierarchy);

         property ProblemObject : TObjectInHierarchy read FProblemObject;
   end; { EHierarchyException class definition }

   THierarchyViewerInterface = class(TComponent)

   private

   protected

   public
      function SelectedObject : TObjectInHierarchy; virtual; abstract;
      function RootObject : TObjectInHierarchy; virtual; abstract;
      procedure AddLastChildToSelected(AChild : TObjectInHierarchy); virtual; abstract;
      procedure AddSiblingToSelected(ASibling : TObjectInHierarchy); virtual; abstract;
      procedure updateSelectedTitle; virtual; abstract;

   end; { THierarchyViewerInterface }

   TObjectInHierarchy = class(TComponent)

      private

         FObjectLevel : integer;
         FParentNode : TObjectInHierarchy; { the parent we belong to }
         FChildNodes : TStringList; { a list of objects and their ID's }

         FOnHierarchyChanged : TNotifyEvent;

         FViewerInterface : THierarchyViewerInterface; { usually only the root will need this }
         FUpdateLevel : integer;

      protected

         function GetCanDemote : boolean; virtual; { can it be a child of sibling ? }
         function GetCanPromote : boolean; virtual; { can it be promoted ? }
         function GetCanMoveBeforeSibling : boolean; virtual; { can it move up ? }
         function GetCanMoveAfterSibling : boolean; virtual; { can it move down ? }
         function GetCanDelete : boolean; virtual; { can this object be deleted from the hierarchy ? }

         function AcceptAsChild(ANode : TObjectInHierarchy) : boolean; virtual;
         function AcceptSiblingBefore(Anode : TObjectInHierarchy) : boolean; virtual;
         function AcceptSiblingAfter(ANode : TObjectInHierarchy) : boolean; virtual;

         procedure ChangeObjectLevelBy(delta : integer); virtual; 
            { change the object's level, and all of it's childs recursivly }
         procedure hierarchyChanged; virtual; { call this after the hierarchy was changed }
         procedure notifyHierarchyChange(var handled : boolean); virtual;

         function ChildIndexInHierList(ANode : TObjectInHierarchy) : integer; virtual;
         function GetIndexInParentList : integer;

         function GetDisplayValue : String; virtual; abstract;
         procedure SetObjectLevel(l : integer);
         function GetChildsCount : integer;
         
         function GetVerbCount : integer; virtual;
         function GetRootVerbCount : integer; virtual;

         function GetRoot : TObjectInHierarchy; virtual;
         function GetNextSibling : TObjectInHierarchy; virtual;
         function GetPrevSibling : TObjectInHierarchy; virtual;

         function GetViewerInterface : THierarchyViewerInterface; virtual;


         property UpdateLevel : integer read FUpdateLevel write FUpdateLevel;
         property ObjectLevel : integer read FObjectLevel write SetObjectLevel;

      public
         constructor create(AOwner : TComponent); override;
         destructor destroy; override;

         procedure Demote;
         procedure Promote;
         procedure MoveBeforeSibling;
         procedure MoveAfterSibling;

         function ChildObjectAt(idx : integer) : TObjectInHierarchy;
         procedure AddLastChild(ANode : TObjectInHierarchy);
         procedure AddFirstChild(ANode : TObjectInHierarchy);
         procedure AddSiblingAfter(ANode : TObjectInHierarchy);
         procedure AddSiblingBefore(ANode : TObjectInHierarchy);
         procedure InsertChild(ANode : TObjectInHierarchy; idx : integer);
         procedure removeChild(ANode : TObjectInHierarchy);

         procedure beginUpdate;  { do not call OnHierarchyCahnged too many times }
         procedure endUpdate;

         function VerbText(idx : integer) : string; virtual; abstract;
         function RootVerbText(idx : integer) : string; virtual; abstract;
         procedure ExecuteVerb(idx : integer); virtual; abstract;
         procedure ExecuteRootVerb(idx : integer); virtual; abstract;

         procedure WriteToStream(Stream : TStream); virtual; 
         procedure WriteProperties(Stream : TStream); virtual; { write class properties }
         procedure WriteFields(Stream : TStream); virtual; { write object instance fields  - see TObjectWithFields }
         procedure WriteChilds(Stream : TStream); virtual; { write the object's childs }

         procedure ReadFromStream(Stream : TStream); virtual;
         procedure ReadProperties(Stream : TStream); virtual;
         procedure ReadFields(Stream : TStream); virtual;
         procedure ReadChilds(Stream : TStream); virtual;

         procedure SaveToFile(const fileName : string);
         procedure LoadFromFile(const fileName : String);

         function CreateChild(const ChildType : String) : TObjectInHierarchy; virtual; abstract;
            { given a child name, return an instance of the child - 
               should be implemented only by objects that can be a root object }
         procedure ClearHierarchy; virtual;

         property DistanceFromRoot : integer read FObjectLevel;
         property Root : TObjectInHierarchy read GetRoot;
         property NextSibling : TObjectInHierarchy read GetNextSibling;
         property PrevSibling : TObjectInHierarchy read GetPrevSibling;

         property CanDemote : boolean read GetCanDemote;
         property CanPromote : boolean read GetCanPromote;
         property CanMoveBeforeSibling : boolean read GetCanMoveBeforeSibling;
         property CanMoveAfterSibling : boolean read GetCanMoveAfterSibling;
         property CanDelete : boolean read GetCanDelete;

         property ParentNode : TObjectInHierarchy read FParentNode;
         property ChildNodes : TStringList read FChildNodes;
         property ChildsCount : integer read GetChildsCount;
         property IndexInParentList : integer read GetIndexInParentList;

         property DisplayValue : String read GetDisplayValue; { the value that will be displayed in the UI for this object }

         property VerbCount : integer read GetVerbCount;
         property RootVerbCount : integer read GetRootVerbCount; { verbs to display if object is the root of view }

         property OnHierarchyChanged : TNotifyEvent read FOnHierarchyChanged write FOnHierarchyChanged;
         property ViewerInterface : THierarchyViewerInterface read GetViewerInterface
                                       write FViewerInterface;

   end; { TObjectInHierarchy class definition }

implementation

uses
   StreamUt
   ;

(******************************************************************************
*                         EHierarchyException.create                          *
******************************************************************************)
constructor EHierarchyException.create;
begin
   inherited create(msg);
   FProblemObject := TheObject;
end; { EHierarchyException.create }

(******************************************************************************
*                          TObjectInHierarchy.create                          *
******************************************************************************)
constructor TObjectInHierarchy.create;
begin
   inherited create(AOwner);
   FChildNodes := TStringList.create;
   FObjectLevel := 0; { root by default }
end; { TObjectInHierarchy.create }

(******************************************************************************
*                         TObjectInHierarchy.destroy                          *
******************************************************************************)
destructor TObjectInHierarchy.destroy;
var
   ANode : TObjectInHierarchy;
   i : integer;
begin
   for i := 0 to FChildNodes.count - 1 do begin
      Anode := FChildNodes.objects[i] as TObjectInHierarchy;
      ANode.FParentNode := nil;
   end;
   if (assigned(FParentNode)) then
      FParentNode.removeChild(self);
   FChildNodes.free;
   inherited destroy;
end; { TObjectInHierarchy.destroy }

(******************************************************************************
*                      TObjectInHierarchy.GetCanPromote                       *
******************************************************************************)
function TObjectInHierarchy.GetCanPromote;
var
   newParent : TObjectInHierarchy;
begin
   if (objectLevel > 1) then begin
      newParent := ParentNode.ParentNode; { we share the parent now }
      result := newParent.AcceptAsChild(self);
   end else
      result := false; 
        { notice that at level 1 you can not promote - there is only one root }
end; { TObjectInHierarchy.GetCanPromote }

(******************************************************************************
*                       TObjectInHierarchy.GetCanDemote                       *
******************************************************************************)
function TObjectInHierarchy.GetCanDemote;
var
   NewParent : TObjectInHierarchy;
begin
   result := false;
   if (not assigned(FParentNode)) then
      exit; { can not demote a root }
   if (IndexInParentList > 0) then begin
      NewParent := ParentNode.ChildObjectAt(indexInParentList - 1);
      result := NewParent.AcceptAsChild(self); 
         { we have a sibling before us that can become our new parent node, ask him if he agrees }
   end;
end; { TObjectInHierarchy.GetCanDemote }

(******************************************************************************
*                       TObjectInHierarchy.GetCanDelete                       *
******************************************************************************)
function TObjectInHierarchy.GetCanDelete;
begin
   result := (ObjectLevel <> 0); { can not delete the root, but other than that.. }
end; { TObjectInHierarchy.GetCanDelete }

(******************************************************************************
*                   TObjectInHierarchy.ChildIndexInHierList                   *
******************************************************************************)
function TObjectInHierarchy.ChildIndexInHierList;
var
   i : integer;
begin
   result := -1;
   for i := 0 to FChildNodes.count - 1 do
      if (FChildNodes.objects[i] = ANode) then begin
         result := i;
         exit;
      end;
end; { TObjectInHierarchy.IndexInHierList }

(******************************************************************************
*                 TObjectInHierarchy.GetCanMoveBeforeSibling                  *
******************************************************************************)
function TObjectInHierarchy.GetCanMoveBeforeSibling;
var
   siblingBefore : TObjectInHierarchy;
begin
   result := false;
   if (not assigned(ParentNode)) then
      exit; { no parent, no siblings, can't be moved }
   if (IndexInParentList > 0) then begin
      siblingBefore := TObjectInHierarchy(ParentNode.childNodes.objects[indexInParentList - 1]);
      result := siblingBefore.AcceptSiblingBefore(self); 
       { we have a sibling before us that we can move before, ask for his permission }
   end;
end; { TObjectInHierarchy.GetCanMoveBeforeSibling }

(******************************************************************************
*                  TObjectInHierarchy.GetCanMoveAfterSibling                  *
******************************************************************************)
function TObjectInHierarchy.GetCanMoveAfterSibling;
var
   siblingAfter : TObjectInHierarchy;
begin
   result := false;
   if (not assigned(ParentNode)) then
      exit; { no parent, no siblings, can't be moved }
   if (indexInParentList < ParentNode.ChildsCount - 1) then begin
      siblingAfter := TObjectInHierarchy(ParentNode.childNodes.objects[indexInParentList + 1]);
      result := siblingAfter.AcceptSiblingAfter(self); 
       { there is a sibling after us, we can move if he allows us.. }
   end; 
end; { TObjectInHierarchy.GetCanMoveAfterSibling }

(******************************************************************************
*                   TObjectInHierarchy.ChangeObjectLevelBy                    *
******************************************************************************)
procedure TObjectInHierarchy.ChangeObjectLevelBy;
var
   i : integer;
begin
   for i := 0 to ChildNodes.count - 1 do
      TObjectInHierarchy(ChildNodes[i]).ChangeObjectLevelBy(delta);
   FObjectLevel := FObjectLevel + delta;
end; { TObjectInHierarchy.ChangeObjectLevelBy }

(******************************************************************************
*                         TObjectInHierarchy.Promote                          *
******************************************************************************)
procedure TObjectInHierarchy.Promote;
var
   newParent : TObjectInHierarchy;
   insertAt : integer;
begin
   if (not CanPromote) then
      raise EHierarchyException.create('Can not promote object', self);
   { .. do it }
   beginUpdate;
   try
      newParent := ParentNode.ParentNode; { we share the parent now }
      insertAt := ParentNode.IndexInParentList + 1; 
      { we want to appear right after the current parent }
      ParentNode.removeChild(self);
      NewParent.InsertChild(self, insertAt);
   finally
      endUpdate;
   end;
   hierarchyChanged;
end; { TObjectInHierarchy.Promote }

(******************************************************************************
*                          TObjectInHierarchy.Demote                          *
******************************************************************************)
procedure TObjectInHierarchy.Demote;
var
   NewParent : TObjectInHierarchy;
begin
   if (not CanDemote) then
      raise EHierarchyException.create('Can not demote object', self);
   { .. do it }
   beginUpdate;
   try
      NewParent := ParentNode.ChildObjectAt(indexInParentList - 1);
      ParentNode.removeChild(self);
      NewParent.AddLastChild(self);
   finally;
      endUpdate;
   end;
   hierarchyChanged;
end; { TObjectInHierarchy.Demote }

(******************************************************************************
*                    TObjectInHierarchy.MoveBeforeSibling                     *
******************************************************************************)
procedure TObjectInHierarchy.MoveBeforeSibling;
var
   myIndex : integer;
begin
   if (not CanMoveBeforeSibling) then
      raise EHierarchyException.create('Can not move object before sibling', self);
   { .. do it }
   beginUpdate;
   try
      myIndex := indexInParentList;
      ParentNode.childNodes.exchange(myIndex, myIndex - 1);
   finally
      endUpdate;
   end;
   hierarchyChanged;
end; { TObjectInHierarchy.MoveBeforeSibling }

(******************************************************************************
*                     TObjectInHierarchy.MoveAfterSibling                     *
******************************************************************************)
procedure TObjectInHierarchy.MoveAfterSibling;
var
   myIndex : integer;
begin
   if (not CanMoveAfterSibling) then
      raise EHierarchyException.create('Can not move object after sibling', self);
   { .. do it }
   beginUpdate;
   try
      myIndex := indexInParentList;
      ParentNode.childNodes.exchange(myIndex, myIndex + 1);
   finally
      endUpdate;
   end;
   hierarchyChanged;
end; { TObjectInHierarchy.MoveAfterSibling }

(******************************************************************************
*                     TObjectInHierarchy.hierarchyChanged                     *
******************************************************************************)
procedure TObjectInHierarchy.hierarchyChanged;
var
   handled : boolean;
begin
   handled := false;
   if (UpdateLevel = 0) then
      notifyHierarchyChange(handled);
end; { TObjectInHierarchy.hierarchyChanged }

(******************************************************************************
*                  TObjectInHierarchy.notifyHierarchyChange                   *
******************************************************************************)
procedure TObjectInHierarchy.notifyHierarchyChange(var handled : boolean);
begin
   if (updateLevel > 0) then
      exit; { we are in a beginUpdate .. endUpdate block, do not fire event }
   if (assigned(FParentNode)) then begin
      FParentNode.notifyHierarchyChange(handled);
      if ((not handled) and (assigned(FOnHierarchyChanged))) then begin
         FOnHierarchyChanged(self);
         handled := true;
      end;
   end else begin
      if (assigned(FOnHierarchyChanged)) then begin
         FOnHierarchyChanged(self);
         handled := true;
      end;
   end;
end; { TObjectInHierarchy.notifyHierarchyChange }

(******************************************************************************
*                   TObjectInHierarchy.GetIndexInParentList                   *
******************************************************************************)
function TObjectInHierarchy.GetIndexInParentList;
begin
   if (not assigned(FParentNode)) then
      result := -1
   else
      result := ParentNode.ChildIndexInHierList(self);
end; { TObjectInHierarchy.GetIndexInParentList }

(******************************************************************************
*                      TObjectInHierarchy.ChildObjectAt                       *
******************************************************************************)
function TObjectInHierarchy.ChildObjectAt;
begin
   result := TObjectInHierarchy(FChildNodes.objects[idx]);
end; { TObjectInHierarchy.childObjectAt }

(******************************************************************************
*                      TObjectInHierarchy.SetObjectLevel                      *
******************************************************************************)
procedure TObjectInHierarchy.SetObjectLevel;
var
   i : integer;
begin
   beginUpdate;
   try
      FObjectLevel := l;
      for i := 0 to FChildNodes.count - 1 do
         TObjectInHierarchy(FChildNodes[i]).ObjectLevel := l + 1;
   finally
      endUpdate;
   end;
   hierarchyChanged;
end; { TObjectInHierarchy.SetObjectLevel }

(******************************************************************************
*                       TObjectInHierarchy.AddLastChild                       *
******************************************************************************)
procedure TObjectInHierarchy.AddLastChild;
begin
   beginUpdate;
   try
      FChildNodes.addObject(ANode.DisplayValue, ANode);
      ANode.ObjectLevel := 1 + ObjectLevel;
      ANode.FParentNode := self;
   finally
      endUpdate;
   end; { finally }
   hierarchyChanged;
end; { TObjectInHierarchy.AddLastChild }

(******************************************************************************
*                      TObjectInHierarchy.AddFirstChild                       *
******************************************************************************)
procedure TObjectInHierarchy.AddFirstChild;
begin
   beginUpdate;
   try
      insertChild(ANode, 0);
   finally
      endUpdate;
   end;
   hierarchyChanged;
end; { TObjectInHierarchy.AddFirstChild }

(******************************************************************************
*                     TObjectInHierarchy.AddSiblingAfter                      *
******************************************************************************)
procedure TObjectInHierarchy.AddSiblingAfter;
var
   NewIndex : integer;
begin
     if (assigned(ParentNode)) then begin
        NewIndex := 1 + IndexInParentList;
        ParentNode.InsertChild(ANode, NewIndex);
     end;
end; { TObjectInHierarchy.AddSiblingAfter }

(******************************************************************************
*                     TObjectInHierarchy.AddSiblingBefore                     *
******************************************************************************)
procedure TObjectInHierarchy.AddSiblingBefore;
begin
     if (assigned(ParentNode)) then begin
        ParentNode.InsertChild(ANode, IndexInParentList);
     end;
end; { TObjectInHierarchy.AddSiblingBefore }

(******************************************************************************
*                       TObjectInHierarchy.insertChild                        *
******************************************************************************)
procedure TObjectInHierarchy.insertChild;
begin
   beginUpdate;
   try
      FChildNodes.InsertObject(idx, ANode.displayValue, ANode);
      ANode.ObjectLevel := 1 + ObjectLevel;
      ANode.FParentNode := self;
   finally
      endUpdate;
   end;
   hierarchyChanged;
end; { TObjectInHierarchy.insertChild }

(******************************************************************************
*                       TObjectInHierarchy.RemoveChild                        *
******************************************************************************)
procedure TObjectInHierarchy.RemoveChild;
var
   idx : integer;
begin
   beginUpdate;
   try
      idx := childIndexInHierList(ANode);
      ANode.FParentNode := nil;
      if (idx <> -1) then begin
         FChildNodes.delete(idx);
      end;
   finally
      endUpdate;
   end;
   hierarchyChanged;
end; { TObjectInHierarchy.RemoveChild }

(******************************************************************************
*                       TObjectInHierarchy.beginUpdate                        *
******************************************************************************)
procedure TObjectInHierarchy.beginUpdate;
begin
   inc(FUpdateLevel);
end; { TObjectInHierarchy.beginUpdate }

(******************************************************************************
*                        TObjectInHierarchy.endUpdate                         *
******************************************************************************)
procedure TObjectInHierarchy.endUpdate;
begin
   dec(FUpdateLevel);
end; { TObjectInHierarchy.endUpdate }

(******************************************************************************
*                     TObjectInHierarchy.AcceptAsChild                        *
* Descendents can refuse to accept childs based on type or whatever           *
******************************************************************************)
function TObjectInHierarchy.AcceptAsChild;
begin
   result := true;
end; { TObjectInHierarchy.AcceptAsChild }

(******************************************************************************
*                   TObjectInHierarchy.AcceptSiblingBefore                    *
******************************************************************************)
function TObjectInHierarchy.AcceptSiblingBefore;
begin
   result := true;
end; { TObjectInHierarchy.AcceptSiblingBefore }

(******************************************************************************
*                    TObjectInHierarchy.AcceptSiblingAfter                    *
******************************************************************************)
function TObjectInHierarchy.AcceptSiblingAfter;
begin
   result := true;
end; { TObjectInHierarchy.AcceptSiblingAfter }

(******************************************************************************
*                      TObjectInHierarchy.GetChildsCount                      *
******************************************************************************)
function TObjectInHierarchy.GetChildsCount;
begin
   result := childNodes.count;
end; { TObjectInHierarchy.GetChildsCount }

(******************************************************************************
*                       TObjectInHierarchy.GetVerbCount                       *
******************************************************************************)
function TObjectInHierarchy.GetVerbCount;
begin
     result := 0; { by default no verbs .. }
end; { GetVerbCount }

(******************************************************************************
*                     TObjectInHierarchy.GetRootVerbCount                     *
******************************************************************************)
function TObjectInHierarchy.GetRootVerbCount;
begin
     result := 0;
end; { GetRootVerbCount }

(******************************************************************************
*                         TObjectInHierarchy.GetRoot                          *
******************************************************************************)
function TObjectInHierarchy.GetRoot;
begin
   if (assigned(ParentNode)) then
      result := ParentNode.Root
   else
      result := self;
end; { TObjectInHierarchy.GetRoot }

function TObjectInHierarchy.GetViewerInterface;
begin
     result := root.FViewerInterface;
end; { TObjectInHierarchy.GetViewerInterface }

(******************************************************************************
*                      TObjectInHierarchy.WriteToStream                       *
******************************************************************************)
procedure TObjectInHierarchy.WriteToStream;
begin
   WriteStrToStream(Stream, ClassName);
   writeProperties(Stream);
   writeFields(Stream);
   writeChilds(Stream);
end; { TObjectInHierarchy.WriteToStream }

(******************************************************************************
*                     TObjectInHierarchy.WriteProperties                      *
******************************************************************************)
procedure TObjectInHierarchy.WriteProperties;
begin
   { empty, descendents might want to write some of the properties }
end; { TObjectInHierarchy.WriteProperties }

(******************************************************************************
*                       TObjectInHierarchy.WriteFields                        *
******************************************************************************)
procedure TObjectInHierarchy.WriteFields;
begin
   { empty, descendents might want to write the fields - see TObjectWithFields }
end; { TObjectInHierarchy.WriteFields }

(******************************************************************************
*                       TObjectInHierarchy.WriteChilds                        *
******************************************************************************)
procedure TObjectInHierarchy.WriteChilds;
var
   i : integer;
begin
   WriteIntToStream(Stream, ChildNodes.count);
   for i := 0 to ChildNodes.count - 1 do
      TObjectInHierarchy(ChildNodes.objects[i]).WriteToStream(Stream);
end; { TObjectInHierarchy.WriteChilds }

(******************************************************************************
*                     TObnjectInHierarchy.ReadFromStream                      *
******************************************************************************)
procedure TObjectInHierarchy.ReadFromStream;
begin
   if (Root = Self) then begin
      ReadStrFromStream(Stream); { skip the class name if it is the root object! }
   end;
   ReadProperties(Stream); 
   ReadFields(Stream);
   ReadChilds(Stream);
end; { TObnjectInHierarchy.ReadFromStream }

(******************************************************************************
*                      TObjectInHierarchy.ReadProperties                      *
******************************************************************************)
procedure TObjectInHierarchy.ReadProperties;
begin
   { descendents will read actual properties if any, and call inherited }
end; { TObjectInHierarchy.ReadProperties }

(******************************************************************************
*                        TObjectInHierarchy.ReadFields                        *
******************************************************************************)
procedure TObjectInHierarchy.ReadFields;
begin
   { descendents will read actual fields if any, and call inherited }
end; { TObjectInHierarchy.ReadFields }

(******************************************************************************
*                        TObjectInHierarchy.ReadChilds                        *
******************************************************************************)
procedure TObjectInHierarchy.ReadChilds;
var
   i, count : integer;
   AChild : TObjectInHierarchy;
   ChildType : String;
begin
   count := readIntFromStream(Stream);
   for i := 0 to count - 1 do begin
      childType := ReadStrFromStream(Stream); { read the object's type - classname }
      AChild := Root.CreateChild(ChildType);
      addLastChild(AChild);
      AChild.ReadFromStream(Stream);
   end;
end; { TObjectInHierarchy.ReadChilds }

(******************************************************************************
*                        TObjectInHierarchy.SaveToFile                        *
******************************************************************************)
procedure TObjectInHierarchy.SaveToFile;
var
   f : TFileStream;
begin
   f := TFileStream.create(fileName, fmCreate or fmOpenWrite);
   try
      writeToStream(f);
   finally
      f.free;
   end;
end; { TObjectInHierarchy.SaveToFile }

(******************************************************************************
*                       TObjectInHierarchy.loadFromFile                       *
******************************************************************************)
procedure TObjectInHierarchy.loadFromFile;
var
   f : TFileStream;
begin
   f := TFileStream.create(fileName, fmOpenRead);
   try
      clearHierarchy;
      readFromStream(f);
   finally
      f.free;
   end;
end; { TObjectInHierarchy.loadFromFile }

(******************************************************************************
*                      TObjectInHierarchy.ClearHierarchy                      *
******************************************************************************)
procedure TObjectInHierarchy.ClearHierarchy;
var
   i : integer;
   ANode : TObjectInHierarchy;
begin
   while (childNodes.count <> 0) do begin
      ANode := TObjectInHierarchy(childNodes.objects[0]);
      ANode.clearHierarchy;
      removeChild(ANode);
   end;
end; { TObjectInHierarchy.ClearHierarchy }

(******************************************************************************
*                      TObjectInHierarchy.GetNextSibling                      *
******************************************************************************)
function TObjectInHierarchy.GetNextSibling;
begin
   result := nil;
   if (not assigned(ParentNode)) then
      exit; { no parent, no siblings }
   if (indexInParentList < ParentNode.ChildsCount - 1) then begin
      result := TObjectInHierarchy(ParentNode.childNodes.objects[indexInParentList + 1]);
   end; 
end; { TObjectInHierarchy.GetNextSibling }

(******************************************************************************
*                      TObjectInHierarchy.GetPrevSibling                      *
******************************************************************************)
function TObjectInHierarchy.GetPrevSibling;
begin
   result := nil;
   if (not assigned(ParentNode)) then
      exit; { no parent, no siblings }
   if (IndexInParentList > 0) then begin
      result := TObjectInHierarchy(ParentNode.childNodes.objects[indexInParentList - 1]);
   end;
end; { TObjectInHierarchy.GetPrevSibling }

(******************************************************************************
*                                    end.                                     *
******************************************************************************)
end.
