%
% "moonup.t" computes the times of moonrise and moonset 
% anywhere in the world.
%
% Adapted from a BASIC program from SKY & TELESCOPE, July, 1989, page 78
%
% Enter north latitudes positive, west longitudes negative.  
%
% For the time zone, enter the number of hours west of Greenwich 
% (e.g., 5 for EST, 4 for EDT).  
%
%   Sample program for the T Interpreter by:
%
%   Stephen R. Schmitt
%   962 Depot Road
%   Boxborough, MA 01719
%

const PI : real := 2 * arcsin( 1 )
const DR : real := PI / 180
const K1 : real := 15 * DR * 1.0027379

var Moonrise, Moonset : boolean := false

type rvector : array[3] of real

program

    var k, zone : int
    var jd, lat, lon, ph, t0, tz : real
    var ra0, dec0, plx0 : real
    var mp : array[3, 3] of real
    var ra, dec, v : rvector
    
    prompt "Latitude, +N, -S (deg):"
    get lat
    prompt "Longitude, +E, -W (deg):"
    get lon
    prompt "Time zone +West of GMT, -East (hrs):"
    get zone

    jd := julian_day - 2451545  % Julian day relative to Jan 1.5, 2000
    show_input( lat, lon, zone )
    
    lon := lon / 360
    tz := zone / 24

    t0 := lst( lon, jd, tz )
    jd := jd + tz

    for k := 0...2 do

        moon( ra0, dec0, plx0, jd )
        mp[k,0] := ra0
        mp[k,1] := dec0
        mp[k,2] := plx0
        jd := jd + 0.5        
        
    end for

    if mp[1,0] <= mp[0,0] then
        mp[1,0] := mp[1,0] + 2 * PI
    end if

    if mp[2,0] <= mp[1,0] then
        mp[2,0] := mp[2,0] + 2 * PI
    end if

    ra[0]  := mp[0,0]
    dec[0] := mp[0,1]

    put ""

    for k := 0...23 do

        ph := ( k + 1 ) / 24
        
        ra[2]  := interpolate( mp[0,0], mp[1,0], mp[2,0], ph )
        dec[2] := interpolate( mp[0,1], mp[1,1], mp[2,1], ph )
        
        test( k, zone, t0, lat, mp[1,2], ra, dec, v )

        ra[0]  := ra[2]
        dec[0] := dec[2]
    
    end for

    if ( not Moonrise ) and ( not Moonset ) then
    
        if v[2] < 0 then
            put "Moon down all day"
        else
            put "Moon up all day"
        end if

    else

        if not Moonrise then
            put "No moonrise this date"
        elsif not Moonset then
            put "No moonset this date"
        end if

    end if
    
end program

%
% display latitude, longitude and time zone
%
procedure show_input( lat, lon : real, zone : int )

    var deg, min : int
    
    if sgn( zone ) = sgn( lon ) and zone ~= 0 then
        put "WARNING: time zone and longitude are incompatible"
    end if

    if abs( zone ) > 12 then
        put "WARNING: invalid time zone"
    end if        

    if abs( lat ) > 90 then
        put "WARNING: invalid latitude"
    end if

    if abs( lon ) > 180 then
        put "WARNING: invalid longitude"
    end if

    if lat > 0.0 then
        deg := floor( lat )
        min := floor( 60 * ( lat - deg ) )
        put deg:4, ":",min:2, " N "...
    else
        deg := floor( -lat )
        min := floor( 60 * ( -lat - deg ) )
        put deg:4, ":",min:2, " S "...
    end if

    if lon > 0.0 then
        deg := floor( lon )
        min := floor( 60 * ( lon - deg ) )
        put deg:4, ":",min:2, " E"
    else
        deg := floor( -lon )
        min := floor( 60 * ( -lon - deg ) )
        put deg:4, ":",min:2, " W"
    end if
    
end procedure

%
% 3-point interpolation
%
function interpolate( f0, f1, f2, p : real ) : real

    var a, b, f : real
    
    a := f1 - f0
    b := f2 - f1 - a
    f := f0 + p * ( 2 * a + b * ( 2 * p - 1 ) )

    return f

end function

%
% LST at 0h zone time
%
function lst( lon, jd, z : real ) : real

    var s, t0 : real

    s := 24110.5 + 8640184.812999999 * jd / 36525
    s := s + 86636.6 * z + 86400 * lon
    
    s := s / 86400
    s := s - floor( s )

    t0 := s * 360 * DR

    return t0
    
end function

%
% test an hour for an event
%
procedure test( k, zone : int, 
                t0, lat, plx : real,
                var ra, dec, v : rvector )

    var ha : rvector
    var a, b, c, d, e, s, z : real
    var hr, min, time : real
    var az, hz, nz, dz : real
    var zchar : string
    var zlet : string := "MLKIHGFEDCBAZNOPQRSTUVWXY"
    label test_exit :

    if ra[2] < ra[0] then
        ra[2] := ra[2] + 2 * PI
    end if
    
    ha[0] := t0 - ra[0] + k * K1
    ha[2] := t0 - ra[2] + k * K1 + K1
    
    ha[1]  := ( ha[2] + ha[0] ) / 2         % hour angle at half hour
    dec[1] := ( dec[2] + dec[0] ) / 2       % declination at half hour

    s := sin( DR * lat )
    c := cos( DR * lat )
    z := cos( DR * ( 90.567 - 41.685 / plx ) )

    if k <= 0 then
        v[0] := s * sin( dec[0] ) + c * cos( dec[0] ) * cos( ha[0] ) - z
    end if

    v[2] := s * sin( dec[2] ) + c * cos( dec[2] ) * cos( ha[2] ) - z
    
    if sgn( v[0] ) = sgn( v[2] ) then
        goto test_exit
    end if
    
    v[1] := s * sin( dec[1] ) + c * cos( dec[1] ) * cos( ha[1] ) - z

    a := 2 * v[2] - 4 * v[1] + 2 * v[0]
    b := 4 * v[1] - 3 * v[0] - v[2]
    d := b * b - 4 * a * v[0]

    if d < 0 then
        goto test_exit
    end if
    
    d := sqrt( d )
    
    if v[0] < 0 and v[2] > 0 then
        put "Moonrise at "...
        Moonrise := true
    end if

    if v[0] > 0 and v[2] < 0 then
        put "Moonset at  "...
        Moonset := true
    end if

    e := ( -b + d ) / ( 2 * a )

    if e > 1 or e < 0 then
        e := ( -b - d ) / ( 2 * a )
    end if

    time := k + e + 1 / 120       % round off

    hr := floor( time )
    min := floor( ( time - hr ) * 60 )

    zchar := "    "
    if abs( zone ) <= 12 then
        zchar[1] := zlet[zone+12]
    end if

    put hr:2, ":", min:2, zchar...

    % azimuth of the moon at the event
    
    hz := ha[0] + e * ( ha[2] - ha[0] )
    nz := -cos( dec[1] ) * sin( hz )
    dz := c * sin( dec[1] ) - s * cos( dec[1] ) * cos( hz )
    az := arctan( nz / dz ) / DR

    if dz < 0 then
        az := az + 180
    end if
    
    if az < 0 then
        az := az + 360
    end if

    if az > 360 then
        az := az - 360
    end if

    put "azimuth: ", az:5:1

test_exit :

    v[0] := v[2]

end procedure

% 
% moon's position using fundamental arguments
%
procedure moon( var ra, dec, plx : real, jd : real )

    var d, f, g, h, m, n, s, u, v, w : real 

    h := 0.606434 + 0.03660110129 * jd
    m := 0.374897 + 0.03629164709 * jd
    f := 0.259091 + 0.0367481952  * jd
    d := 0.827362 + 0.03386319198 * jd
    n := 0.347343 - 0.00014709391 * jd
    g := 0.993126 + 0.0027377785  * jd

    h := h - floor( h )
    m := m - floor( m )
    f := f - floor( f )
    d := d - floor( d )
    n := n - floor( n )
    g := g - floor( g )

    h := h * 2 * PI
    m := m * 2 * PI
    f := f * 2 * PI
    d := d * 2 * PI
    n := n * 2 * PI
    g := g * 2 * PI

    v := 0.39558 * sin( f + n )
    v := v + 0.082   * sin( f )
    v := v + 0.03257 * sin( m - f - n )
    v := v + 0.01092 * sin( m + f + n )
    v := v + 0.00666 * sin( m - f )
    v := v - 0.00644 * sin( m + f - 2 * d + n )
    v := v - 0.00331 * sin( f - 2 * d + n )
    v := v - 0.00304 * sin( f - 2 * d )
    v := v - 0.0024  * sin( m - f - 2 * d - n )
    v := v + 0.00226 * sin( m + f )
    v := v - 0.00108 * sin( m + f - 2 * d )
    v := v - 0.00079 * sin( f - n )
    v := v + 0.00078 * sin( f + 2 * d + n )
    
    u := 1 - 0.10828 * cos( m )
    u := u - 0.0188  * cos( m - 2 * d )
    u := u - 0.01479 * cos( 2 * d )
    u := u + 0.00181 * cos( 2 * m - 2 * d )
    u := u - 0.00147 * cos( 2 * m )
    u := u - 0.00105 * cos( 2 * d - g )
    u := u - 0.00075 * cos( m - 2 * d + g )
    
    w := 0.10478 * sin( m )
    w := w - 0.04105 * sin( 2 * f + 2 * n )
    w := w - 0.0213  * sin( m - 2 * d )
    w := w - 0.01779 * sin( 2 * f + n )
    w := w + 0.01774 * sin( n )
    w := w + 0.00987 * sin( 2 * d )
    w := w - 0.00338 * sin( m - 2 * f - 2 * n )
    w := w - 0.00309 * sin( g )
    w := w - 0.0019  * sin( 2 * f )
    w := w - 0.00144 * sin( m + n )
    w := w - 0.00144 * sin( m - 2 * f - n )
    w := w - 0.00113 * sin( m + 2 * f + 2 * n )
    w := w - 0.00094 * sin( m - 2 * d + g )
    w := w - 0.00092 * sin( 2 * m - 2 * d )

    %    compute right ascension, declination, and parallax
    s  := w / sqrt( u - v * v )
    ra := h + arctan( s / sqrt( 1 - s * s ) )

    s   := v / sqrt( u )
    dec := arctan( s / sqrt( 1 - s * s ) )

    plx := 60.40974 * sqrt( u )

end procedure

%
% determine Julian day from calendar date
% ref: Jean Meeus, "Astronomical Algorithms", Willmann-Bell, 1991
%
function julian_day : real

    var a, b, day, jd : real
    var month, year : int
    var gregorian : boolean

    prompt "Year:"
    get year
    prompt "Month:"
    get month
    prompt "Day:"
    get day

    date( floor( day ), month, year )
    
    if year < 1583 then
        gregorian := false
    else
        gregorian := true
    end if
    
    if month = 1 or month = 2 then
        year := year - 1
        month := month + 12
    end if

    a := floor( year / 100 )
    if gregorian then
        b := 2 - a + floor( a / 4 )
    else
        b := 0.0
    end if

    jd := floor( 365.25 * ( year + 4716 ) ) + 
          floor( 30.6001 * ( month + 1 ) ) + 
          day + b - 1524.5
    
    return jd
    
end function

%
% display the date
%
procedure date( d, m, y : int )

    var day, month : array[12] of string
    var mi, yi, i : int

    month[0]  := "January"
    month[1]  := "February"
    month[2]  := "March"
    month[3]  := "April"
    month[4]  := "May"
    month[5]  := "June"
    month[6]  := "July"
    month[7]  := "August"
    month[8]  := "September"
    month[9]  := "October"
    month[10] := "November"
    month[11] := "December"

    day[0] := "Monday"
    day[1] := "Tuesday"
    day[2] := "Wednesday"
    day[3] := "Thursday"
    day[4] := "Friday"
    day[5] := "Saturday"
    day[6] := "Sunday"

    mi := m
    yi := y

    if y < 1752 then

        put month[mi-1], " ", d, ", ", yi

    else

        if m = 1 or m = 2 then

            m := m + 12
            y := y - 1

        end if

        i := d + 2*m + 3*(m+1) div 5 + y + y div 4 - y div 100 + y div 400
        i := i mod 7

        put day[i], ", ", month[mi-1], " ", d, ", ", yi

    end if

end procedure

%
% returns value for sign of argument
%
function sgn( x : real ) : int

    var rv : int
    
    if x > 0.0 then
        rv :=  1
    elsif x < 0.0 then
        rv := -1
    else
        rv :=  0
    end if

    return rv

end function

%
% absolute value of argument
%
function abs( x : real ) : real

    if x < 0.0 then
        x := -x
    end if

    return x

end function