/********************************* XFIND.C **********************************
*   Author:  Bill Lee.
*   Copyright (c) 1989 Bill Lee.  All rights reserved.  Use freely for any
*       non-military purpose; acknowledgment of copyright and retention of
*       copyright notice is required.
*   Purpose:  Search one or more text files for multiple keywords
*       simultaneously.
*   Version:  1.0 (see VERSION macro), 89aug26.
*   Compilers: Microsoft C v5.10.
*              Turbo C 2.0.
*   Memory model:  Small, but should work with any model.
*   Operating systems:  Tested with IBM DOS v3.21, DOS v4.01, and
*       OS/2 Standard Edition v1.10 (protected mode and DOS mode).
*
*   Compiler switches:  none.
*   Compiling and linking:  cl xfind.c msrch.obj expandwi.obj
*
*   Usage:  XFIND "text string A" ["text string B"...]
*                   {[/Ffilespec [/Ffilespec...] | < stdin} [> stdout]
*                   [/H] [/I] [/O[C][L][N][S][1]] [/S] [/V]
*
******************************************************************************/

#include <conio.h>
#include <ctype.h>
#include <stddef.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <msrch.h>

/*  Manifest constants, macros: */
#define COPYRIGHT   "Copyright (c) 1989 Bill Lee.  All Rights Reserved."
#define VERSION     "v1.0"                                /* current version */
#define SWITCHAR    "/"                          /* default switch character */
#define SWITCHAR_ENV_VAR_NAME   "SWITCHAR"      /* env variable (upper case) */
#define INBUFSIZE   8192                                /* input buffer size */
#define TRUE        (BOOL) 1
#define FALSE       (BOOL) 0

/*  Derived types, from OS/2 Presentation Manager's Toolkit v1.1.
    Copyright (c) 1987,1989 Microsoft Corporation. */

#define                 VOID    void
#define                 SHORT   short
typedef unsigned short  BOOL;
typedef unsigned char   UCHAR;
typedef unsigned int    UINT;
typedef unsigned short  USHORT;
typedef unsigned long   ULONG;

/*  Prototypes for internal functions and for those not in header files: */
VOID XfdErrExit1(USHORT);                                        /* internal */
VOID XfdErrExit2(USHORT, char *);                                /* internal */
VOID XfdUsage(USHORT);                                           /* internal */
int MshRetrieveChar(void);                                       /* internal */
void MshFoundWord(char *);                                       /* internal */
int expandwild(unsigned int *, char **[], int, char *[], int);   /* external */

/*  Global data: */

static char *npszSwitchar;     /* ptr to SWITCHAR environment variable value */

static struct KWORD2 {                    /* struct for keyword text strings */
    struct kword ms;                              /* msrch kword linked list */
    BOOL fCntLine1;                  /* flag: keyword counted once this line */
    BOOL fOutLine1;                   /* flag: line output once this keyword */
    /* "hit" counts for this keyword: */
    ULONG cKeywordFile;                                   /* total this file */
    ULONG cKeywordAccum;                                  /* total all files */
    ULONG cKeywordFile1;                              /* lines hit this file */
    ULONG cKeywordAccum1;                             /* lines hit all files */
} (*pakW1st)[];                /* ptr to 1st element of keyword struct array */

/*  Flag and counts for hits on lines:  */
static BOOL fHitLine1 = FALSE;                /* flag: current line hit once */
static ULONG cHitLines;                   /* hit count on lines in this file */
static ULONG cHitLinesAccum = 0L;         /* hit count on lines in all files */

static USHORT cTxt = 0;                   /* count of text strings in argv[] */

/*  Command line switches:  */
static BOOL fShowSwitches = FALSE;               /* /S see switches flag     */
static BOOL fCaseIns  = FALSE;                   /* /I case Insensitive flag */
static BOOL freVerse  = FALSE;                   /* /V reVerse flag          */
static BOOL fOutCount = FALSE;                   /* /OC Output Counts flag   */
static BOOL fOutLine  = FALSE;                   /* /OL Output Lines flag    */
static BOOL fOutNum   = FALSE;                   /* /ON Output line Num flag */
static BOOL fOutStr   = FALSE;                   /* /OS Output String flag   */
static BOOL fOutOnce  = FALSE;             /* /O1 Output each line once flag */

/*  Input file stuff:  */
static char **npaszFname = NULL;      /* ptr to array of expanded file names */
static SHORT iszFname = -1;                  /* index into *npaszFname array */
static FILE *hInput;                                    /* input file handle */
static ULONG cLine = 0L;                      /* line count in current input */
static ULONG cLineAccum = 0L;                        /* line count all input */
static USHORT iInBuffEnd;           /* index to last non-\0 char of szInBuff */
static char *npszInBuff = NULL;                     /* ptr into input buffer */
static UCHAR szInBuff[INBUFSIZE];              /* input buffer (ended by \0) */
static BOOL fFnameOut = FALSE;           /* flag indicating file name output */

/*---------------------------------------------------------------------------*/

int main(USHORT argc, char *argv[])
{
    USHORT ich0, ich1;                                  /* indices into char */
    USHORT ikW0;                           /* index into KWORD2 struct array */
    USHORT cFspec = 0;                                 /* count of filespecs */
    char **npaszFspec = NULL;                   /* ptr to array of filespecs */
    UINT cFname = 0;                         /* count of expanded file names */
    int iszExpWildCode;                          /* expandwild() return code */
    static char *npaszExpWildErr[4] = { "",  /* texts for expandwild() codes */
        "Insufficient memory",
        "Number of names changed while expanding",
        "Length of names changed while expanding"
    };

    /*  If switch character environment variable is defined, use it; otherwise
        use default. */
    if ((npszSwitchar = getenv(SWITCHAR_ENV_VAR_NAME)) == NULL)/* undefined? */
        npszSwitchar = SWITCHAR;                              /* set default */
    else                                            /* defined, check it out */
        if (strlen(npszSwitchar) != 1)
            XfdErrExit1(11);

    if (argc < 2)                           /* there must be some parameters */
        XfdErrExit1(1);                         /* output error msg and code */

    /*  Scan all input arguments for the text strings and filespecs, and
        count the number of each.  If Help switch found, issue help info and
        exit.  */
    for (ich0 = 1; ich0 < argc; ++ich0) {
        if (argv[ich0][0] == '\0')                      /* null text string? */
            XfdErrExit1(2);                     /* output error msg and code */
        if (argv[ich0][0] != *npszSwitchar) {              /* a text string? */
            ++cTxt;                                     /* count text string */
            continue;
        }
        if (argv[ich0][1] == 'f' || argv[ich0][1] == 'F') {   /* a filespec? */
            ++cFspec;                                            /* count it */
            continue;
        }
        if (argv[ich0][1] == 'h' || argv[ich0][1] == 'H') {         /* help? */
            XfdUsage(2);
            return (0);
        }
    }

    /*  There must be at least 1 text string. */
    if (cTxt < 1)
        XfdErrExit1(3);               /* output error msg and exit with code */

    /*  Obtain and partially initialize structs and counts for text strings. */
    /*  Allocate structs and zero counts, ptrs, and flags: */
    pakW1st = (struct KWORD2 (*)[]) calloc(cTxt, sizeof(struct KWORD2));
    if (pakW1st == NULL)                             /* no memory available? */
        XfdErrExit1(4);                                   /* exit with error */
    /*  Build linked list: */
    for (ikW0 = 0; ikW0 < cTxt-1; ++ikW0)
        (*pakW1st)[ikW0].ms.next = &(*pakW1st)[ikW0+1].ms;   /* link to next */

    /*  Obtain zeroed array of ptrs for filespecs if any specified. */
    if (cFspec > 0) {
        npaszFspec = (char **) malloc((cFspec+1) * sizeof(char *));
        if (npaszFspec == NULL)                      /* no memory available? */
            XfdErrExit1(4);                               /* exit with error */
        hInput = NULL;                  /* say there are files, need to open */
        npaszFspec[cFspec] = NULL;                   /* zero terminating ptr */
    }
    else {
        hInput = stdin;                 /* indicate no disk files, use stdin */
        fFnameOut = TRUE;          /* so file name is never output for stdin */
    }

    /*  Scan all input arguments again.  Build linked list of text strings,
        array of filespecs, and set option switches. */
    ikW0 = 0;                                           /* beginning of list */
    cFspec = 0;                                     /* reset for array index */

    for (ich0 = 1; ich0 < argc; ++ich0) {

        if (argv[ich0][0] != *npszSwitchar) {              /* a text string? */
            (*pakW1st)[ikW0++].ms.word = argv[ich0];       /* save ptr to it */
            continue;
        }

        switch (argv[ich0][1]) {

            case 'F':                                         /* a Filespec? */
            case 'f':
                if (argv[ich0][2] == '\0')                /* null file name? */
                    XfdErrExit1(12);
                /*  Save ptr to filespec, increment for next. */
                npaszFspec[cFspec++] = &argv[ich0][2];
                break;

            case 'I':                        /* the case Insensitive switch? */
            case 'i':
                if (argv[ich0][2] != '\0')        /* switch string too long? */
                    XfdErrExit2(6, argv[ich0]);
                fCaseIns = TRUE;
                break;

            case 'S':                                      /* show switches? */
            case 's':
                if (argv[ich0][2] != '\0')        /* switch string too long? */
                    XfdErrExit2(6, argv[ich0]);
                fShowSwitches = TRUE;
                break;

            case 'V':                                     /* reVerse search? */
            case 'v':
                if (argv[ich0][2] != '\0')        /* switch string too long? */
                    XfdErrExit2(6, argv[ich0]);
                freVerse = TRUE;
                break;

            case 'O':                                    /* an Output switch */
            case 'o':
                if (argv[ich0][2] == '\0')                /* switch missing? */
                    XfdErrExit2(6, argv[ich0]);
                for (ich1 = 2; argv[ich0][ich1] != '\0'; ++ich1) {
                    switch (argv[ich0][ich1]) {
                        case 'C':         /* Output Counts rather than lines */
                        case 'c':
                            fOutCount = TRUE;
                            break;
                        case 'L':                            /* Output Lines */
                        case 'l':
                            fOutLine = TRUE;
                            break;
                        case 'N':                     /* Output line Numbers */
                        case 'n':
                            fOutNum = TRUE;
                            break;
                        case 'S':            /* Output text String with line */
                        case 's':
                            fOutStr = TRUE;
                            break;
                        case '1':        /* Output each input line once only */
                            fOutOnce = TRUE;
                            break;
                        default:                    /* invalid Output switch */
                            XfdErrExit2(6, argv[ich0]);
                    }
                }
                break;

            default:
                XfdErrExit2(5, argv[ich0]);             /* invalid parameter */

        }   /* switch (argv[ich0][1]) */

    }   /* for (ich0 = 1; ich0 < argc; ++ich0) */

    /*  Set default switch if it and related switch not set. */
    fOutLine = (!fOutCount && !fOutLine) ? TRUE : fOutLine;

    /*  Set to FALSE any switches that are ignored when other switches are set.
        This may save testing multiple switches later. */
    fOutStr = freVerse ? FALSE : fOutStr;
    fOutNum = (fOutCount && !fOutLine) ? FALSE : fOutNum;

    /*  If strings not being output, turn on output-once switch as part of
        logic to output a line only once regardless of number of hits on it. */
    fOutOnce = fOutStr ? fOutOnce : TRUE;

    /*  Expand any wild card filespecs into actual file names. */
    if (npaszFspec != NULL)
        if ((iszExpWildCode =
            expandwild(&cFname, &npaszFname, -1, npaszFspec, 0)) != 0)
            XfdErrExit2(7, npaszExpWildErr[iszExpWildCode]);

    /*  Output to stderr the command line parameters for this execution of
        XFIND. */
    if (fShowSwitches) {
        /*  Option switches:  */
        fprintf(stderr, "XFIND " VERSION " switch settings: ");
        if (fCaseIns)
            fprintf(stderr, " %cI", *npszSwitchar);
        fprintf(stderr, " %cS", *npszSwitchar);
        if (freVerse)
            fprintf(stderr, " %cX", *npszSwitchar);
        if (fOutCount || fOutLine || fOutNum || fOutStr || fOutOnce) {
            fprintf(stderr, " %cO", *npszSwitchar);
            if (fOutCount)    fprintf(stderr, "C");
            if (fOutLine)     fprintf(stderr, "L");
            if (fOutNum)      fprintf(stderr, "N");
            if (fOutStr)      fprintf(stderr, "S");
            if (fOutOnce)     fprintf(stderr, "1");
        }
        fprintf(stderr, "\n");
        /*  Target text strings:  */
        fprintf(stderr, "%u target text strings:\n", cTxt);
        for (ikW0 = 0; ikW0 < cTxt; ++ikW0)
            fprintf(stderr, "    \"%s\"\n", (*pakW1st)[ikW0].ms.word);
        /*  Names of input files:  */
        if (hInput == stdin)
            fprintf(stderr, "Input file is stdin.\n");
        else {
            fprintf(stderr, "%u input file names:\n", cFname);
            for (ich0 = 0; npaszFname[ich0] != NULL; ++ich0)
                fprintf(stderr, "    \"%s\"\n", npaszFname[ich0]);
        }
    }

    /*  If case insensitivity specified, convert text strings to lower. */
    if (fCaseIns)
        for (ikW0 = 0; ikW0 < cTxt; ++ikW0)
            strlwr((*pakW1st)[ikW0].ms.word);

    /*  Initialize, do the searching, and clean up. */
    msrch_init(&(*pakW1st)[0].ms);                      /* initialize search */
    msrch_go(MshRetrieveChar, MshFoundWord);               /* perform search */
    msrch_end();                                           /* clean up after */

    /*  Output accumulator(s) if requested and if more than 1 file. */
    if (fOutCount && cFname > 1) {
        printf("==== %u-File Total ", cFname);                     /* marker */
        if (fOutStr) {
            if (cHitLinesAccum > 0)
                printf("Counts:\n");                /* title only if strings */
            for (ikW0 = 0; ikW0 < cTxt; ++ikW0) {
                if ((*pakW1st)[ikW0].cKeywordAccum > 0)
                    printf("\"%s\": %lu hits in %lu lines\n",
                        (*pakW1st)[ikW0].ms.word,
                        (*pakW1st)[ikW0].cKeywordAccum,
                        (*pakW1st)[ikW0].cKeywordAccum1);
            }
        }
        printf("Lines: %lu hits in %lu lines\n", cHitLinesAccum, cLineAccum);
    }

    return (0);                                         /* XFIND normal exit */

}   /* main() */

/*---------------------------------------------------------------------------*/

/*  Return next input character or EOF to msrch_go(), opening and reading
    file(s) as needed. */
int MshRetrieveChar(void)
{
    USHORT ikW0;                                 /* index into KWORD2 struct */
    USHORT uchInBuff;                              /* current char in buffer */

    /*  Note: fgets() includes the \n at the end of every line (except the line
    at EOF).  The \n has the effect of terminating the search for that line
    because the DOS or OS/2 command line doesn't permit embedded \n.  Thus a
    target string is not found if it is broken between two lines.  */

    if (hInput == NULL) {                                  /* file not open? */
        /*  If no more file names, indicate EOF. */
        if (npaszFname == NULL || *(npaszFname+(++iszFname)) == NULL) {
            return (EOF);
        }
        if ((hInput = fopen(*(npaszFname+iszFname), "r")) == NULL)
            XfdErrExit1(8);
        npszInBuff = NULL;                                   /* buffer empty */
    }

    /*  If buffer ptr not initialized or input line all used, refill buffer. */
    if (npszInBuff == NULL || *(++npszInBuff) == '\0') {
        if (npszInBuff != NULL) {                      /* end of input line? */
            if (fHitLine1 && !freVerse)
                ++cHitLines;                                     /* count it */
            /*  No hits; an excluded line?  If yes, count and output it. */
            else if (!fHitLine1 && freVerse) {
                ++cHitLines;                                     /* count it */
                if (!fFnameOut) {           /* file name not already output? */
                    printf("========== %s\n",
                        *(npaszFname+iszFname));         /* output file name */
                    fFnameOut = TRUE;       /* remember file name was output */
                }
                if (fOutLine) {                              /* output line? */
                    if (fOutNum)
                        printf("[%lu] ", cLine);              /* line number */
                    printf("%s%s", szInBuff,                    /* line text */
                        (szInBuff[iInBuffEnd] != '\n') ? "\n" : "");
                }
            }
        }
        /*  Get an input line. */
        if (fgets(szInBuff, INBUFSIZE, hInput) == NULL) {   /* EOF or error? */
            /*  If EOF, output counts & recurse to open next file.  */
            if (feof(hInput) != 0) {                                 /* EOF? */
                if (fOutCount && cHitLines > 0) {          /* output counts? */
                    printf("---------- ");                         /* marker */
                    for (ikW0 = 0; ikW0 < cTxt; ++ikW0) {
                        if (fOutStr) {
                            if (ikW0 == 0)  /* title only if strings output: */
                                printf("Counts:\n");
                            if ((*pakW1st)[ikW0].cKeywordFile > 0)
                                printf("\"%s\": %lu hits in %lu lines\n",
                                    (*pakW1st)[ikW0].ms.word,        /* text */
                                    (*pakW1st)[ikW0].cKeywordFile,  /* count */
                                    (*pakW1st)[ikW0].cKeywordFile1);/* count */
                        }
                        (*pakW1st)[ikW0].cKeywordAccum +=      /* accumulate */
                            (*pakW1st)[ikW0].cKeywordFile;
                        (*pakW1st)[ikW0].cKeywordFile = 0L;         /* reset */
                        (*pakW1st)[ikW0].cKeywordAccum1 +=     /* accumulate */
                            (*pakW1st)[ikW0].cKeywordFile1;
                        (*pakW1st)[ikW0].cKeywordFile1 = 0L;        /* reset */
                    }
                    /*  Output line counts for the file. */
                    printf("Lines: %lu hits in %lu-line file\n",
                        cHitLines, cLine);
                    cHitLinesAccum += cHitLines;               /* accumulate */
                    cHitLines = 0L;                                 /* reset */
                }
                cLineAccum += cLine;                /* accumulate line count */
                cLine = 0L;                              /* reset line count */
                /*  Return to caller, after closing input if necessary.  */
                if (hInput == stdin)
                    return (EOF);         /* if stdin, no more files to open */
                fclose(hInput);                       /* not stdin, close it */
                hInput = NULL;                      /* indicate no file open */
                fFnameOut = FALSE; /* clear flag indicating file name output */
                return (MshRetrieveChar());
                                        /* recurse to open next file, if any */
            }
            else                            /* not EOF, must be a read error */
                XfdErrExit1(9);
        }
        /*  If input line is complete, it will have \n at the end except
            perhaps when we are at EOF. */
        iInBuffEnd = strlen(szInBuff) - 1;             /* index of last char */
        if (szInBuff[iInBuffEnd] != '\n')
            if (feof(hInput) == 0)                               /* not EOF? */
                XfdErrExit1(10);                   /* input buffer too short */
        /*  Set and reset everything for the new input line. */
        npszInBuff = szInBuff;              /* point back to start of buffer */
        ++cLine;                                                /* next line */
        fHitLine1 = FALSE;                                     /* reset flag */
        for (ikW0 = 0; ikW0 < cTxt; ++ikW0) {
            (*pakW1st)[ikW0].fCntLine1 = FALSE;                /* reset flag */
            (*pakW1st)[ikW0].fOutLine1 = FALSE;                /* reset flag */
        }
    }

    /*  Return character to caller as lower case or unchanged. */
    uchInBuff = *npszInBuff & 0x00FF;           /* must not be sign-extended */
    return (fCaseIns ? tolower(uchInBuff) : uchInBuff);

}   /*  MshRetrieveChar() */

/*---------------------------------------------------------------------------*/

/*  Called by msrch_go() for each hit in input. */
void MshFoundWord(char *npszWord)      /* npszWord is ptr to the text found. */
{
    USHORT ikW0, ikW1;                         /* indices into KWORD2 struct */

    fHitLine1 = TRUE;                       /* remember the hit on this line */

    if (freVerse)                    /* if a reVerse search, ignore this one */
        return;

    /*  Find same ptr in list to determine index into struct array.
        for should always break and never fall thru.  */
    for (ikW0 = 0; ikW0 < cTxt; ++ikW0)
        if ((*pakW1st)[ikW0].ms.word == npszWord)                  /* equal? */
            break;

    ++(*pakW1st)[ikW0].cKeywordFile;                    /* increment counter */
    /*  Increment line hit count if not already counted for this line. */
    if (!(*pakW1st)[ikW0].fCntLine1) {               /* not already counted? */
        ++(*pakW1st)[ikW0].cKeywordFile1;               /* increment counter */
        (*pakW1st)[ikW0].fCntLine1 = TRUE;               /* remember counted */
    }

    if (!fFnameOut) {                       /* file name not already output? */
        printf("========== %s\n", *(npaszFname+iszFname));/* output file name*/
        fFnameOut = TRUE;                   /* remember file name was output */
    }

    if (fOutLine)                                 /* are we outputing lines? */
        /*  Output line if not already output. */
        if (!(*pakW1st)[ikW0].fOutLine1) {            /* not already output? */
            if (fOutOnce)                        /* set flag only if we care */
                (*pakW1st)[ikW0].fOutLine1 = TRUE;        /* remember output */
            if (!fOutStr) {                        /* not outputing strings? */
                /*  If line already output for another string, return. */
                for (ikW1 = 0; ikW1 < cTxt; ++ikW1)
                    if (ikW1 != ikW0 && (*pakW1st)[ikW1].fOutLine1)
                        return;
            }
            else                                            /* output string */
                printf("\"%s\": ", (*pakW1st)[ikW0].ms.word);
            if (fOutNum)                          /* output file line number */
                printf("[%lu] ", cLine);
            printf("%s%s", szInBuff,                     /* output line text */
                (szInBuff[iInBuffEnd] != '\n') ? "\n" : "");
        }

    return;                                                /* back to caller */

}   /*  MshFoundWord() */

/*---------------------------------------------------------------------------*/

/*  Output error msg on stderr and exit with error code. */
VOID XfdErrExit1(USHORT usErrCode)                /* usErrCode is error code */
{
    switch (usErrCode) {
        case 1:
            fprintf(stderr, "\a");
            XfdUsage(2);
            break;
        case 2:
            fprintf(stderr, "\aXFIND--"
                "Null text string not allowed.\n");
            break;
        case 3:
            fprintf(stderr, "\aXFIND--"
                "At least 1 text string must be specified.\n");
            XfdUsage(1);
            break;
        case 4:
            fprintf(stderr, "\aXFIND--"
                "Insufficient memory.\n");
            break;
        case 8:
            fprintf(stderr, "\aXFIND--"
                "Error opening input file \"%s\":  %s.\n",
                *(npaszFname+iszFname), strerror(errno));
            break;
        case 9:
            fprintf(stderr, "\aXFIND--"
                "Error reading %s at line %lu:  %s.\n",
                hInput == stdin ? "stdin" : *(npaszFname+iszFname), cLine,
                ferror(hInput) ?
                    (errno != 0 ? strerror(errno) : "unknown cause") :
                    "unknown cause");
            break;
        case 10:
            fprintf(stderr, "\aXFIND--"
                "Length of line %lu in %s exceeds input buffer size of "
                "%u bytes.\n", cLine+1,
                hInput == stdin ? "stdin" : *(npaszFname+iszFname),
                INBUFSIZE-1);
            break;
        case 11:
            fprintf(stderr, "\aXFIND--"
                "Value of environment variable " SWITCHAR_ENV_VAR_NAME
                " must be 1 character in length.\n");
            break;
        case 12:
            fprintf(stderr, "\aXFIND--"
                "null file name.\n");
            break;
    }
    exit(usErrCode);                                      /* terminate XFIND */
}   /* XfdErrExit() */

/*---------------------------------------------------------------------------*/

/*  Output error msg on stderr and exit with error code. */
VOID XfdErrExit2(USHORT usErrCode,                /* usErrCode is error code */
    char *npszArg)                      /* npszArg is text specific to error */
{
    switch (usErrCode) {
        case 5:
            fprintf(stderr, "\aXFIND--"
                "Invalid parameter:  %s.\n", npszArg);
            XfdUsage(1);
            break;
        case 6:
            fprintf(stderr, "\aXFIND--"
                "Switch invalid or missing:  %s.\n", npszArg);
            XfdUsage(1);
            break;
        case 7:
            fprintf(stderr, "\aXFIND--"
                "Error during expansion of wildcard file names:  %s.\n",
                npszArg);
            break;
    }
    exit(usErrCode);                                      /* terminate XFIND */
}   /* XfdErrExit() */

/*---------------------------------------------------------------------------*/

/*  Output usage syntax via stderr. */
VOID XfdUsage(USHORT sCode)               /* 1 == short help, 2 == long help */
{
    if (sCode == 2)
        fprintf(stderr, "XFIND " VERSION " " COPYRIGHT "\n\n"
            "XFIND searches one or more files for one or more strings "
                "simultaneously.\n"
            "Output consists of the file names, \"hit\" lines, "
                "and/or counts of \"hits\".\n\n");

    fprintf(stderr,
        "Usage:  XFIND \"text string A\" [\"text string B\"...]\n"
        "              {%cFfilespec [%cFfilespec...] | < stdin} [> stdout]\n"
        "              [%cH] [%cI] [%cO[C][L][N][S][1]] [%cS] [%cV]\n",
        *npszSwitchar, *npszSwitchar, *npszSwitchar, *npszSwitchar,
        *npszSwitchar, *npszSwitchar, *npszSwitchar);

    if (sCode == 2) {

        fprintf(stderr,                           /* \xFE is a square bullet */
            "\n\xFE   Press any key for remainder of help.");
        getch();
        fprintf(stderr, "\r");      /* return to start of "Press any key..." */

        fprintf(stderr,
            /*  (Next line erases above help prompt, watch length.) */
            "Enclose target text strings in double quotes "
            "if they contain white space.\n");

        fprintf(stderr,
            "\n%cFfilespec [%cFfilespec...] | < stdin\n"
            "    Input can be 1 or more file specs (wildcards o.k.) or "
                "via redirection or\n"
            "    piping of stdin.  Maximum line length is %u bytes.\n",
            *npszSwitchar, *npszSwitchar, INBUFSIZE-1);

        fprintf(stderr,
            "\nOutput is to stdout; defaults to screen, can be redirected or "
                "piped.\n");

        fprintf(stderr,
            "\nSwitches (%cO switches can be combined, "
                "and in any order, e.g., %cOLCN):\n"
            "%cH  This Help info output to stderr.  "
                "(All other parameters ignored.)\n"
            "%cI  Insensitive to case during searching.\n",
            *npszSwitchar, *npszSwitchar, *npszSwitchar, *npszSwitchar);
        fprintf(stderr,
            "%cOC Output \"hit\" Counts at EOF.  "
                "(Lines not output unless %cOL specified.)\n"
            "%cOL Output Lines.  (Default if neither "
                "%cOC nor %cOL specified.)\n"
            "%cON Output relative line Number with each line.\n"
            "%cOS Output text String with line and/or count.  "
                "(Ignored with %cV.)\n"
            "%cO1 Output hits for a string only once for each line.  "
                "(Useful only with %cOS).\n",
            *npszSwitchar, *npszSwitchar, *npszSwitchar, *npszSwitchar,
            *npszSwitchar, *npszSwitchar, *npszSwitchar, *npszSwitchar,
            *npszSwitchar, *npszSwitchar);
        fprintf(stderr,
            "%cS  Show switches, strings, and "
                "expanded input file names on stderr.\n"
            "%cV  A reVerse search, i.e., lines excluding strings "
                "are output and/or counted.\n",
            *npszSwitchar, *npszSwitchar);

        fprintf(stderr,
            "\nDefault switch character is %c; change by setting "
                "environment variable " SWITCHAR_ENV_VAR_NAME ".",
            *npszSwitchar);
    }

}   /* XfdUsage() */

/*---------------------------------------------------------------------------*/
