/****************************  SAVMOUSE.C  ************************
*   This program saves and restores the mouse state.
*   Verified with Mouse drivers from Logitech and Microsoft
*   Compilers: Turbo C, Microsoft, Watcom
*   Author: Andrew Binstock, Copyright (c) 1989
*   Use freely as long as authorship is acknowledged.
*******************************************************************/

#include <stdio.h>
#include <dos.h>
#include <stdlib.h>

#define DRIVER 100

#define MOUSE_INT  0x33         /* Mouse interrupt number */

#if ! defined MK_FP             /* For Microsoft especially */
#define MK_FP(seg,ofs)  ((void far *) \
               (((unsigned long)(seg) << 16) | (unsigned)(ofs)))
#endif

union  REGS   in, out;
struct SREGS  segs;


int check_mouse (void)    /* Returns -1 if no driver, 0 if no mouse */
{                         /* Returns  1 if driver and mouse present */

    /* Get the address of MOUSE_INT             */
    /* We do it the long way for portability.   */

    char far * mouse_vector;

    in.h.ah = 0x35;       /* Get-vector service */
    in.h.al = MOUSE_INT;  /* Get mouse vector   */
    int86x (0x21, &in, &out, &segs);

    mouse_vector = MK_FP(segs.es, out.x.bx);

    if ((long) mouse_vector == 0L || *mouse_vector == 0xCF)
        return (-1);      /* No driver found    */

    /* Driver found, let us check and reset the mouse */
    in.x.ax = 0;
    int86 (MOUSE_INT, &in, &out);

    /* AX = -1, all went well, otherwise no mouse found */
    return ((int) out.x.ax == -1 ? 1 : 0);
}

size_t create_save_area (void far **mouse_ptr)  /* Get save area size */
{                                               /* and allocate it.   */
    size_t buffer_size;

    in.x.ax = 21;        /* Service 21d of mouse calls gets size */
    int86 (MOUSE_INT, &in, &out);
    buffer_size = out.x.bx;

    *mouse_ptr = (void far *) malloc (buffer_size);

    return (*mouse_ptr == NULL ? 0 : buffer_size);
}

void save_mouse_state (void far * save_area)    /* Save mouse info. */
{
    in.x.ax = 22;
    in.x.dx = FP_OFF (save_area);
    segs.es = FP_SEG (save_area);
    int86x (MOUSE_INT, &in, &out, &segs);
}

void restore_mouse_state (void far * save_area) /* Restore saved info. */
{
    in.x.ax = 23;
    in.x.dx = FP_OFF (save_area);
    segs.es = FP_SEG (save_area);
    int86x (MOUSE_INT, &in, &out, &segs);
}

#ifdef DRIVER

#include <conio.h>

int main (void)
{
    size_t     mouse_area_size;
    void far * mouse_area_ptr;

    switch (check_mouse())
    {
        case -1: fputs ("No Mouse Driver", stderr);
                 exit (5);
        case  0: fputs ("No Mouse Attached", stderr);
                 exit (6);
        case  1: fputs ("Driver and Mouse Found\n", stderr);
                 break;
    }

    if ((mouse_area_size = create_save_area (&mouse_area_ptr)) == 0)
    {
        fputs ("Mouse save area could not be allocated", stderr);
        exit (7);
    }
    else 
        printf ("Mouse area occupies %u bytes\n", mouse_area_size);

    in.x.ax = 0x01;   /* show mouse cursor */
    int86 (MOUSE_INT, &in, &out);

    fputs ("Press 's' to save mouse ", stdout);
    while (getche() != 's') ;

    save_mouse_state (mouse_area_ptr);

    in.x.ax = 8;     /* Lock the cursor into a couple of rows */
    in.x.cx = 15;
    in.x.dx = 22;
    int86(MOUSE_INT, &in, &out);

    fputs ("\nYour cursor is locked into a narrow band\n", stdout);

    fputs ("Press 'r' to restore mouse ", stdout);
    while (getche() != 'r') ;

    restore_mouse_state (mouse_area_ptr);

    fputs ("\nYour cursor is freed. Press any key to exit\n", stdout);
    getche();

    in.x.ax = 0x02;   /* hide mouse cursor */
    int86 (MOUSE_INT, &in, &out);

    return (1);
}
#endif