/********************** HELPMAKE.C *****************************
* Author: David Michmerhuizen
*
* Function: Compile help screen data for help()
*
* Compilers:    Turbo C  V1.5
*
* Memory models:    all
*
* Usage:  HELPMAKE helpsource, [helpsource...]
*
*         where 'helpsource' is the filename of one or more
*         help source files.
*
*         Helpmake then creates three files:
*
*               help.h       for use by application subroutines
*                            that call help().
*
*               helpstru.h   used by help.c
*
*               help.hlp     data file used by help.c
*
***************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>


#ifndef TRUE
#define TRUE  1
#define FALSE 0
#endif


char    help_fname[]    = "help.hlp";
char    struct_fname[]  = "helpstru.h";
char    defs_fname[]    = "help.h";

FILE    *helpout;
FILE    *structout;
FILE    *defsout;

int startfiles();
int endfiles( int );
int makehelp( FILE *, long *, int *);


/*--------------------------------------------------------------------------*/

void main( int argc, char *argv[])
{
    extern  FILE    *helpout;
    extern  FILE    *structout;
    extern  FILE    *defsout;
    auto    FILE    *helpsrc;
    auto    int     i;
    auto    int     outnum = 0;
    auto    long    outpos = 0L;


    /* need at least one argument */

    if  ( argc == 1)
    {
        fputs("\nFormat: HELPMAKE helpsource, [helpsource...]\n\n", stdout);
        exit(1);
    }

    /* open output files */

    if  ( startfiles())
        exit(2);

    /* process multiple input files */

    for( i=1; i<argc; i++)
    {
        if  ( !(helpsrc = fopen( argv[i], "rt")))
        {
            fputs("Error opening input file ",stdout);
            fputs(argv[i],stdout);
            fputc('\n',stdout);
            fcloseall();
            exit(3);
        }
        else
        {
            fputs( "Compiling ", stdout);
            fputs( argv[i],      stdout);
            fputs( "...\n",      stdout);
            if  ( makehelp( helpsrc, &outpos, &outnum))
            {
                fcloseall();
                exit(4);
            }
            else
                fclose( helpsrc);
        }
    }

    /* finish off output files */

    endfiles( outnum);

} /* end of main */




/*--------------------------------------------------------------------------
**  startfiles -- opens and puts initial output to output files
*/


int startfiles()
{
    extern  FILE    *helpout;
    extern  FILE    *structout;
    extern  FILE    *defsout;
    extern  char    help_fname[];
    extern  char    struct_fname[];
    extern  char    defs_fname[];


    if  ( !(helpout = fopen( help_fname, "wb")))
    {
        fputs("Error opening output file ", stdout);
        fputs( help_fname, stdout);
        fputc('\n', stdout);
        return(1);
    }
    if  ( !(structout = fopen( struct_fname, "wt")))
    {
        fputs("Error opening output file ", stdout);
        fputs( struct_fname, stdout);
        fputc('\n', stdout);
        fcloseall();
        return(1);
    }
    if  ( !(defsout = fopen( defs_fname, "wt")))
    {
        fputs("Error opening output file ", stdout);
        fputs( defs_fname, stdout);
        fputc('\n', stdout);
        fcloseall();
        return(1);
    }


    fputs( "/*\n",              structout);
    fputs( "** helpstru.h\n",   structout);
    fputs( "*/\n\n",            structout);
    fputs( "HELPSCREEN helpscreens[] = {\n", structout);

    fputs( "/*\n",              defsout);
    fputs( "** help.h\n",       defsout);
    fputs( "*/\n\n",            defsout);

    return(0);

} /* end of startfiles */



/*--------------------------------------------------------------------------
** endfiles -- finishes off and closes output files
*/

int endfiles( int outnum)
{
    extern  FILE    *helpout;
    extern  FILE    *structout;
    extern  FILE    *defsout;

    fputs( "       };\n\n\n", structout);
    fprintf( structout,  " #define HELPSCREENS %d\n\n", outnum);
    fcloseall();

    return(0);

} /* end of endfiles */



/*--------------------------------------------------------------------------
** makehelp  -- processes an input file into output file items
*/

int makehelp( FILE *helpsrc, long *outpos, int *outnum)
{
    #define linemax 128
    extern  FILE    *helpout;
    static  char    delims[] = "\x20\t\n";
    auto    char    *c;
    auto    int     i, line = 0;
    auto    char    inline[ linemax];
    auto    char    outdef[ linemax];
    auto    char    outnam[ linemax];
    auto    int     outlen;
    auto    char    tmpline[ linemax];
    auto    char    *tmp;
    auto    int     screen = FALSE;
    auto    long    scrnpos;


    while( fgets( inline, linemax, helpsrc))    /* while can get a line     */
    {                                           /*                          */
        line++;
        strcpy( tmpline, inline);               /* make working copy        */
        tmp = strtok( tmpline, delims);         /* get first token          */
        if  ( tmp && !stricmp( tmp, ".SCREEN")) /* if .screen command       */
        {
            if  ( screen)                       /* if screen already active */
            {
                printf(".SCREEN before .END, line %d\n", line);
                return(1);                      /* error.                   */
            }
            else                                /* otherwise..              */
            {                                   /* start gathering info     */
                outdef[0] = outnam[0] = '\0';
                tmp = strtok( NULL, delims); /* -- first the define name */
                if  ( tmp)
                    strcpy( outdef, tmp);
                while ( tmp = strtok( NULL, delims))
                {
                    strcat( outnam, tmp);
                    strcat( outnam, " ");
                }
                if  ( strlen( outnam))
                    outnam[strlen(outnam)-1] = '\0';

                if  ( !strlen( outdef) || !strlen( outnam))
                {
                    printf("Incomplete .SCREEN def, line %d\n", line);
                    return(1);
                }
                strupr( outdef);
                strlwr( outnam);
                outlen = 0;
                scrnpos = *outpos;
                screen = TRUE;
            }
        }                                       /* end if .screen command   */
        else if  ( tmp && !stricmp( tmp, ".END")) /* else if .end command   */
        {
            if  ( !screen)                      /* if no screen active      */
            {
                printf(".END before .SCREEN, line %d\n", line);
                return(1);                      /* error.                   */
            }
            else                                /* else write defs & struct */
            {
                fprintf( defsout, "#define %-30s %3d\n",
                    outdef, *outnum);
                fprintf( structout, "       \"%s\",%*s %8luL, %4d,\n",
                    outnam, 35-strlen(outnam), " ", scrnpos, outlen);
                ++*outnum;
                *outpos += outlen;
                screen = FALSE;
            }
        }                                       /* end if .end command      */
        else                                    /* else not a command, so   */
            if  ( screen)                           /* if screen is active      */
            {
                c = inline;                         /* output the line, with    */
                while( *c)                          /* repeating chars compresd */
                {
                    if  ( *c == '|')                /* (except inside links)    */
                    {
                        do  {
                            fputc( *(c++), helpout);
                            outlen++;
                        }
                        while( *c && *c != '|');
                        if  ( *c)
                        {
                            fputc( *(c++), helpout);
                            outlen++;
                        }
                    }
                    for ( i=1; *c == *(c+i); i++)   /* find numb of same chars  */
                        ;
                    if  ( i > 1)
                    {
                        fputc( '\xFF',   helpout);
                        fputc( (char) i, helpout);
                        outlen += 2;
                    }
                    fputc( *c, helpout);
                    outlen++;
                    c += i;
                }
            }                                       /* end if screen active     */
    }                                           /* end while                */

    return(0);

} /* end of makehelp */





















