/************************** DIF.C ***************************
 *
 * Purpose: translate .dif files to .db3 files
 * Author:  Victor Volkman
 * Compilers: MSC 5.1
 *            TurboC 2.0 - be sure to read TC usage note in
 *                         read_dif_cell() at about line 310
 *
 * Usage: dif filein.dif fileout.db3 [-v]
 *   translates from filein.dif to fileout.db3
 *   -v flag is for verbose mode--progress messages
 *       are written to stdout
 *
 * Source code is (c) copyright 1989 Victor Volkman
 * Permission is granted to incorporate the source code into
 * a larger work and to distribute compiled forms of the code.
 **********************************************************/

#include <stdlib.h>    /* System #includes from your compiler */
#include <io.h>
#include <conio.h>
#include <dos.h>
#include <fcntl.h>
#include <stdio.h>
#include <string.h>

#include "common.h"   /* DIF and DB3 subsystem */
#include "difdata.h"
#include "db3data.h"
#include "difproto.h"

char *dif_hdr_strs[] = {
    "TABLE", "VECTORS", "TUPLES", "DATA",
    "LABEL", "COMMENT", "SIZE", "PERIDOCITY", "MAJORSTART",
    "MINORSTART", "TRUELENGTH", "UNITS", "DISPLAYUNITS"
};

char *dif_val_strs[] = {
    "V", "NA", "ERROR", "TRUE", "FALSE", "BOT", "EOD"
};

static int verbose_mode = FALSE;
int error_msg();
int calculate_db3_header();
int dump_db3_hdr();
int free_dif_hdr(), free_db3_hdr();
int read_dif_cell(), check_dif_type_numeric(), check_dif_type_string();

main(argc,argv)
int argc;
char **argv;
{
    char filename1[80];
    char filename2[80];

    if (argc < 2) {
        printf("filename [.DIF]: ");
        gets(filename1);
    }
    else
        strcpy(filename1,argv[1]);

    if (argc < 3) {
        printf("filename [.DBF]: ");
        gets(filename2);
    }
    else
        strcpy(filename2,argv[2]);

    verbose_mode = (argc==4) && strcmp(argv[3],"-v")==0;
    return dif_to_db3(filename1,filename2);
}


/* The first filename is the .DIF file to read and the second filename
   is the .DB3 file to write. */

int dif_to_db3(filename1, filename2)
char *filename1;
char *filename2;
{
    dif_hdr inp_hdr;
    db3_hdr out_hdr;
    FILE *inp_file;
    int out_file;

    if ((inp_file=fopen(filename1,"r")) == NULL) {
        error_msg("Unable to open <%s> for input.\n",filename1);
        return FAIL;
    }
    if ((out_file=open(filename2,(int)(O_CREAT+O_WRONLY+O_TRUNC+O_BINARY))) < 0) {
        error_msg("Unable to open <%s> for output.\n",filename2);
        return FAIL;
    }
    if (read_reqd_dif_header(inp_file,&inp_hdr,&out_hdr) != OK)
        return FAIL;
    if (read_opt_dif_header(inp_file,&inp_hdr) != OK)
        return FAIL;
    if (read_dif_data_for_types(inp_file,&out_hdr) != OK)
        return FAIL;

    calculate_db3_header(&inp_hdr,&out_hdr);
    if (write_db3_header(out_file,&inp_hdr,&out_hdr) != OK)
        return FAIL;

    if (verbose_mode) dump_db3_hdr(&out_hdr);

    fseek(inp_file,0L,SEEK_SET);  /* Ready for pass two */
    if (read_dif_data_and_write_db3(inp_file,out_file,&out_hdr) != OK)
        return FAIL;

    fclose(inp_file);
    close(out_file);
    free_dif_hdr(&inp_hdr);
    free_db3_hdr(&out_hdr);
    return OK;
}



/*  The first header item must be the TABLE record
 *
 *  The second and third items must be TUPLES and VECTORS,
 *  although either item may appear first.
 *
 *  All remaining items are optional and may include:
 *  LABEL, COMMENT, SIZE, PERIDOCITY, MAJORSTART,
 *  MINORSTART, TRUELENGTH, UNITS, DISPLAYUNITS
*/

int read_reqd_dif_header(inp_file,inp_hdr_p,out_hdr_p)
FILE *inp_file;
dif_hdr_p inp_hdr_p;
db3_hdr_p out_hdr_p;
{
    dif_elt inp_hdr_elt;
    enum dif_hdr_vals inp_hdr_val;
    int i;

    if (verbose_mode) printf("Starting pass one...\n\n");

    if (read_dif_hdr_elt(inp_file,&inp_hdr_elt,&inp_hdr_val) != OK)
        return FAIL;

    if (inp_hdr_val != DIF_TABLE) {
        error_msg("Expected TABLE keyword in DIF header, found <%s>.\n",
            dif_hdr_strs[ (int) inp_hdr_val]);
        return FAIL;
    }

    strcpy(inp_hdr_p->title, inp_hdr_elt.str_val);

    for (i=0; i<2; i++) {
        if (read_dif_hdr_elt(inp_file,&inp_hdr_elt,&inp_hdr_val) != OK)
            return FAIL;

        if (inp_hdr_val == DIF_VECTORS)
            inp_hdr_p->num_vectors = (int)inp_hdr_elt.float_val;
        else if (inp_hdr_val == DIF_TUPLES)
            inp_hdr_p->num_tuples = (int)inp_hdr_elt.float_val;
        else {
            error_msg("Expected TUPLES or VECTORS keyword in header\n",NULL);
            return FAIL;
        }
    }

    if (inp_hdr_p->num_vectors == 0) {  /* can't continue if VECTORS unknown */
        error_msg("VECTORS keyword not found in first header items\n",NULL);
        return FAIL;
    }

    /* Allocate vector descriptors based on info from first three header
     * records
    */

    if ((inp_hdr_p->vec = (vector_desc_p) mcalloc(inp_hdr_p->num_vectors+1,
        sizeof(vector_desc))) == NULL)
        return FAIL;

    /* The number of DIF vectors is the same as DB3 fields */
    if ((out_hdr_p->fld = (db3_hdr_elt_p) mcalloc(inp_hdr_p->num_vectors+1,
        sizeof(db3_hdr_elt))) == NULL)
        return FAIL;

    return OK;
}


/* Read the optional portion of the DIF input file and save all information
   given about each tuple.  Not every topic parsed is relevant to the DB3
   translation.  Although the DIF specification allows multiple LABEL and
   COMMENT topics per tuple, the translator will only use the first one. */

int read_opt_dif_header(inp_file,inp_hdr_p)
FILE *inp_file;
dif_hdr_p inp_hdr_p;
{
    dif_elt inp_hdr_elt;
    enum dif_hdr_vals inp_hdr_val;
    vector_desc_p vec_p;
    int numer_val;
    char *str_val;

    inp_hdr_val = DIF_TABLE;
    while (inp_hdr_val != DIF_DATA) {
        if (read_dif_hdr_elt(inp_file,&inp_hdr_elt,&inp_hdr_val) != OK)
            return FAIL;
        vec_p = &inp_hdr_p->vec[inp_hdr_elt.vector];
        numer_val = (int)inp_hdr_elt.float_val;
        str_val = inp_hdr_elt.str_val;
        switch (inp_hdr_val) {
            case DIF_LABEL:
                if (vec_p->num_labels == 0)
                    if ((vec_p->labels = (char **) mcalloc(NUM_STRS,
                        sizeof(char **))) == NULL)
                        return FAIL;
                if ((vec_p->labels[vec_p->num_labels] =
                    mcalloc(DIF_STR_LEN,sizeof(char))) == NULL)
                    return FAIL;
                strcpy(vec_p->labels[vec_p->num_labels++], str_val);
                break;

            case DIF_COMMENT:
                if (vec_p->num_comments == 0)
                    if ((vec_p->comments = (char **) mcalloc(NUM_STRS,
                        sizeof(char **))) == NULL)
                        return FAIL;
                if ((vec_p->comments[vec_p->num_comments] =
                    mcalloc(DIF_STR_LEN,sizeof(char))) == NULL)
                    return FAIL;
                strcpy(vec_p->comments[vec_p->num_comments++], str_val);
                break;

            case DIF_SIZE:
                vec_p->size = numer_val;
                break;
            case DIF_PERIODICITY:
                vec_p->period = numer_val;
                break;
            case DIF_MAJORSTART:
                vec_p->majorstart = numer_val;
                break;
            case DIF_MINORSTART:
                vec_p->minorstart = numer_val;
                break;
            case DIF_TRUELENGTH:
                vec_p->truelength = numer_val;
                break;
            case DIF_UNITS:
                strcpy(vec_p->units, str_val);
                break;
            case DIF_DISPLAYUNITS:
                strcpy(vec_p->displayunits, str_val);
                break;
            case DIF_DATA: /* state transition: do nothing */
                break;
            default:
                printf("Internal error. %d = <%s>\n",inp_hdr_val,dif_hdr_strs[(int) inp_hdr_val]);
                return FAIL;
                /* break; */
        }
    }

    return OK;
}



/* This top level of input reads a DIF header cell.  A cell in the DIF
   header is preceded by a topic line.  */

int read_dif_hdr_elt(inp_file,inp_hdr_elt_p,inp_hdr_val_p)
FILE *inp_file;
dif_elt_p inp_hdr_elt_p;
enum dif_hdr_vals *inp_hdr_val_p;
{
    char inp_line[80];
    int found = FALSE;
    int i;

    /* First, read the topic line */
    if (read_dif_line(inp_file,inp_line) != OK)
        return FAIL;

    for (i=0; i<=NUM_HDR_VALS; i++) {
        if (stricmp(inp_line,dif_hdr_strs[i]) == 0) {
            found = TRUE;
            break;
        }
    }

    if (!found)
        return FAIL;

    *inp_hdr_val_p = i;
    /* Last, read the actual DIF cell */
    return read_dif_cell(inp_file,inp_hdr_elt_p);
}


/* This second-to-lowest level of input reads an entire DIF cell (via
   read_dif_line).  Please see the text for explanation of DIF cell
   formats. */

int read_dif_cell(inp_file,cell_p)
FILE *inp_file;
dif_elt_p cell_p;
{
    char inp_line[80];
    char *comma;
    char *period;
    int len;

/* this piece of useless code is needed to force Turbo C 2.0 to
   link in the floating point scanf routines.  Otherwise TC2.0 does
   not think they are needed, despite the clear usage of a double
   in other scanf() calls. This code can be deleted if you have patched
   your copy of TC 2.0 with the official Borland patches. These can
   be obtained from:

     CompuServe: bprogb forum, lib 5: TC2PAT.ARC and PATCH.ARC

*/
#if defined(__TURBOC__)
    float useless;
    sscanf(inp_line, "%f", &useless);
#endif

    if (read_dif_line(inp_file,inp_line) != OK)
        return FAIL;

    if (sscanf(inp_line,"%d",&cell_p->vector) != 1) {
        error_msg("Numeric value expected <%s>\n",inp_line);
        return FAIL;
    }

    if ((comma=strchr(inp_line,COMMA)) == NULL) {
        error_msg("Comma expected <%s>\n",inp_line);
        return FAIL;
    }

    if ((period=strchr(inp_line,PERIOD)) == NULL)  /* Numeric Integer */
        cell_p->dec_cnt = 0;
    else
        cell_p->dec_cnt = strlen(period+1);

    if (sscanf(comma+1,"%lf",&cell_p->float_val) != 1) {
        error_msg("Float value expected <%s>\n",inp_line);
        return FAIL;
    }

    cell_p->numer_len = strlen(comma+1);
    if (read_dif_line(inp_file,inp_line) != OK)
        return FAIL;

    len = strlen(inp_line);
    if (inp_line[0]!=QUOTE || inp_line[len-1]!=QUOTE) {
        strcpy(cell_p->str_val,inp_line);  /* No quotes, copy as is */
        return OK;
    }

    inp_line[len-1] = '\0';    /* copy inp_line minus quotes */
    strcpy(cell_p->str_val,inp_line+1);
    return OK;
}



int read_dif_line(inp_file,inp_line)  /* Lowest level of DIF input */
FILE *inp_file;
char *inp_line;
{
    static int cur_line=0;

    if (fgets(inp_line, DIF_STR_LEN, inp_file) == NULL) {
        error_msg("Premature end-of-file on input.\n",NULL);
        return FAIL;
    }
    inp_line[strlen(inp_line)-1] = '\0';

    if (verbose_mode) printf("%05d: %s\n",cur_line++,inp_line);
    return OK;
}


/* The primary task of pass one is to determine the DB3 field types
   which correspond to the DIF vectors.  The DB3 field types are
   determined gradually by check_dif_type_numic() and
   check_dif_type_string() */

int read_dif_data_for_types(inp_file,out_hdr_p)
FILE *inp_file;
db3_hdr_p out_hdr_p;
{
    db3_hdr_elt_p fld_p;
    enum dif_val_vals inp_val;
    dif_elt inp_cell;
    int vector = 0;
    int done = FALSE;

    while (!done) {
        read_dif_cell(inp_file,&inp_cell);
        fld_p = &out_hdr_p->fld[vector];
        switch (inp_cell.vector) {
            case DIF_TYPE_SPECIAL:
                if (lookup_val_strs(inp_cell.str_val,&inp_val)!=OK)
                    return FAIL;
                switch (inp_val) {
                    case DIF_VAL_BOT:  /* Beginning of Tuple */
                        vector = 0;
                        continue;
                    case DIF_VAL_EOD:  /* End of Data        */
                        done = TRUE;
                        break;
                    default:
                        error_msg("Expected BOT or EOD val\n",NULL);
                }
                break;

            case DIF_TYPE_NUMERIC:
                check_dif_type_numeric(&inp_cell,fld_p);
                break;

            case DIF_TYPE_STRING:
                check_dif_type_string(&inp_cell,fld_p);
                break;
        }

        vector++;
    }

    return OK;
}


/* Calculate the remaining fields in the DB3 header so we can write it.
   Note that a DIF tuple is indeed the sames as a DB3 record.  Also,
   DIF LABELS are converted to their DB3 equivalents as field names. */

int calculate_db3_header(inp_hdr_p,out_hdr_p)
dif_hdr_p inp_hdr_p;
db3_hdr_p out_hdr_p;
{
    int i;
    db3_hdr_elt_p fld_p;
    vector_desc_p vec_p;

#if defined(__TURBOC__)
    struct date cur_date;
    getdate(&cur_date);
    out_hdr_p->last_upd[UPD_YY] = (char)(cur_date.da_year - 1900);
    out_hdr_p->last_upd[UPD_MM] = (char)cur_date.da_mon;
    out_hdr_p->last_upd[UPD_DD] = (char)cur_date.da_day;

#else /* MSC */
    struct dosdate_t cur_date;    /* from INT 21h */
    _dos_getdate(&cur_date);  /* Use DOS date as internal file date */
    out_hdr_p->last_upd[UPD_YY] = (char)(cur_date.year - 1900);
    out_hdr_p->last_upd[UPD_MM] = (char)cur_date.month;
    out_hdr_p->last_upd[UPD_DD] = (char)cur_date.day;

#endif

    out_hdr_p->ver_no = DB3_FILE_TYPE;
    out_hdr_p->num_recs = inp_hdr_p->num_tuples;
    out_hdr_p->size_hd = DB3_HDR_SIZE * (inp_hdr_p->num_vectors + 1) + 2;

    out_hdr_p->size_rec = 1;     /* 1 byte for active record flag */
    for (i=0; i < inp_hdr_p->num_vectors; i++) {
        fld_p = &out_hdr_p->fld[i];
        vec_p = &inp_hdr_p->vec[i];

        /* Attach DIF LABEL as field name, if any */
        if (vec_p->num_labels)
            build_db3_name_from_label(fld_p->fld_name, *vec_p->labels);
        else
            sprintf(fld_p->fld_name,"Field_%03d",i);

        /* Setup field sizes for DATE and LOGICAL fields */
        if (fld_p->fld_type == DB3_DATE)
            fld_p->fld_size = DB3_DATE_SIZE;
        else if (fld_p->fld_type == DB3_LOGICAL)
            fld_p->fld_size = DB3_LOGICAL_SIZE;
        else if (fld_p->fld_type == DB3_NONE) {  /* empty vector, set defs. */
            fld_p->fld_type = DB3_CHAR;
            fld_p->fld_size = DB3_DEFAULT_SIZE;
        }

        /* Compute record size, data address is offset into the record */
        fld_p->fld_data_addr = out_hdr_p->size_rec;
        out_hdr_p->size_rec += fld_p->fld_size;
    }

    /* This buffer will store one record during I/O */
    if ((out_hdr_p->data=mcalloc(out_hdr_p->size_rec,sizeof(char))) == NULL)
        return FAIL;
    *out_hdr_p->data = BLANK;  /* mark as non-deleted */

    return OK;
}

/* Create a valid DB3 field name based on the DIF label. Note that DB3
   names may contain a maximum of 10 characters with no embedded blanks. */

int build_db3_name_from_label(db3_name, dif_name)
char *db3_name;      /* DB3 name we're going to build   */
char *dif_name;      /* DIF LABEL that name is based on */
{
    int i=0;

    while (*dif_name != '\0') {
        if (*dif_name != BLANK)
            *db3_name++ = *dif_name++;
        else {
            *db3_name = UNDERSCORE;  /* replace BLANK */
            dif_name++;
        }
        if (i++ == DB3_NAME_LEN-2)
            break;
    }

    *db3_name = '\0';
    return OK;
}



/* Write out the DB3 from the information collected in pass one.
   Note the trailing zeroes following the actual header data. */

int write_db3_header(out_file,inp_hdr_p,out_hdr_p)
int out_file;           /* DB3 file that we're writing */
dif_hdr_p inp_hdr_p;    /* DIF header from input file  */
db3_hdr_p out_hdr_p;    /* DB3 header for output file  */
{
    int i;
    int end_of_headers = 0x000D;
    char zeroes[DB3_HDR_SIZE-DB3_HDR_ACTUAL_SIZE];

    memset(zeroes, 0, sizeof(zeroes));
    write(out_file, (char *)out_hdr_p, DB3_HDR_SIZE);
    for (i=0; i < inp_hdr_p->num_vectors; i++) {
        write(out_file, (char *)&out_hdr_p->fld[i], DB3_HDR_ACTUAL_SIZE);
        write(out_file, zeroes, DB3_HDR_SIZE-DB3_HDR_ACTUAL_SIZE);
    }

    write(out_file, (char *)&end_of_headers, sizeof(int));
    return OK;
}

/* During pass two over the input file, we will skip over the header
   and convert each cell into the native DB3 format */

int read_dif_data_and_write_db3(inp_file,out_file,out_hdr_p)
FILE *inp_file;
int out_file;
db3_hdr_p out_hdr_p;
{
    enum dif_hdr_vals inp_hdr_val;
    enum dif_val_vals inp_val;
    dif_elt inp_cell;
    int vector = 0;
    int tuple = 0;
    int done = FALSE;

    if (verbose_mode) printf("Starting pass two...\n\n");

    /* First, skip past header records */
    inp_hdr_val = DIF_TABLE;
    while (inp_hdr_val != DIF_DATA)   /* Flag for start of data */
        if (read_dif_hdr_elt(inp_file,&inp_cell,&inp_hdr_val) != OK)
            return FAIL;

    while (!done) {
        read_dif_cell(inp_file,&inp_cell);
        switch (inp_cell.vector) {
            case DIF_TYPE_SPECIAL:
                lookup_val_strs(inp_cell.str_val,&inp_val);
                switch (inp_val) {
                    case DIF_VAL_BOT:  /* Beginning of Tuple */
                        if (tuple)
                            write(out_file, out_hdr_p->data, out_hdr_p->size_rec);
                        vector = 0;
                        tuple++;
                        continue;
                    case DIF_VAL_EOD:  /* End of Data        */
                        write(out_file, out_hdr_p->data, out_hdr_p->size_rec);
                        done = TRUE;
                        break;
                    default:
                        error_msg("Expected BOT or EOD val, found <%s>\n",
                            dif_val_strs[inp_val]);
                }
                break;

            case DIF_TYPE_NUMERIC:  /* Insert cell into output buffer */
                add_dif_numeric_to_db3_record(vector,&inp_cell,out_hdr_p);
                break;

            case DIF_TYPE_STRING:   /* Insert cell into output buffer */
                add_dif_string_to_db3_record(vector,&inp_cell,out_hdr_p);
                break;
        }

        vector++;
    }
    return OK;
}


/* Format a DIF string cell into the record I/O buffer */

int add_dif_numeric_to_db3_record(vector,inp_cell_p,out_hdr_p)
int vector;                /* current DIF vector            */
dif_elt_p inp_cell_p;      /* DIF cell we're going to add   */
db3_hdr_p out_hdr_p;       /* DB3 header where data will go */
{
    db3_hdr_elt_p fld_p;
    char *data_p;
    char temp[8];
    char format[10];
    char local[20];
    int size;

    fld_p = &out_hdr_p->fld[vector];
    data_p = out_hdr_p->data + fld_p->fld_data_addr;
    size = fld_p->fld_size;
    memset(data_p, BLANK, size);

    switch(fld_p->fld_type) {
        case DB3_NUMERIC:
        case DB3_CHAR:
            strcpy(format,"%16.");
            itoa((int)fld_p->fld_dec_cnt,temp,4);
            strcat(format,temp);
            strcat(format,"f");
            sprintf(local,format,inp_cell_p->float_val);
            strncpy(data_p,local+strlen(local)-size,size);
            break;
        case DB3_LOGICAL:
            *data_p = inp_cell_p->float_val ? (char)'T' : (char)'F';
            break;
        default:
            error_msg("Type determination failure, non-numeric data\n",NULL);
    }
    return OK;
}


/* Format a DIF string cell into the record I/O buffer */

int add_dif_string_to_db3_record(vector,inp_cell_p,out_hdr_p)
int vector;               /* current DIF vector            */
dif_elt_p inp_cell_p;     /* DIF cell we're going to add   */
db3_hdr_p out_hdr_p;      /* DB3 header where data will go */
{
    db3_hdr_elt_p fld_p;
    char *data_p;
    int size;
    int len;

    fld_p = &out_hdr_p->fld[vector];
    data_p = out_hdr_p->data + fld_p->fld_data_addr;
    size = fld_p->fld_size;
    len = strlen(inp_cell_p->str_val);
    memset(data_p, BLANK, size);

    switch(fld_p->fld_type) {
        case DB3_CHAR:
            strncpy(data_p,inp_cell_p->str_val,len);
            break;
        case DB3_DATE:
            format_db3_date_from_string(data_p,inp_cell_p->str_val);
            break;
        case DB3_NUMERIC:
            error_msg("a db3 numeric field cannot receive a string value\n",NULL);
            break;
        case DB3_LOGICAL:
            format_db3_logical_from_string(data_p,inp_cell_p->str_val);
            break;
        default:
            error_msg("Type determination failure, non-numeric data\n",NULL);
    }
    return OK;
}


/* Examine the DIF number in the cell and try to find which type of
   DB3 field would best accommodate it.  */

int check_dif_type_numeric(inp_cell_p,fld_p)
dif_elt_p inp_cell_p;
db3_hdr_elt_p fld_p;
{
    enum dif_val_vals inp_val;

    if (lookup_val_strs(inp_cell_p->str_val,&inp_val)!=OK)
        return FAIL;
    /* Actions based on previous notion of field type */
    switch (fld_p->fld_type) {
        case DB3_NONE:
            switch (inp_val) {
                case DIF_VAL_V:
                case DIF_VAL_NA:
                case DIF_VAL_ERROR:
                    fld_p->fld_type = DB3_NUMERIC;
                    break;
                case DIF_VAL_TRUE:
                case DIF_VAL_FALSE:
                    fld_p->fld_type = DB3_LOGICAL;
                    break;
            }
            break;

        case DB3_CHAR:
            if (inp_cell_p->numer_len > fld_p->fld_size)
                fld_p->fld_size = (char)inp_cell_p->numer_len;
            break;

        case DB3_DATE:
            fld_p->fld_type = DB3_CHAR;    /* DATE not conformable to NUMERIC */
            break;

        case DB3_LOGICAL:
            switch (inp_val) {
                case DIF_VAL_V:
                case DIF_VAL_NA:
                case DIF_VAL_ERROR:
                    fld_p->fld_type = DB3_NUMERIC;
                    break;
            }
            break;
    }

    if (fld_p->fld_type == DB3_NUMERIC) {
        if (inp_cell_p->dec_cnt > fld_p->fld_dec_cnt)
            fld_p->fld_dec_cnt = (char)inp_cell_p->dec_cnt;
        if (inp_cell_p->numer_len > fld_p->fld_size)
            fld_p->fld_size = (char)inp_cell_p->numer_len;
    }

    return OK;
}


/* Examine the DIF string in the cell and try to find which type of
   DB3 field would best accommodate it.  */

int check_dif_type_string(inp_cell_p,fld_p)
dif_elt_p inp_cell_p;   /* DIF cell to check   */
db3_hdr_elt_p fld_p;    /* DB3 field to modify */
{
    char *str;
    int len;
    str = inp_cell_p->str_val;

    /* Actions based on previous notion of field type */
    switch (fld_p->fld_type) {
        case DB3_NONE:
            if (strlen(str)==0)   /* Can't make a decision on empty strings */
                break;
            if (is_logical(str))
                fld_p->fld_type = DB3_LOGICAL;
            else if (is_date(str))
                fld_p->fld_type = DB3_DATE;
            else
                fld_p->fld_type = DB3_CHAR;
            break;

        case DB3_DATE:
            if (!is_date(str))            /* Not a DATE string    */
                fld_p->fld_type = DB3_CHAR;
            break;

        case DB3_LOGICAL:
            if (!is_logical(str))         /* Not a LOGICAL string */
                fld_p->fld_type = DB3_CHAR;
            break;

        case DB3_NUMERIC:
            fld_p->fld_type = DB3_CHAR;
            break;
    }

    if (fld_p->fld_type == DB3_CHAR) {
        len = strlen(inp_cell_p->str_val);
        if (len > fld_p->fld_size)
            fld_p->fld_size = (char)len;
    }

    return OK;
}


/* Translate the string into its corresponding enumerated value */

int lookup_val_strs(str,val_p)
char *str;
enum dif_val_vals *val_p;
{
    int found = FALSE;
    int i;

    for (i=0; i<=NUM_VAL_VALS; i++)
        if (stricmp(str,dif_val_strs[i]) == 0) {
            found = TRUE;
            *val_p = i;
            break;
        }

    if (!found) {
        error_msg("Unknown value indicator <%s>\n",str);
        return FAIL;
    }

    return OK;
}



/* Free all vectors and vector data (labels and comments) in the
   DIF header */

int free_dif_hdr(inp_hdr_p)
dif_hdr_p inp_hdr_p;
{
    vector_desc_p vec_p;
    int i;
    int cur_item;

    for (i=0; i < inp_hdr_p->num_vectors; i++) {
        vec_p = &inp_hdr_p->vec[i];
        for (cur_item=0; cur_item < vec_p->num_labels; cur_item++)
            mfree(vec_p->labels[cur_item]);
        for (cur_item=0; cur_item < vec_p->num_comments; cur_item++)
            mfree(vec_p->comments[cur_item]);
        if (vec_p->labels != NULL)
            mfree((char *)vec_p->labels);
        if (vec_p->comments != NULL)
            mfree((char *)vec_p->comments);
    }

    mfree((char *)inp_hdr_p->vec);  /* Lastly, free all the fields */
    inp_hdr_p->vec = NULL;
    return OK;
}


int free_db3_hdr(out_hdr_p)   /* Free the DB3 header and all fields */
db3_hdr_p out_hdr_p;
{
    mfree(out_hdr_p->data);
    mfree((char *)out_hdr_p->fld);
    out_hdr_p->data = NULL;
    out_hdr_p->fld = NULL;
    return OK;
}


char *mcalloc(num,size)  /* Just one level above ordinary calloc() */
int num;
int size;
{
    char *retval;

    retval = calloc(num,size);
    if (retval == NULL)
        error_msg( "Insufficient memory\n",NULL);
    return retval;
}


int mfree(ptr)    /* Just one level above ordinary free() */
char *ptr;
{
    if (ptr==NULL)
        error_msg( "Attempt to free NULL ptr\n",NULL);
    else
        free(ptr);
    return OK;
}


int error_msg(format_str,data_str)  /* customize for your application */
char *format_str;
char *data_str;
{
    fprintf(stdout,format_str,data_str);
    return OK;
}


int is_date(test_str)  /* See if this string contains date data */
char *test_str;
{
    char dest_str[20];

    return format_db3_date_from_string(dest_str, test_str);
}



/* convert from any valid date format in the source string ("MM/DD/YY",
   "MM/DD/YYYY", "MM-DD-YY", "MM-DD-YYYY", "MM.DD.YY", or "MM.DD.YYY")
   to the DB3 destination date format "YYMMDD". */

int format_db3_date_from_string(dest_str, source_str)
char *dest_str;
char *source_str;
{
    char local[80];
    char *ptr;
    int month, day, year;
    int dashes=0;
    int slashes=0;
    int periods=0;

    /* Convert the source string to a parsable format, while checking
       validity along the way */
    strcpy(local,source_str);
    for (ptr=local; *ptr!='\0'; ptr++) {
        if (*ptr == PERIOD) {
            *ptr = SLASH;
            periods++;
        }
        else if (*ptr == SLASH)
            slashes++;
        else if (*ptr == DASH) {
            *ptr = SLASH;
            dashes++;
        }
        else if (*ptr < '0' || *ptr > '9')
            return FAIL;
    }

    /* Valid date formats:  mm/dd/yy, mm-dd-yy, mm.dd.yy  */
    if ((periods==2 && slashes==0 && dashes==0) ||
        (periods==0 && slashes==2 && dashes==0) ||
        (periods==0 && slashes==0 && dashes==2))  {

        if (sscanf(local,"%d/%d/%d",&month,&day,&year) < 3)
            return FAIL;
        if (year < 100)     /* convert two digit to four digit years */
            year += 1900;
        if (!month || !day)
            return FAIL;
        sprintf(local,"%04d%02d%02d",year,month,day);
        strncpy(dest_str, local, strlen(local));
    }
    else
        return FAIL;

    return OK;
}


int is_logical(str)  /* See if this string contains logical data */
char *str;
{
    char dest_str[2];

    return format_db3_logical_from_string(dest_str,str);
}

/* If the source string is a DB3 logical (Y, T, N, or F) then
   copy it to the destination string, else the result is blank */

int format_db3_logical_from_string(dest_str, source_str)
char *dest_str;
char *source_str;
{
    *dest_str = BLANK;
    if (strlen(source_str) != 1)
        return FAIL;

    if (strpbrk(source_str,"yYtTnNfF") != NULL) {
        *dest_str = *source_str;
        return OK;
    }
    return FAIL;
}


#define calc_num_flds(size_hd)  ((size_hd - 2) / DB3_HDR_SIZE - 1)

int dump_db3_hdr(inp_hdr_p)
db3_hdr_p inp_hdr_p;
{
    int num_flds;
    int i;

    num_flds = calc_num_flds(inp_hdr_p->size_hd);
    printf("DBase III  -  .DBF Header Report\n\n\n");
    printf("DBase version:  %d\n",inp_hdr_p->ver_no & 0x7F);
    if (inp_hdr_p->ver_no & 0x80)
        printf("This file contains memo fields and requires ancilliary .DBT file\n");
    printf("Last update: %02d/%02d/%02d\n",inp_hdr_p->last_upd[UPD_MM],
        inp_hdr_p->last_upd[UPD_DD], inp_hdr_p->last_upd[UPD_YY]);
    printf("Number of records:    %ld\n",inp_hdr_p->num_recs);
    printf("Size of header area:  %d bytes\n",inp_hdr_p->size_hd);
    printf("Size of a record:     %d bytes\n",inp_hdr_p->size_rec);

    printf("\n\n\nDBase III  -  .DBF Field Header Report\n\n\n");
    printf("Name       | Type | Data Addr | Size | Dec Cnt \n");
    printf("-----------+------+-----------+------+---------\n");

    for (i=0; i<num_flds; i++)
        dump_db3_hdr_elt(&inp_hdr_p->fld[i]);

    return OK;
}


dump_db3_hdr_elt(fld_p)  /* Dump a report of a single DB3 field */
db3_hdr_elt_p fld_p;
{
    printf("%-11s|  %c   | %8lX  | %3d  |%2d\n",fld_p->fld_name,
        fld_p->fld_type, fld_p->fld_data_addr, fld_p->fld_size,
        fld_p->fld_dec_cnt);
    return OK;
}
