/***************************** NEWTOUCH.C *************************
*   Program to expand the functions of the Unix 'touch' utility
*
*   Usage:   newtouch  filename  new-date new-time
*
*            filename may contain wildcards. With wildcards, all
*                     matching files will be set to the specified
*                     date and time. With no wildcards, only the
*                     specified file is set; if the file does not
*                     exist, it is created.
*            new-date [optional]  date in the format MM/DD/YY
*                     YY is 20th century; if not specified,
*                     the existing file date is used
*            new-time [optional]  time in the format HH:MM:SS
*                     uses a 24-hour clock; if not specified,
*                     the existing file time is used.
*
*            Note: if time and date are omitted, the current time
*                  and date are used.
*
*   Copyright (c) 1989 Andrew Binstock. May be distributed as is
*   as long as no fee of any kind is charged. C Gazette, Spring 1989
*******************************************************************/

/* OK for Microsoft and Turbo. For Microsoft, you must #define MSC */

/* #define MSC 1 */

#include <stdio.h>
#include <dos.h>
#include <stdlib.h>
#include <string.h>

#if __TURBOC__
#include <dir.h>
#endif

#define YES  1
#define NO   0

int hh, min, ss,
    yy, mm,  dd,
    wildcards;

FILE *touch_me;

void show_usage(void),
     current_date_and_time(void),
     set_file_date_time (int, char *);

int  get_file_name  (char *, char *),
     get_next_name (char *, char *);

int  time_specified,    /* Switched to YES if true */
     date_specified;

main (int argc, char *argv[])
{
    int i;

    if (argc < 2)
    {
        show_usage();
        exit (5);
    }

    wildcards = (((strpbrk (argv[1], "?*")) == NULL) ? NO : YES);

    for (i = 2; i < argc; i++)
    {
        if (strchr (argv[i], ':') != NULL)      /* time found */
        {
            sscanf(argv[i], "%d:%d:%d", &hh, &min, &ss);
            time_specified = YES;
        }
        else
        if (strpbrk (argv[i], "/-") != NULL)    /* date found */
        {
            char c;
            sscanf (argv[i], "%d%c%d%c%d", &mm, &c, &dd, &c, &yy);
            date_specified = YES;
        }
        else
        {
            show_usage();
            exit (6);
        }
    }   /* end of argument scan */

    for (;;)
    {
        char filename[80];

        if (get_file_name (filename, argv[1]) == NO) /*If no more files, quit */
            break;
        else
        {
            if (wildcards == NO) /* Files can only be created if = NO */
            {                    /* so create file as needed.         */
                if ((touch_me = fopen (filename, "r")) == NULL)
                {
                    if ((touch_me = fopen (filename, "w")) == NULL)
                    {
                        fprintf (stderr, "Cannot create %s\n", filename);
                        exit (7);
                    }
                }
            }
            else
            {
                if ((touch_me = fopen (filename, "r")) == NULL)
                {
                    fprintf (stderr, "Cannot open %s\n", filename);
                    continue;
                }
            }

            /* At this point, the file is open */

            set_file_date_time (fileno(touch_me), filename);
            fclose (touch_me);
        }
    }   /* end of file names - fall through to exit */

    exit (0);
}

int get_file_name(char *name, char *spec)    /* Get name of next file */
{
    static int no_more = NO;

    if (no_more)
        return NO;

    if (wildcards == NO)
    {
        strcpy (name, spec);    /* No wildcards, use the name as is */
        no_more = YES;          /* and stop file name generation.   */
        return (YES);
    }
    else
    {
        if (get_next_name (name, spec) == NO)
        {
            no_more = YES;          /* No more matching files */
            return (NO);
        }
        else
            return (YES);           /* Matching files still found */
    }
}

int get_next_name (char *name, char *spec)/* Get next matching wildcard name */
{

#define NO_MORE  -1

	static struct ffblk {
		char reserved [21];
		char attribute;
		unsigned write_time;
		unsigned write_date;
		long sizeof_file;
		char ff_name [13];
		}
			 file_info;     /* structure to hold file match info */
    static int been_here = NO;

    if (been_here == NO)        /* On first time, do findfirst() */
	{
#if __TURBOC__
        if (findfirst (spec, &file_info, 0) == NO_MORE)
#elif MSC
        if (_dos_findfirst (spec, 0x7F, &file_info) != 0)
#endif
			return (NO);
		else
			strcpy (name, file_info.ff_name);

        been_here = YES;
        return (YES);
    }
    else
	{
#if __TURBOC__
		if (findnext (&file_info) == NO_MORE)
#elif MSC
		if (_dos_findnext (&file_info) != 0)
#endif
            return (NO);
        else
        {
            strcpy (name, file_info.ff_name);
            return (YES);
        }
    }
}

void set_file_date_time (int file_handle, char *name)
{
    /* First, get the file's present date and time,
       then overlay with any specified date and time */

    union REGS inreg, outreg;

    struct set_time {               /* Bit fields  */
        unsigned set_sec:  5;       /* seconds / 2 */
        unsigned set_min:  6;       /* minutes     */
        unsigned set_hrs:  5;       /* hours       */
        }
            st;

     struct set_date {
        unsigned set_day:  5;       /* day of month*/
        unsigned set_mon:  4;       /* month       */
        unsigned set_yr:   7;       /* year - 1980 */
        }
            sd;

    /* Get the file's present date and time */

    inreg.h.ah = 0x57;      /* function for get/set file date and time */
    inreg.h.al = 0x00;      /* get */
    inreg.x.bx = file_handle;
    int86 (0x21, &inreg, &outreg);

    st = *(struct set_time *) &outreg.x.cx;
    sd = *(struct set_date *) &outreg.x.dx;

    /* If nothing specified, use the current date and time */

    if (!time_specified && !date_specified)
    {
        current_date_and_time();
        st.set_sec = ss / 2;
        st.set_min = min;
        st.set_hrs = hh;
        sd.set_day = dd;
        sd.set_mon = mm;
        sd.set_yr  = yy - 1980;
     }

    /* If something specified, use that */

    if (time_specified)               /* If a time was specified,  */
    {                                 /* use it, else use defaults */
        st.set_sec = ss / 2;
        st.set_min = min;
        st.set_hrs = hh;
    }

    if (date_specified)               /* If a date was specified,   */
    {                                 /* use it, else use defaults  */
        sd.set_day = dd;
        sd.set_mon = mm;
        sd.set_yr  = (yy - 80);
    }

    /* At last, set the values */

    inreg.h.ah = 0x57;      /* function for get/set file date and time */
    inreg.h.al = 0x01;      /* set */
    inreg.x.bx = file_handle;
    inreg.x.cx = * ((unsigned int *) &st);
    inreg.x.dx = * ((unsigned int *) &sd);

    int86 (0x21, &inreg, &outreg);

    printf ("%s set to %02d-%02d-%02d %02d:%02d:%02d\n", name,
            sd.set_mon, sd.set_day, sd.set_yr,
            st.set_hrs, st.set_min, st.set_sec);
}

void current_date_and_time (void)  /* Get them */
{
#if __TURBOC__
    struct date curr_date;
    struct time curr_time;

    getdate (&curr_date);
    gettime(&curr_time);

    yy = curr_date.da_year; mm = curr_date.da_mon; dd = curr_date.da_day;
    hh = curr_time.ti_hour; min= curr_time.ti_min; ss = curr_time.ti_sec;

#elif MSC
    struct dosdate_t curr_date;
    struct dostime_t curr_time;

    _dos_getdate (&curr_date);
    _dos_gettime (&curr_time);

    yy = curr_date.year; mm = curr_date.month; dd = curr_date.day;
    hh = curr_time.hour; min= curr_time.minute; ss = curr_time.second;
#endif
}

void show_usage(void)        /* Explain specification format */
{
    fputs ("Usage: newtouch filename new-date new-time\n"      , stderr);
    fputs ("  filename may contain wildcards\n"                , stderr);
    fputs ("  date is optional using MM/DD/YY or MM-DD-YY\n"   , stderr);
    fputs ("  time is optional using HH:MM:SS and 24-hr. clock", stderr);
}


