{****************************************************************************}
{*                                                                          *}
{*  TTTypes                                                                 *}
{*                                                                          *}
{*  This unit contains the data types used by the TrueType engine           *}
{*                                                                          *}
{****************************************************************************}

unit TTTypes;

interface

type

  (* IntN types :                                                       *)
  (*                                                                    *)
  (*  These types are used as a way to garantee the size of some        *)
  (*  specific integers.                                                *)
  (*                                                                    *)

  Int16 = Integer;            (* 16 bits integer *)
  Int32 = LongInt;            (* 32 ""           *)
  Int64 = record              (* 64 ""           *)
            Hi,
            Lo  : LongInt;
          end;


  (* BYTE is already defined in Pascal       *)
  (* They are equivalent to C unsigned chars *)

  UShort   = Word;          (* unsigned short integer, must be on 16 bits *)
  Short    = Integer;       (* signed short integer,   must be on 16 bits *)
  ShortRec = record
               Low,         (* a structure used for various type casts *)
               High : Byte;
             end;

  ULong = LongInt;         (* unsigned long integer, must be on 32 bits *)
                           (* NOTE : There is no 'LongWord' in Pascal,  *)
                           (*        but the unsigned ops are all in    *)
                           (*        the inline assembly routines       *)

  Long  = LongInt;         (* signed long integer,   must be on 32 bits *)

  Fixed    = Longint;      (* Signed Fixed 16.16 Float *)
  FixedRec = record        (* Structure used for various typecasts *)
               Low,
               High  : Integer;
             end;

  FixedPoint = FixedRec;

{$IFDEF OS2}
  Int = LongInt;      (* the 'int' type is used for loop counters and  *)
{$ELSE}               (* indexes.. Their size must be the one a given  *)
  Int = Integer;      (* system handles most easily ( 16 bits on Turbo *)
{$ENDIF}              (* and 32 on Virtual Pascals )                   *)

  (* FUnits are the distance unit used for the EM square *)

  FWord = Short;      (* Distance in FUnits *)

  UFWord = UShort;    (* unsigned distance *)

  F2Dot14 = Short;    (* signed fixed float 2.14 used for unary vectors *)
                      (* Layout :                                       *)
                      (*                                                *)
                      (*  s : 1  -- sign bit                            *)
                      (*  m : 1  -- mantissa bit                        *)
                      (*  f : 14 -- unsigned fractional part            *)
                      (*                                                *)
                      (*  's:m' is the 2-bit signed integer value to    *)
                      (*  which the positive fractional part should be  *)
                      (*  added.                                        *)
                      (*                                                *)

  TUnitVector = record
                 x, y : F2Dot14; (* guess what ? *)
                end;

  F26Dot6 = LongInt;  (* 26.6 Fixed float, used for glyph points'  *)
                      (* pixel coordinates                         *)
  TVector = record
              x, y : F26Dot6;  (* Simple pixel vector *)
            end;

  PVecTable = ^TVecTable;
  TVecTable  = Array[0..1000] of TVector;
  (* a vector table type *)

  PTouchTable = ^TTouchTable;
  TTouchTable = Array[0..1000] of Byte;
  (* a flag table type *)

  TVecRecord = record
                N     : int;         (* points number                *)
                Org   : PVecTable;   (* original coordinates from EM *)
                Cur   : PVecTable;   (* current coordinates          *)
                Touch : PTouchTable; (* flags table                  *)
               end;

  (* This type defining a set of glyph points will be used to represent *)
  (* each zones ( regular and twilight ) during instructions decoding   *)

  TContour = record
              First, Last : Int;
             end;
  (* First and Last point index of a contour in a glyph *)

  PContourTable = ^TContourTable;
  TContourTable = array[0..15] of TContour;
  (* A contour bounds table *)

  TContourRecord = record
                    N : int;           (* Contours numbers *)
                    C : PContourTable; (* Contours table   *)
                   end;

  (* This type is used to define the contours of each glyph *)


  (* Simple access types : pointers and tables *)

  PUShort     = ^UShort;
  PShort      = ^Short;

  PULong      = ^ULong;
  PLong       = ^Long;

  PFixed      = ^Fixed;
  PFixedRec   = ^FixedRec;

  TByteArray  = array[0..63999] of Byte;
  PByteArray  = ^TByteArray;

  PShortArray = ^TShortArray;
  TShortArray = array[0..1023] of Integer;

  TStorage    = array[0..16000] of Long;
  PStorage    = ^TStorage;

  TPoint = Record
             GlyphNum : integer; (* this point's glyph number   *)
             V        : TVector; (* current pixel position      *)
             Touch    : Byte;    (* touch flags                 *)
            end;


  PTTZone = ^TTTZone;
  TTTZone = Array[0..1023] of TPoint;



  TCVTRecord  = record
                 N : int;       (* size in 32 bits elements *)
                 A : PStorage;  (* table address            *)
                end;
  (* The CVT is a simple storage whose values are defined and used *)
  (* by the font and glyphs programs                               *)



const
  CTTFlagTouchedX = $01;  (* X touched flag *)
  CTTFlagTouchedY = $02;  (* Y touched flag *)




(*****************************************************)
(*                                                   *)
(*              TrueType Tables Types                *)
(*                                                   *)
(*****************************************************)

type
  (* Graphics State                            *)
  (*                                           *)
  (* The Graphics State (GS) is managed by the *)
  (* instruction field, but does not come from *)
  (* the font file. Thus, we can use 'int's    *)
  (* where needed.                             *)
  (*                                           *)

  PGraphicsState = ^TGraphicsState;
  TGraphicsState = record
                     autoFlip                : boolean;
                     controlValueCutIn       : F26dot6;
                     deltaBase               : int;
                     deltaShift              : int;

                     dualVector,
                     projVector,
                     freeVector              : TUnitVector;

                     gep0,
                     gep1,
                     gep2                    : int;

                     instructControl         : byte;
                     loop                    : Int32;

                     minimumDistance         : F26dot6;
                     roundState              : int;

                     rp0,
                     rp1,
                     rp2                     : int;

                     scanControl             : boolean;
                     singleWidthCutIn        : F26dot6;
                     singleWidthValue        : F26dot6;
                   end;



  (* TrueType Table Directory type *)

  TTableDir = Record
                version     : FixedPoint;   (* should be $10000 *)
                numTables   : word;         (* Tables number    *)

                searchRange,           (* These parameters are only used  *)
                entrySelector,         (* for a dichotomy search in the   *)
                rangeShift     : Word; (* directory. We ignore them       *)
               end;

  (* The 'TableDir' is followed by 'numTables' TableDirEntries *)

  TTableDirEntry = Record
                     Tag      : array[0..3] of Char; (*        table type *)
                     CheckSum : Long;                (*    table Checksum *)
                     Offset   : Long;                (* Table file offset *)
                     Length   : Long;                (*      Table length *)
                    end;

  TTableDirEntries = array[0..100] of TTableDirEntry;
  PTableDirEntries = ^TTableDirEntries;

  (* "cmap" Table *)

  TCMapDir = Record
               TableVersionNumber : UShort;  (* should be 0 *)
               cMapNum            : UShort;  (* number of entries *)
              end;

  (* The "cmap" is followed by cMapNum TCMapDirEntries *)

  TCMapDirEntry = Record
                    PlatformID,                   (* Windows = 3          *)
                    PlatformEncodingID : UShort;  (* Microsoft UGL = 1    *)
                    offset             : Long;    (*       offset         *)
                   end;

  (* There exists 4 distinct formats : 0,2,4 et 6
     Until now, we only process format 4 *)

  TCMap4 = Record
             Format,                (* should be 4 *)
             Length,                (* size in bytes *)
             Version,               (* version number, starting at 0 *)
             segCountX2,            (* segments number * 2 *)

             SearchRange,           (* these parameteres can be useful *)
             EntrySelector,         (* for a dichotomy search *)
             RangeShift     : UShort;
            end;

  TCMap4Segment = Record
                    endCount,
                    startCount,
                    idDelta,
                    idRangeOffset : UShort;
                   end;

  (* table "maxp" of 'Maximum Profiles' *)

  TMaxProfile = Record
                  Version                 : Fixed;
                  numGlyphs,
                  maxPoints,
                  maxContours,
                  maxCompositePoints,
                  maxCompositeContours,
                  maxZones,
                  maxTwilightPoints,
                  maxStorage,
                  maxFunctionDefs,
                  maxInstructionDefs,
                  maxStackElements,
                  maxSizeOfInstructions,
                  maxComponentElements,
                  maxComponentDepth       : UShort;
                end;


  (* table of tag "glyph" *)

  TPointRec = record
               x, y, flag : integer;
              end;

  TPoints = array[0..1023] of TPointRec;
  PPoints = ^TPoints;

  PGlyphContour = ^TGlyphContour;
  TGlyphContour = record
                   Start,
                   Finish : word;
                  end;

  PGlyphContours = ^TGlyphContours;
  TGlyphContours = array[0..1023] of TGlyphContour;

  PGlyph = ^TGlyph;
  TGlyph = Record
            numberOfContours,
            xMin,
            yMin,
            xMax,
            yMax,
            numberOfPoints : Short;

            Contours : PGlyphContours;
            Points   : PPoints;
           end;

  TGlyphs = array[0..1000] of TGlyph;
  PGlyphs = ^TGlyphs;

  (* table of type "head" *)

  TLongDateTime = record
                    L1,
                    L2 : long;
                  end;

  THeader = Record
             TableVersion : FixedPoint;
             FontRevision : FixedPoint;

             CheckSumAdjust : Long;
             MagicNumber    : Long;

             Flags      : UShort;
             UnitsPerEM : UShort;

             Created  : TLongDateTime;
             Modified : TLongDateTime;

             xMin : Short;
             yMin : Short;
             xMax : Short;
             yMax : Short;

             MacStyle      : UShort;
             LowestRecPPEM : UShort;

             FontDirection     : Short;
             IndexToLocFormat  : Short;
             GlyphDataFormat   : Short;
            end;

  (* The engine provides its own memory management, it uses a simple   *)
  (* growing heap, with only Alloc/Mark/Release operations. That's the *)
  (* fastest heap you can have, but forces you to manage it gently. We *)
  (* do !!                                                             *)

  TMarkRecord = record
                  Magic : Long;    (* Magic number to identify a valid *)
                  Top   : Int;     (* Mark Record                      *)
                end;

var
  MaxProfile : ^TMaxProfile;         (* Maximum Profile is a global *)
                                     (* ( for now )                 *)

const
  Font_Pool_Allocated : boolean = False;

procedure InitBuffer ( var Buff; size : longint );
(* Inits the Font Pool's memory management *)

function  Alloc  ( Size : int; var P: Pointer ): boolean;
(* Allocate a block a 'Size' bytes from the heap, and returns its *)
(* address in 'P'. Returns False on failure                       *)

procedure Mark( var M : TMarkRecord );
(* Mark the Font Pool's current state into 'M' *)

function  Release( M : TMarkRecord ) : boolean;
(* Release the Font Pool to its previous state recorded in 'M' *)
(* Returns False on failure                                    *)
(* WARNING: 'M' is no longer valid after the call              *)
(*          this being useful for debugging purposes           *)

implementation

const
  Mark_Magic = $BABE0007;

(***********************************)
(*                                 *)
(*   FONT POOL MANAGEMENT          *)
(*                                 *)
(***********************************)

var
  Buffer   : PByteArray;
  SizeBuff : longint;
  CurBuff  : longint;

procedure InitBuffer;
begin
  Buffer   := PByteArray(@Buff);
  SizeBuff := Size;
  CurBuff  := 0;

  Font_Pool_Allocated := True;
end;


function Alloc( Size : int; var P: Pointer ): boolean;
var
  L : int;
begin
  P     := nil;
  Alloc := False;
  L     := CurBuff + ( Size+3 ) and -4;

  if L >= SizeBuff then Alloc:=False
   else
    begin
     P       := @Buffer^[CurBuff];
     CurBuff := L;
     Alloc   := True;
    end
end;

(**********)
(*  Mark  *)
(*        ***********************************************)
(*                                                      *)
(*  Gets the current heap top position and put it       *)
(*  into a mark record.                                 *)
(*                                                      *)
(********************************************************)

procedure Mark( var M : TMarkRecord );
begin
  M.Magic := Mark_Magic;
  M.Top   := CurBuff;
end;

(*************)
(*  Release  *)
(*           ********************************************)
(*                                                      *)
(*  Sets the heap top to a previously saved position    *)
(*  with 'Mark'. Returns FALSE is the mark is invalid   *)
(*  or has been already released.                       *)
(*                                                      *)
(********************************************************)

function Release( M : TMarkRecord ) : boolean;
begin
  if M.Magic = Mark_Magic then
    begin
      M.Magic := 0;
      CurBuff := M.Top;
      M.Top   := -1;
      Release := True;
    end
  else
    Release := False;
end;


end.
