{****************************************************************************}
{*                                                                          *}
{*  CodeView.PAS                                                            *}
{*                                                                          *}
{*  This unit implements a simple TrueType bytecode viewer for the          *}
{*  FREETYPE project debugger.                                              *}
{*                                                                          *}
{****************************************************************************}

Unit CodeTV;

interface

uses Objects, Views, Drivers, TTTypes, TTDebug;

{$I DEBUGGER.INC}

type

  { TCodeViewer }

  { This TView is a simple code list viewer ( IP + focused + breaks ) }

  PCodeViewer = ^TCodeViewer;
  TCodeViewer = object( TListViewer )

                  constructor Init( var Bounds : TRect );
                  procedure Draw; virtual;
                  procedure HandleEvent( var Event : TEvent  ); virtual;

                  procedure Set_Range( var ARange : TRangeRec );
                  procedure Set_IP   ( AIP : Int );

                private
                  CodeRange : TRangeRec;
                  IP        : Int;
                end;

  { TCodeWindow }

  PCodeWindow = ^TCodeWindow;
  TCodeWindow = object( TWindow )
                  CodeView : PCodeViewer;
                  constructor Init( var Bounds : TRect );
                end;

implementation

{ TCodeViewer }

constructor TCodeViewer.Init;
begin
  inherited Init( Bounds, 1, nil, nil );
  SetRange(0);

  GrowMode  := gfGrowHiX or gfGrowHiY;
  DragMode  := dmDragGrow or dmLimitLoX or dmLimitLoY;
  EventMask := EventMask or evCommand;

  with CodeRange do
  begin
    Disassembled := nil;
    NLines       := 0;
  end;

  IP := -1;
end;


procedure TCodeViewer.HandleEvent( var Event : TEvent );
var
  Limits     : TRect;
  Mini, Maxi : Objects.TPoint;
begin

  if Event.What = evKeydown then

    case Event.KeyCode of

      kb_val_ToggleBreak : begin
                             Event.What    := evCommand;
                             Event.Command := cmToggleBreak;
                           end;
    end;

  inherited HandleEvent(Event);

  if (Event.What = evCommand) then

    case Event.Command of

      cmResize: begin
                  Owner^.GetExtent(Limits);
                  SizeLimits( Mini, Maxi );
                  DragView(Event, DragMode, Limits, Mini, Maxi );
                  ClearEvent(Event);
                end;

      cmToggleBreak: if State and sfSelected <> 0 then
                     begin
                       Toggle_Break( CodeRange,
                                     CodeRange.Disassembled^[Focused] );
                       DrawView;
                     end;
    end;
end;


procedure TCodeViewer.Draw;
const
  Colors : array[0..3] of byte
         = ($1E,$40,$0E,$30);
var
  I, J, Item : Int;
  B          : TDrawBuffer;
  S          : String;
  Indent     : Int;
  Ligne      : Int;

  Color  : word;

  On_BP : boolean;
  BP    : PBreakPoint;

begin

{
  Colors[0] := GetColor(1);  (* Normal line *)
  Colors[1] := GetColor(2);  (* Normal breakpoint *)
  Colors[2] := GetColor(3);  (* Focused line *)
  Colors[3] := GetColor(4);  (* Focused breakpoint *)
}
  if HScrollBar <> nil then Indent := HScrollBar^.Value else Indent := 0;

  BP := CodeRange.Breaks;

  with CodeRange do
  begin

    if (BP<>nil) and (NLines>TopItem) then
      while ( BP<>nil ) and ( BP^.Adresse < Disassembled^[TopItem] ) do
        BP := BP^.Next;

    for I := 0 to Self.Size.Y-1 do
    begin

      Item := TopItem + I;

      Color := 0;

      if Item < NLines then
        begin

          Ligne := Disassembled^[Item];

          if ( BP<>nil ) and ( BP^.Adresse = Ligne ) then
            begin
              Color := 1;
              Repeat
                BP := BP^.Next
              until ( BP=nil ) or ( BP^.Adresse > Ligne );
            end;

          if (Range > 0) and
             (State and (sfSelected+sfActive) = (sfSelected+sfActive)) and
             (Focused = Item ) then

            Color := Color or 2;

          S := ' ' + Cur_U_Line( Code, Ligne );
          S := copy( S, 1 + Indent, Self.Size.X );

          if Ligne = IP then S[1] := '>';
        end
      else
        begin
          S := '';
        end;

      Color := Colors[Color];

      MoveChar( B, ' ', Color, Self.Size.X );
      MoveStr( B, S, Color );

      WriteLine( 0, I, Self.Size.X, 1, B );
    end;
  end;
end;


procedure TCodeViewer.Set_Range;
begin
  CodeRange := ARange;
  SetRange( CodeRange.NLines );
  FocusItem(0);
  DrawView;
end;


procedure TCodeViewer.Set_IP;
begin
  IP := AIP;
  DrawView;
end;

{ TCodeWindow }

constructor TCodeWindow.Init;
begin
  inherited Init( Bounds,'Code',wnNoNumber );
  GetClipRect( Bounds );
  Bounds.Grow(-1,-1);
  New( CodeView, Init( Bounds ) );
  Insert( CodeView );
end;

end.
