/*rexx*/
/*_____________________________________________________________________________
FreeCell
Philip R Brenan, 1996, phil@bga.com
_____________________________________________________________________________*/
 
call setUpSystem

/*_____________________________________________________________________________
The game
_____________________________________________________________________________*/

do game = 1 by 1
  call initializeGame(randomGame())

  do forever
    if countEmptyColumn() = game.!columns then game.!msg = 'You Won!'
    call save
    call drawboard
    if /*game.!turn > 1 &*/ game.!turn = game.!maxTurn then if autoMove() then iterate
    call input
    call update
  end
end 

/*_____________________________________________________________________________
Update board
_____________________________________________________________________________*/

update: procedure expose game.
  game.!depth = 0
  do j = 1 to game.!columns
    game.!depth = max(game.!depth, game.!depth.j)
  end
return

/*_____________________________________________________________________________
Auto move
_____________________________________________________________________________*/

autoMove: procedure expose game.
  a.1 = min(cardNo(game.!home.2), cardNo(game.!home.3)) + 1
  a.2 = min(cardNo(game.!home.1), cardNo(game.!home.4)) + 1
  a.3 = a.2
  a.4 = a.1

  do f = 1 to game.!suits
    c = game.!freecell.f
    s = suitNo(c)
    if c > 0 & cardNo(c) <= a.s & homeable(c) > 0 then do; call homeFreeCmd f; call sleep; return 1; end
  end

  do j = 1 to game.!columns
    d = game.!depth.j
    if d > 0 then do
      c = game.!board.d.j
      s = suitNo(c)
      if c > 0 & cardNo(c) <= a.s & homeable(c) > 0 then do
        call homeCmd j
        call sleep
        return 1
      end
    end
  end
return 0

/*_____________________________________________________________________________
Get user input
_____________________________________________________________________________*/

input: procedure expose game.
  game.!msg = ''
  pull in
  if length(in) < 1 then in = '?'
  s = translate(left(in, 1))

  if abbrev(s, 'X') then exit

  if abbrev(s, '?') then do
    call sysCls
    say 'FreeCell!'
    say
    say 'x   - eXit'
    say 's   - reStart current game'
    say 'gN  - play Game N'
    say 'C   - move column C to free cell'
    say 'CC  - move column C to home'
    say 'CD  - move column C to column D'
    say 'FFC - move free cell F to column C'
    say 'hF  - move free cell F to Home'
    say 'u   - undo last move'
    say 'r   - redo last move'
    say 'o   - OK, resume play after undo, redo'
    say
    say 'Freeware: Philip R Brenan, 1996, phil@bga.com'
    say
    say 'any key to continue'
    pull .
    return
  end
 
  if abbrev(s, 'S') then do
    call initializeGame(game.!game)
    return
  end
 
  if abbrev(s, 'G') then do
    n = randomGame()
    if length(in) > 1 then if datatype(substr(in, 2)) = 'NUM' then n = abs(left(substr(in, 2), 5))
    call initializeGame(n)
    return
  end
 
  if abbrev(s, 'H') then do
    if length(in)= 2 then if datatype(substr(in, 2)) = 'NUM' then call homeFreeCmd substr(in, 2)
    else do
      game.!msg = 'Invalid free cell for hN command - move free cell N to home'
      return
    end
  end
 
  if abbrev(s, 'U') then do
    call undo
    return
  end
 
  if abbrev(s, 'R') then do
    call redo
    return
  end
 
  if abbrev(s, 'O') then do
    game.!maxTurn = game.!turn
    return
  end

  drop a.; a. = ''; do i = 1 to length(in); a.i = substr(in, i, 1); a.0 = i; end

  if datatype(in) = 'NUM' then do
    if      length(in) = 1              then call freeCmd     a.1
    else if length(in) = 2 & a.1 \= a.2 then call moveCmd     a.1, a.2
    else if length(in) = 2 & a.1  = a.2 then call homeCmd     a.1
    else if length(in) = 3              then call getFreeCmd  a.1, a.3
    else game.!msg = 'Invalid Move command' in 
  end
  else game.!msg = 'Invalid command' in 
return

/*_____________________________________________________________________________
Move column to column
_____________________________________________________________________________*/

moveCmd: procedure expose game.
  c = arg(1)
  d = arg(2)

  if invalidColumn(c) | invalidColumn(d) | errorEmptyColumn(c) then return

  di = game.!depth.d
  dc = game.!board.di.d
  if di = 0 then target = 'onto column' d; else target = 'onto' cardLongName(dc);

  ci = game.!depth.c
  sc = game.!board.ci.c
  ci = ci + 1

  if di = 0,
  then maxCards = min((countEmptyFreeCell() + 1) * 2 ** countEmptyColumn() - 1, game.!depth.c)
  else maxCards = min((countEmptyFreeCell() + 1) * 2 ** countEmptyColumn() - 0, game.!depth.c)

  do j = 1 to game.!depth.c
    if j > 1 & \onto4(ci, c, ci - 1, c) then leave
    ci = ci - 1
    cc = game.!board.ci.c

    if j > maxCards then do
      game.!msg = 'I can move' maxCards 'but column' c target 'requires' j 'free cells'
      return;
    end
    if (di = 0 & (j = game.!depth.c | \onto4(ci, c, ci - 1, c))) | onto2(cc, dc) then do
      do k = 1 to j
        si = game.!depth.c - k + 1
        sc = game.!board.si.c
        ti = game.!depth.d + j - k + 1
        game.!board.ti.d = sc
        game.!board.si.c = 0
      end
      game.!depth.d  = game.!depth.d + j
      game.!depth.c  = game.!depth.c - j
      if j > 1 then game.!msg = 'Moved' j 'cards from column' c target
      else          game.!msg = 'Moved' cardLongName(sc) ||     target
      return
    end
  end
  game.!msg = 'Cannot move' cardLongName(sc) 'onto' cardLongName(dc) 
return

/*_____________________________________________________________________________
Move card to free cell
_____________________________________________________________________________*/

freeCmd: procedure expose game.
  j = arg(1)
  if invalidColumn(j) | errorEmptyColumn(j) then return

  do f = 1 to game.!suits
    if game.!freecell.f = 0 then do
      i = game.!depth.j
      game.!freecell.f = game.!board.i.j
      game.!board.i.j  = 0
      game.!depth.j    = game.!depth.j - 1
      call madeMove cardLongName(game.!freecell.f) 'to free cell' f
      return
    end
  end
  game.!msg = 'No more free cells'
return

/*_____________________________________________________________________________
Move card to home
_____________________________________________________________________________*/

homeCmd: procedure expose game.
  j = arg(1)
  
  if invalidColumn(j) | errorEmptyColumn(j) then return

  i = game.!depth.j
  c = game.!board.i.j
  s = homeable(c)

  if s > 0 then do
    game.!home.s    = c
    game.!board.i.j = 0
    game.!depth.j   = game.!depth.j - 1
    call madeMove cardLongName(c) 'home'
    return
  end
  game.!msg = 'Cannot move' cardLongName(c) 'home yet'
return

/*_____________________________________________________________________________
Can card be moved home yet? Return suit if possible
_____________________________________________________________________________*/

homeable: procedure expose game.
  c = arg(1)
  s = suitNo(c)
  h = game.!home.s

  if (h = 0 & cardNo(c) = 1) | (cardNo(c) = cardNo(h) + 1) then return s
return 0

/*_____________________________________________________________________________
Move free cell to column
_____________________________________________________________________________*/

getFreeCmd: procedure expose game.
  f = arg(1); j = arg(2)

  if invalidFreeCell(f) | errorEmptyFreeCell(f) | invalidColumn(j) then return

  i  = game.!depth.j
  fc = game.!freecell.f
  jc = game.!board.i.j

  if i > 0 then if \onto2(fc, jc) then do
    game.!msg = 'Cannot move' cardLongName(fc) 'from free cell onto' cardLongName(jc)
    return
  end

  i = i + 1
  game.!depth.j    = i
  game.!board.i.j  = game.!freecell.f
  game.!freecell.f = 0
  call madeMove cardLongName(fc) 'onto' cardLongName(jc)
return

/*_____________________________________________________________________________
Move free cell to home
_____________________________________________________________________________*/

homeFreeCmd: procedure expose game.
  f = arg(1)

  if invalidFreeCell(f) | errorEmptyFreeCell(f) then return

  c = game.!freecell.f
  s = suitNo(c)
  h = game.!home.s

  if (h = 0 & cardNo(c) = 1) | (cardNo(c) = cardNo(h) + 1) then do
    game.!home.s     = c
    game.!freecell.f = 0
    call madeMove cardLongName(c) 'home'
  end
  else game.!msg = 'Cannot move' cardLongName(c) 'home yet'
return

/*_____________________________________________________________________________
Made a move
_____________________________________________________________________________*/

madeMove: procedure expose game.
  t = arg(1)
  turn          = game.!turn + 1
  game.!turn    = turn
  game.!maxTurn = turn
  game.!msg     = 'Moved' t
return

/*_____________________________________________________________________________
Count empty free cells, columns
_____________________________________________________________________________*/

countEmptyFreeCell: procedure expose game.
  n = 0
  do f = 1 to game.!suits
    if game.!freecell.f = 0 then n = n + 1
  end
return n

countEmptyColumn: procedure expose game.
  n = 0
  do j = 1 to game.!columns
    if game.!depth.j = 0 then n = n + 1
  end
return n

/*_____________________________________________________________________________
Error if there are no cards in a free cell or a column
_____________________________________________________________________________*/

errorEmptyFreeCell: procedure expose game.
  f = arg(1)
  if game.!freecell.f = 0 then do
    game.!msg = 'No cards in free cell' f
    return 1
  end
return 0

errorEmptyColumn: procedure expose game.
  j = arg(1)
  if game.!depth.j = 0 then do
    game.!msg = 'No cards in column' j
    return 1
  end
return 0

/*_____________________________________________________________________________
Invalid column or free cell?
_____________________________________________________________________________*/

invalidFreeCell: procedure expose game.
  f = arg(1)

  if f < 0 | f > game.!suits then do
    game.!msg = 'Invalid free cell' f 'specified'
    return 1
  end
return 0

invalidColumn: procedure expose game.
  j = arg(1)

  if j < 0 | j > game.!columns then do
    game.!msg = 'Invalid column' j 'specified'
    return 1
  end
return 0

/*_____________________________________________________________________________
Check whether one card can be place on top of another.
2 - Card number
4 - Board Cordinates
_____________________________________________________________________________*/

onto2: procedure expose game.
 sc = arg(1)
 tc = arg(2)

 if cardColor(sc) \= cardColor(tc) & cardNo(sc) = cardNo(tc) - 1 then return 1
return 0 

onto4: procedure expose game.
 sr = arg(1); sc = arg(2)
 tr = arg(3); tc = arg(4)

 sc = game.!board.sr.sc
 tc = game.!board.tr.tc

return onto2(sc, tc)

/*_____________________________________________________________________________
Draw the current state of the game
_____________________________________________________________________________*/

drawboard: procedure expose game.
  parse value SysTextScreenSize() with game.!rows game.!cols
  game.!board = ''; k = copies('.', (game.!rows - 10) * game.!cols)

  row = 1; cols = game.!columns; colw = game.!cols / game.!suits / 2

  call out center('Free Cell !', game.!cols), row, 1
  call out 'Game' game.!game', turn' game.!turn', max' game.!maxTurn, row, 1

  row = row + 1
  call out copies(copies('-', colw - 1)'+', game.!suits), row, 1
  call out copies('=', colw * game.!suits), row, game.!cols / 2 + 1

  do i = 1 to game.!suits;
    call out i, row, 1 + (i - 1) * colw + colw / 2 - length(i) / 2 - 1
  end

  row = row + 1
  do i = 1 to game.!suits;
    if game.!freecell.i > 0 then do
      parse value cardName(game.!freecell.i) with suit card
      if suit \= '' then do
        col = 1 + (i - 1) * colw
        call out center(card, colw), row,     col
        call out center('of', colw), row + 1, col
        call out center(suit, colw), row + 2, col
      end
    end

    parse value cardName(game.!home.i) with suit card
    if suit \= '' then do
      col = game.!cols / 2 + 1 + (i - 1) * colw
      call out center(card, colw), row,     col
      call out center('of', colw), row + 1, col
      call out center(suit, colw), row + 2, col
    end
  end

  do i = 0 to 2
    call out '|', row + i, game.!cols / 2
  end

  row = row + i
  call out copies(copies('-', colw - 1)'+', cols), row, 1

  do i = 1 to game.!columns;
    call out i, row, 1 + (i - 1) * colw + colw / 2 - length(i) / 2 - 1
  end

  do i = 1 to game.!depth
    do j = 1 to cols
      if i <= game.!depth.j then do
        card = game.!board.i.j
        if card > 0 then do
          col = 1 + (j - 1) * colw
          parse value cardName(card) with suit card
          call out card, row + i,     col
        end
      end
    end
  end

  row = row + game.!depth + 3
  call out game.!msg, row, 1

  row = row + 2
  text = 'Enter Command, X to exit, ENTER for help:'
  call out text, row, 1

  if game.!turn = 1 then call syscls
  call sysCurPos 0, 0
  call charout , left(game.!board, (game.!rows - 1) * game.!cols)
  call sysCurPos row - 1, length(text) + 2
return

/*_____________________________________________________________________________
Write a string into the output buffer
_____________________________________________________________________________*/

out: procedure expose game.
  game.!board = overlay(arg(1), game.!board, (arg(2) - 1) * game.!cols + format(arg(3),,0))
return

/*_____________________________________________________________________________
Generate a random game
_____________________________________________________________________________*/

randomGame: return random(1, 99999)

/*_____________________________________________________________________________
Initialize a game
_____________________________________________________________________________*/

initializeGame: procedure expose game.
  drop game.; game. = 0; game.!msg = 'New Game' arg(1); game.!game = arg(1)
  game.!turn = 1; game.!maxTurn = 1

  call cards; cards = game.!suits * game.!cards
  game.!columns = game.!suits * 2

  do i = 1 to cards; place.i = i; end

  j = random(1, cards, game.!game)
  do i = 1 to 1000 
    j = random(1, cards)
    k = random(1, cards)
    t = place.j; place.j = place.k; place.k = t
  end

  do i = 1 to game.!suits; game.!freecell.i = 0; game.!home.i = 0; end

  cardNo = 0
  do i = 1 by 1
    do j = 1 to game.!columns
      cardNo = cardNo + 1
      if cardNo <= cards then do
        game.!board.i.j = place.cardNo
        game.!depth.j = i
        game.!depth   = max(i, game.!depth)
      end
      else leave i
    end
  end
return

/*_____________________________________________________________________________
The cards
_____________________________________________________________________________*/

cards: procedure expose game.
  s = 'spades hearts diamonds clubs'

  game.!suits = words(s)
  do i = 1 to words(s)
    game.!suit.i = word(s, i)
  end

  s = '01-ace 02-two 03-three 04-four 05-five 06-six 07-seven 08-eight 09-nine 10-ten 11-jack 12-queen 13-king'
 
  game.!cards = words(s)
  do i = 1 to words(s)
    game.!card.i = word(s, i)
  end
return

/*_____________________________________________________________________________
Card name from card number
_____________________________________________________________________________*/

cardName: procedure expose game.
  n    = arg(1)
  if n = 0 then return ''
  card = cardNo(n); 
  suit = suitNo(n); 
  card = game.!card.card
  suit = game.!suit.suit

  select
    when abbrev(suit, 's') then card = translate(overlay(d2c(6), card, 3));
    when abbrev(suit, 'h') then card = overlay(d2c(3), card, 3);
    when abbrev(suit, 'd') then card = overlay(d2c(4), card, 3);
    when abbrev(suit, 'c') then card = translate(overlay(d2c(5), card, 3));
    otherwise
  end  
return suit card

/*_____________________________________________________________________________
Card long name from card number
_____________________________________________________________________________*/

cardLongName: procedure expose game.
  parse value cardName(arg(1)) with suit card
return ''''card 'of' suit''''

/*_____________________________________________________________________________
Card color
_____________________________________________________________________________*/

cardColor: procedure expose game.
  n = suitNo(arg(1))
return n = 2 | n = 3

/*_____________________________________________________________________________
Card/Suite number from card number
_____________________________________________________________________________*/

cardNo: procedure expose game. ; return (arg(1) - 1) // game.!cards + 1
suitNo: procedure expose game. ; return (arg(1) - 1) %  game.!cards + 1

/*_____________________________________________________________________________
System set up
_____________________________________________________________________________*/

setUpSystem:
  call RxFuncAdd 'SysLoadFuncs', 'RexxUtil', 'SysLoadFuncs'
  call SysLoadFuncs
return

/*_____________________________________________________________________________
Sleep
_____________________________________________________________________________*/

sleep:
  call sysSleep 1
return

/*_____________________________________________________________________________
Save
_____________________________________________________________________________*/

save: procedure expose game.
  state =           game.!turn game.!maxTurn game.!game game.!columns game.!rows game.!cols game.!depth game.!suits game.!cards
  do i = 1 to game.!suits
    state = state game.!suite.i game.!freecell.i game.!home.i
  end
  do i = 1 to game.!cards
    state = state game.!cards.i
  end
  do i = 1 to game.!columns
    state = state game.!depth.i
    do j = 1 to game.!depth.i
      state = state game.!board.i.j
    end
  end
  state = state game.!msg
  turn = game.!turn
  game.!state.turn = state
return

/*_____________________________________________________________________________
Undo
_____________________________________________________________________________*/

undo: procedure expose game.
  if game.!turn > 1 then do
    turn = game.!turn - 1
    state = game.!state.turn

    parse var state game.!turn .             game.!game game.!columns game.!rows game.!cols game.!depth game.!suits game.!cards state
    do i = 1 to game.!suits
      parse var state game.!suite.i game.!freecell.i game.!home.i state
    end
    do i = 1 to game.!cards
      parse var state game.!cards.i state
    end
    do i = 1 to game.!columns
      parse var state game.!depth.i state
      do j = 1 to game.!depth.i
        parse var state game.!board.i.j state
      end
    end
    parse var state game.!msg
  end
return

/*_____________________________________________________________________________
Redo
_____________________________________________________________________________*/

redo: procedure expose game.
  if game.!turn < game.!maxTurn then do
    game.!turn = game.!turn + 2
    call undo
  end
return

