// hintbar.cpp -- Member function definitions for hint tag control bar classes.

#include "hintbar.h"

// Required Constants

const int SPACE = 4;     // space around text in hint tags
const int OFFSET_X = 5;  // horizontal offset of hint tag from cursor
const int OFFSET_Y = 20; // vertical offset of hint tag from cursor
const int TIMER_ID = 1;  // ID of timer used for hint tag appearance delay

//----------------------------------------------------------------------------
//--- THintButtonGadget Implementation
//----------------------------------------------------------------------------

THintButtonGadget::THintButtonGadget(TResId bmpResId, int id, char far *tag,
   TType type, bool enabled, TState state, bool repeat) :
   TButtonGadget(bmpResId, id, type, enabled, state, repeat)
{
   text = 0;
   if (tag) {
      text = new char[strlen(tag)+1];
      strcpy(text, tag);
   }
}

THintButtonGadget::~THintButtonGadget(void)
{
   delete text;
}

void THintButtonGadget::MouseEnter(uint modKeys, TPoint& pt)
{
   TButtonGadget::MouseEnter(modKeys, pt);
   TYPESAFE_DOWNCAST(Window, THintTagBar)->CreateTag(text);
}

void THintButtonGadget::MouseLeave(uint modKeys, TPoint& pt)
{
   TButtonGadget::MouseLeave(modKeys, pt);
   TYPESAFE_DOWNCAST(Window, THintTagBar)->DestroyTag();
}


//----------------------------------------------------------------------------
//--- TTagWindow Definition
//----------------------------------------------------------------------------

class TTagWindow : public TWindow
{
public:
   TTagWindow(TWindow *parent, const char far* title);
   virtual bool Create(void);
   virtual void Paint(TDC& dc, bool erase, TRect& rect);
};


//----------------------------------------------------------------------------
//--- TTagWindow Implementation
//----------------------------------------------------------------------------

TTagWindow::TTagWindow (TWindow *parent, const char far* title)
   : TWindow(parent, title)
{
   Attr.Style = WS_POPUP | WS_DISABLED;   // invisible window
}

bool TTagWindow::Create(void)
{
   bool result;
   TPoint point = TYPESAFE_DOWNCAST(Parent, THintTagBar)->GetLastPoint();
   TRect r = Parent->GetWindowRect();
   Attr.X = point.x + r.left + OFFSET_X;
   Attr.Y = point.y + r.top + OFFSET_Y;
   Attr.W = 1;       // to be calculated at Paint time
   Attr.H = 1;
   result = TWindow::Create();
   Show(SW_SHOWNOACTIVATE);   // make visible without changing active
   return result;
}

void TTagWindow::Paint(TDC& dc, bool erase, TRect& rect)
{
   TWindow::Paint(dc, erase, rect);

   TRect r;
   TSize sz;
   THintTagBar *owner = TYPESAFE_DOWNCAST(Parent, THintTagBar);
   dc.SelectObject(owner->GetFont());
   sz = dc.GetTextExtent(Title, strlen(Title));
   Attr.W = sz.cx + SPACE;
   Attr.H = sz.cy + SPACE;
   MoveWindow(Attr.X, Attr.Y, Attr.W, Attr.H, true);
   dc.SelectObject(*(owner->GetBackgroundBrush()));
   dc.SelectObject(*(owner->GetBorderPen()));
   r = TRect(0, 0, Attr.W, Attr.H);
   dc.Rectangle(r);
   dc.SetTextColor(owner->GetTextColor());
   dc.SetBkMode(TRANSPARENT);
   dc.DrawText(Title, -1, r, DT_CENTER | DT_VCENTER | DT_NOCLIP |
      DT_EXTERNALLEADING | DT_SINGLELINE);
   dc.RestoreObjects();
}



//----------------------------------------------------------------------------
//--- THintTagBar Implementation
//----------------------------------------------------------------------------

DEFINE_RESPONSE_TABLE1(THintTagBar, TControlBar)
  EV_WM_LBUTTONDOWN,
  EV_WM_MOUSEMOVE,
  EV_WM_TIMER,
END_RESPONSE_TABLE;

THintTagBar::THintTagBar(TWindow *parent, TTileDirection direction,
   TFont *font, TColor textColor, TColor borderColor, TColor brushColor,
   TModule *module) : TControlBar(parent, direction, font, module),
   lastPoint(0,0)
{
   tag = 0;
   delayTag = 0;
   tagsActive = false;
   colorTag = textColor;
   penTag = new TPen(borderColor);
   brushTag = new TBrush(brushColor);
}

THintTagBar::~THintTagBar(void)
{
   DestroyTag();
   delete penTag;
   delete brushTag;
}

bool THintTagBar::IdleAction(long idleCount)
{
   if (idleCount == 0) {
// If the mouse is no longer inside the control bar, turn off hint tags
      TPoint pt;
      GetCursorPos(pt);
      if (!GetWindowRect().Contains(pt)) ShutDownTags();
   }
   return TControlBar::IdleAction(idleCount);
}

void THintTagBar::EvLButtonDown(uint modKeys, TPoint& point)
{
// Left button down anywhere in control bar turns off hints
   ShutDownTags();
   TControlBar::EvLButtonDown(modKeys, point);
}

void THintTagBar::EvMouseMove(uint modKeys, TPoint& point)
{
   if (delayTag > 0) {  // none of this matters if hint tags are disabled
      if (point != lastPoint) {
         lastPoint = point;
         if (!tagsActive) {      // if hints are not yet active then...
            KillTimer(TIMER_ID); // ...stop any timer in progress and...
            if (tag) SetTimer(TIMER_ID, delayTag);
         }                       // ...restart if there's a tag waiting
      }
   }
   TControlBar::EvMouseMove(modKeys, point);
}

void THintTagBar::EvTimer(uint timerId)
// Timer has fired -- user calmed down long enough to see the tags!
{
   if (timerId == TIMER_ID) {     // ensure this is the timer we expected
      KillTimer(TIMER_ID);
      tagsActive = true;
      if (tag) tag->Create();  // show tag for gadget the user lingered on
   }
}

void THintTagBar::CreateTag(const char far *text)
{
   if (delayTag > 0) {     // do nothing if hint tags disabled
      DestroyTag();        // there can be only one!
      tag = new TTagWindow(this, text);
      if (tagsActive) tag->Create();    // only show the tag if active
   }
}

void THintTagBar::DestroyTag(void)
{
   delete tag;
   tag = 0;
}

void THintTagBar::ShutDownTags(void)
{
   KillTimer(TIMER_ID);
   DestroyTag();
   tagsActive = false;
}

