// hinttag.cpp -- Example application using a control bar with hint tags
//  by Brent W. York (sage@ameritel.net)

#include <owl\applicat.h>
#include <owl\decframe.h>
#include <owl\statusba.h>
#include <owl\inputdia.h>
#include <owl\chooseco.h>
#include <owl\choosefo.h>
#pragma hdrstop

#include "hinttag.rh"
#include "hintbar.h"


// Global custom color array for TChooseColorDialog

TColor custColors[16];


//----------------------------------------------------------------------------
//--- hintApp Definition
//----------------------------------------------------------------------------

class hintApp : public TApplication
{
public:
   hintApp(void);
   virtual void InitMainWindow(void);

   void SetDelay(int n);
   void SetTextColor(TColor color);
   void SetBrushColor(TColor color);
   TFont &GetFont(void) { return bar->GetFont(); }
   void SetFont(TFont *font) { bar->SetFont(font); }

   int delay;
   TColor textColor, brushColor;

private:
   THintTagBar *bar;
};


//----------------------------------------------------------------------------
//--- hintClient Definition
//----------------------------------------------------------------------------

class hintClient : public TWindow
{
public:
   hintClient(TModule* module = 0) : TWindow(0, 0, module) {}

protected:
   void CmBkColor(void);
   void CmTxColor(void);
   void CmFont(void);
   void CmDelay(void);
   DECLARE_RESPONSE_TABLE(hintClient);
};

//----------------------------------------------------------------------------
//--- hintClient Implementation
//----------------------------------------------------------------------------

DEFINE_RESPONSE_TABLE1(hintClient, TWindow)
   EV_COMMAND(CM_BKCOLOR, CmBkColor),
   EV_COMMAND(CM_TXCOLOR, CmTxColor),
   EV_COMMAND(CM_FONT, CmFont),
   EV_COMMAND(CM_DELAY, CmDelay),
END_RESPONSE_TABLE;

void hintClient::CmBkColor(void)
{
   hintApp *theApp = TYPESAFE_DOWNCAST(GetApplication(), hintApp);
   TChooseColorDialog::TData data;
   data.Color = theApp->brushColor;
   data.CustColors = custColors;
   data.Flags = CC_RGBINIT;
   if ((TChooseColorDialog(this, data, 0, "Hint Tag Background")).Execute()
      == IDOK) theApp->SetBrushColor(data.Color);
}

void hintClient::CmTxColor(void)
{
   hintApp *theApp = TYPESAFE_DOWNCAST(GetApplication(), hintApp);
   TChooseColorDialog::TData data;
   data.Color = theApp->textColor;
   data.CustColors = custColors;
   data.Flags = CC_RGBINIT;
   if ((TChooseColorDialog(this, data, 0, "Hint Tag Text")).Execute()
      == IDOK) theApp->SetTextColor(data.Color);
}

void hintClient::CmFont(void)
{
   hintApp *theApp = TYPESAFE_DOWNCAST(GetApplication(), hintApp);
   TChooseFontDialog::TData data;
   theApp->GetFont().GetObject(data.LogFont);
   data.Flags = CF_SCREENFONTS | CF_INITTOLOGFONTSTRUCT;
   if ((TChooseFontDialog(this, data, 0, "Hint Tag Font")).Execute() == IDOK)
      theApp->SetFont(new TFont(&(data.LogFont)));
}

void hintClient::CmDelay(void)
{
   hintApp *theApp = TYPESAFE_DOWNCAST(GetApplication(), hintApp);
   char buf[18];
   itoa(theApp->delay, buf, 10);
   if ((TInputDialog(this, "Hint Tag Delay",
      "Enter new delay in milliseconds.", buf, 17)).Execute() == IDOK)
         theApp->SetDelay(atoi(buf));
}


//----------------------------------------------------------------------------
//--- hintApp Implementation
//----------------------------------------------------------------------------

hintApp::hintApp(void) : TApplication("Hint Tag Example")
{
   delay = 1000;
   textColor = TColor::Black;
   brushColor = TColor(255,255,115);
   custColors[0] = brushColor;
}

void hintApp::SetDelay(int n)
{
   delay = n;
   bar->SetTagDelay(delay);
}

void hintApp::SetBrushColor(TColor color)
{
   brushColor = color;
   bar->SetBrushColor(brushColor);
}

void hintApp::SetTextColor(TColor color)
{
   textColor = color;
   bar->SetTextColor(textColor);
}

void hintApp::InitMainWindow(void)
{
   hintClient *client = new hintClient;
   TDecoratedFrame *frame = new TDecoratedFrame(0, Name, client, true);
   frame->AssignMenu(HINT_MENU);
   bar = new THintTagBar(frame, TGadgetWindow::Horizontal,
      new TGadgetWindowFont(6, true, false));
   bar->Insert(*new THintButtonGadget(CM_BKCOLOR, CM_BKCOLOR,
      "Background Color"));
   bar->Insert(*new THintButtonGadget(CM_TXCOLOR, CM_TXCOLOR, "Text Color"));
   bar->Insert(*new THintButtonGadget(CM_FONT, CM_FONT, "Font"));
   bar->Insert(*new THintButtonGadget(CM_DELAY, CM_DELAY, "Hint Delay"));
   bar->Insert(*new TSeparatorGadget(8));
   bar->Insert(*new THintButtonGadget(CM_EXIT, CM_EXIT, "Exit"));
   bar->SetHintMode(TGadgetWindow::EnterHints);
   bar->Attr.Id = IDW_TOOLBAR;
   bar->SetTagDelay(delay);
   frame->Insert(*bar, TDecoratedFrame::Top);
   TStatusBar *status = new TStatusBar(frame, TGadget::Plain, 0);
   frame->Insert(*status, TDecoratedFrame::Bottom);
   SetMainWindow(frame);
   EnableCtl3d(true);
}


//----------------------------------------------------------------------------
//--- OwlMain Function
//----------------------------------------------------------------------------

int OwlMain (int, char* [])
{
   try {
      hintApp app;
      return app.Run();
   }
   catch (xmsg& x) {
      ::MessageBox(0, x.why().c_str(), "Exception", MB_OK);
   }
   return -1;
}

