// ========================================================
// tracex definition
// Author: Keith Elliott  5/95
//
// tracex can be used to produce formatted trace entries
// with indentations to indicate the call stack.  To use, 
// simply instantiate a CTracex object on the stack at the 
// beginning of those functions that you want to trace.  
// Use the function name in the CTracex constructor.  The
// CTracex object will automatically display a message to 
// the Microsoft TRACE output at the current static 
// indentation level.  It will then increase the static 
// indentation level.  When the object is deleted as it 
// goes out of scope, it will decrease the static 
// indentation level and display a message that the exit 
// was reached.
//
// The TRACEX macro has been provided to allow conditional 
// compilation. In order to display a message at the 
// current indentation level, use the TRACEXMSG, 
// TRACEXMSGVAL, or TRACEXWATCH macros.
//
// Output example:
// ---------------
// CTestApp:InitInstance
//   CQri::CQri
//     CRepPage::CRepPage
//     CRepPage::CRepPage exit
//     CQrSheet::CQrSheet
//     CQrSheet::CQrSheet exit
//     CRepPage::SetDocTemplate
//       CQrdvView::CQrdvView
//       CQrdvView::CQrdvView exit
//       CQrdvView::setQrSheet
//       CQrdvView::setQrSheet exit
//     CRepPage::SetDocTemplate exit
//   CQri::CQri exit
// CTestApp:InitInstance exit
//
// The macros will expand out to a semi-colon unless 
// TRACEXON has been defined.  You can use these macros 
// throughout the implementation of a class/module and can 
// enable trace output for a module simply by defining 
// TRACEXON before the header statement.  This provides an 
// easier mechanism for placing a class/module into 
// "trace debug" mode that having to repetatively 
// comment/un-comment code.
// =========================================================

#undef TRACEX
#undef TRACEXMSG
#undef TRACEXMSGVAL
#undef TRACEXWATCH

#if defined (TRACEXON)
  #define TRACEX(pString) CTracex tAutoTrace(pString);
  #define TRACEXMSG(pString) {tAutoTrace.msg(pString);}
  #define TRACEXMSGVAL(pString, pVal) \
    {tAutoTrace.msg(pString, pVal);}
  #define TRACEXWATCH(pVar) \
    {tAutoTrace.msg( #pVar " == ", pVar);}

#else
  #define TRACEX(pString) {;}
  #define TRACEXMSG(pString) {;}
  #define TRACEXMSGVAL(pString, pVal) {;}
  #define TRACEXWATCH {;}
#endif

#ifndef TRACEX_H
#define TRACEX_H

#include <afx.h>

class CTracex : public CObject
 {private:
   CString mMsg;
 
  protected:
   void traceIndent();
    
  public:
   static int mIndent;
   static BOOL mTracing;

   // When we have the keyword explicit, this single 
   // argument constructor should be declared explicit to 
   // prevent unintentional implicit type conversion.
   // ----------------------------------------------------
   CTracex(CString pMsg);
   virtual ~CTracex();

   virtual void AssertValid() const;
   void msg(const CString pMsg);
   void msg(const CString pMsg, const int pInt);
   void msg(const CString pMsg, const unsigned int pUint);
   void msg(const CString pMsg, const long pLong);
   void msg(const CString pMsg, const unsigned long pUlong);
   void msg(const CString pMsg, const double pDouble);
   void msg(const CString pMsg, const CString pString);
   void msg(const CString pMsg, const CObject* pObject);
 };
 
#endif //TRACEX_H
