//@doc		VIEW
//@module	VIEW.H - View inline declarations |
//	
//	This file contains inline definitions for the view classes.
//		
//	Only very small members are selected for inline code.
//	See the class listing for a description of these functions.
//	
//@devnote	Version 1.2, jcw@meta4.nl
//@normal	Copyright <cp> 1996 Meta Four Software. All rights reserved.

/////////////////////////////////////////////////////////////////////////////
// c4_View

	//@mfunc Destructor.
d4_inline c4_View::~c4_View ()
{
	_DecSeqRef();
}

	//@mfunc Returns the number of entries in this view.
	//@rdesc A non-negative integer.
d4_inline int c4_View::GetSize() const
{
	return _seq->Size();
}

	//@mfunc Returns the highest index in this view (same as: GetSize() - 1).
	//@rdesc One less than the number of entries.
	// If there are no entries, the value -1 is returned.
d4_inline int c4_View::GetUpperBound() const
{
	return GetSize() - 1;
}

	//@mfunc Changes the size of this view.
	//@parm int | newSize_ | the new size of this view
	//@parm int | growBy_ | the granularity of size increases (not used)
d4_inline void c4_View::SetSize(int newSize_, int growBy_)
{
	_seq->Resize(newSize_, growBy_);
}

	//@mfunc Removes all entries (sets size to zero).
d4_inline void c4_View::RemoveAll()
{
	SetSize(0);
}

	//@mfunc Returns a reference to specified entry.
	//@parm int | index_ | the zero-based row index
	//@rdesc A reference to the specified row in the view.
d4_inline c4_RowRef c4_View::GetAt(int index_) const
{
	return * c4_Cursor (*_seq, index_);
}

	//@mfunc Shorthand for <mf c4_View::GetAt>.
	//@parm int | index_ | the zero-based row index
	//@rdesc A reference to the specified row in the view.
	// This reference can be used on either side of the assignment operator.
d4_inline c4_RowRef c4_View::operator[] (int index_) const
{
	return GetAt(index_);
}
    
	//@mfunc Changes the value of the specified entry. If the new value has
	// other properties, these will be added to the underlying view.
	//@parm int | index_ | the zero-based row index
	//@parm c4_RowRef | newElem_ | the row to copy to this view
d4_inline void c4_View::SetAt(int index_, c4_RowRef newElem_)
{
	_seq->SetAt(index_, &newElem_);
}

	//@mfunc Element access, for use as RHS or LHS. Shorthand for GetAt().
	//@parm int | index_ | the zero-based row index
	//@rdesc A reference to the specified row in the view.
d4_inline c4_RowRef c4_View::ElementAt(int index_)
{
	return GetAt(index_);
}

	//@mfunc Inserts one or more copies of an entry. This is identical to
	// inserting the specified number of default entries and then setting
	// each of them to the new element value passed as argument.
	//@parm int | index_ | the zero-based row index
	//@parm c4_RowRef | newElem_ | the value to insert
	//@parm int | count_ | the nuber of entries to insert (default is 1)
d4_inline void c4_View::InsertAt(int index_, c4_RowRef newElem_, int count_)
{
	_seq->InsertAt(index_, &newElem_, count_);
}

	//@mfunc Removes entries starting at the given index. Entries which have
	// other view references may cause these views to be deleted if their
	// reference counts drop to zero because of this removal.
	//@parm int | index_ | the zero-based row index
	//@parm int | count_ | the nuber of entries to remove (default is 1)
d4_inline void c4_View::RemoveAt(int index_, int count_)
{
	_seq->RemoveAt(index_, count_);
}

	//@mfunc Returns the number of properties present in this view.
	//@rdesc A non-negative integer.
d4_inline int c4_View::NumProperties() const
{
	return _seq->NumHandlers();
}

	//@mfunc Returns the N-th property (using zero-based indexing).
	//@parm int | index_ | the zero-based property index
	//@rdesc A reference to the specified property.
d4_inline int c4_View::NthProperty(int index_) const
{
	return _seq->NthProperty(index_);
}

	//@mfunc Finds a property, given its id, as returned by NthProperty().
	//@rdesc The index of the property, or -1 of it was not found.
d4_inline int c4_View::FindProperty(int propId_)
{
	return _seq->PropIndex(propId_);
}

d4_inline c4_String c4_View::Describe() const
{
	return _seq->Describe();
}

d4_inline void c4_View::_IncSeqRef()
{
	_seq->IncRef();
}

d4_inline void c4_View::_DecSeqRef()
{
	_seq->DecRef();
}

/////////////////////////////////////////////////////////////////////////////
// c4_Cursor

d4_inline c4_Cursor::c4_Cursor (c4_Sequence& seq_, int index_)
	: _seq (&seq_), _index (index_)
{
}

d4_inline c4_RowRef c4_Cursor::operator* () const
{
	return *this;
}

d4_inline c4_Cursor& c4_Cursor::operator++ ()
{
	++_index;
	return *this;
}

d4_inline c4_Cursor c4_Cursor::operator++ (int)
{
	return c4_Cursor (*_seq, _index++);
}

d4_inline c4_Cursor& c4_Cursor::operator-- ()
{
	--_index;
	return *this;
}

d4_inline c4_Cursor c4_Cursor::operator-- (int)
{
	return c4_Cursor (*_seq, _index--);
}

d4_inline c4_Cursor& c4_Cursor::operator+= (int offset_)
{
	_index += offset_;
	return *this;
}

d4_inline c4_Cursor& c4_Cursor::operator-= (int offset_)
{
	_index -= offset_;
	return *this;
}

d4_inline c4_Cursor c4_Cursor::operator- (int offset_) const
{
	return c4_Cursor (*_seq, _index - offset_);
}

d4_inline int c4_Cursor::operator- (c4_Cursor cursor_) const
{
	return _index - cursor_._index;
}

d4_inline c4_Cursor operator+ (c4_Cursor cursor_, int offset_)
{
	return c4_Cursor (*cursor_._seq, cursor_._index + offset_);
}

d4_inline c4_Cursor operator+ (int offset_, c4_Cursor cursor_)
{
	return cursor_ + offset_;
}

d4_inline bool operator== (c4_Cursor a_, c4_Cursor b_)
{
	return a_._seq == b_._seq && a_._index == b_._index;
}

d4_inline bool operator!= (c4_Cursor a_, c4_Cursor b_)
{
	return !(a_ == b_);
}

d4_inline bool operator< (c4_Cursor a_, c4_Cursor b_)
{
	return a_._seq < b_._seq ||
			a_._seq == b_._seq && a_._index < b_._index;
}

d4_inline bool operator> (c4_Cursor a_, c4_Cursor b_)
{
	return b_ < a_;
}

d4_inline bool operator<= (c4_Cursor a_, c4_Cursor b_)
{
	return !(b_ < a_);
}

d4_inline bool operator>= (c4_Cursor a_, c4_Cursor b_)
{                     
	return !(a_ < b_);
}

/////////////////////////////////////////////////////////////////////////////
// c4_RowRef

d4_inline c4_RowRef::c4_RowRef (c4_Cursor cursor_)
	: _cursor (cursor_)
{
}

d4_inline c4_RowRef c4_RowRef::operator= (const c4_RowRef& rowRef_)
{
	if (_cursor != rowRef_._cursor)
		_cursor._seq->SetAt(_cursor._index, &rowRef_);
	
	return *this;
}

d4_inline c4_Cursor c4_RowRef::operator& () const
{
	return _cursor;
}

d4_inline c4_View c4_RowRef::Container() const
{
	return _cursor._seq;
}

d4_inline bool operator== (c4_RowRef a_, c4_RowRef b_)
{
	return (&a_)._seq->Compare((&a_)._index, &b_) == 0;
}               

d4_inline bool operator!= (c4_RowRef a_, c4_RowRef b_)
{
	return !(a_ == b_);
}

d4_inline bool operator< (c4_RowRef a_, c4_RowRef b_)
{
	return (&a_)._seq->Compare((&a_)._index, &b_) < 0;
}               

d4_inline bool operator> (c4_RowRef a_, c4_RowRef b_)
{
	return b_ < a_;
}

d4_inline bool operator<= (c4_RowRef a_, c4_RowRef b_)
{
	return !(b_ < a_);
}

d4_inline bool operator>= (c4_RowRef a_, c4_RowRef b_)
{                     
	return !(a_ < b_);
}

/////////////////////////////////////////////////////////////////////////////
// c4_Storage

    //@mfunc Used to get or set a named view in this storage object.
    //@parm const char* | name_ | the name of the view property to use
    //@rdesc Returns a reference to the specified view.
d4_inline c4_ViewRef c4_Storage::View(const char* name_) const
{
    c4_ViewProp prop (name_);
    return prop (Contents());
}

    //@mfunc Retrieves a named view from this storage object.
    //@parm const char* | name_ | the name of the view property to get
    //@rdesc Returns a reference to the specified view.
d4_inline c4_View c4_Storage::Get(const char* name_) const
{
    return View(name_);
}

    //@mfunc Stores a view under a specified name in this storage object.
    //@parm const char* | name_ | the name of the view property to set
    //@parm const c4_View& | view_ | the view to be stored
d4_inline void c4_Storage::Set(const char* name_, const c4_View& view_)
{
    View(name_) = view_;
}

/////////////////////////////////////////////////////////////////////////////
// c4_Property

d4_inline c4_Property::~c4_Property ()
{
	Refs(-1);
}

d4_inline int c4_Property::GetId() const
{
	return _id;
}

d4_inline c4_Reference c4_Property::operator() (c4_RowRef rowRef_) const
{
	return c4_Reference (rowRef_, GetId());
}

/////////////////////////////////////////////////////////////////////////////
// c4_IntProp

	//@mfunc Constructs a new integer property.
	//@parm const char* | name_ | the name of this property
d4_inline c4_IntProp::c4_IntProp (const char* name_) 
	: c4_Property ('I', name_)
{
}

d4_inline c4_Row c4_IntProp::operator[] (long value_)
{
	c4_Row row;
	operator() (row) = value_;
	return row;
}
	
d4_inline c4_IntRef c4_IntProp::operator() (c4_RowRef rowRef_)
{
	return c4_Reference (rowRef_, GetId());
}

/////////////////////////////////////////////////////////////////////////////
// c4_FloatProp

	//@mfunc Constructs a new floating point property.
	//@parm const char* | name_ | the name of this property
d4_inline c4_FloatProp::c4_FloatProp (const char* name_) 
	: c4_Property ('F', name_)
{
}

d4_inline c4_Row c4_FloatProp::operator[] (double value_)
{
	c4_Row row;
	operator() (row) = value_;
	return row;
}
	
d4_inline c4_FloatRef c4_FloatProp::operator() (c4_RowRef rowRef_)
{
	return c4_Reference (rowRef_, GetId());
}

/////////////////////////////////////////////////////////////////////////////
// c4_StringProp

	//@mfunc Constructs a new string property.
	//@parm const char* | name_ | the name of this property
d4_inline c4_StringProp::c4_StringProp (const char* name_) 
	: c4_Property ('S', name_)
{
}
	
d4_inline c4_Row c4_StringProp::operator[] (const char* value_)
{
	c4_Row row;
	operator() (row) = value_;
	return row;
}
	
d4_inline c4_StringRef c4_StringProp::operator() (c4_RowRef rowRef_)
{
	return c4_Reference (rowRef_, GetId());
}

/////////////////////////////////////////////////////////////////////////////
// c4_ViewProp

	//@mfunc Constructs a new view property.
	//@parm const char* | name_ | the name of this property
d4_inline c4_ViewProp::c4_ViewProp (const char* name_)
	: c4_Property ('V', name_)
{
}
	
d4_inline c4_Row c4_ViewProp::operator[] (const c4_View& value_)
{
	c4_Row row;
	operator() (row) = value_;
	return row;
}
	
d4_inline c4_ViewRef c4_ViewProp::operator() (c4_RowRef rowRef_)
{
	return c4_Reference (rowRef_, GetId());
}

/////////////////////////////////////////////////////////////////////////////
// c4_Sequence

d4_inline c4_Dependencies* c4_Sequence::GetDependencies() const
{
	return _dependencies;
}

/////////////////////////////////////////////////////////////////////////////
// c4_Reference

d4_inline c4_Reference::c4_Reference (c4_RowRef rowRef_, int propId_)
	: _cursor (&rowRef_), _propId (propId_)
{
}

d4_inline bool c4_Reference::GetData(c4_Bytes& buf_) const
{
	return _cursor._seq->Get(_cursor._index, _propId, buf_);
}

d4_inline void c4_Reference::SetData(const c4_Bytes& buf_) const
{
	_cursor._seq->Set(_cursor._index, _propId, buf_);
}

/////////////////////////////////////////////////////////////////////////////
// c4_IntRef

d4_inline c4_IntRef::c4_IntRef (const c4_Reference& value_)
	: c4_Reference (value_)
{
}

/////////////////////////////////////////////////////////////////////////////
// c4_FloatRef

d4_inline c4_FloatRef::c4_FloatRef (const c4_Reference& value_)
	: c4_Reference (value_)
{
}

/////////////////////////////////////////////////////////////////////////////
// c4_StringRef

d4_inline c4_StringRef::c4_StringRef (const c4_Reference& value_)
	: c4_Reference (value_)
{
}

/////////////////////////////////////////////////////////////////////////////
// c4_ViewRef

d4_inline c4_ViewRef::c4_ViewRef (const c4_Reference& value_)
	: c4_Reference (value_)
{
}

/////////////////////////////////////////////////////////////////////////////
