.data

; ##### VESAOffs is a table of Lineoffset values.
; ##### For example VESAOffs[100] is the Offset of Screenline 100.
; ##### With VESAOffs there are much less IMULs.

EXTRN _VESA_OFSTable:DWORD
EXTRN _VESA_MinX:DWORD
EXTRN _VESA_MaxX:DWORD
EXTRN _VESA_MinY:DWORD
EXTRN _VESA_MaxY:DWORD
EXTRN _VESA_BpL:DWORD
EXTRN _VESA_ScreenSize_Byte:DWORD
EXTRN _VESA_Selector:WORD

VESAOffs EQU _VESA_OFSTable
VESAMinX EQU _VESA_MinX
VESAMinY EQU _VESA_MinY
VESAMaxX EQU _VESA_MaxX
VESAMaxY EQU _VESA_MaxY
VESABpL  EQU _VESA_BpL
VESAScrSize EQU _VESA_ScreenSize_Byte
VESASel EQU _VESA_Selector

CODE SEGMENT PUBLIC DWORD USE32

ASSUME CS:CODE,FS:CODE

PUBLIC VESA_Line8B
PUBLIC VESA_HLine8B
PUBLIC VESA_HLineG8B
PUBLIC VESA_HLineG8BFast
PUBLIC VESA_HLineTex8B
PUBLIC VESA_HLineTex8BFast
PUBLIC VESA_HLineZ8B
PUBLIC VESA_HLineGZ8B
PUBLIC VESA_HLineTexZ8B
PUBLIC VESA_FilledTriangle8B
PUBLIC VESA_FilledTriangleG8B
PUBLIC VESA_SetColor8B
PUBLIC VESA_GetCodeAlias
PUBLIC VESA_CopyScreen8B
PUBLIC VESA_ClearScreen8B
PUBLIC VESA_SetClipArea
PUBLIC VESA_SyncDisplay
PUBLIC VESA_SetDisplayStart
PUBLIC VESA_ClearZBuffer

; ###### Assume FS=CODE for Selfmodification.
; ###### Without this you get Compiler-Warnings.
ASSUME FS:CODE

CodeAlias DW ?

VESA_SetDisplayStart PROC PASCAL X:WORD,Y:WORD
 
    ; ##### After calling this PROC the Graphic Adapter starts 
    ; ##### scanning the Video-Memory from Row Y and Column X.

    MOV AX,04f07h
    MOV BL,80h
    MOV BH,0
    MOV CX,X
    MOV DX,Y
    INT 10h
    RET

VESA_SetDisplayStart ENDP

VESA_SyncDisplay PROC PASCAL

    ; ##### Waits for Vertical Retrace
    ; ##### Maybe needed for 'no-flicker'.

    MOV DX,03DAh
    
  sd_Wait0:
  
    IN AL,DX
    TEST AL,08h
    JNZ sd_Wait0
    
  sd_Wait1:
  
    IN AL,DX
    TEST AL,08h
    JZ sd_Wait1
    
    RET

VESA_SyncDisplay ENDP

VESA_SetClipArea PROC PASCAL X1:DWORD,Y1:DWORD,X2:DWORD,Y2:DWORD

    ; ##### Changes the VESAMin- and VESAMax-Values.
    ; ##### Fast routines can plug these Values directly
    ; ##### in their code. This is much faster then
    ; ##### CMPring to memory values. 
  
    ; ##### CodeAlias needed for Code Segment
    ; ##### writes.
    MOV FS,CS:CodeAlias

    MOV EAX,X1
    MOV VESAMinX,EAX
    MOV FS:Li8B_XCMPMinX,EAX
    MOV FS:Li8B_YCMPMinX,EAX
    MOV FS:HLi8BR_X2CMPMinX,EAX    
    MOV FS:HLi8BR_X1CMPMinX,EAX
    MOV FS:HLi8BR_MOVMinX,EAX
    MOV FS:HLiG8B_CMPMinX,EAX
    MOV FS:HLiG8B_CMPMinX2,EAX
    MOV FS:HLiG8B_SUBMinX,EAX
    MOV FS:HLiG8B_MOVMinX,EAX
    
    MOV EAX,Y1
    MOV VESAMinY,EAX
    MOV FS:Li8B_XCMPMinY,EAX
    MOV FS:Li8B_YCMPMinY,EAX
    MOV FS:HLi8BR_CMPMinY,EAX
    MOV FS:HLiG8B_CMPMinY,EAX
    
    MOV EAX,X2
    MOV VESAMaxX,EAX
    MOV FS:Li8B_XCMPMaxX,EAX
    MOV FS:Li8B_YCMPMaxX,EAX
    MOV FS:HLi8BR_X2CMPMaxX,EAX    
    MOV FS:HLi8BR_X1CMPMaxX,EAX
    MOV FS:HLi8BR_MOVMaxX,EAX
    MOV FS:HLiG8B_CMPMaxX,EAX
    MOV FS:HLiG8B_CMPMaxX2,EAX
    MOV FS:HLiG8B_MOVMaxX,EAX
    
    MOV EAX,Y2
    MOV VESAMaxY,EAX
    MOV FS:Li8B_XCMPMaxY,EAX
    MOV FS:Li8B_YCMPMaxY,EAX
    MOV FS:HLi8BR_CMPMaxY,EAX
    MOV FS:HLiG8B_CMPMaxY,EAX
    
    RET
    
VESA_SetClipArea ENDP

VESA_SetColor8B PROC PASCAL NR:WORD,R:WORD,G:WORD,B:WORD

    ; ##### Sets the Palette Entry for Color NR.
    ; ##### Red,Green and Blue Values from 0 to 255.

    MOV BL,BYTE PTR NR
    MOV BH,0
    MOV DH,BYTE PTR R
    SHR DH,2
    MOV CH,BYTE PTR G
    SHR CH,2
    MOV CL,BYTE PTR B
    SHR CL,2
    MOV AX,1010h
    INT 10h
    RET

VESA_SetColor8B ENDP

VESA_ClearScreen8B PROC PASCAL CO:DWORD,S:WORD

    ; ##### Clears the Screen 'S' in all 256-Color-Modes.
    ; ##### Fills the Screen with Color CO.
    ; ##### Uses fast DWORD STOS.
    ; ##### The number of bytes cleared are taken from
    ; ##### VESAScrSize=VESABpL*YResolution of the
    ; ##### current Video Mode.
    
    MOV ECX,VESAScrSize
    MOV EBX,ECX
    AND EBX,11b
    SHR ECX,2
    MOV ES,S
    XOR EDI,EDI
    MOV EAX,CO
    CLD
    REP STOS DWORD ES:[EDI]
    MOV ECX,EBX
    REP STOS BYTE ES:[EDI]
    
    RET
    
VESA_ClearScreen8B ENDP

VESA_CopyScreen8B PROC PASCAL S:WORD,D:WORD

    ; ##### Copies the Screen 'S' to the Screen 'D' in all
    ; ##### 256-Color-Modes.
    ; ##### Uses fast DWORD MOVS.
    ; ##### The number of bytes copied are taken from
    ; ##### VESAScrSize=VESABpL*YResolution of the
    ; ##### current Video Mode.
    
    MOV ECX,VESAScrSize
    MOV EBX,ECX
    AND EBX,11b
    SHR ECX,2
    MOV FS,S
    XOR ESI,ESI
    MOV ES,D
    XOR EDI,EDI
    CLD
    REP MOVS DWORD ES:[EDI],FS:[ESI]
    MOV ECX,EBX
    REP MOVS BYTE ES:[EDI],FS:[ESI]
  
    RET
    
VESA_CopyScreen8B ENDP

VESA_GetCodeAlias PROC PASCAL

    ; ##### Because writing to Code Segment causes a protection fault,
    ; ##### we have to create a Code Alias Selector, to which we can
    ; ##### write. This is done by DPMI Function 000ah.
    
    MOV AX,0000ah
    MOV BX,CS
    INT 031h
    MOV FS,AX
    MOV FS:CodeAlias,AX
    RET
    
VESA_GetCodeAlias ENDP

VESA_Line8B PROC PASCAL X1:DWORD,Y1:DWORD,X2:DWORD,Y2:DWORD,C:DWORD,S:WORD

    ; ##### 
    ; ##### Line8B draws a Line from (X1,Y1) to (X2,Y2) in all
    ; ##### 256-Color-Modes. C is used as color and S as the 
    ; ##### Screenselector.
    ; ##### 

    ; ##### FS = CodeAlias for Self-Modification
    ; ##### GS = Selector of Screen
    MOV FS,CS:CodeAlias
    MOV GS,S

    ; ##### ECX  = X2-X1    
    ; ##### EAX  = ABS( ECX )
    ; ##### AddX = ( X2-X1 > 0 ? 1 : -1 )
    MOV EAX,X2
    SUB EAX,X1
    MOV ECX,EAX
    MOV DWORD FS:Li8B_AddX,1
    JNS Li8B_DXPos
    MOV DWORD FS:Li8B_AddX,-1
    NEG EAX
    
  Li8B_DXPos: 
  
    ; ##### EDX  = Y2-Y1
    ; ##### EBX  = ABS( EDX )
    ; ##### AddY = ( Y2-Y1 > 0 ? 1 : -1 )
    MOV EBX,Y2
    SUB EBX,Y1
    MOV EDX,EBX
    MOV DWORD FS:Li8B_AddY,1
    JNS Li8B_DYPos
    MOV DWORD FS:Li8B_AddY,-1
    NEG EBX
    
  Li8B_DYPos: 
  
    ; ##### Walk along X-Axis or Y-Axis ?
    CMP EBX,EAX
    JLE Li8B_ADXBigger
    
      ; ##### Calculate M = (X2-X1)/ABS(Y2-Y1) * 2^16
      OR EBX,EBX
      JZ Li8B_DYZero
      MOV EAX,ECX
      SHL EAX,16
      CDQ
      IDIV EBX   
      MOV FS:Li8B_YM,EAX
      
    Li8B_DYZero:

      ; ##### ECX = Number of Pixs  
      ; ##### EAX = Color
      ; ##### EDX = X1 * 2^16
      ; ##### EBX = Y1  
      MOV ECX,EBX
      INC ECX
      MOV EAX,C 
      MOV EDX,X1
      SHL EDX,16
      MOV EBX,Y1
      
    Li8B_LoopY:
    
      ; ##### ESI = INT( X );
      MOV ESI,EDX
      SAR ESI,16
      
      ; ##### Clipping ...
      ; ##### The Clipping-Values are set by 'SetClipArea'
      
      ; ##### CMP ESI,YCMPMinX 
      DB 081h, 0feh
      Li8B_YCMPMinX DD ?               
      JL Li8B_YNoPix
      
      ; ##### CMP ESI,YCMPMaxX
      DB 081h, 0feh
      Li8B_YCMPMaxX DD ?               
      JG Li8B_YNoPix
      
      ; ##### CMP EBX,YCMPMinY
      DB 081h, 0fbh
      Li8B_YCMPMinY DD ?               
      JL Li8B_YNoPix
      
      ; ##### CMP EBX,YCMPMaxY
      DB 081h, 0fbh
      Li8B_YCMPMaxY DD ?               
      JG Li8B_YNoPix
      
        ; ##### ' Pix( ESI, EBX, AL ) '
        ; ##### These two are the only memory references
        ; ##### in the LOOP. It should be fast.
        MOV EDI,VESAOffs[EBX*4]
        MOV GS:[EDI+ESI],AL
        
    Li8B_YNoPix:  
 
      ; ##### Calculate new X- and Y-Values and go on... 
    
      ; ##### ADD EDX,M means 'ADD X,M'
      DB 081h, 0c2h
      Li8B_YM DD ?                     
      
      ; ##### ADD EBX,AddY means 'ADD Y,AddY'
      DB 081h, 0c3h
      Li8B_AddY DD ?                   
      
      LOOP Li8B_LoopY   
      
      RET
      
  Li8B_ADXBigger: 
  
      ; ##### ECX = Number of Pixs
      MOV ECX,EAX
      INC ECX   
      
      ; ##### Calculate M = (Y2-Y1)/ABS(X2-X1) * 2^16
      OR EAX,EAX
      JZ Li8B_DXZero
      MOV EBX,EAX
      MOV EAX,EDX
      SHL EAX,16
      CDQ
      IDIV EBX   
      MOV FS:Li8B_XM,EAX
      
    Li8B_DXZero:
    
      ; ##### EAX = Color
      ; ##### ESI = X1
      ; ##### EDX = Y1 * 2^16
      MOV EAX,C 
      MOV ESI,X1
      MOV EDX,Y1
      SHL EDX,16
      
    Li8B_LoopX:
    
      ; ##### EBX = INT( Y )    
      MOV EBX,EDX
      SAR EBX,16
      
      ; ##### Clipping ...
      ; ##### The Clipping-Values are set by 'SetClipArea'
      
      ; ##### CMP ESI,XCMPMinX
      DB 081h, 0feh
      Li8B_XCMPMinX DD ?               
      JL Li8B_XNoPix
      
      ; ##### CMP ESI,XCMPMaxX
      DB 081h, 0feh
      Li8B_XCMPMaxX DD ?               
      JG Li8B_XNoPix
      
      ; ##### CMP EBX,XCMPMinY
      DB 081h, 0fbh
      Li8B_XCMPMinY DD ?               
      JL Li8B_XNoPix
      
      ; ##### CMP EBX,XCMPMaxY
      DB 081h, 0fbh
      Li8B_XCMPMaxY DD ?               
      JG Li8B_XNoPix
      
        ; ##### ' Pix( ESI, EBX, AL ) '
        ; ##### These two are the only memory references
        ; ##### in the LOOP. It should be fast.
        MOV EDI,VESAOffs[EBX*4]
        MOV GS:[EDI+ESI],AL
        
    Li8B_XNoPix:  
    
      ; ##### Calculate new X- and Y-Values and go on... 
    
      ; ##### ADD EDX,M means 'ADD Y,M'
      DB 081h, 0c2h
      Li8B_XM DD ?                     
      
      ; ##### ADD ESI,AddX means ADD 'X,AddX'
      DB 081h, 0c6h
      Li8B_AddX DD ?                   
      
      LOOP Li8B_LoopX   
      
      RET
      
VESA_Line8B ENDP
   
VESA_HLine8B PROC PASCAL X1:DWORD,X2:DWORD,Y:DWORD,C:DWORD,S:WORD
    
    ; #####
    ; ##### HLine8B draws a horizontal line from (X1,Y) to
    ; ##### (X2,Y) in all 256-Color-Modes. 
    ; ##### C is used as Color, S as the Screenselector.
    ; #####
    
    ; ##### Clip before Drawing.
    MOV EAX,Y
    CMP EAX,VESAMinY
    JL HLi8B_Exit
    
    CMP EAX,VESAMaxY
    JG HLi8B_Exit
    
    MOV EBX,X1
    MOV ECX,X2
    CMP ECX,EBX
    JGE HLi8B_XOk
    XCHG EBX,ECX
    
  HLi8B_XOk:  
  
    CMP ECX,VESAMinX
    JL HLi8B_Exit
    
    CMP EBX,VESAMaxX
    JG HLi8B_Exit
    
    CMP EBX,VESAMinX
    JGE HLi8B_X1Ok
    MOV EBX,VESAMinX
    
  HLi8B_X1Ok:  
    
    CMP ECX,VESAMaxX
    JLE HLi8B_X2Ok
    MOV ECX,VESAMaxX
    
  HLi8B_X2Ok:

    ; ##### ECX = Number of Bytes  
    SUB ECX,EBX
    INC ECX
    
    ; ##### ES:[EDI] = Screenoffset
    MOV ES,S
    MOV EDI,VESAOffs[EAX*4]
    ADD EDI,EBX
    
    ; ##### Put AL in all bytes of EAX for STOS DWORD's.
    MOV EAX,C
    MOV AH,AL
    MOV BX,AX
    SHL EAX,16
    MOV AX,BX
 
    ; ##### EBX = Number of STOS BYTE's.
    ; ##### ECX = Number of STOS DWORD's.
    MOV EBX,ECX
    SHR ECX,2
    AND EBX,11b
    
    ; ##### Fill.
    CLD
    REP STOS DWORD ES:[EDI]
    MOV ECX,EBX
    REP STOS BYTE ES:[EDI]
    
  HLi8B_Exit:  
  
    RET
    
VESA_HLine8B ENDP   
   
VESA_HLine8BReg PROC
    
    ; #####
    ; ##### HLine8BReg draws a horizontal line from (EBX,EDX) to
    ; ##### (ECX,EDX) in all 256-Color-Modes. 
    ; ##### EAX is used as Color, ES as the Screenselector.
    ; ##### Clipping Values are set by SetClipArea.
    ; #####
    ; ##### These registers are changed :
    ; ##### EBX,ECX,EDI
    ; #####

    ; ##### Clip before Drawing.
    
    ; ##### This means 'CMP EDX,CMPMinY'
    DB 081h,0fah
    HLi8BR_CMPMinY DD ?
    JL HLi8BR_Exit
    
    ; ##### This means 'CMP EDX,CMPMaxY'
    DB 081h,0fah
    HLi8BR_CMPMaxY DD ?
    JG HLi8BR_Exit
    
    CMP ECX,EBX
    JGE HLi8BR_XOk
    XCHG EBX,ECX
    
  HLi8BR_XOk:  
  
    ; ##### This means 'CMP ECX,X2CMPMinX'
    DB 081h,0f9h
    HLi8BR_X2CMPMinX DD ?    
    JL HLi8BR_Exit
    
    ; ##### This means 'CMP EBX,X1CMPMaxX'
    DB 081h,0fbh
    HLi8BR_X1CMPMaxX DD ?
    JG HLi8BR_Exit
    
    ; ##### This means 'CMP EBX,X1CMPMinX'
    DB 081h,0fbh
    HLi8BR_X1CMPMinX DD ?
    JGE HLi8BR_X1Ok
    ; ##### This means 'MOV EBX,MOVMinX'
    DB 0bbh
    HLi8BR_MOVMinX DD ?
    
  HLi8BR_X1Ok:  
    
    ; ##### This means 'CMP ECX,X2CMPMaxX'
    DB 081h,0f9h
    HLi8BR_X2CMPMaxX DD ?
    JLE HLi8BR_X2Ok
    ; ##### This means 'MOV ECX,MOVMaxX'
    DB 0b9h
    HLi8BR_MOVMaxX DD ?
    
  HLi8BR_X2Ok:

    ; ##### ECX = Number of Bytes  
    SUB ECX,EBX
    INC ECX
    
    ; ##### ES:[EDI] = Screenoffset
    MOV EDI,VESAOffs[EDX*4]
    ADD EDI,EBX
    
    ; ##### Put AL in all bytes of EAX for STOS DWORD's.
    MOV AH,AL
    MOV BX,AX
    SHL EAX,16
    MOV AX,BX
 
    ; ##### EBX = Number of STOS BYTE's.
    ; ##### ECX = Number of STOS DWORD's.
    MOV EBX,ECX
    SHR ECX,2
    AND EBX,11b
    
    ; ##### Fill.
    CLD
    REP STOS DWORD ES:[EDI]
    MOV ECX,EBX
    REP STOS BYTE ES:[EDI]
    
  HLi8BR_Exit:  
  
    RET

VESA_HLine8BReg ENDP   
   
VESA_FilledTriangle8B PROC PASCAL X1:DWORD,Y1:DWORD,X2:DWORD,Y2:DWORD,X3:DWORD,Y3:DWORD,
                                  C:DWORD,S:WORD
                        
    ; #####
    ; ##### FilledTriangle8B draws a filled, clipped Triangle in all
    ; ##### 256-Color-Modes. C is used as Color. S as the Screen-
    ; ##### selector.
    ; #####
    
    ; ##### FS = CodeAlias for Selfmodification.
    MOV FS,CS:CodeAlias                        
                        
    ; ##### Bubblesort the Points (X1,Y1),(X2,Y2) and (X3,Y3)
    ; ##### to increasing Y-Values.
    MOV EAX,Y1
    MOV EBX,Y2
    MOV ECX,Y3
    
    ; ##### IF Y1 > Y2 THEN SWAP(Y1,Y2);SWAP(X1,X2)
    CMP EAX,EBX
    JLE FTri8B_Sort1
    XCHG EAX,EBX
    MOV EDX,X1
    XCHG EDX,X2
    MOV X1,EDX
    
  FTri8B_Sort1: 
    
    ; ##### IF Y2 > Y3 THEN SWAP(Y2,Y3);SWAP(X2,X3)
    CMP EBX,ECX
    JLE FTri8B_Sort2
    XCHG EBX,ECX
    MOV EDX,X2
    XCHG EDX,X3
    MOV X2,EDX
    
  FTri8B_Sort2: 
                        
    ; ##### Once again : IF Y1 > Y2 THEN SWAP(Y1,Y2);SWAP(X1,X2)
    CMP EAX,EBX
    JLE FTri8B_Sort3
    XCHG EAX,EBX
    MOV EDX,X1
    XCHG EDX,X2
    MOV X1,EDX
    
  FTri8B_Sort3: 
    
    MOV Y1,EAX
    MOV Y2,EBX
    MOV FS:FTri8B_CMPY2,EBX    
    MOV Y3,ECX
    MOV FS:FTri8B_CMPY3,ECX    
    
    ; ##### IF Y1<>Y2 THEN M12=(X2-X1)/(Y2-Y1)*2^16
    SUB EBX,EAX
    JZ FTri8B_CalcM13
    MOV EAX,X2
    SUB EAX,X1
    SHL EAX,16
    CDQ
    IDIV EBX
    MOV FS:FTri8B_M12,EAX
    
  FTri8B_CalcM13:
  
    ; ##### IF Y1<>Y3 THEN M13=(X3-X1)/(Y3-Y1)*2^16
    MOV EBX,Y3
    SUB EBX,Y1
    JZ FTri8B_CalcM23
    MOV EAX,X3
    SUB EAX,X1
    SHL EAX,16
    CDQ
    IDIV EBX
    MOV FS:FTri8B_L1M13,EAX
    MOV FS:FTri8B_L2M13,EAX
    
  FTri8B_CalcM23:
  
    ; ##### IF Y2<>Y3 THEN M23=(X3-X2)/(Y3-Y2)*2^16
    MOV EBX,Y3
    SUB EBX,Y2
    JZ FTri8B_Draw
    MOV EAX,X3
    SUB EAX,X2
    SHL EAX,16
    CDQ
    IDIV EBX
    MOV FS:FTri8B_M23,EAX

  FTri8B_Draw:

    ; ##### Two loops follow, in wich (XA,Y) and (XB,Y) walk along the
    ; ##### edges of the triangle.
    ; ##### In the first, Y walks from Y1 to Y2, XA walks from X1 to X2
    ; ##### and XB starts to walk from X1 to X3.
    ; ##### In the second loop, Y continues to Y3, XA walks from X2 to X3
    ; ##### and XB continues his journey to X3.
    
    ; ##### Initialize registers:
    ; ##### EDX = Y-Counter.
    ; ##### EAX = Color.
    ; ##### EBX = XA = X1*2^16
    ; ##### ECX = XB = X1*2^16
    MOV EDX,Y1
    MOV EAX,C
    MOV ES,S
    MOV EBX,X1
    MOV ECX,EBX
    SHL EBX,16
    SHL ECX,16
    MOV FTri8B_MEBP,EBP

  FTri8B_Loop1:  
    
    ; ##### This means 'CMP EDX,CMPY2'
    DB 081h,0fah
    FTri8B_CMPY2 DD ?
    JE FTri8B_Loop2Init
    
    ; ##### Save EBX and ECX.
    MOV ESI,EBX
    MOV EBP,ECX
    
    ; ##### Parameters for HLine8BReg:
    ; ##### EBX = 'XA'
    ; ##### ECX = 'XB'
    SAR EBX,16
    SAR ECX,16
    
    CALL VESA_HLine8BReg
    
    ; ##### Restore EBX and ECX.
    MOV EBX,ESI
    MOV ECX,EBP
    
    ; ##### Calculate new X-Values.    
    ; ##### This means 'ADD EBX,M12'
    DB 081h, 0c3h
    FTri8B_M12 DD ?

    ; ##### This means 'ADD ECX,L1M13'
    DB 081h, 0c1h
    FTri8B_L1M13 DD ?
    
    ; ##### Next Line...
    INC EDX
    JMP FTri8B_Loop1

  FTri8B_Loop2Init:
  
    MOV EBP,FTri8B_MEBP
    MOV EBX,X2
    SHL EBX,16
  
  FTri8B_Loop2:  
  
    ; ##### This means 'CMP EDX,CMPY3'
    DB 081h,0fah
    FTri8B_CMPY3 DD ?
    JG FTri8B_LoopExit
    
    ; ##### Save EBX and ECX.
    MOV ESI,EBX
    MOV EBP,ECX
    
    ; ##### Parameters for HLine8BReg:
    ; ##### EBX = XA
    ; ##### ECX = XB
    SAR EBX,16
    SAR ECX,16
    
    CALL VESA_HLine8BReg
    
    ; ##### Restore EBX and ECX.
    MOV EBX,ESI
    MOV ECX,EBP
    
    ; ##### Calculate new X-Values.    
    ; ##### This means 'ADD EBX,M23'
    DB 081h, 0c3h
    FTri8B_M23 DD ?

    ; ##### This means 'ADD ECX,L2M13'
    DB 081h, 0c1h
    FTri8B_L2M13 DD ?
    
    ; ##### Next Line...
    INC EDX
    JMP FTri8B_Loop2

  FTri8B_LoopExit:
  
    MOV EBP,FTri8B_MEBP
    
  FTri8B_Exit:  

    RET                
    
    FTri8B_MEBP DD ?

VESA_FilledTriangle8B ENDP   
   
VESA_HLineG8B PROC PASCAL X1:DWORD,X2:DWORD,Y:DWORD,C1:DWORD,C2:DWORD,S:WORD
    
    ; #####
    ; ##### HLine8B draws a gouraudshaded horizontal line from (X1,Y) to
    ; ##### (X2,Y) in all 256-Color-Modes. 
    ; ##### Interpolates from Color C1 to C2, S is used as the Screenselector.
    ; #####
    
    LOCAL StartC:DWORD
    
    ; ##### Clip before Drawing.
    MOV EAX,Y

    ; ##### This means CMP EAX,VESAMinY
    DB 03dh
    HLiG8B_CMPMinY DD ?
    JL HLiG8B_Exit
     
    ; ##### This means CMP EAX,VESAMaxY
    DB 03dh
    HLiG8B_CMPMaxY DD ?
    JG HLiG8B_Exit
    
    MOV EBX,X1
    MOV ECX,X2
    MOV ESI,C1
    MOV EDX,C2
    CMP ECX,EBX
    JGE HLiG8B_XOk
    XCHG EBX,ECX
    MOV X1,EBX
    MOV X2,ECX
    XCHG EDX,ESI

  HLiG8B_XOk:  
  
    ; ##### This means CMP ECX,VESAMinX
    DB 081h,0f9h
    HLiG8B_CMPMinX DD ?
    JL HLiG8B_Exit
    
    ; ##### This means CMP ECX,VESAMaxX
    DB 081h,0fbh
    HLiG8B_CMPMaxX DD ?
    JG HLiG8B_Exit
    
    ; ##### Here I have to say some words. If X1 is left of
    ; ##### the clip border, X1 is set to VESAMinX. This
    ; ##### is simple. But the Color Value of X1 must also
    ; ##### be increased. So the difference between X1 and
    ; ##### VESAMinX in pixels is saved in HLiG8B_Mul.
    ; ##### Later we IMUL this with DC/DX and add it to
    ; ##### the starting color value. If no clipping is done,
    ; ##### HLiG8B_Mul is set to zero and nothing is added.

    MOV StartC,0

    ; ##### This means CMP EBX,VESAMinX
    DB 081h,0fbh
    HLiG8B_CMPMinX2 DD ?
    JGE HLiG8B_X1Ok
    
    ; ##### This means SUB EBX,VESAMinX
    DB 081h,0ebh
    HLiG8B_SUBMinX DD ?
    
    NEG EBX
    MOV StartC,EBX
    
    ; ##### This means MOV EBX,VESAMinX
    DB 0bbh
    HLiG8B_MOVMinX DD ?
    
  HLiG8B_X1Ok:  
    
    ; ##### This means CMP ECX,VESAMaxX
    DB 081h,0f9h
    HLiG8B_CMPMaxX2 DD ?
    JLE HLiG8B_X2Ok

    ; ##### This means MOV ECX,VESAMaxX
    DB 0b9h
    HLiG8B_MOVMaxX DD ?
    
  HLiG8B_X2Ok:

    ; ##### ES:[EDI] = Screenoffset
    MOV ES,S
    MOV EDI,VESAOffs[EAX*4]
    ADD EDI,EBX
    
    ; ##### ECX = Number of Bytes.
    ; ##### EBX = X2 - X1.
    ; ##### EAX = (C2 - C1)*2^16.
    SUB ECX,EBX
    INC ECX

    MOV EBX,X2
    SUB EBX,X1
    JZ HLiG8B_InitFill
    
    MOV EAX,EDX
    SUB EAX,ESI
    CDQ
    IDIV EBX
    ; ##### EBX = (C2 - C1)/(X2 - X1)*2^16
    MOV EBX,EAX
    
  HLiG8B_InitFill:  

    ; ##### In case of clipping, correct the start color. (see above)
    IMUL EAX,StartC
    ADD ESI,EAX
    MOV EAX,ESI
    CLD

    ; ##### Fill.

  HLiG8B_FillLoop:  
  
    ; ##### EAX = Color.
    BSWAP EAX
    MOV ES:[EDI],AH
    INC EDI
    BSWAP EAX
    
    ; ##### Calculate new Color.
    ADD EAX,EBX
    
    ; ##### Go on.
    LOOP HLiG8B_FillLoop
    
  HLiG8B_Exit:  
  
    RET

VESA_HLineG8B ENDP   
   
VESA_FilledTriangleG8B PROC PASCAL X1:DWORD,Y1:DWORD,X2:DWORD,Y2:DWORD,X3:DWORD,Y3:DWORD,
                                   C1:DWORD,C2:DWORD,C3:DWORD,S:WORD
                        
    ; #####
    ; ##### FilledTriangle8B draws a filled, clipped and gouraudshaded
    ; ##### Triangle in all 256-Color-Modes.
    ; ##### C1,C2,C3 are the Color Values of each edge. S the Screen-
    ; ##### Selector.
    ; #####
    
    LOCAL M12:DWORD,M13:DWORD,M23:DWORD
    LOCAL MC12:DWORD,MC13:DWORD,MC23:DWORD
    LOCAL Y:DWORD,XA:DWORD,XB:DWORD,CA:DWORD,CB:DWORD
  
    ; ##### FS = CodeAlias for Selfmodification.
    MOV FS,CS:CodeAlias                        
                        
    ; ##### Bubblesort the Points (X1,Y1),(X2,Y2) and (X3,Y3)
    ; ##### to increasing Y-Values.
    MOV EAX,Y1
    MOV EBX,Y2
    MOV ECX,Y3
    
    ; ##### IF Y1 > Y2 THEN SWAP(Y1,Y2);SWAP(X1,X2);SWAP(C1,C2)
    CMP EAX,EBX
    JLE FTriG8B_Sort1
    XCHG EAX,EBX
    MOV EDX,X1
    XCHG EDX,X2
    MOV X1,EDX
    MOV EDX,C1
    XCHG EDX,C2
    MOV C1,EDX
    
  FTriG8B_Sort1: 
    
    ; ##### IF Y2 > Y3 THEN SWAP(Y2,Y3);SWAP(X2,X3);SWAP(C2,C3)
    CMP EBX,ECX
    JLE FTriG8B_Sort2
    XCHG EBX,ECX
    MOV EDX,X2
    XCHG EDX,X3
    MOV X2,EDX
    MOV EDX,C2
    XCHG EDX,C3
    MOV C2,EDX
    
  FTriG8B_Sort2: 
                        
    ; ##### Once again : IF Y1 > Y2 THEN SWAP(Y1,Y2);SWAP(X1,X2);SWAP(C1,C2)
    CMP EAX,EBX
    JLE FTriG8B_Sort3
    XCHG EAX,EBX
    MOV EDX,X1
    XCHG EDX,X2
    MOV X1,EDX
    MOV EDX,C1
    XCHG EDX,C2
    MOV C1,EDX
    
  FTriG8B_Sort3: 
    
    MOV Y1,EAX
    MOV Y2,EBX
    MOV Y3,ECX
    
    ; ##### IF Y1<>Y2 THEN M12 =(X2-X1)/(Y2-Y1)*2^16
    ; #####                MC12=(C2-C1)/(Y2-Y1)*2^16
    SUB EBX,EAX
    JZ FTriG8B_CalcM13
    MOV EAX,X2
    SUB EAX,X1
    SHL EAX,16
    CDQ
    IDIV EBX
    MOV M12,EAX
    MOV EAX,C2
    SUB EAX,C1
    SHL EAX,16
    CDQ
    IDIV EBX
    MOV MC12,EAX
    
  FTriG8B_CalcM13:
  
    ; ##### IF Y1<>Y3 THEN M13 =(X3-X1)/(Y3-Y1)*2^16
    ; #####                MC13=(C3-C1)/(Y3-Y1)*2^16
    MOV EBX,Y3
    SUB EBX,Y1
    JZ FTriG8B_CalcM23
    MOV EAX,X3
    SUB EAX,X1
    SHL EAX,16
    CDQ
    IDIV EBX
    MOV M13,EAX
    MOV EAX,C3
    SUB EAX,C1
    SHL EAX,16
    CDQ
    IDIV EBX
    MOV MC13,EAX
    
  FTriG8B_CalcM23:
  
    ; ##### IF Y2<>Y3 THEN M23 =(X3-X2)/(Y3-Y2)*2^16
    ; #####                MC23=(C3-C2)/(Y3-Y2)*2^16
    MOV EBX,Y3
    SUB EBX,Y2
    JZ FTriG8B_Draw
    MOV EAX,X3
    SUB EAX,X2
    SHL EAX,16
    CDQ
    IDIV EBX
    MOV M23,EAX
    MOV EAX,C3
    SUB EAX,C2
    SHL EAX,16
    CDQ
    IDIV EBX
    MOV MC23,EAX

  FTriG8B_Draw:

    ; ##### Two loops follow, in wich (XA,Y) and (XB,Y) walk along the
    ; ##### borders of the triangle.
    ; ##### As a bonus, Colors (CA and CB) are interpolated between the
    ; ##### edges.
    ; ##### In the first, Y walks from Y1 to Y2, XA walks from X1 to X2
    ; ##### and XB starts to walk from X1 to X3.
    ; ##### In the second loop, Y continues to Y3, XA walks from X2 to X3
    ; ##### and XB continues his journey to X3.
    
    ; ##### Initialize Y.
    MOV EAX,Y1
    MOV Y,EAX
    
    ; ##### Initialize XA and XB.
    MOV EAX,X1
    SHL EAX,16
    MOV XA,EAX
    MOV XB,EAX
    
    ; ##### Initialize CA and CB.
    MOV EAX,C1
    SHL EAX,16
    MOV CA,EAX
    MOV CB,EAX
    
  FTriG8B_Loop1:
    
    ; ##### Ready with first loop?
    MOV EAX,Y
    CMP EAX,Y2
    JE FTriG8B_Loop2Init
    
    ; ##### Call HLineG8B.
    MOV EAX,XA
    SAR EAX,16
    PUSH EAX
    MOV EAX,XB
    SAR EAX,16
    PUSH EAX
    MOV EAX,Y
    PUSH EAX
    MOV EAX,CA
    PUSH EAX
    MOV EAX,CB
    PUSH EAX
    MOVZX EAX,S
    PUSH EAX
    CALL VESA_HLineG8B

    ; ##### Calculate new X- and Color-Values.
    MOV EAX,M12
    ADD XA,EAX
    MOV EAX,M13
    ADD XB,EAX
    MOV EAX,MC12
    ADD CA,EAX
    MOV EAX,MC13
    ADD CB,EAX
    
    INC Y
    
    JMP FTriG8B_Loop1
    
  FTriG8B_Loop2Init:
  
    MOV EAX,C2
    SHL EAX,16
    MOV CA,EAX
    MOV EAX,X2
    SHL EAX,16
    MOV XA,EAX
  
  FTriG8B_Loop2:
    
    ; ##### Ready ?
    MOV EAX,Y
    CMP EAX,Y3
    JG FTriG8B_Exit
    
    ; ##### Call HLineG8B.
    MOV EAX,XA
    SAR EAX,16
    PUSH EAX
    MOV EAX,XB
    SAR EAX,16
    PUSH EAX
    MOV EAX,Y
    PUSH EAX
    MOV EAX,CA
    PUSH EAX
    MOV EAX,CB
    PUSH EAX
    MOVZX EAX,S
    PUSH EAX
    CALL VESA_HLineG8B

    ; ##### Calculate new X- and Color-Values.
    MOV EAX,M23
    ADD XA,EAX
    MOV EAX,M13
    ADD XB,EAX
    MOV EAX,MC23
    ADD CA,EAX
    MOV EAX,MC13
    ADD CB,EAX
    
    INC Y
    
    JMP FTriG8B_Loop2

  FTriG8B_Exit:  

    RET                        

VESA_FilledTriangleG8B ENDP   
   
VESA_HLineTex8B PROC PASCAL X1:DWORD,X2:DWORD,Y:DWORD,TX1:DWORD,TY1:DWORD,TX2:DWORD,TY2:DWORD,
                            MulTable:DWORD,S:WORD
            
    ; ##### Draws a textured HLine from (X1,Y) to (X2,Y). Texture coordinates
    ; ##### are interpolated from (TX1,TY1) to (TX2,TY2). S is the Screen
    ; ##### Selector, MulTable is a pointer to a DWORD-array, in which every 
    ; ##### DWORD is the Memory-Offset for the corresponding line. 
            
    LOCAL StartTX:DWORD
           
    MOV FS,CS:CodeAlias           
     
    ; ##### Clipping...      
    MOV EDX,X1
    MOV EBX,X2
    MOV ECX,Y
    
    CMP ECX,VESAMinY
    JL HLiT8B_Exit
    
    CMP ECX,VESAMaxY
    JG HLiT8B_Exit
    
    CMP EDX,EBX
    JLE HLiT8B_XOk
    ; ##### Swaping...
    XCHG EDX,EBX    
    MOV X1,EDX
    MOV X2,EBX
    MOV EAX,TX1
    XCHG EAX,TX2
    MOV TX1,EAX
    MOV EAX,TY1
    XCHG EAX,TY2
    MOV TY1,EAX
    
  HLiT8B_XOk:         

    CMP EDX,VESAMaxX
    JG HLiT8B_Exit
    
    CMP EBX,VESAMinX
    JL HLiT8B_Exit
    
    ; ##### Prepare to correct the start-tex-coors if clipping is necessary.
    MOV StartTX,0
    CMP EDX,VESAMinX
    JGE HLiT8B_X1OK
    SUB EDX,VESAMinX
    NEG EDX
    MOV StartTX,EDX
    MOV EDX,VESAMinX
    
  HLiT8B_X1OK:  
  
    CMP EBX,VESAMaxX
    JLE HLiT8B_X2OK
    MOV EBX,VESAMaxX
    
  HLiT8B_X2OK:  

    ; ##### Get screen offset.
    MOV ES,S
    MOV EDI,VESAOffs[ECX*4]
    ADD EDI,EDX
    
    ; #### ECX = Number of pixs.
    SUB EBX,EDX
    MOV ECX,EBX
    INC ECX
    
    MOV EBX,X2
    SUB EBX,X1
    JZ HLiT8B_InitLoop
    
    ; ##### Calc dTX/dX.
    MOV EAX,TX2
    SUB EAX,TX1
    CDQ
    IDIV EBX
    MOV FS:HLiT8B_ADDTX,EAX
    ; ##### Correct start-TX if clipping was necessary.
    IMUL StartTX
    ADD TX1,EAX
    
    ; ##### Calc dTY/dX.
    MOV EAX,TY2
    SUB EAX,TY1
    CDQ
    IDIV EBX           
    MOV FS:HLiT8B_ADDTY,EAX
    ; ##### Correct start-TY if clipping was necessary.
    IMUL StartTX
    ADD TY1,EAX
    
  HLiT8B_InitLoop:   

    ; ##### Prepare to LOOP.
    MOV EBX,MulTable
    MOV HLiT8B_MULTable,EBX
    MOV EDX,TX1
    MOV EBX,TY1
    CLD
    
    ; ##### Main LOOP :
    ; ##### EDX = TX in 16.16 fixed points.
    ; ##### EBX = TY in 16.16 fixed points.
    
  HLiT8B_Loop:

    ; ESI = TX + MulTable[TY];
    MOV EAX,EBX
    SHR EAX,16
    
    MOV ESI,EDX
    SHR ESI,16
    ; ##### ADD ESI,MULTable[4*EAX]
    db 003h,034h,085h
    HLiT8B_MULTable DD ?

    MOVSB
    
    ; ##### ADD EDX,_ADDTX
    DB 081h, 0c2h
    HLiT8B_ADDTX DD ?
    
    ; ##### ADD EBX,_ADDTY
    DB 081h, 0c3h
    HLiT8B_ADDTY DD ?
    
    LOOP HLiT8B_Loop

  HLiT8B_Exit:

    RET
  
VESA_HLineTex8B ENDP   

VESA_ClearZBuffer PROC PASCAL M:DWORD,C:DWORD

    ; ##### Clears the entire Z-Buffer pointed to by M.
    ; ##### C is the number of z-values in M.
 
    PUSH DS
    POP ES
    MOV EDI,M
    MOV ECX,C
    ; ##### 0ffffffffh is VERY far away.
    MOV EAX,0ffffffffh
    CLD
    REP STOSD
    RET

VESA_ClearZBuffer ENDP
   
VESA_HLineZ8B PROC PASCAL X1:DWORD, X2:DWORD, Y:DWORD, Z1:DWORD, Z2:DWORD, C:DWORD, 
                          ZSEL:WORD, S:WORD

    ; ##### Draws a zbuffered-HLine from (X1,Y(,Z1)) to (X2,Y(,Z2)) with Color C
    ; ##### in all 256-color-modes. ZSEL is the z-buffer-Selector, S the 
    ; ##### screen-selector.

    LOCAL Correction:DWORD

    MOV FS,CS:CodeAlias

    ; ##### Clipping...
    MOV EAX,X1
    MOV EBX,X2
    MOV ECX,Y
    
    CMP ECX,VESAMinY
    JL HLiZ8B_Exit
    
    CMP ECX,VESAMaxY
    JG HLiZ8B_Exit
    
    CMP EAX,EBX
    JLE HLiZ8B_XOk
    ; ##### Swaping...
    XCHG EAX,EBX    
    MOV X1,EAX
    MOV X2,EBX
    MOV EDX,Z1
    XCHG EDX,Z2
    MOV Z1,EDX
    
  HLiZ8B_XOk:         

    CMP EAX,VESAMaxX
    JG HLiZ8B_Exit
    
    CMP EBX,VESAMinX
    JL HLiZ8B_Exit
    
    ; ##### Prepare to correct start-Z-value if clipping is necessary.
    MOV Correction,0
    CMP EAX,VESAMinX
    JGE HLiZ8B_X1OK
    SUB EAX,VESAMinX
    NEG EAX
    MOV Correction,EAX
    MOV EAX,VESAMinX
    
  HLiZ8B_X1OK:  
  
    CMP EBX,VESAMaxX
    JLE HLiZ8B_X2OK
    MOV EBX,VESAMaxX
    
  HLiZ8B_X2OK:  

    ; ##### Get screen offset and z-buffer-selector.
    MOV ES,S
    MOV EDI,VESAOffs[ECX*4]
    ADD EDI,EAX
    MOV GS,ZSEL
    
    ; ##### ECX = nr. of pixs.
    SUB EBX,EAX
    MOV ECX,EBX
    INC ECX
    
    MOV EBX,X2
    SUB EBX,X1
    JZ HLiZ8B_DXZero

    ; ##### Calc dZ/dX.    
    MOV EAX,Z2
    SUB EAX,Z1
    CDQ
    IDIV EBX
    MOV FS:HLiZ8B_ADDZ,EAX    
    
  HLiZ8B_DXZero:

    ; ##### Correct start-Z-value if clipping was necessary.
    IMUL EAX,Correction   
    MOV EDX,Z1
    ADD EDX,EAX
    MOV EAX,C
    
    ; ##### Main LOOP:
    ; ##### EDX = Z in 16.16 fixed point or 8.24 or whatever you like... ;)
    ; ##### EDI = Offset to pixel in screen.
    ; ##### EDI*4 = Offset to value in Z-Buffer.
    
  HLiZ8B_Loop:
 
    ; ##### Compare Z with value in z-buffer.
    ; ##### If behind, don't pix.
    CMP EDX,GS:[EDI*4]
    JAE HLiZ8B_Behind
    ; ##### If before, set new z-buffer-value and pix.
    MOV GS:[EDI*4],EDX
    MOV ES:[EDI],AL
    
  HLiZ8B_Behind:
  
    db 081h, 0c2h
    HLiZ8B_ADDZ dd ?

    INC EDI    
  
    LOOP HLiZ8B_Loop
  
  HLiZ8B_Exit:
  
    RET  

VESA_HLineZ8B ENDP   
   
VESA_HLineGZ8B PROC PASCAL X1:DWORD, X2:DWORD, Y:DWORD, Z1:DWORD, Z2:DWORD, C1:DWORD, 
                           C2:DWORD, ZSEL:WORD, S:WORD

    ; ##### Draws a zbuffered/gouraudshaded HLine from (X1,Y(,Z1)) to (X2,Y(,Z2)) 
    ; ##### with colors interpolated from C1 to C2
    ; ##### in all 256-color-modes. ZSEL is the z-buffer-Selector, S the 
    ; ##### screen-selector.

    LOCAL Correction:DWORD

    MOV FS,CS:CodeAlias

    ; ##### Clipping...
    MOV EAX,X1
    MOV EBX,X2
    MOV ECX,Y
    
    CMP ECX,VESAMinY
    JL HLiGZ8B_Exit
    
    CMP ECX,VESAMaxY
    JG HLiGZ8B_Exit
    
    CMP EAX,EBX
    JLE HLiGZ8B_XOk
    ; ##### Swaping...
    XCHG EAX,EBX    
    MOV X1,EAX
    MOV X2,EBX
    MOV EDX,Z1
    XCHG EDX,Z2
    MOV Z1,EDX
    MOV EDX,C1
    XCHG EDX,C2
    MOV C1,EDX
    
  HLiGZ8B_XOk:         

    CMP EAX,VESAMaxX
    JG HLiGZ8B_Exit
    
    CMP EBX,VESAMinX
    JL HLiGZ8B_Exit
    
    ; ##### Prepare to correct start-Z-value if clipping is necessary.
    MOV Correction,0
    CMP EAX,VESAMinX
    JGE HLiGZ8B_X1OK
    SUB EAX,VESAMinX
    NEG EAX
    MOV Correction,EAX
    MOV EAX,VESAMinX
    
  HLiGZ8B_X1OK:  
  
    CMP EBX,VESAMaxX
    JLE HLiGZ8B_X2OK
    MOV EBX,VESAMaxX
    
  HLiGZ8B_X2OK:  

    ; ##### Get screen offset and z-buffer-selector.
    MOV ES,S
    MOV EDI,VESAOffs[ECX*4]
    ADD EDI,EAX
    MOV GS,ZSEL
    
    ; ##### ECX = nr. of pixs.
    SUB EBX,EAX
    MOV ECX,EBX
    INC ECX
    
    MOV EBX,X2
    SUB EBX,X1
    JZ HLiGZ8B_DXZero

    ; ##### Calc dZ/dX.    
    MOV EAX,Z2
    SUB EAX,Z1
    CDQ
    IDIV EBX
    MOV FS:HLiGZ8B_ADDZ,EAX    
    IMUL EAX,Correction
    ADD Z1,EAX
    
    MOV EAX,C2
    SUB EAX,C1
    CDQ
    IDIV EBX
    MOV FS:HLiGZ8B_ADDC,EAX
    IMUL EAX,Correction
    ADD C1,EAX
    
  HLiGZ8B_DXZero:

    ; ##### Correct start-Z-value if clipping was necessary.
    MOV EDX,Z1
    MOV EAX,C1
    
    ; ##### Main LOOP:
    ; ##### EDX = Z in 16.16 fixed point or 8.24 or whatever you like... ;)
    ; ##### EDI = Offset to pixel in screen.
    ; ##### EDI*4 = Offset to value in Z-Buffer.
    
  HLiGZ8B_Loop:
 
    ; ##### Compare Z with value in z-buffer.
    ; ##### If behind, don't pix.
    CMP EDX,GS:[EDI*4]
    JAE HLiGZ8B_Behind
    ; ##### If before, set new z-buffer-value and pix.
    MOV GS:[EDI*4],EDX
    BSWAP EAX
    MOV ES:[EDI],AH
    BSWAP EAX
    
  HLiGZ8B_Behind:
  
    ; ##### Get next Z- and C-Value.    
    ; ##### ADD EDX,_ADDZ
    db 081h, 0c2h
    HLiGZ8B_ADDZ dd ?
    
    ; ##### ADD EAX,_ADDC
    db 05h
    HLiGZ8B_ADDC dd ?

    INC EDI    
  
    LOOP HLiGZ8B_Loop
  
  HLiGZ8B_Exit:
  
    RET  

VESA_HLineGZ8B ENDP   
   
VESA_HLineTexZ8B PROC PASCAL X1:DWORD,X2:DWORD,Y:DWORD,TX1:DWORD,TY1:DWORD,TX2:DWORD,TY2:DWORD,
                             Z1:DWORD,Z2:DWORD,MulTable:DWORD,ZSEL:WORD,S:WORD
            
    ; ##### Draws a textured HLine from (X1,Y) to (X2,Y). Texture coordinates
    ; ##### are interpolated from (TX1,TY1) to (TX2,TY2). S is the Screen
    ; ##### Selector, MulTable is a pointer to a DWORD-array, in which every 
    ; ##### DWORD is the Memory-Offset for the corresponding line.
    ; ##### ZSEL is the Z-Buffer-Selector.
    ; #####
    ; ##### The TX- and Z-Values must all be 16.16-Fixed-Points.
            
    LOCAL StartTX:DWORD
           
    MOV FS,CS:CodeAlias           
     
    ; ##### Clipping...      
    MOV EDX,X1
    MOV EBX,X2
    MOV ECX,Y
    
    CMP ECX,VESAMinY
    JL HLiTZ8B_Exit
    
    CMP ECX,VESAMaxY
    JG HLiTZ8B_Exit
    
    CMP EDX,EBX
    JLE HLiTZ8B_XOk
    ; ##### Swaping...
    XCHG EDX,EBX    
    MOV X1,EDX
    MOV X2,EBX
    MOV EAX,TX1
    XCHG EAX,TX2
    MOV TX1,EAX
    MOV EAX,TY1
    XCHG EAX,TY2
    MOV TY1,EAX
    MOV EAX,Z1
    XCHG EAX,Z2
    MOV Z1,EAX
    
  HLiTZ8B_XOk:         

    CMP EDX,VESAMaxX
    JG HLiTZ8B_Exit
    
    CMP EBX,VESAMinX
    JL HLiTZ8B_Exit
    
    ; ##### Prepare to correct the start-values if clipping is necessary.
    MOV StartTX,0
    CMP EDX,VESAMinX
    JGE HLiTZ8B_X1OK
    SUB EDX,VESAMinX
    NEG EDX
    MOV StartTX,EDX
    MOV EDX,VESAMinX
    
  HLiTZ8B_X1OK:  
  
    CMP EBX,VESAMaxX
    JLE HLiTZ8B_X2OK
    MOV EBX,VESAMaxX
    
  HLiTZ8B_X2OK:  

    ; ##### Get screen offset.
    MOV ES,S
    MOV EDI,VESAOffs[ECX*4]
    ADD EDI,EDX
    MOV GS,ZSEL
    
    ; #### ECX = Number of pixs.
    SUB EBX,EDX
    MOV ECX,EBX
    INC ECX
    
    MOV EBX,X2
    SUB EBX,X1
    JZ HLiTZ8B_InitLoop
    
    ; ##### Calc dTX/dX.
    MOV EAX,TX2
    SUB EAX,TX1
    CDQ
    IDIV EBX
    MOV FS:HLiTZ8B_ADDTX,EAX
    ; ##### Correct start-TX if clipping was necessary.
    IMUL StartTX
    ADD TX1,EAX
    
    ; ##### Calc dTY/dX.
    MOV EAX,TY2
    SUB EAX,TY1
    CDQ
    IDIV EBX           
    MOV FS:HLiTZ8B_ADDTY,EAX
    ; ##### Correct start-TY if clipping was necessary.
    IMUL StartTX
    ADD TY1,EAX
    
    ; ##### Calc dZ/dX.
    MOV EAX,Z2
    SUB EAX,Z1
    CDQ
    IDIV EBX
    MOV FS:HLiTZ8B_ADDZ,EAX
    ; ##### Correct start-Z if clipping was necessary.
    IMUL StartTX
    ADD Z1,EAX
    
  HLiTZ8B_InitLoop:   

    ; ##### Prepare to LOOP.
    MOV EAX,MulTable
    MOV FS:HLiTZ8B_MULTable,EAX
    MOV EBX,Z1
    MOV EDX,TX1
    PUSH EBP
    MOV EBP,TY1
    
    ; ##### Main LOOP :
    ; ##### EDX = TX in 16.16 fixed points.
    ; ##### EBP = TY in 16.16 fixed points.
   
  HLiTZ8B_Loop:

    CMP EBX,GS:[EDI*4]
    JAE HLiTZ8B_Behind
    MOV GS:[EDI*4],EBX
    ; ESI = TX + MulTable[TY];
    MOV EAX,EBP
    SAR EAX,16
    MOV ESI,EDX
    SAR ESI,16
    ; ##### ADD ESI,DWORD DS:[_MULTable+4*EAX]
    db 003h,034h,085h
    HLiTZ8B_MULTable DD ?

    MOV AL,DS:[ESI]
    MOV ES:[EDI],AL
    
  HLiTZ8B_Behind:  
    
    ; ##### ADD EDX,_ADDTX
    DB 081h, 0c2h
    HLiTZ8B_ADDTX DD ?
    
    ; ##### ADD EBP,_ADDTY
    DB 081h, 0c5h
    HLiTZ8B_ADDTY DD ?
    
    ; ##### ADD EBX,_ADDZ
    DB 081h,0c3h
    HLiTZ8B_ADDZ DD ?
    
    INC EDI
    
    LOOP HLiTZ8B_Loop

    POP EBP

  HLiTZ8B_Exit:

    RET
  
VESA_HLineTexZ8B ENDP   

VESA_HLineG8BFast PROC PASCAL X1:DWORD,X2:DWORD,Y:DWORD,C1:DWORD,C2:DWORD,DC:DWORD,S:WORD

    MOV EAX,Y
    MOV EBX,X1
    MOV ECX,X2
    MOV EDX,C1
    CMP EBX,ECX
    JLE HLiG8BF_XOk
    XCHG EBX,ECX
    MOV EDX,C2
    
  HLiG8BF_XOk:  
 
    MOV ES,S
    MOV EDI,VESAOffs[EAX*4]
    ADD EDI,EBX

    SUB ECX,EBX
    INC ECX

    MOV EBX,DC
    CLD
    
  HLiG8BF_Loop:  
    
    MOV EAX,EDX
    SAR EAX,16
    
    STOS BYTE ES:[EDI]
    
    ADD EDX,EBX
    
    LOOP HLiG8BF_Loop

  HLiG8BF_Exit:

    RET

VESA_HLineG8BFast ENDP

VESA_HLineTex8BFast PROC PASCAL X1:DWORD,X2:DWORD,Y:DWORD,
                                TX1:DWORD,TY1:DWORD,TX2:DWORD,TY2:DWORD,
                                DTX:DWORD, DTY:DWORD,
                                MUL:DWORD, S:WORD

    MOV FS,CS:CodeAlias

    MOV EAX,Y
    MOV EBX,X1
    MOV ECX,X2
    MOV EDX,TX1
    MOV ESI,TY1
    CMP EBX,ECX
    JLE HLiT8BF_XOk
    XCHG EBX,ECX
    MOV EDX,TX2
    MOV ESI,TY2
    
  HLiT8BF_XOk:  
 
    MOV ES,S
    MOV EDI,VESAOffs[EAX*4]
    ADD EDI,EBX

    SUB ECX,EBX
    INC ECX
    
    MOV EAX,MUL
    MOV FS:HLiT8BF_MULTable,EAX
    MOV EAX,DTX
    MOV FS:HLiT8BF_ADDTX,EAX
    MOV EAX,DTY
    MOV FS:HLiT8BF_ADDTY,EAX

    MOV EBX,ESI
    CLD
    
  HLiT8BF_Loop:  
    
    ; ESI = TX + MulTable[TY];
    MOV EAX,EBX
    SHR EAX,16
    MOV ESI,EDX
    SHR ESI,16
    ; ##### ADD ESI,_MulTable[4*EAX]
    db 003h,034h,085h
    HLiT8BF_MULTable DD ?

    MOVSB
    
    ; ##### ADD EDX,_ADDTX
    DB 081h, 0c2h
    HLiT8BF_ADDTX DD ?
    
    ; ##### ADD EBX,_ADDTY
    DB 081h, 0c3h
    HLiT8BF_ADDTY DD ?
    
    LOOP HLiT8BF_Loop

  HLiT8BF_Exit:

    RET

VESA_HLineTex8BFast ENDP

CODE ENDS   
   
END

