Option Explicit

' Image indices inside the standard bitmap provided with bttncurr.dll
Global Const TOOLIMAGE_EDITCUT = 0
Global Const TOOLIMAGE_EDITCOPY = 1
Global Const TOOLIMAGE_EDITPASTE = 2
Global Const TOOLIMAGE_FILENEW = 3
Global Const TOOLIMAGE_FILEOPEN = 4
Global Const TOOLIMAGE_FILESAVE = 5
Global Const TOOLIMAGE_FILEPRINT = 6
Global Const TOOLIMAGE_HELP = 7
Global Const TOOLIMAGE_HELPCONTEXT = 8

' Standard sizes for toolbar buttons and bitmaps
Global Const TOOLBUTTON_STDWIDTH = 24
Global Const TOOLBUTTON_STDHEIGHT = 22
Global Const TOOLBUTTON_STDIMAGEWIDTH = 16
Global Const TOOLBUTTON_STDIMAGEHEIGHT = 15
Global Const TOOLBUTTON_STDSEPERATOR = 6

'
' Values for button display states.  Each value is mutually
' exclusive and contains one or more grouping bits.  Each group
' represents buttons that share some sub-state in common.
'
Const BUTTONGROUP_DOWN = &H1
Const BUTTONGROUP_ACTIVE = &H2
Const BUTTONGROUP_DISABLED = &H4
Const BUTTONGROUP_LIGHTFACE = &H8
Const BUTTONGROUP_BLANK = &H10

' Command buttons only
Global Const COMMANDBUTTON_UP = (BUTTONGROUP_ACTIVE)
Global Const COMMANDBUTTON_MOUSEDOWN = (BUTTONGROUP_ACTIVE Or BUTTONGROUP_DOWN)
Global Const COMMANDBUTTON_DISABLED = (BUTTONGROUP_DISABLED)

' Attribute buttons only
Global Const ATTRIBUTEBUTTON_UP = (BUTTONGROUP_ACTIVE)
Global Const ATTRIBUTEBUTTON_MOUSEDOWN = (BUTTONGROUP_ACTIVE Or BUTTONGROUP_DOWN)
Global Const ATTRIBUTEBUTTON_DISABLED = (BUTTONGROUP_DISABLED)
Global Const ATTRIBUTEBUTTON_DOWN = (BUTTONGROUP_ACTIVE Or BUTTONGROUP_DOWN Or BUTTONGROUP_LIGHTFACE)
Global Const ATTRIBUTEBUTTON_INDETERMINATE = (BUTTONGROUP_ACTIVE Or BUTTONGROUP_LIGHTFACE)
Global Const ATTRIBUTEBUTTON_DOWNDISABLED = (BUTTONGROUP_DISABLED Or BUTTONGROUP_DOWN Or BUTTONGROUP_LIGHTFACE)

Declare Function UIToolButtonDraw% Lib "bttncur.dll" (ByVal hDC%, ByVal x%, ByVal y%, ByVal dxButton%, ByVal dyButton%, ByVal hBmp%, ByVal dxBitmap%, ByVal dyBitmap%, ByVal iImage%, ByVal uState%)


' **
' ** Windows definitions used by this module
' **
Type RECT
    left As Integer
    top As Integer
    right As Integer
    bottom As Integer
End Type

Type POINTAPI
    x As Integer
    y As Integer
End Type

Type msg
    hWnd As Integer
    message As Integer
    wparam As Integer
    lParam As Long
    time As Long
    pt As POINTAPI
End Type

Const WM_LBUTTONUP = &H202
Const WM_MOUSEMOVE = &H200
Const WM_MOUSEFIRST = &H200
Const WM_MOUSELAST = &H209
Const PM_REMOVE = 1

Declare Sub GetCursorPos Lib "User" (lpPoint As POINTAPI)
Declare Sub ScreenToClient Lib "User" Alias "ScreenToClient" (ByVal hWnd As Integer, MyPoint As POINTAPI)
Declare Sub ClientToScreen Lib "User" (ByVal hWnd As Integer, lpPoint As POINTAPI)
Declare Function SetCapture Lib "User" (ByVal hWnd%) As Integer
Declare Sub ReleaseCapture Lib "User" ()
Declare Function PeekMessage Lib "User" (lpMsg As msg, ByVal hWnd As Integer, ByVal wMsgFilterMin As Integer, ByVal wMsgFilterMax As Integer, ByVal wRemoveMsg As Integer) As Integer
Declare Function PtInRect Lib "User" (lpRect As RECT, ByVal ptRect As Any) As Integer


' **
' ** Private Data for ToolBar module
' **
Type ToolBarButtonInfo
    ID As Integer
    Index As Integer
    rc As RECT
    bAttributeButton As Integer
    state As Integer
End Type

Const TOOLBAR_BORDER = 2

Dim Toolbar As PictureBox
Dim ToolBarButtons()  As ToolBarButtonInfo
Dim ToolBarButtonIndex As Integer
Dim hToolBarBitmap As Integer
Dim ToolBarImageWidth As Integer
Dim ToolBarImageHeight As Integer
Dim ToolBarNumButtons As Integer
Dim ToolBarButtonWidth As Integer
Dim ToolBarButtonHeight As Integer

Function POINTAPItoLong (pt As POINTAPI) As Long
' Helper routine to convert POINTAPI routine to a long

    POINTAPItoLong = pt.x + CLng(pt.y) * &H10000
End Function

Function ToolBarDepressButton () As Integer
' Determines which toolbar button (if any) was presssed.
' Call this function in the toolbar mouse down event.
'
' Parameters:
'   return value - the button index of the button that was pressed. Zero if not button pressed.

    Dim ButtonIndex As Integer
    Dim pt As POINTAPI
    Dim message As msg
    Dim bDepressed As Integer
    Dim bToolBarCapture As Integer
    Dim UpState As Integer
    Dim DownState As Integer
    Dim NewState As Integer
    Dim i As Integer
    Dim ec As Integer

    ' Get current position of the mouse pointer in pixels
    Call GetCursorPos(pt)
    Call ScreenToClient(Toolbar.hWnd, pt)

    ' First find the button (if any) that is under the mouse
    ButtonIndex = 0
    For i = 1 To ToolBarNumButtons
	If PtInRect(ToolBarButtons(i).rc, POINTAPItoLong(pt)) Then
	    ButtonIndex = i
	    Exit For
	End If
    Next i
    
    If ButtonIndex > 0 Then
	
	' If button is disabled then leave
	UpState = ToolBarButtons(ButtonIndex).state
	If (UpState And BUTTONGROUP_DISABLED) <> 0 Then
	    ToolBarDepressButton = 0
	    Exit Function
	End If

	' Draw button in down state
	If ToolBarButtons(ButtonIndex).bAttributeButton Then
	    DownState = ATTRIBUTEBUTTON_MOUSEDOWN
	Else
	    DownState = COMMANDBUTTON_MOUSEDOWN
	End If
	bDepressed = True
	Call ToolBarDrawButton(ButtonIndex, DownState)

	' Capture the mouse
	bToolBarCapture = True
	ec = SetCapture(Toolbar.hWnd)
	
	' Wait until the mouse button is released
	While (bToolBarCapture)
	    If PeekMessage(message, Toolbar.hWnd, WM_MOUSEFIRST, WM_MOUSELAST, PM_REMOVE) Then
		Select Case message.message
		    Case WM_LBUTTONUP
			' Check to see if mouse is still over button and release the mouse capture
			bDepressed = PtInRect(ToolBarButtons(ButtonIndex).rc, message.lParam)
			Call ReleaseCapture
			bToolBarCapture = False
		    
		    Case WM_MOUSEMOVE
			' While tracking the mouse, draw the button down if the mouse
			' is over button, otherwise draw the button up.
			If PtInRect(ToolBarButtons(ButtonIndex).rc, message.lParam) Then
			    If Not bDepressed Then
				bDepressed = True
				Call ToolBarDrawButton(ButtonIndex, DownState)
			    End If
			Else
			    If bDepressed Then
				bDepressed = False
				Call ToolBarDrawButton(ButtonIndex, UpState)
			    End If
			End If
		    
		    Case Else
		End Select
	    End If
	Wend

	' Determine the new state of button
	If bDepressed Then
	    ToolBarDepressButton = ToolBarButtons(ButtonIndex).ID
	    If ToolBarButtons(ButtonIndex).bAttributeButton Then
		If (UpState = ATTRIBUTEBUTTON_UP) Or (UpState = ATTRIBUTEBUTTON_INDETERMINATE) Then
		    NewState = ATTRIBUTEBUTTON_DOWN
		Else
		    NewState = ATTRIBUTEBUTTON_UP
		End If
	    Else
		NewState = COMMANDBUTTON_UP
	    End If
	Else
	    ToolBarDepressButton = 0
	    NewState = UpState
	End If
	
	' Set the new state of the button
	Call ToolBarSetButtonState(ButtonIndex, NewState)
    
    Else
	ToolBarDepressButton = 0
    End If
End Function

Sub ToolBarDrawButton (ButtonIndex As Integer, state As Integer)
' Internal sub that calls UIToolButtonDraw to draw a toolbar button
'
' Parameters:
'   ButtonIndex - Index of the toolbar button in the button array
'   state       - the state that the button will be drawn in

    Dim ec As Integer

    ec = UIToolButtonDraw%(Toolbar.hDC, ToolBarButtons(ButtonIndex).rc.left, ToolBarButtons(ButtonIndex).rc.top, ToolBarButtonWidth, ToolBarButtonHeight, hToolBarBitmap, ToolBarImageWidth, ToolBarImageHeight, ToolBarButtons(ButtonIndex).Index, state)
End Sub

Function ToolBarGetButtonState (ButtonID As Integer) As Integer
' Returns the current state of a toolbar button
'
' Parameters:
'   ButtonID     - ID of the toolbar button
'   return value - the state of the toolbar button
    
    Dim ButtonIndex As Integer

    For ButtonIndex = 1 To ToolBarNumButtons
	If ToolBarButtons(ButtonIndex).ID = ButtonID Then
	    Exit For
	End If
    Next ButtonIndex

    ToolBarGetButtonState = ToolBarButtons(ButtonIndex).state
End Function

Sub ToolBarInit (ToolbarControl As PictureBox, ButtonImages As Image, ImageWidth As Integer, ImageHeight As Integer, NumButtons As Integer, ButtonWidth As Integer, ButtonHeight As Integer)
' Initializes the toolbar.
' Call this routine in form load event of the form that contains the toolbar
'
' Parameters:
'   ToolbarControl - PictureBox control that is the toolbar
'   ButtonImages   - Image control that holds the bitmap images of the toolbar buttons
'   ImageWidth     - Width of each image in ButtonImages.  All images must be the same width.
'   ImageHeight    - Height of each image in ButtonImages.  All images must be the same Height.
'   NumButtons     - Number of toolbar buttons that will be displayed in the toolbar
'   ButtonWidth    - Width of each button.  All buttons must be the same width.
'   ButtonHeight   - Height of each button.  All buttons must be the same Height.

    Dim OldScaleMode As Integer

    ' Initialize ToolBar data
    ReDim ToolBarButtons(1 To NumButtons)
    Set Toolbar = ToolbarControl
    hToolBarBitmap = ButtonImages.Picture
    ToolBarImageWidth = ImageWidth
    ToolBarImageHeight = ImageHeight
    ToolBarNumButtons = NumButtons
    ToolBarButtonWidth = ButtonWidth
    ToolBarButtonHeight = ButtonHeight
    ToolBarButtonIndex = 0

    ' Initialize the Toolbar control
    Toolbar.Align = 1                       ' Align top
    Toolbar.AutoRedraw = False              ' Draw directly to window
    Toolbar.BackColor = RGB(192, 192, 192)  ' Light gray
    Toolbar.BorderStyle = 0                 ' No border
    Toolbar.ScaleMode = 3                   ' Pixels
    
    ' Set height of Toolbar based upon height of the buttons.
    ' Since our dimensions are in pixels,  we need to temporarly set the
    '   scalemode of the toolbar's parent form to pixels.
    OldScaleMode = Toolbar.Parent.ScaleMode
    Toolbar.Parent.ScaleMode = 3
    Toolbar.Height = ButtonHeight + TOOLBAR_BORDER * 2 + 3
    Toolbar.Parent.ScaleMode = OldScaleMode
End Sub

Sub ToolBarInitButton (ButtonID As Integer, x As Integer, BitmapIndex As Integer, bAttrButton As Integer)
' Initializes a single toolbar button.
' Call this routine in form load event of the form that contains the toolbar.
' ToolbarInit must be called before calling this sub.
'
' Parameters:
'   ButtonID    - ID of the toolbar button.  This ID is returned by ToolBarDepressButton when the button is pressed.
'                 Value must be non-zero.  Zero is reserved to indicate that no button was pressed.
'   x           - x coordinate of the button, in pixels, relative to the toolbar control
'   BitmapIndex - Index into ButtonImages (see ToolbarInit) for this button's image
'   bAttrButton - Set to True if this is an attribute button, False if this is a command button.
    
    ToolBarButtonIndex = ToolBarButtonIndex + 1
    If ToolBarButtonIndex > ToolBarNumButtons Then
	Exit Sub
    End If

    ' Initialize button data
    ToolBarButtons(ToolBarButtonIndex).ID = ButtonID
    ToolBarButtons(ToolBarButtonIndex).Index = BitmapIndex
    ToolBarButtons(ToolBarButtonIndex).bAttributeButton = bAttrButton
    
    ' Set initial state of button
    If bAttrButton Then
	ToolBarButtons(ToolBarButtonIndex).state = ATTRIBUTEBUTTON_UP
    Else
	ToolBarButtons(ToolBarButtonIndex).state = COMMANDBUTTON_UP
    End If
    
    ' Save button location relative to toolbar
    ToolBarButtons(ToolBarButtonIndex).rc.left = x
    ToolBarButtons(ToolBarButtonIndex).rc.top = TOOLBAR_BORDER + 1
    ToolBarButtons(ToolBarButtonIndex).rc.right = x + ToolBarButtonWidth
    ToolBarButtons(ToolBarButtonIndex).rc.bottom = TOOLBAR_BORDER + 1 + ToolBarButtonHeight
End Sub

Sub ToolBarPaint ()
' Paints all the toolbar buttons and also adds 3-D affect to toolbar
' Call this routine from the toolbar paint event.
    
    Dim i As Integer
    
    ' Give the toolbar a 3-D raised look (assumes toolbar.scalemode = 3 (pixels))
    Toolbar.Line (0, 0)-(Toolbar.ScaleWidth, 0), RGB(255, 255, 255)
    Toolbar.Line (0, Toolbar.ScaleHeight - 2)-(Toolbar.ScaleWidth, Toolbar.ScaleHeight - 2), RGB(128, 128, 128)
    Toolbar.Line (0, Toolbar.ScaleHeight - 1)-(Toolbar.ScaleWidth, Toolbar.ScaleHeight - 1), RGB(0, 0, 0)
    
    ' Draw all the buttons
    For i = 1 To ToolBarNumButtons
	Call ToolBarDrawButton(i, ToolBarButtons(i).state)
    Next i
End Sub

Sub ToolBarSetButtonState (ButtonID As Integer, state As Integer)
' Sets the current state of a toolbar button
'
' Parameters:
'   ButtonID  - ID of the toolbar button
'   state     - the new state of the toolbar button
    
    Dim ButtonIndex As Integer
    Dim i As Integer

    ButtonIndex = 0
    For i = 1 To ToolBarNumButtons
	If ToolBarButtons(i).ID = ButtonID Then
	    ButtonIndex = i
	    Exit For
	End If
    Next i

    If ButtonIndex > 0 Then
	ToolBarButtons(ButtonIndex).state = state
	Call ToolBarDrawButton(ButtonIndex, state)
    End If
End Sub

