
/*-
 * Copyright (c) 1995 The Apache Group. All rights reserved.
 * 
 *
 * Apache httpd license
 * ====================
 * 
 *
 * This is the license for the Apache Server. It covers all the
 * files which come in this distribution, and should never be removed.
 * 
 * The "Apache Group" has based this server, called "Apache", on
 * public domain code distributed under the name "NCSA httpd 1.3".
 * 
 * NCSA httpd 1.3 was placed in the public domain by the National Center 
 * for Supercomputing Applications at the University of Illinois 
 * at Urbana-Champaign.
 * 
 * As requested by NCSA we acknowledge,
 * 
 *  "Portions developed at the National Center for Supercomputing
 *   Applications at the University of Illinois at Urbana-Champaign."
 *
 * Copyright on the sections of code added by the "Apache Group" belong
 * to the "Apache Group" and/or the original authors. The "Apache Group" and
 * authors hereby grant permission for their code, along with the
 * public domain NCSA code, to be distributed under the "Apache" name.
 * 
 * Reuse of "Apache Group" code outside of the Apache distribution should
 * be acknowledged with the following quoted text, to be included with any new
 * work;
 * 
 * "Portions developed by the "Apache Group", taken with permission 
 *  from the Apache Server   http://www.apache.org/apache/   "
 *
 *
 * Permission is hereby granted to anyone to redistribute Apache under
 * the "Apache" name. We do not grant permission for the resale of Apache, but
 * we do grant permission for vendors to bundle Apache free with other software,
 * or to charge a reasonable price for redistribution, provided it is made
 * clear that Apache is free. Permission is also granted for vendors to 
 * sell support for for Apache. We explicitly forbid the redistribution of 
 * Apache under any other name.
 * 
 * The "Apache Group" makes no promises to support "Apache". Users and
 * sellers of Apache support, and users of "Apache Group" code, should be 
 * aware that they use "Apache" or portions of the "Apache Group" code at 
 * their own risk. While every effort has been made to ensure that "Apache"
 * is robust and safe to use, we will not accept responsibility for any
 * damage caused, or loss of data or income which results from its use.
 * 
 */



/*
 * http_log.c: Dealing with the logs and errors
 * 
 * Rob McCool
 * 
 */


#include "httpd.h"

#ifdef VIRTUAL_HOST
FILE *nonvirtual_error_log = NULL;
static int nonvirtual_xfer_log = -1;
#endif /* VIRTUAL_HOST */
FILE *error_log;
static int xfer_log;
static int xfer_flags = ( O_WRONLY | O_APPEND | O_CREAT );
static mode_t xfer_mode = ( S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH );

  /* Apache. Define two new strings to hold the URL and args of the incoming
      request. We need these 'cos they will be lost if we redirect in the
      event of an error/problem which results in a call to die().
      
      Also define a new int to hold the original status value and
      a small string to holf the method (it'll be a valid HTTP method)
  */
char original_url[HUGE_STRING_LEN];
char original_args[HUGE_STRING_LEN];
char original_method[20];
int original_status;
char auth_string[MAX_STRING_LEN];

#ifdef VIRTUAL_HOST
/*
 * switch logs for the virtual host
 */
 
void use_virtual_logs(virtual_host_data *vhd)
{
	/* if a different error log, use it */
	if (vhd->error_fname != NULL) {
	    /* if we have not opened the log, do so */
	    if (vhd->error_log == NULL)
		open_virtual_logs(vhd);
	    error_log = vhd->error_log;
	 }

	/* if a different xfer log, use it */
	if (vhd->xfer_fname != NULL) {
	    /* if we have not opened the log, do so */
	    if (vhd->xfer_log == -1)
		open_virtual_logs(vhd);
	    xfer_log = vhd->xfer_log;
	}
}

/* open logs for a virtual host */
void open_virtual_logs(virtual_host_data *vhd)
{
    if(!(vhd->error_log = fopen(vhd->error_fname,"a"))) {
        fprintf(stderr,"httpd: could not open virtual error log file %s.\n",
                vhd->error_fname);
        perror("fopen");
        exit(1);
    }
    if((vhd->xfer_log = open(vhd->xfer_fname,xfer_flags,xfer_mode)) < 0) {
        fprintf(stderr,"httpd: could not open vitrtual transfer log file %s.\n",
                vhd->xfer_fname);
        perror("open");
        exit(1);
    }
    if (fcntl(vhd->xfer_log, F_SETFD, (int)FD_CLOEXEC) == -1)
    {
	perror("fcntl");
	fprintf(stderr, "httpd: could not set close-on-exec on transfer log file.\n", vhd->xfer_fname);
	exit(1);
    }
}
#endif /* VIRTUAL_HOST */

void open_logs() {
    if(!(error_log = fopen(error_fname,"a"))) {
        fprintf(stderr,"httpd: could not open error log file %s.\n",
                error_fname);
        perror("fopen");
        exit(1);
    }
    if((xfer_log = open(xfer_fname,xfer_flags,xfer_mode)) < 0) {
        fprintf(stderr,"httpd: could not open transfer log file %s.\n",
                xfer_fname);
        perror("open");
        exit(1);
    }
    if (fcntl(xfer_log, F_SETFD, (int)FD_CLOEXEC) == -1)
    {
	perror("fcntl");
	fprintf(stderr, "httpd: could not set close-on-exec on transfer log file.\n", xfer_fname);
	exit(1);
    }

#ifdef VIRTUAL_HOST
    nonvirtual_xfer_log = xfer_log;
    nonvirtual_error_log = error_log;
#endif /* VIRTUAL_HOST */
}

#ifdef VIRTUAL_HOST
void close_virtual_logs()
{
    int x;

    for (x=0; x<num_virt && x<MAX_VIRTUAL_HOSTS; x++) {
	if (virt[x].error_log != NULL) {
	    fclose(virt[x].error_log);
	    virt[x].error_log = NULL;
	}
	if (virt[x].xfer_log > -1) {
	    close(virt[x].xfer_log);
	    virt[x].xfer_log = -1;
	}
    }
}
#endif /* VIRTUAL_HOST */

void close_logs() {
    close(xfer_log);
    fclose(error_log);
}


void error_log2stderr() {
    if(fileno(error_log) != STDERR_FILENO)
        dup2(fileno(error_log),STDERR_FILENO);
}

void log_pid() {
    FILE *pid_file;

    if(!(pid_file = fopen(pid_fname,"w"))) {
        fprintf(stderr,"httpd: could not log pid to file %s\n",pid_fname);
        exit(1);
    }
    fprintf(pid_file,"%ld\n",(long)getpid());
    fclose(pid_file);
}

  /* the_request shouldn't be static any more, we want to see it elsewhere */
char the_request[HUGE_STRING_LEN];
int status;
int bytes_sent;

void record_request(char *cmd_line) {
    status = -1;
    bytes_sent = -1;

    strcpy(the_request,cmd_line);
}

#ifdef LOGUSER
	/* B00042
	 * Keep a copy of the first 'user' id provided for .htaccess
	 */
	extern char log_user[MAX_STRING_LEN];
#endif

   /* Apache. Add new argument to allow the log file to stay open */
void log_transaction(int closeit) {
    char str[HUGE_STRING_LEN];
    long timz;
    struct tm *t;
    char tstr[MAX_STRING_LEN],sign;

    t = get_gmtoff(&timz);
    sign = (timz < 0 ? '-' : '+');
    if(timz < 0) 
        timz = -timz;

    strftime(tstr,MAX_STRING_LEN,"%d/%b/%Y:%H:%M:%S",t);

    sprintf(str,"%s %s %s [%s %c%02ld%02ld] \"%s\" ",
            remote_name,
            (do_rfc931 ? remote_logname : "-"),
#ifdef LOGUSER
 	/* B00042
	 * Keep a copy of the first 'user' id provided for .htaccess
	 */
           (log_user[0] ? log_user : "-"),
#else
            (user[0] ? user : "-"),
#endif
            tstr,
            sign,
            timz/3600,
            timz%3600,
            the_request);
    if(status != -1)
        sprintf(str,"%s%d ",str,status);
    else
        strcat(str,"- ");

    if(bytes_sent != -1)
        sprintf(str,"%s%d\n",str,bytes_sent);
    else
        strcat(str,"-\n");

    write(xfer_log,str,strlen(str));
    if (closeit) close(xfer_log);    /* Only close if close wanted */
}

void log_error(char *err) {
    fprintf(error_log, "[%s] %s\n",get_time(),err);
    fclose(error_log);
}

void log_error_noclose(char *err) {
    fprintf(error_log, "[%s] %s\n",get_time(),err);
    fflush(error_log);
}

void log_reason(char *reason, char *file) {
    char t[MAX_STRING_LEN];

    sprintf(t,"httpd: access to %s failed for %s, reason: %s",
            file,remote_name,reason);
    log_error(t);
}

/* Apache. When custom error response is for Authorization, always send the
    right status */
void begin_http_header(FILE *fd, char *msg) {
    if (original_status == AUTH_REQUIRED) {
        fprintf(fd,"%s %d %s\015\012",SERVER_PROTOCOL,AUTH_REQUIRED, "Unauthorized");
	fprintf(fd,"WWW-Authenticate: %s\015\012",auth_string);
	status = AUTH_REQUIRED;
    } else
        fprintf(fd,"%s %s\015\012",SERVER_PROTOCOL,msg);
       
    dump_default_header(fd);
}

void error_head(FILE *fd, char *err) {
    if(!assbackwards) {
        begin_http_header(fd,err);
        fprintf(fd,"Content-type: text/html\015\012\015\012");
    }
    if(!header_only) {
        fprintf(fd,"<HEAD><TITLE>%s</TITLE></HEAD>%c",err,LF);
        fprintf(fd,"<BODY><H1>%s</H1>%c",err,LF);
    }
}

void title_html(FILE *fd, char *msg) {
    fprintf(fd,"<HEAD><TITLE>%s</TITLE></HEAD>%c",msg,LF);
    fprintf(fd,"<BODY><H1>%s</H1>%c",msg,LF);
}

void die(int type, char *err_string, FILE *fd) {
    char t[MAX_STRING_LEN];
    int error_index; /* Apache. holds the array index of a response code */
    
      /* The following takes care of Apache redirects to custom response URLs */
    error_index = index_of_response(type);
    if (original_url[0] == '\0') {
       if (response_code_strings[error_index] != NULL) {

             /* Only text beginning with a quote (") should be given to
                  the user as-is. Everything else is assumed to be a URL -
                  whether it be local or external.
             */
          if ((char) *response_code_strings[error_index] != '"') {
          
              if (is_url(response_code_strings[error_index])) {
                 /* The URL isn't local, so lets drop through the rest of
                     this apache code, and continue with the usual REDIRECT
                     handler
                 */
                 type = REDIRECT;
                    /* The external URL is expected in "err_string"  */
                 err_string = response_code_strings[error_index];
                 
              } else if ( *response_code_strings[error_index] == '/') {
              
                      /* All local redirect should start with a '/' */

                   status = type;        /* Log the current URL */
                   log_transaction(0);   
                               
                     /* flag the fact that we are redirecting from this URL */
                     /*  - not very elegant, but it save 1 global variable  */ 
                   original_url[0] = ' ';
 
                     /* Save existing info prior to the redirect */
                   prepare_for_redirect(fd, type, response_code_strings[error_index]);
                   
                     /* if we are redirecting from a server error, log it */
                   if (type == SERVER_ERROR) log_error_noclose(err_string);

                     /* remember the AUTH_REQ string */
                   if (type == AUTH_REQUIRED) strcpy(auth_string, err_string);
                     
     
                    /* redirect to the new URL, no args this time. */
                   process_get(0,fd,"GET",response_code_strings[error_index],"");
                     /* if by some chance we return here, it's time to die */

                   fflush(fd);
                   htexit(0,fd); 
              } else {
                     /* dumb user has given us a bad url to redirect to */
                   type = SERVER_ERROR;
                   err_string = "Invalid attempt to redirect.";
                   log_error(response_code_strings[error_index]);
             }       
          } 
       }
    }   

    switch(type) {
      case REDIRECT:
        status = 302;
        if(!assbackwards) {
            begin_http_header(fd,"302 Found");
            fprintf(fd,"Location: %s\015\012",err_string);
            fprintf(fd,"Content-type: text/html\015\012\015\012");
        }
        if(header_only) break;
           /* Display our custom message if we have one */
        if (response_code_strings[error_index] != NULL) {
           fprintf(fd,response_code_strings[error_index]+1,err_string);
           break;
        }
        title_html(fd,"Document moved");
        fprintf(fd,"This document has moved <A HREF=\"%s\">here</A>.<P>%c",
                err_string,LF);
        break;
      case USE_LOCAL_COPY:
        status = USE_LOCAL_COPY;
        begin_http_header(fd,"304 Not modified\015\012");
        header_only = 1;
        break;
      case AUTH_REQUIRED:
        status = 401;
        if(!assbackwards) {
            begin_http_header(fd,"401 Unauthorized");
            fprintf(fd,"Content-type: text/html\015\012");
            fprintf(fd,"WWW-Authenticate: %s\015\012\015\012",err_string);
        }
        if(header_only) break;
           /* Display our custom message if we have one */
        if (response_code_strings[error_index] != NULL) {
           fprintf(fd,response_code_strings[error_index]+1,err_string);
           break;
        }        
        title_html(fd,"Authorization Required");
        fprintf(fd,"Browser not authentication-capable or %c",LF);
        fprintf(fd,"authentication failed.%c",LF);
        break;
      case BAD_REQUEST:
        status = 400;
        error_head(fd,"400 Bad Request");
        if(header_only) break;
           /* Display our custom message if we have one */
        if (response_code_strings[error_index] != NULL) {
           fprintf(fd,response_code_strings[error_index]+1,err_string);
           break;
        }    
        fprintf(fd,"Your client sent a query that this server could not%c",LF);
        fprintf(fd,"understand.<P>%c",LF);
        fprintf(fd,"Reason: %s<P>%c",err_string,LF);
        break;
      case FORBIDDEN:
        status = 403;
        error_head(fd,"403 Forbidden");
        if(header_only) break;
           /* Display our custom message if we have one */
        if (response_code_strings[error_index] != NULL) {
           fprintf(fd,response_code_strings[error_index]+1,err_string);
           break;
        }  
        fprintf(fd,"Your client does not have permission to get URL %s ",
                err_string);
        fprintf(fd,"from this server.<P>%c",LF);
        break;
      case NOT_FOUND:
        status = 404;
        error_head(fd,"404 Not Found");
        if(header_only) break;
           /* Display our custom message if we have one */
        if (response_code_strings[error_index] != NULL) {
              /* .. but don't display a redirected URL that has failed */
           if (original_url[0] && *response_code_strings[error_index] == '/') 
              fprintf(fd,"Tried to redirect to %s after some earlier problem with %s, but %s didn't exist\n",response_code_strings[error_index],original_url+1,response_code_strings[error_index]);
           else
              fprintf(fd,response_code_strings[error_index]+1,err_string);
           break;
        }  
        fprintf(fd,"The requested URL %s was not found on this server.<P>%c",
                err_string,LF);
        break;
      case SERVER_ERROR:
        status = 500;
        error_head(fd,"500 Server Error");
        log_error(err_string);
        if(header_only) 
            break;
           /* Display our custom message if we have one */
        if (response_code_strings[error_index] != NULL) {
           fprintf(fd,response_code_strings[error_index]+1,err_string);
           break;
        }
        fprintf(fd,"The server encountered an internal error or%c",LF);
        fprintf(fd,"misconfiguration and was unable to complete your%c",LF);
        fprintf(fd,"request.<P>%c",LF);
        fprintf(fd,"Please contact the server administrator,%c",LF);
        fprintf(fd," %s ",server_admin);
        fprintf(fd,"and inform them of the time the error occurred, and%c",LF);
        fprintf(fd,"anything you might have done that may have caused%c",LF);
        fprintf(fd,"the error.<P>%c",LF);
        break;
      case NOT_IMPLEMENTED:
        status = 501;
        error_head(fd,"501 Not Implemented");
        if(header_only) break;
           /* Display our custom message if we have one */
        if (response_code_strings[error_index] != NULL) {
           fprintf(fd,response_code_strings[error_index],err_string);
           break;
        }
        fprintf(fd,"We are sorry to be unable to perform the method %s",
                err_string);
        fprintf(fd," at this time.<P>%c",LF);
        fprintf(fd,"If you would like to see this capability in future%c",LF);
        fprintf(fd,"releases, send the method which failed, why you%c",LF);
        fprintf(fd,"would like to have it, and the server version %s%c",
                SERVER_VERSION,LF);
        fprintf(fd,"to <ADDRESS>%s</ADDRESS><P>%c",SERVER_SUPPORT,LF);
        break;
      case NO_MEMORY:
        log_error("httpd: memory exhausted");
        status = 500;
        error_head(fd,"500 Server Error");
        if(header_only) break;
        fprintf(fd,"The server has temporarily run out of resources for%c",LF);
        fprintf(fd,"your request. Please try again at a later time.<P>%c",LF);
        break;
    }
    if(!header_only)
        fprintf(fd,"</BODY>%c",LF);
    fflush(fd);
    log_transaction(1);
    htexit(1,fd);
}
