#ifndef TEMPLATE_MAP_H
#define TEMPLATE_MAP_H

#include <assert.h>
#include <iostream.h>


//-----------------------------------------------------------------------
//-----------------------------------------------------------------------
//     MAP TEMPLATE
//-----------------------------------------------------------------------
//-----------------------------------------------------------------------

template <class K, class V> class MapOfIter;
template <class K, class V> class MapOf {
	friend class MapOfIter<K,V>;
	public:
		MapOf() : theKeys(0), theValues(0), limit(0), idx(0) {}
		MapOf(const K& key, const V& value) : limit(0), idx(0) {
			enter(key,value);
		}
		virtual ~MapOf(){ clear(); }
		MapOf(const MapOf<K,V>& map);
		MapOf<K,V>& operator=(const MapOf<K,V>& map);
		MapOf<K,V>& operator+=(const MapOf<K,V>& map);
		void enter(const K& key, const V& value, int replace=1);
		int contains(const K& key) const;
		int size() const { return idx; }
		void flush() { idx = 0; }
		int empty() const { return idx == 0; }
		V valueOf(const K& key) const;
		V valueOfFirst() const { return theValues[0]; }
		V valueOfLast() const { return theValues[idx-1]; }
		void dropLast() { if( idx > 0 ) --idx; }	
		void clear() {
		  delete [] theKeys;
			theKeys = 0;
			delete [] theValues;
			theValues = 0;
		  idx = 0;
		  limit = 0;
		}
		friend ostream& operator << (ostream& os, const MapOf<K,V>& map);
      const K* keys() const { return theKeys; }
	private:
		K *theKeys;
		V *theValues;
		int limit, idx, locale;
		// locale is for replacing value whose key already exists
};


template<class K, class V>
V MapOf<K,V>::valueOf(const K& k) const
{
	for(int i=0; i<idx; i++) {
	   if(theKeys[i]==k) return theValues[i];
	}
	assert(!"Map did not contain the requested key->value pair");
   return theValues[0];
}


template<class K, class V>
MapOf<K,V>::MapOf(const MapOf<K,V>& map) : limit(map.limit), idx(map.idx),
locale(map.locale), theKeys(0), theValues(0)
{
	theKeys = new K [limit];
	assert(theKeys!=0);
	theValues = new V [limit];
	assert(theValues!=0);
	for(int i=0; i<idx; i++) {
		theKeys[i] = map.theKeys[i];
		theValues[i] = map.theValues[i];
	}
}

template<class K, class V>
MapOf<K,V>& MapOf<K,V>::operator=(const MapOf<K,V>& map)
{
   limit = map.limit;
   idx = map.idx;
	locale = map.locale;

	delete [] theKeys;
	theKeys = 0;
	theKeys = new K [limit];
	assert(theKeys!=0);

	delete [] theValues;
	theValues = 0;
	theValues = new V [limit];
	assert(theValues!=0);

	for(int i=0; i<idx; i++) {
		theKeys[i] = map.theKeys[i];
		theValues[i] = map.theValues[i];
	}
	return *this;
}

template<class K, class V>
MapOf<K,V>& MapOf<K,V>::operator+=(const MapOf<K,V>& map)
{
	for(int i=0; i<map.idx; i++) {
		K tmpKey = map.theKeys[i];
		V tmpValue = map.theValues[i];
      this->enter(tmpKey,tmpValue,0);
	}
	return *this;
}

template<class K, class V>
ostream& operator << (ostream& os, const MapOf<K,V>& map)
{
	int j=1;	
	os << "[\n";
   for(int i=0; i<map.idx; i++) {
      os << map.theKeys[i] << "->" << map.theValues[i];
		if(j++ < map.idx) os << endl;
	}
	os << "\n]";
   return os;
}


template<class K, class V>
void MapOf<K,V>::enter(const K& key, const V& value, int replace)
{
	if(contains(key)){
   	if(!replace) return;
		theKeys[locale] = key;
		theValues[locale] = value;
		return;
	}
	if(idx==limit){
		limit+=50;
		K *newK = new K [limit];
		assert(newK!=0);
		V *newV = new V [limit];
		assert(newV != 0);
		for(int i=0; i<idx; i++){
			newK[i] = theKeys[i];
			newV[i] = theValues[i];
		}
		delete [] theKeys;
		delete [] theValues;
		theKeys = newK;
		theValues = newV;
	}
   theKeys[idx] = key;
   theValues[idx] = value;
   idx++;
}

template<class K, class V>
int MapOf<K,V>::contains(const K& k) const
{
   for(int i=0; i<idx; i++){
		if(theKeys[i]==k){
         MapOf<K,V> *ptr = (MapOf<K,V>*) this;
			ptr->locale = i;
			return 1;
		}
	}
   return 0;
}


//-----------------------------------------------------------------------
//-----------------------------------------------------------------------
//     MAP ITERATOR TEMPLATE
//-----------------------------------------------------------------------
//-----------------------------------------------------------------------

template <class K, class V> class MapOfIter {
   public:
      MapOfIter() : Kptr(0), Vptr(0), map(0), position(0), ptr(0) {}

      MapOfIter(const MapOf<K,V>* m) :
			Kptr(m->theKeys),	 Vptr(m->theValues), map(m), position(0) {
				ptr=&position;
				if(m->limit==0) ptr=0;
			}

      void rewind() {
         assert(Kptr!=0);
         assert(Vptr!=0);
         ptr=&position;
         position=0;
      }

      void goEnd() {
         assert(Kptr!=0);
         assert(Vptr!=0);
         ptr=&position;
         position=map->idx-1;
      }

		int last() const { return position==map->size()-1; }
      int offEnd() const {
         if(position<0 || position>=map->idx) return 1;
         return 0;
      }
      operator void* () { return ptr; }
      K& key();
      V& value();
      void next() { operator++(); }
      void prev() { operator--(); }
	   virtual void operator++() { position++; if(offEnd()) ptr=0;}
      virtual void operator--() { position--; if(offEnd()) ptr=0;}
      ~MapOfIter() { map = 0; Kptr = 0; Vptr = 0; ptr = 0; }
   private:
      const MapOf<K,V> *map;
      K *Kptr;
      V *Vptr;
      int *ptr;
   protected:
      int position;
};

template<class K, class V>
K& MapOfIter<K,V>::key() {
   assert(!offEnd());
   assert(Kptr!=0);
	return Kptr[position];
}

template<class K, class V>
V& MapOfIter<K,V>::value() {
   assert(!offEnd());
   assert(Vptr!=0);
	return Vptr[position];
}

#endif // TEMPLATE_MAP_H
