/* 
ITERATOR.C -- Tim Coe

---  failing operations found by iterator.c  ---

---  Nicely's Prime  ---
8 bfffffb829  decimal=824633702441
8 a7fff6d6128
8 8fffe35e151

800bf6 bffffc
a00ef6 effffc

8 bffffb82200-bffffb831f3
80000035f8 bfffee5dc

a808d2 8fffe
e00bd2 bfffe

a7ffd2 8fffe
c3ffd2 a7ffe
dfffd2 bfffe
fbffd2 d7ffe

f9ffdc7 efffe

b9ffab0e-b9ffab7f 8fffc

c3ffd2eb0d2eb0d2 a7ffe
e00bd229315 bfffe

9fffef5-9fffeff effff4
9ffff21-9ffff3f effff8
9ffff4d-9ffff7f effffc

f008e35-f008e3f 8ffff4
f008e6d-f008e7f 8ffff6
f008ea1-f008ebf 8ffff8
f008ed9-f008eff 8ffffa
f008f0d-f008f3f 8ffffc
f008f45-f008f7f 8ffffe
f008f7e 8ffffff1
f0023e 8fffff8

effff0d 8ffffc

9fffec effff88

a808d1b-a808d3f 8fffe
a808d67-a808d7f 8fffe4
a808db3-a808dbf 8fffe8
a808dff 8fffec

---  six 1 cases  ---
9efa99444688 efe00ff8036
9efa994446f1 efe00ff803fe
9efac64141c efe00ff81f
9faeb98c604 efe00ff81f8
9faeb98c608 efe00ff81fc
8f12c7c141c d7e00ff81f
9efac73141c efe00ff81f

---  eight 1 cases  ---
bfef557bf8-bfef557dff 8ff8
bfdd55efc-bfdd55eff 8ff8
c02aa6c 8ffbffc
c02ce6c 8ffbffc
80000022ab8a0f bffbf03ffffffff

---  nine 1 cases  ---
9f0d60f effc0f8
bffafe7c 8ffc

---  ten 1 cases  ---
9fff52e effeff8
9fff5218 effefbf
9fff3ff76 effedfffc
9fff3ff32e effedfeff8
9fff3ff3f776 effedfedfffc
9fff3ff3fd32e effedfedfeff8
9fff3ff3ff37776 effedfedfedfffc

---  eleven 1 cases  ---
b9feab7-b9feabf 8fff
be7eaade-be7eab7f 8fff

---  end failing operations  ---
*/

#include <stdio.h>

main()
{
    /*
     * Documentation and variable names were provided
     * by David Goldberg of Xerox PARC
     *
     * SRT division sets r(0) = dividend and iterates
     *
     *   r(i+1) = 4*[r(i) - q(i)*divisor]
     *
     * where q(i) is the i-th quotient digit, -2 <= q(i) <= 2.  The
     * proper value of q(i) is obtained from a lookup table indexed by
     * the top 5 bits of divisor and the top 7 bits of r(i) (including
     * the sign bit).  The value of q(i) in the lookup table ensures
     * that |r(i+1)| <= (8/3)*divisor.  So if dividend, divisor are
     * between 1.0 and 2.0, r(i) ranges between 2*(8/3) =
     * 0101.01010... and 2*(8/3) = 1010.10101...  Thus the index into
     * the lookup table [top 7 bits of r(i)] are 4 bits left of binary
     * point, 3 bits right of it.
     *
     * Double precision significands have 53 bits. They are stored in
     * a pair of unsigned variables, xxx_hi, xxx_lo.  Assuming IEEE
     * normalization with dividend, divisor between 1.0 and 2.0, then
     * the binary point is between bits 27 and 26 of xxx_hi:
     *
     *  |----------    xxx_hi     -------------|    |-- xxx_lo
     *  31 30 29 28   27 26 25 24  ....  3 2 1 0    31 30 29 28 ...
     *                  ^
     *              binary point
     *
     * Thus the initial dividend and divisor are stored in the least
     * significant 28 bits of xxx_hi, and the most significant 25
     * bits of xxx_lo.
     */

    unsigned div_hi, div_lo;          /* divisor */
    unsigned rem_sum_hi, rem_sum_lo;  /* remainder: sum word */
    unsigned rem_car_hi, rem_car_lo;  /* remainder: carry word */
    unsigned qd_hi, qd_lo;            /* -q*div, quotient times divsor */
    unsigned rem_approx;              /* chopped remainder sum, to select digit
*/

    unsigned t0, t1, t2, t3, cycle, f, incorrect;
    unsigned thr_m2_m1, thr_m1_0, thr_0_1, thr_1_2, positive, bad_remainder;
    char line[30], *linepoint;

    /*  The following section is the input routine.  */
    /*  Entering q's gives the popular magic pair.   */
    div_hi = 0x0bffffc0;
    div_lo = 0;
    rem_sum_hi = 0x0800bf60;
    rem_sum_lo = 0;
    printf("Most significant digits must be between 8 and f inclusive\n");
    printf("Enter dividend mantissa in hex: ");

    /*
     * Read dividend into char array line[], pad with zeros on the
     * right, and then since the dividend is also the initial value
     * of remainder, store into rem_sum.  The first 7 bytes (28 bits)
     * are stored into rem_sum_hi, the next 8 bytes into rem_sum_lo.
     */
    *(line+15) = '0';
    scanf("%s", line);
    linepoint = line;
    while (*linepoint != '\0') linepoint++;
    while (linepoint < line + 15) *linepoint++ = '0';
    *(line+15) = '\0';
    sscanf(line+7, "%x", &rem_sum_lo);
    *(line+7) = '\0';
    sscanf(line, "%x", &rem_sum_hi);

    /* do the same for divisor */
    printf("Enter divisor  mantissa in hex: ");
    scanf("%s", line);
    linepoint = line;
    while (*linepoint != '\0') linepoint++;
    while (linepoint < line + 15) *linepoint++ = '0';
    *(line+15) = '\0';
    sscanf(line+7, "%x", &div_lo);
    *(line+7) = '\0';
    sscanf(line, "%x", &div_hi);
    rem_car_hi = 0;
    rem_car_lo = 0;

    /*  I print out the high portion of the operands as integers.  */
    t0 = rem_sum_hi;
    while (!(t0 & 1)) t0 = t0 >> 1;
    printf("%d\n", t0);
    t0 = div_hi;
    while (!(t0 & 1)) t0 = t0 >> 1;
    printf("%d\n", t0);

    /*
     * Lookup table uses top 5 bits of divisor, top 7 bits of of
     * remainder.  Load top 5 bits of divisor into t0 (recall divisor
     * only occupies top 28 bits of div_hi), and use it to compute
     * (from lookup table) the range of remainder that corresponds to
     * each quotient digit.
     *
     * rem < thr_1_2   => q <= 1
     * rem < thr_0_1   => q <= 0
     * rem > thr_m1_0  => q >= 0
     * rem > thr_m2_m1 => q >= -1
     *
     * These numbers are 7 bit two's complement numbers, represented
     * as unsigned.  That is, -5 is represented as 128 - 5 = 123.
     */

    t0 = div_hi >> 23;

    /*  Next threshold is strongly indicated */
    /*  by the failure of 1/9895574626641    */
    if (t0 < 18) thr_0_1 = 3;
    /*  Next threshold is strongly indicated */
    /*  by the failure of 1/824633702441     */
    else if (t0 < 24) thr_0_1 = 4;
    /*  Next threshold is strongly indicated */
    /*  by the failure of 5244795/3932159    */
    else if (t0 < 30) thr_0_1 = 5;
    else thr_0_1 = 6;
    /* thr_m1_0 = -thr_0_1 - 2 in two's complement */
    thr_m1_0 = 126 - thr_0_1;

    /*  Next threshold is strongly indicated     */
    /*  by the failure of 41.999999/35.9999999   */
    if (t0 < 18) thr_1_2 = 12;
    /*  Next threshold is strongly indicated     */
    /*  by the failure of 1/1443107810341 and    */
    /*  by the failure of 48.999999/41.9999999   */
    else if (t0 < 21) thr_1_2 = 14;
    /*  Next threshold is strongly indicated     */
    /*  by the failure of 55.999999/47.9999999   */
    else if (t0 < 24) thr_1_2 = 16;
    /*  Next threshold is strongly indicated     */
    /*  by the failure of 62.999999/53.9999999   */
    else if (t0 < 27) thr_1_2 = 18;
    /*  Next threshold is strongly indicated     */
    /*  by the failure of 54.999999/59.9999999   */
    else if (t0 < 30) thr_1_2 = 20;
    else thr_1_2 = 22;
    thr_m2_m1 = 126 - thr_1_2;

    /*
     * Treating the top 5 bits of div, and top 7 bits of rem as
     * integers, the bad entries in lookup table are (d=17,r=23),
     * (d=20,r=27), etc. Set bad_remainder = 64 if no possibility of
     * error.
     */
    if (t0 == 17) bad_remainder = 23;
    else if (t0 == 20) bad_remainder = 27;
    else if (t0 == 23) bad_remainder = 31;
    else if (t0 == 26) bad_remainder = 35;
    else if (t0 == 29) bad_remainder = 39;
    else bad_remainder = 64;
    /*
     ***  Original failure remainders follow.
     *
     * if (t0 == 17) bad_remainder = 22;
     * else if (t0 == 20) bad_remainder = 26;
     * else if (t0 == 23) bad_remainder = 30;
     * else if (t0 == 26) bad_remainder = 34;
     * else if (t0 == 29) bad_remainder = 38;
     * else bad_remainder = 64;
     */

    incorrect = 0;
    cycle = 1;
    /*  make loop counter large enough to catch the Nicely prime */
    while (cycle < 17) {

        /*
         * The following calculates the chopped sum of the sum and
         * carry words of the remainder, and performs the equivalent
         * of the remainder portion of the table lookup.  It then
         * calculates qd (-quotient_digit * divisor).
         *
         * The lookup table uses the top 7 bits of the SHIFTED remainder,
         * which will occupy the top 31 bits of rem_sum.  So the top
         * 7 bits live in the top byte (i.e. shift by 24).  This is
         * stored in rem_approx.
         */

        rem_approx = 0x7f & ((rem_sum_hi >> 24) + (rem_car_hi >> 24));
        if ((rem_approx > thr_m1_0) || (rem_approx < thr_0_1) ||
            /* The expression below catches the case when the
             * remainder has gone out of bounds [ > (8/3)*divisor ]
             * because the bug generated a bad quotient digit in a
             * previous cycle.
             */
           ((rem_approx > bad_remainder) && (rem_approx < 126 - bad_remainder))) {
            qd_hi = 0;
            qd_lo = 0;
            positive = 0;
            printf("next digit 0\n");
        }
        else if (rem_approx > thr_m2_m1) {
            qd_hi = div_hi;
            qd_lo = div_lo;
            positive = 0;
            printf("next digit -1\n");
        }
        else if (rem_approx < thr_1_2) {
            qd_hi = ~div_hi;
            qd_lo = ~div_lo;
            /* 'positive' will be added later to make qd a two's complement */
            positive = 4;
            printf("next digit 1\n");
        }
        else if (rem_approx & 0x40) {  /* rem_approx negative */
            qd_hi = (div_hi << 1) | (div_lo >> 31);
            qd_lo = div_lo << 1;
            positive = 0;
            printf("next digit -2\n");
        }
        else {
            qd_hi = ~((div_hi << 1) | (div_lo >> 31));
            qd_lo = ~(div_lo << 1);
            positive = 4;
            if (rem_approx == bad_remainder) {
                printf("A bug condition has been detected.\n");
                printf("Enter 0 for correct result or 1 for incorrect result: ");
                scanf("%d", &incorrect);
                if (incorrect) {
                    qd_hi = 0;
                    qd_lo = 0;
                    positive = 0;
                    printf("next digit 0\n");
                }
                else printf("next digit 2\n");
            }
            /*
             ***  The original error detection and generation went as follows.
             *
             * if ((rem_approx == bad_remainder) &&
             *     (((rem_sum_hi >> 21) & 7) == 7) && (((rem_car_hi >> 21) & 7)
{
             *   printf("A bug condition has been detected.\n");
             *   printf("Enter 0 for correct result or 1 for incorrect result: "
             *   scanf("%d", &incorrect);
             *   if (incorrect) {
             *     if (bad_remainder == 22) qd_hi = qd_hi - (3 << 25);
             *     else qd_hi = qd_hi - (4 << 25);
             *     }
             *   }
             */
            else printf("next digit 2\n");
        }

        /* Compute r = 4*(r - q*d) */

        /* sum bits of (-q*d + remainder sum + remainder carry) */
        t0 = qd_hi ^ rem_sum_hi ^ rem_car_hi;
        t1 = qd_lo ^ rem_sum_lo ^ rem_car_lo;

        /* carry bits of same: will be shifted into carry position below */
        t2 = (qd_hi & rem_sum_hi) | (qd_hi & rem_car_hi) |
                (rem_sum_hi & rem_car_hi);
        t3 = (qd_lo & rem_sum_lo) | (qd_lo & rem_car_lo) |
                (rem_sum_lo & rem_car_lo);

        /* new rem_sum: just take sum bits and multiply by 4 (<< 2) */
        rem_sum_hi = (t0 << 2) | (t1 >> 30);
        rem_sum_lo = t1 << 2;

        /*
         * new rem_carry: shift 1 because its a carry, 2 more
         * for multiplication by 4.  'positive' completes the
         * computation of the two's complement negation of qd = -q*d
         */
        rem_car_hi = (t2 << 3) | (t3 >> 29);
        rem_car_lo = (t3 << 3) | positive;

        /*  The following section outputs rem_sum, rem_car, and rem_approx in bi */
        t0 = rem_sum_hi;
        f = 32;
        while (f--) {
            if (t0 & (1 << 31)) putchar('1');
            else putchar('0');
            t0 = t0 << 1;
        }
        t0 = rem_sum_lo;
        f = 32;
        while (f--) {
            if (t0 & (1 << 31)) putchar('1');
            else putchar('0');
            t0 = t0 << 1;
        }
        putchar('\n');
        t0 = rem_car_hi;
        f = 32;
        while (f--) {
            if (t0 & (1 << 31)) putchar('1');
            else putchar('0');
            t0 = t0 << 1;
        }
        t0 = rem_car_lo;
        f = 32;
        while (f--) {
            if (t0 & (1 << 31)) putchar('1');
            else putchar('0');
            t0 = t0 << 1;
        }
        putchar('\n');
        putchar(' ');
        rem_approx = (127 & ((rem_sum_hi >> 24) + (rem_car_hi >> 24))) << 25;
        f = 7;
        while (f--) {
            if (rem_approx & (1 << 31)) putchar('1');
            else putchar('0');
            rem_approx <<= 1;
        }
        printf(" iteration number %d\n", cycle++);

    }
}

