// Socket
// socket.cpp
//
// access to WinSockets
//
//

#include <owl\owlpch.h>
#pragma hdrstop

// misc system includes

// our includes
#include "socket.hpp"

// constants
const unsigned int VERSION_REQUIRED = 0x101;  // winsock version we need
const unsigned int MAX_MESSAGE = 32767;       // max message size --old limit--
const int MAXHOSTNAMELEN = 256;               // max host name length
const int MAX_RETRIES = 5;                    // max retries
const int TIMEOUT_MS = 1000;                  // 1000 millisecs --1 sec--
const int BLOCK_TIMEOUT = 10;                 // in seconds
const int TIMER_ID = 1;                       // internal timer id
const int WORKBUFF_SIZE = 1024;               // work buffer size

// pop responses
const char OK_RESPONSE[] = "+OK";             // ok
const char ERR_RESPONSE[] = "-ERR";           // error
const char TERMINATION_OCTET[] = "\r\n.\r\n"; // message terminator

// values returned by MailResponse() , could be an enum
const int POP_NOTHING = 0;                    // nothing received
const int POP_OK = 1;                         // received an ok
const int POP_ERR = 2;                        // received an error
const int POP_LINE = 3;                       // response looked like a plain line

// POP commands
const char POP_QUIT[] = "QUIT";               // terminate POP session
const char POP_RETR[] = "RETR ";               // retrieve a message
const char POP_STAT[] = "STAT ";               // message stats for a user
const char POP_USER[] = "USER ";               // set user name
const char POP_PASS[] = "PASS ";               // set user password

// standard pop port --POP can live on other ports but is almost always here--
const int POP_PORT = 110;

// temp file name for message saves
const char TEMP_MAILNAME[] = "tempmail.";

// message id's
enum
{
  SM_HOSTNAME = WM_USER + 2100,
  SM_DATAREADY,
  SM_POPCONNECT,
  SM_POPAUTHORIZE,
  SM_POPPASSWORD,
  SM_POPMESSAGECOUNT,
  SM_POPGETMESSAGE,
  SM_POPSENDMESSAGE,
  SM_POPDELETEMESSAGE,
  SM_POPSTARTMESSAGE,
};

// Misc socket error classes, routines and constants
//
// 
enum // private socket errors
{
  SOCKERR_INVALID = 20000,
  SOCKERR_NOSOCKET
};


class SocketError
{
public:
  SocketError(int n,char* p) {nError = n; pMsg = p;};
  int nError;
  char* pMsg;
};

//const char UNKNOWN_ERROR[] = "Unknown error code";

SocketError SocketErrors[] =
{
  {SocketError(WSAEINTR,"WSAEINTR")},
  {SocketError(WSAEBADF,"WSAEBADF")},
  {SocketError(WSAEACCES,"WSAEACCES")},
  {SocketError(WSAEFAULT,"WSAEFAULT")},
  {SocketError(WSAEINVAL,"WSAEINVAL")},
  {SocketError(WSAEMFILE,"WSAEMFILE")},
  {SocketError(WSAEWOULDBLOCK,"WSAEWOULDBLOCK")},
  {SocketError(WSAEINPROGRESS,"WSAEINPROGRESS")},
  {SocketError(WSAEALREADY,"WSAEALREADY")},
  {SocketError(WSAENOTSOCK,"WSAENOTSOCK")},
  {SocketError(WSAEDESTADDRREQ,"WSAEDESTADDRREQ")},
  {SocketError(WSAEMSGSIZE,"WSAEMSGSIZE")},
  {SocketError(WSAEPROTOTYPE,"WSAEPROTOTYPE")},
  {SocketError(WSAENOPROTOOPT,"WSAENOPROTOOPT")},
  {SocketError(WSAEPROTONOSUPPORT,"WSAEPROTONOSUPPORT")},
  {SocketError(WSAESOCKTNOSUPPORT,"WSAESOCKTNOSUPPORT")},
  {SocketError(WSAEOPNOTSUPP,"WSAEOPNOTSUPP")},
  {SocketError(WSAEPFNOSUPPORT,"WSAEPFNOSUPPORT")},
  {SocketError(WSAEAFNOSUPPORT,"WSAEAFNOSUPPORT")},
  {SocketError(WSAEADDRINUSE,"WSAEADDRINUSE")},
	{SocketError(WSAEADDRNOTAVAIL,"WSAEADDRNOTAVAIL")},
  {SocketError(WSAENETDOWN,"WSAENETDOWN")},
  {SocketError(WSAENETUNREACH,"WSAENETUNREACH")},
  {SocketError(WSAENETRESET,"WSAENETRESET")},
  {SocketError(WSAECONNABORTED,"WSAECONNABORTED")},
  {SocketError(WSAECONNRESET,"WSAECONNRESET")},
  {SocketError(WSAENOBUFS,"WSAENOBUFS")},
  {SocketError(WSAEISCONN,"WSAEISCONN")},
  {SocketError(WSAENOTCONN,"WSAENOTCONN")},
  {SocketError(WSAESHUTDOWN,"WSAESHUTDOWN")},
  {SocketError(WSAETOOMANYREFS,"WSAETOOMANYREFS")},
  {SocketError(WSAETIMEDOUT,"WSAETIMEDOUT")},
  {SocketError(WSAECONNREFUSED,"WSAECONNREFUSED")},
  {SocketError(WSAELOOP,"WSAELOOP")},
  {SocketError(WSAENAMETOOLONG,"WSAENAMETOOLONG")},
  {SocketError(WSAEHOSTDOWN,"WSAEHOSTDOWN")},
  {SocketError(WSAEHOSTUNREACH,"WSAEHOSTUNREACH")},
  {SocketError(WSAENOTEMPTY,"WSAENOTEMPTY")},
  {SocketError(WSAEPROCLIM,"WSAEPROCLIM")},
  {SocketError(WSAEUSERS,"WSAEUSERS")},
	{SocketError(WSAEDQUOT,"WSAEDQUOT")},
  {SocketError(WSAESTALE,"WSAESTALE")},
  {SocketError(WSAEREMOTE,"WSAEREMOTE")},
  {SocketError(WSASYSNOTREADY,"WSASYSNOTREADY")},
  {SocketError(WSAVERNOTSUPPORTED,"WSAVERNOTSUPPORTED")},
  {SocketError(WSANOTINITIALISED,"WSANOTINITIALISED")},
  {SocketError(WSAHOST_NOT_FOUND,"WSAHOST_NOT_FOUND")},
  {SocketError(WSATRY_AGAIN,"WSATRY_AGAIN")},
  {SocketError(WSANO_RECOVERY,"WSANO_RECOVERY")},
  {SocketError(WSANO_DATA,"WSANO_DATA")},
  {SocketError(WSANO_ADDRESS,"WSANO_ADDRESS")},
  {SocketError(SOCKERR_INVALID,"Socket is invalid")},
  {SocketError(SOCKERR_NOSOCKET,"Socket creation failed")},
};

const int NUM_SOCKET_ERRORS = sizeof(SocketErrors) / sizeof(SocketError*);


// The token and queue routines that follow are
// for utility use and should be turned into real code
//
const char* TOKENS[] =
{
	" ",
	"\t",
	"\n\r",
	"\r\n"
};

const int NUM_TOKENS = sizeof(TOKENS) / sizeof(char*);

void Token(string& source,string& token)
{
	unsigned int i;
	unsigned int Positions[NUM_TOKENS];
	int nTokens = 0;
	while (nTokens < NUM_TOKENS)
	{
		i = source.find(TOKENS[nTokens]);
		Positions[nTokens++] = i;
	}
	i = strlen(source.c_str());
	int p = 0;
	for (int n = 0; n < NUM_TOKENS; n++)
	{
		if (i > Positions[n])
		{
			i = Positions[n];
			p = n;
		}
	}
	token = source.substr(0,i);
	source = source.substr(i+strlen(TOKENS[p]));
}

// removes all messages from a windows queue
static void CLEANQUEUE(HWND hwnd)
{
  MSG msg;
  while (::PeekMessage(&msg,hwnd,0,0,PM_REMOVE))
	;
}


// TAsyncSocket event handlers
DEFINE_RESPONSE_TABLE1(TAsyncSocket,TWindow)
	EV_WM_TIMER,
  EV_MESSAGE(SM_HOSTNAME,SmHostName),
END_RESPONSE_TABLE;

// TAsyncSocket constructor
//
// Expects a ptr to the parent TWindow and the socket catch class
//
// Hides the window, sets valid and connected flags to false
// and sets up catcher ptr.
//
TAsyncSocket::TAsyncSocket(TWindow* pParent,TSocketCatch* pC)
						 :TWindow(pParent)
{
	Attr.Style &= ~WS_VISIBLE;	// force window to be hidden
	bValid = FALSE;							// invalid until SetupWindow is hit
	bConnected = FALSE;			  	// invalid until connection is made
  pCatcher = pC;              // store our catcher 
  pWorkBuffer = new char[WORKBUFF_SIZE]; // allocate work buffer
}

// TAsyncSocket Destructor
//
// delete pWorkBuffer
TAsyncSocket::~TAsyncSocket()
{
  delete []pWorkBuffer;             // delete work buffer
}

// TAsyncSocket::CleanupWindow
//
// Opposite of SetupWindow
//
// Cancels any blocking calls, closes the socket if we are connected
// and kills the timer if necessary
//
void TAsyncSocket::CleanupWindow()
{
	if (WSAIsBlocking())        // if we are blocking
		WSACancelBlockingCall();  // cancel the blocking call

	if (bConnected)             // if we are connected
		closesocket(sock);        // close the socket

	if (bValid)                 // if we are valid
  {                           //
	  WSACleanup();             // clean up the Winsockets
	  KillTimer(1);             // and kill our timer
  }
  TWindow::CleanupWindow();   // call parent cleanupwindow
}

// TAsyncSocket::SetupWindow
//
// Called as part of the normal creation sequence in OWL
//
// 1. Call parent SetupWindow 
// 2. Startup the sockets and check version #
// 3. Start our timer
// 
void TAsyncSocket::SetupWindow()
{
  TWindow::SetupWindow();     // call parent

	int nErr = WSAStartup(VERSION_REQUIRED,&wsaData);
	if (nErr == 0)              // if we have a good version
  {                           //
		bValid = TRUE;            // set validity flag
	  SetTimer(TIMER_ID,TIMEOUT_MS);// set our timer
  }                           //
  else                        // problem with version
  {                           //
    if (pCatcher)             // make sure we have a catcher
      pCatcher->WinSocketError(nErr,""); // and notify them of error
  }
}

// TAsyncSocket::AsyncGetHostByName
//
// Starts an asyncronous host name lookup
// stores the returned handle in hAsync
//
// Expects:
//    pHost -- ptr to host name
//
// Returns:
//    0 if ok, LastError() if a problem was encountered
//
int TAsyncSocket::AsyncGetHostByName(const char* pHost)
{
  hAsync = WSAAsyncGetHostByName(HWindow,SM_HOSTNAME,pHost,pWorkBuffer,WORKBUFF_SIZE);
  if (hAsync == 0)          // if we did not receive a handle
    return LastError();     // return socket error
  else                      // otherwise
    return 0;               // return an error
}

// TAsyncSocket::SmHostName
//
// receives host name notification and connects the socket
//
// Expects:
//    HIWORD(lp) contains error information --see winsock.hlp or winsock.h 
//    for addition info--
//
// Returns:
//    TRUE -- message handled
//
LRESULT TAsyncSocket::SmHostName(WPARAM /* wp */, LPARAM lp)
{
  WSACancelAsyncRequest(hAsync);    // cancel async requests
  if (HIWORD(lp) == 0)              // check for error
  {
    // setup host structures
	  memcpy(&Host,pWorkBuffer,sizeof(Host));
	  memcpy(&inetAddr.sin_addr,Host.h_addr,Host.h_length);

    // connect the socket
	  if (connect(sock,(sockaddr far*)&inetAddr,sizeof(sockaddr)) == 0)
		  bConnected = TRUE;      // set connected flag if operation was ok
  }
  return TRUE;
}


// TAsyncSocket::Connect
//
// high level host connection
//
// Starts the process of an asynchronous connection
// sets up portions of the internet address structure inetAddr
// for use later when the host look up finishes, etc.
//
//  Expects:
//    pHost -- ptr to desired host name
//    nPort -- port number to use
//    nProtocol -- protocol, defaults ti PF_INET
//
//  Returns:
//    integer -- 0 on success, any other value is either a WinSocket error or 
//    one of our defined errors
//
int TAsyncSocket::Connect(const char* pHost, int nPort, int nProtocol = PF_INET)
{
  memset(&inetAddr,0,sizeof(inetAddr));     // cleanup structure
	inetAddr.sin_family = nProtocol;          // set protocol (almost always PF_INET)
	inetAddr.sin_port = htons(nPort);         // setup port
	sock = socket(nProtocol,SOCK_STREAM,0);   // create the socket, passed protocol, STREAM,
														                // no protocol specification (0)
	if (sock != INVALID_SOCKET)               // socket was created ok
  {
	 return AsyncGetHostByName(pHost);        // acquire the host asynchronously
  }
  else
  {
	  return LastError();                     // return WSAError
  }
}

// TAsyncSocket::LastError
//
// Returns the last WinSocket error encountered
//
int TAsyncSocket::LastError()
{
	return WSAGetLastError();
}

// TAsyncSocket::SyncSend
//
// Syncronous send, use for small amounts of data
//
// Expects:
//    pData -- ptr to data to send
//    nLen  -- length of data
//
// Returns:
//    integer -- SOCKET_ERROR - extract error info with LastError()
//            -- Anything else - contains amount of data sent
//
int TAsyncSocket::SyncSend(const char* pData,int nLen)
{
  return send(sock,pData,nLen,0);
}

// TAsyncSocket::SyncRecv
//
// Syncronous recv, use when you expect small amounts of data
//
// Expects:
//    pData -- ptr to data for storage
//    nMaxLen  -- max length of data
//
// Returns:
//    integer -- SOCKET_ERROR - extract error info with LastError()
//            -- Anything else - contains amount of data received
//
int TAsyncSocket::SyncRecv(char* pData,int nMaxLen)
{
  return recv(sock,pData,nMaxLen,0);
}

// TAsyncSocket::EvTimer
//
// Catches potential blocking problems and times out
// after BLOCK_TIMEOUT number of seconds
//
static int bTick = 0;
static BOOL bInTimer = FALSE;
void TAsyncSocket::EvTimer(uint /* uTimerId */)
{
	if (!bInTimer)
	{
		bInTimer = TRUE;
		if (WSAIsBlocking())
		{
			bTick++;
			if (bTick > BLOCK_TIMEOUT)
			{
				WSACancelBlockingCall();
        if (pCatcher)
          pCatcher->WinSocketError(WSAETIMEDOUT,"");
			}
		}
		else bTick = 0;
		bInTimer = FALSE;
	}
}

// TAsyncSocket::SyncSendLine
//
// Appends a cr/lf to the passed data and sends it with SyncSend
// 
// Expects:
//    pData -- null terminated string
//
// Returns:
//    integer -- SOCKET_ERROR - extract error info with LastError()
//            -- Anything else - contains amount of data sent
//
// ToDo: Check for lines that already end with cr/lf and do not append ours?
//
int TAsyncSocket::SyncSendLine(const char* pData)
{
	char* pC = new char[strlen(pData)+3];       // watch OUT FOR THAT CONSTANT
	strcpy(pC,pData);                           // copy original
	strcat(pC,"\r\n");                          // append a cr/lf
	int nReturn = SyncSend(pC,strlen(pC));      // send it along
  delete []pC;                                // delete working buffer
  return nReturn;                             // return send result
}


// TAsyncPOPSocket event handlers
DEFINE_RESPONSE_TABLE1(TAsyncPOPSocket,TAsyncSocket)
  EV_MESSAGE(SM_HOSTNAME,SmHostName),
  EV_MESSAGE(SM_DATAREADY,SmDataReady),
  EV_MESSAGE(SM_POPCONNECT,SmPopConnect),
  EV_MESSAGE(SM_POPAUTHORIZE,SmPopAuthorize),
  EV_MESSAGE(SM_POPPASSWORD,SmPopPassword),
  EV_MESSAGE(SM_POPMESSAGECOUNT,SmPopMessageCount),
  EV_MESSAGE(SM_POPGETMESSAGE,SmPopGetMessage),
  EV_MESSAGE(SM_POPSTARTMESSAGE,SmPopStartMessage),
END_RESPONSE_TABLE;

// TAsyncPOPSocket constructor
//
// 
TAsyncPOPSocket::TAsyncPOPSocket(TWindow* pParent,TPOPSocketCatch* pC)
							  :TAsyncSocket(pParent,pC)
{
	bAuthorized = FALSE;
	pWorkString = new string;
  pCurrentMessage = 0;
  pMyCatcher = pC;
}

// TAsyncPOPSocket destructor
//
//
TAsyncPOPSocket::~TAsyncPOPSocket()
{
	delete pWorkString;
}


// TAsyncPOPSocket::Connect
//
// calls parent Connect --which in return resolves the host name which virtually calls
// our SmHostName()
//
// Expects:
//    ptr to host name
//
// Returns:
//    return value from TAsyncSocket::Connect
//
int TAsyncPOPSocket::Connect(const char* pHost)
{
  return TAsyncSocket::Connect(pHost,POP_PORT,PF_INET);
}


// TAsyncPOPSocket::SmDataReady
//
// place holder for virtual function --does nothing useful--
//
LRESULT TAsyncPOPSocket::SmDataReady(WPARAM /* wp */, LPARAM /* lp */)
{
  WSAAsyncSelect(sock,HWindow,0,0);
  return 0;
}
// TAsyncPOPSocket::SmPopConnect
//
// Receives FD_READ from SmHostName call --should receive general info from POP
// server, sent after a connection--
//
// Expects:
//    HIHWORD(lp) - error code from sockets
//
// 1. check for POP_OK
// 2. if we got it, authorize the account via Authorize(user,password)
//
LRESULT TAsyncPOPSocket::SmPopConnect(WPARAM /* wp */, LPARAM lp)
{
  WSAAsyncSelect(sock,HWindow,0,0);             // turn off async msg's
	if (HIWORD(lp) == 0 && LOWORD(lp) == FD_READ) // check for error
  {                                             //
	 if (MailResponse() == POP_OK)                // did we get an OK
		 Authorize(sUser.c_str(),sPassword.c_str());// yes, authorize the account
	 else                                         //
		return TRUE;                                // ERROR HANDLING needed
  }                                             //
  return TRUE;                                  //
}

// TAsyncPOPSocket::SmPopPassword
//
// response for FD_READ, expects OK or ERR from POP server
// after submission of the password
//
// Expects:
//    HIHWORD(lp) - error code from sockets
//
// 1. check for POP_OK
// 2. if we got it, set our authorized flag and start message
// retrieval with MessageCount()
//
LRESULT TAsyncPOPSocket::SmPopPassword(WPARAM /* wp */, LPARAM lp)
{
  WSAAsyncSelect(sock,HWindow,0,0);     // turn off async msgs
	if (HIWORD(lp) == 0 && LOWORD(lp) == FD_READ) // check for error
  {                                     //
	  if (MailResponse() == POP_OK)       // check for POP_OK
	  {                                   //
		  bAuthorized = TRUE;               // got it, set flag
		  GetMessageCount();                // start message retrieval
		  return TRUE;                      //
	  }                                   //
    else                                //
      return TRUE;                      // ERROR HANDLER
	 }                                    //
  return TRUE;                          // ERROR HANDLER  
}

// TAsyncPOPSocket::RetrieveMail
//
// retrieves mail based on passed host, user, and password
//
// Expects:
//    ptr - host, user, and passowrd
//
// 1.  setup internal variables with information
// 2.  call Connect with host name
//
int TAsyncPOPSocket::RetrieveMail(const char* pHost, const char* pUser, const char* pPassword)
{
  sServer = pHost;
  sUser = pUser;
  sPassword = pPassword;
  return Connect(sServer.c_str());
}

// TAsyncPOPSocket::SmPopAuthorize
//
// Reponse for FD_READ on user name submission, 
//
// Expects:
//    HIHWORD(lp) - error code from sockets
//
// 1. check for POP_OK
// 2. if we got it, send password information
//
LRESULT TAsyncPOPSocket::SmPopAuthorize(WPARAM /* wp */, LPARAM lp)
{
  WSAAsyncSelect(sock,HWindow,0,0);       // turn off async msgs
	if (HIWORD(lp) == 0 && LOWORD(lp) == FD_READ) // check for error
  {                                       //
	  if (MailResponse() == POP_OK)         // did we get POP_OK
	  {                                     // yes
		  string s = POP_PASS;                // format password
		  s += sPassword;                     // string
		  SyncSendLine(s.c_str());            // and send it
		  WSAAsyncSelect(sock,HWindow,SM_POPPASSWORD,FD_READ); // response goes to SmPopPassword
		  return TRUE;                        //
	  }                                     // else needs ERROR HANDLER
  }                                       //   
  return TRUE;                            // ERROR HANDLER  
}

// TAsyncPOPSocket::SmHostName
//
// Reponse for host name resolution, called virtually from TAsyncSocket base routines
//
// Expects:
//    HIHWORD(lp) - error code from sockets
//
// 1. call parent
// 2. if connected flag is on, we need to start our connection process
// 3. otherwise report the error
//
LRESULT TAsyncPOPSocket::SmHostName(WPARAM wp, LPARAM lp)
{
  TAsyncSocket::SmHostName(wp,lp);    // call parent 
  if (bConnected)                     // normal connection was ok
  {                                   // so start our series of commands
		WSAAsyncSelect(sock,HWindow,SM_POPCONNECT,FD_READ);
  }
  else                                // connect did not happen
  {                                   //
    if (pCatcher)                     // if we have a catcher, let them know
      pCatcher->WinSocketError(LastError(),"");
  }
  return TRUE;
}

// TAsyncPOPSocket::MailResponse()
//
// performs a receive and checks the response for a POP related OK or
// error field
// 
// Expects:
//    Nothing
//
// Returns:
//    POP response code --POP_OK, POP_ERR, POP_LINE
//
int TAsyncPOPSocket::MailResponse()
{
  // get data
	if (SyncRecv(pWorkBuffer,WORKBUFF_SIZE) != SOCKET_ERROR)
	{
	  *pWorkString = pWorkBuffer;                   // assing work string
		if (pWorkString->contains(OK_RESPONSE))       // check for OK
		  return POP_OK;                              // yes, return it
		else if (pWorkString->contains(ERR_RESPONSE)) // no, check for ERR
				   return POP_ERR;                        // yes, return it
				 else                                     // no,
			     return POP_LINE;                       // something else, must be a line...
	}                                               // if receive did not get any data
	else                                            // ERROR_HANDLER
  {                                               // 
  }                                               //
  return POP_NOTHING;                             //
}

// TAsyncPOPSocket::Hangup
//
// terminates a POP session
//
// 1. if we are authorized, call Quit()
//
int TAsyncPOPSocket::Disconnect()
{
	if (bAuthorized)
		return Quit();
  return 0;
}


// TAsyncPOPSocket::Authorize
//
// authorizes a POP account with name and password
//
// 1. assemble user string
// 2. start authorization message sequence
//
#pragma warn -par
int TAsyncPOPSocket::Authorize(const char* pName, const char* pPassword)
{
	if (!bConnected)                    // error catch for no connection
  {                                   // ERROR HANDLER  
  }                                   // 

	string s = POP_USER;
	s += pName;
	SyncSendLine(s.c_str());
  WSAAsyncSelect(sock,HWindow,SM_POPAUTHORIZE,FD_READ);
  return 0;
}
#pragma warn .par


// TAsyncPOPSocket::SmPopMessageCount
//
// Handler for FD_READ on POP_STAT request, parses info and begins
// message retrieval
// 
// Notes:
//    See POP RFC for specifics on line formats
//
// 1. check error from sockets
// 2. check mail response for POP_OK
// 3. parse information
// 4. give information catcher the message count
// 5. setup messages pending and start retrieving
//    
LRESULT TAsyncPOPSocket::SmPopMessageCount(WPARAM /* wp */,LPARAM lp)
{
  WSAAsyncSelect(sock,HWindow,0,0);     // turn off msgs
	if (HIWORD(lp) == 0 && LOWORD(lp) == FD_READ) // check for error
  {                                     //
	 if (MailResponse() == POP_OK)			  // we assume the data is in pWorkString 
	 {                                    // 
		pWorkString->remove(0,strlen(OK_RESPONSE)+1); // remove ok response
		int i = pWorkString->contains(" ");           // look for " "
		if (i > 0)                                    // if we have any
		{                                             //
		  string sizeStr = pWorkString->substr(i+1);  // remove size string
		  pWorkString->substr(0,i);                   // excise 
		  lMessageTotal = atol(sizeStr.c_str());      // setup total message sizes
		  lMessageCount = atoi(pWorkString->c_str()); // and count
		}                                             //  
		else                                          //
		{                                             // this ELSE is probably a bug
		  lMessageTotal = atoi(pWorkString->c_str()); // default to total --problematic--
		}                                             //
		if (pMyCatcher)                               // if we have a catcher
		  pMyCatcher->MessageCount((int)lMessageCount); // tell them the count
                                                  //
		nMessagesPending = (int)lMessageCount;        // setup messages pending
		RetrieveMessages();                           // start retrieving
	 }                                              //
  }                                               // ERRORHANDLER 
  return TRUE;                                    //
}

// TAsyncPOPSocket::GetMessageCount()
//
// starts --post authorization-- the chain of events that 
// results in messages being retrieved
//
// 1. check authorization
// 2. send POP_STAT to obtain status
// 3. setup handler 
//
int TAsyncPOPSocket::GetMessageCount()
{
	if (bAuthorized)
  {
	  SyncSendLine(POP_STAT);
    if (WSAAsyncSelect(sock,HWindow,SM_POPMESSAGECOUNT,FD_READ) == 0)
	    return 0;
    else
	   return WSAGetLastError();
  }
  return 0;
}

// TAsyncPOPSocket::SmPopStartMessage
//
// first hit on message retrieval
//
// 1. clean window queue
// 2. validate error and response type
// 3. check mail response
// 4. if it was ok, create temp file and write current data out
// 5. otherwise, try for the next message
//  
LRESULT TAsyncPOPSocket::SmPopStartMessage(WPARAM /* wp */, LPARAM lp)
{
  WSAAsyncSelect(sock,HWindow,0,0);             // end msgs
  CLEANQUEUE(HWindow);                          // clean this windows queue
	if (HIWORD(lp) == 0 && LOWORD(lp) == FD_READ) // check error and response
  {                                             //
    int i = MailResponse();                     // get mail response
	  if (i == POP_OK)                            // ok?
    {                                           //
      sTempName = TEMP_MAILNAME;                // temp name primer
      char c[10];                               //
      itoa(nMessagesPending+1,c,10);            // all sorts of assumptions here, but easy
      sTempName += c;
      out.open(sTempName.c_str(),ios::out | ios::binary); // open file
// un-comment the following if you want to see the POP stuff before the msg
//      out.write(pWorkString->c_str(),pWorkString->length()); // write current data
	    WSAAsyncSelect(sock,HWindow,SM_POPGETMESSAGE,FD_READ); // select next read handler
    }                                           //
	 else                                         // no POP_OK
    {                                           //
      RetrieveNext();                           // next message
    }                                           //
  }                                             // ERRORHANDLER
  return TRUE;                                  //
}


// TAsyncPOPSocket::SmGetMessge
//
// handles FD_READS during message retrieval
//
// 1. check re-entrant flag, exit if set
// 2. set re-entrant flag
// 3. end msgs and clean window queue
// 4. validate error and response
// 5. allocate work buffer, get data and store it
// 6. check for message terminator, if it exists, close file 
// and notify info catcher
// 7. otherwise: reselect ourselves for more data
// 8. clear re-entrant flag
//  
static bool bInGet = FALSE;
static char* pData = 0;
LRESULT TAsyncPOPSocket::SmPopGetMessage(WPARAM /* wp */, LPARAM lp)
{
  if (!bInGet)                                    // make sure we are not already here
  {                                               //
	  bInGet = TRUE;                                // set re-entrant flag
    WSAAsyncSelect(sock,HWindow,0,0);             // turn off msgs
    CLEANQUEUE(HWindow);                          // clean window queue
	  if (HIWORD(lp) == 0 && LOWORD(lp) == FD_READ) // check error and validate response
	  {                                             //
		  pData = new char[MAX_MESSAGE];              // allocate max single chunk
		  int i = SyncRecv(pData,MAX_MESSAGE);        // receive available data
		  if (i != SOCKET_ERROR)                      // error?
		  {                                           // no:
        out.write(pData,i);                       // write data
        out.flush();                              // and flush
			  pData[i] = 0;                             // null terminate data
        int n = strlen(TERMINATION_OCTET);        // look for termination octet
		    if (i >= n && (strcmp((pData + i - n),TERMINATION_OCTET) == 0))
				{                                         // found it: 
          out.close();                            // close the file
          if (pMyCatcher)                         // notify catcher
					  pMyCatcher->MessageRetrieved(nMessagesPending+1,sTempName.c_str());
					RetrieveNext();                         // get next message
				}                                         //
				else                                      // no message terminator
				{                                         // reselect ourselves for more data
				  WSAAsyncSelect(sock,HWindow,SM_POPGETMESSAGE,FD_READ);
				}                                         //
			}                                           //
		  delete []pData;                             // delete work buffer
      pData = 0;                                  //
	 }                                              //
	 else                                           //
    {                                             // ERRORHANDLER
	  }                                             //
    bInGet = FALSE;                               // clear re-entrant flag
  }                                               //
  return 0;                                       //
}                                                 //


// TAsyncPOPSocket::RetrieveNext()
//
// gets the next pending message
void TAsyncPOPSocket::RetrieveNext()
{
  RetrieveMessages();
}

// TAsyncPOPSocket::RetrieveMessages
//
// retrieves messages
//
// 1. check authorization
// 2. do we have messages pending
// 3. yes: compose POP_RETR and select message retriever
// 4. no: Quit()  
//
int TAsyncPOPSocket::RetrieveMessages()
{
	if (bAuthorized)
  {
    if (nMessagesPending > 0)
    {
	    string s = POP_RETR;
	    char c[10];
	    s += itoa(nMessagesPending,c,10);
	    nMessagesPending--;
	    SyncSendLine(s.c_str());
	    WSAAsyncSelect(sock,HWindow,SM_POPSTARTMESSAGE,FD_READ);
	    return 0;
    }
    else
    {
	    Quit();
    }
  }
  return 0;
}

// TAsyncPOPSocket::Quit
//
// Terminates a POP session
// 
// 1. check authorization
// 2. send POP_QUIT
// 3. eat the response
//  
int TAsyncPOPSocket::Quit()
{
	if (bAuthorized)
  {
	  SyncSendLine(POP_QUIT);
		MailResponse();
	}
  return 0;
}


