/* Copyright (C) 1989, 1992, 1993, 1994 Aladdin Enterprises.  All rights reserved.
  
  This file is part of Aladdin Ghostscript.
  
  Aladdin Ghostscript is distributed with NO WARRANTY OF ANY KIND.  No author
  or distributor accepts any responsibility for the consequences of using it,
  or for whether it serves any particular purpose or works at all, unless he
  or she says so in writing.  Refer to the Aladdin Ghostscript Free Public
  License (the "License") for full details.
  
  Every copy of Aladdin Ghostscript must include a copy of the License,
  normally in a plain ASCII text file named PUBLIC.  The License grants you
  the right to copy, modify and redistribute Aladdin Ghostscript, but only
  under certain conditions described in the License.  Among other things, the
  License requires that the copyright notice and this notice be preserved on
  all copies.
*/

/* iref.h */
/* Object structure and type definitions for Ghostscript */

#ifndef iref_INCLUDED
#  define iref_INCLUDED

/* The typedef for object references */
typedef struct ref_s ref;

/* The typedef for packed object references.  This is opaque here: */
/* the details are in packed.h. */
typedef ushort ref_packed;

/*
 * Object types.  The types marked with + use the read/write/execute
 * attributes; the rest only use the executable attribute.
 * The types marked with ! behave differently in the interpreter
 * depending on whether they are executable or literal.
 * The types marked with @ are composite and hence use the a_local and
 * a_foreign attributes; objects of all other types must have a_local cleared.
 * The types marked with # use the size field.
 */
typedef enum {

/*
 * Type 0 must be left unassigned, so that the type (and type_attrs)
 * of a valid ref will never be zero.  This speeds up simultaneous
 * type/space checking in def (see dstack.h for details) and a similar
 * check in ref_save (see store.h for details).  We may as well use
 * type 0 for t__invalid, which will never appear in a real ref.
 *
 * The "invalid" type is only used in a few special places: the guard
 * entries at the bottom of the o-stack that detect stack underflow,
 * and (eventually) the ref that the cached value pointer in names points to
 * if the binding isn't known.  It never appears on a stack or in a
 * program-visible data structure.
 */

	t__invalid,		/*	(no value) */
	t_boolean,		/*	value.boolval */
	t_dictionary,		/* @ +  value.pdict */
	t_file,			/* @!+# value.pfile, uses size for id */

/*
 * The 4 array types must be kept together, and must start at
 * a multiple of 4, for the sake of r_is_array and r_is_proc (see below).
 */

#define _t_array_span 4
	t_array,		/* @!+# value.refs */
		/* The following are the two implementations of */
		/* packed arrays. */
	t_mixedarray,		/* @!+# value.packed */
	t_shortarray,		/* @!+# value.packed */
	  t_unused_array_,	/*	(an unused array type) */

/*
 * t_[a]struct is an "umbrella" for other types that are represented by
 * allocated objects (structures).  Objects of these types are composite
 * and hence use the a_local attribute.  The type name is taken from
 * the allocator template for the structure.  t_astruct objects use the
 * access attributes; t_struct objects do not.  Neither t_struct nor
 * t_astruct objects use the size.
 *
 * t_struct is currently used for the following PostScript types:
 *	condition, lock.
 * We could use it for fontIDs, except that they may have subclasses.
 * Eventually it will also be used for the new 'device' type.
 * t_astruct is currently used for the following PostScript types:
 *	gstate.
 *
 * The 2 structure types must be kept together, and must start at
 * a multiple of 2, for the sake of r_has_stype (see below).
 */

#define _t_struct_span 2
	t_struct,		/* @	value.pstruct */
	t_astruct,		/* @ +	value.pstruct */

/*
 * We now continue with individual types.
 */
	t_fontID,		/* @    value.pstruct */
	t_integer,		/*	value.intval */
	t_mark,			/*        (no value) */
	t_name,			/*  ! # value.pname, uses size for index */
	t_null,			/*  ! # (value.opproc, uses size for mark */
				/*        type, on e-stack only) */
	t_operator,		/*  ! # value.opproc, uses size for index */
	t_real,			/*	value.realval */
	t_save,			/*      value.saveid, see isave.h for why */
				/*        this isn't a t_struct */
	t_string,		/* @!+# value.bytes */
/*
 * The following are extensions to the PostScript type set.
 * When adding new types, be sure to edit the table in gs_init.ps
 * (==only operator), the printing routine in idebug.c, the dispatches
 * in igc.c, igcref.c, and interp.c, obj_eq in iutil.c, restore_check_stack
 * in zvmem.c, and also type_name_strings and type_print_strings below.
 */
	t_device,		/* @     value.pdevice */
	t_oparray,		/* @  #    (no value), uses size for index */
	t_next_index	/*** first available index ***/
} ref_type;
/*
 * The interpreter uses types starting at t_next_index for representing
 * a few high-frequency operators.
 * Since there are no operations specifically on operators,
 * there is no need for any operators to check specifically for these
 * types.  The r_btype macro takes care of the conversion when required.
 */
/*extern const int tx_next_index;*/		/* in interp.c */
/*
 * Define the types that use the size field.
 * (The extended types used by the interpreter also use the size field.)
 */
#define case_types_with_size\
  case t_array: case t_mixedarray: case t_shortarray:\
  case t_file: case t_name: case t_operator: case t_string:\
  case t_oparray
/*
 * Define the type names for debugging printout.
 * All names must be the same length, so that columns will line up.
 */
#define type_print_strings\
  "INVL","bool","dict","file",\
  "arry","mpry","spry","u?ry",\
  "STRC","ASTR",\
  "font","int ","mark","name","null",\
  "oper","real","save","str ",\
  "devc","opry"
/*
 * Define the type names for the type operator.
 */
#define type_name_strings\
  "","booleantype","dicttype","filetype",\
  "arraytype","packedarraytype","packedarraytype","arraytype",\
  "","",\
  "fonttype","integertype","marktype","nametype","nulltype",\
  "operatortype","realtype","savetype","stringtype",\
  "devicetype","operatortype"

/*
 * The following factors affect the encoding of attributes:
 *
 *	- The packed array format requires the high-order bits of the
 *	  type/attributes field to be 0.  (see packed.h)
 *
 *	- The interpreter wants the type, executable bit, and execute
 *	  permission to be adjacent, and in that order from high to low.
 *
 *	- Type testing is most efficient if the type is in a byte by itself.
 *
 * The layout given below results in the most efficient code overall.
 */

/* Location attributes. */
/* Note that these are associated with the *location*, not with the */
/* ref that is *stored* in that location. */
#define l_mark 1			/* mark for garbage collector */
#define l_new 2				/* stored into since last save */
/* Internal attributes. */
#define a_foreign 4			/* object allocated outside */
					/* space managed by Ghostscript */
/* Attributes visible at the PostScript language level. */
#define a_local 8			/* object allocated in local VM */
#define a_space (a_foreign+a_local)
#define a_write 0x10
#define a_read 0x20
#define a_execute 0x40
#define a_executable 0x80
#define a_readonly (a_read+a_execute)
#define a_all (a_write+a_read+a_execute)
#define r_type_shift 8
#define r_type_bits 6

/* Define the attribute names for debugging printout. */
#define attr_print_string "mnflwrxe......??"

/* Abstract types */
typedef struct dict_s dict;
typedef struct name_s name;
struct stream_s;
struct gx_device_s;
#ifndef obj_header_DEFINED
#  define obj_header_DEFINED
typedef struct obj_header_s obj_header_t;
#endif
/* We duplicate the definition of os_ptr (a.k.a. s_ptr) here */
/* so that we can have an accurate typdef for op_proc */
/* without having to drag in istack.h and ostack.h. */
typedef int (*op_proc_p)(P1(ref _ds *));
/* real_opproc is a holdover.... */
#define real_opproc(pref) ((pref)->value.opproc)

/* Object reference */
/*
 * Note that because of the way packed arrays are represented,
 * the type_attrs member must be the first one in the ref structure.
 */
struct tas_s {
	ushort type_attrs;
	ushort rsize;
};
struct ref_s {

	struct tas_s tas;

#define r_size(rp) ((rp)->tas.rsize)
#define r_inc_size(rp,inc) ((rp)->tas.rsize += (inc))
#define r_dec_size(rp,dec) ((rp)->tas.rsize -= (dec))
#define r_set_size(rp,siz) ((rp)->tas.rsize = (siz))
/* type_attrs is a single element for fast dispatching in the interpreter */
#if r_type_shift == 8
#  if arch_is_big_endian
#    define r_type(rp) (((byte *)&((rp)->tas.type_attrs))[sizeof(ushort)-2])
#  else
#    define r_type(rp) (((byte *)&((rp)->tas.type_attrs))[1])
#  endif
#  define r_has_type(rp,typ) (r_type(rp) == (typ))
#else
#  define r_type(rp) ((rp)->tas.type_attrs >> r_type_shift)
#  define r_has_type(rp,typ) r_has_type_attrs(rp,typ,0)	/* see below */
#endif
/* A special macro for testing arrayhood. */
#define r_is_array(rp) _r_has_masked_type_attrs(rp,t_array,_t_array_span,0)
#define r_set_type(rp,typ) ((rp)->tas.type_attrs = (typ) << r_type_shift)
#define r_btype(rp)\
 ((rp)->tas.type_attrs >= (t_next_index << r_type_shift) ?\
  t_operator : r_type(rp))
#define r_type_xe_shift (r_type_shift - 2)
#define type_xe_(tas) ((tas) >> r_type_xe_shift)	/* internal use only */
#define r_type_xe(rp) type_xe_((rp)->tas.type_attrs)
#define type_xe_value(t,xe) type_xe_(((t) << r_type_shift) + (xe))
#define r_type_attrs(rp) ((rp)->tas.type_attrs)	/* reading only */
#define r_has_attrs(rp,mask) !(~r_type_attrs(rp) & (mask))
#define r_has_attr(rp,mask1)		/* optimize 1-bit case */\
   (r_type_attrs(rp) & (mask1))
/* The following macro is not for external use. */
#define _r_has_masked_type_attrs(rp,typ,tspan,mask)\
 (((rp)->tas.type_attrs &\
   ((((1 << r_type_bits) - (tspan)) << r_type_shift) + (mask))) ==\
  (((typ) << r_type_shift) + (mask)))
#define r_has_type_attrs(rp,typ,mask)\
  _r_has_masked_type_attrs(rp,typ,1,mask)
/* A special macro for testing procedurehood. */
#define r_is_proc(rp)\
  _r_has_masked_type_attrs(rp,t_array,_t_array_span,a_execute+a_executable)
#define r_set_attrs(rp,mask) ((rp)->tas.type_attrs |= (mask))
#define r_clear_attrs(rp,mask) ((rp)->tas.type_attrs &= ~(mask))
#define r_store_attrs(rp,mask,attrs)\
  ((rp)->tas.type_attrs = ((rp)->tas.type_attrs & ~(mask)) | (attrs))
#define r_copy_attrs(rp,mask,sp)\
  r_store_attrs(rp,mask,(sp)->tas.type_attrs & (mask))
#define r_set_type_attrs(rp,typ,mask)\
  ((rp)->tas.type_attrs = ((typ) << r_type_shift) + (mask))
/* Macros for t_[a]struct objects. */
#define r_is_struct(rp) _r_has_masked_type_attrs(rp,t_struct,_t_struct_span,0)
#define r_has_stype(rp,mem,styp)\
  (r_is_struct(rp) && gs_object_type(mem, (rp)->value.pstruct) == &styp)
#define r_ptr(rp,typ) ((typ *)((rp)->value.pstruct))
#define r_set_ptr(rp,ptr) ((rp)->value.pstruct = (void *)(ptr))

	union v {			/* name the union to keep gdb happy */
		long intval;
		ushort boolval;
		float realval;
		ulong saveid;
		byte *bytes;
		const byte *const_bytes;
		ref *refs;
		const ref *const_refs;
		name *pname;
		const name *const_pname;
		dict *pdict;
		const dict *const_pdict;
		const ref_packed *packed;
		op_proc_p opproc;
		struct stream_s *pfile;
		struct gx_device_s *pdevice;
		obj_header_t *pstruct;
	} value;
};

/* Define the required alignment for refs. */
/* We assume all alignment values are powers of 2. */
#define align_ref_mod\
 (((arch_align_long_mod - 1) | (arch_align_float_mod - 1) |\
   (arch_align_ptr_mod - 1)) + 1)

/* Define the maximum size of an array or a string. */
/* The maximum array size is determined by the fact that */
/* the allocator cannot allocate a block larger than max_uint. */
#define max_array_size min(max_ushort, max_uint / (uint)arch_sizeof_ref)
#define max_string_size max_ushort

#endif					/* iref_INCLUDED */
