/* Copyright (C) 1992, 1994 Aladdin Enterprises.  All rights reserved.
  
  This file is part of Aladdin Ghostscript.
  
  Aladdin Ghostscript is distributed with NO WARRANTY OF ANY KIND.  No author
  or distributor accepts any responsibility for the consequences of using it,
  or for whether it serves any particular purpose or works at all, unless he
  or she says so in writing.  Refer to the Aladdin Ghostscript Free Public
  License (the "License") for full details.
  
  Every copy of Aladdin Ghostscript must include a copy of the License,
  normally in a plain ASCII text file named PUBLIC.  The License grants you
  the right to copy, modify and redistribute Aladdin Ghostscript, but only
  under certain conditions described in the License.  Among other things, the
  License requires that the copyright notice and this notice be preserved on
  all copies.
*/

/* zimage2.c */
/* image operator extensions for Level 2 PostScript */
#include "math_.h"
#include "memory_.h"
#include "ghost.h"
#include "errors.h"
#include "oper.h"
#include "gscolor.h"
#include "gscspace.h"
#include "gscolor2.h"
#include "gsmatrix.h"
#include "idict.h"
#include "idparam.h"
#include "ilevel.h"
#include "igstate.h"		/* for igs */

/* Imported from zpaint.c */
extern int zimage_setup(P10(int width, int height, gs_matrix *pmat,
  ref *sources, int bits_per_component,
  bool multi, const gs_color_space *pcs, int masked,
  const float *decode, int npop));
extern int zimage(P1(os_ptr));
extern int zimagemask(P1(os_ptr));

/* Define a structure for acquiring image parameters. */
typedef struct image_params_s {
	int Width;
	int Height;
	gs_matrix ImageMatrix;
	bool MultipleDataSources;
	ref DataSource[4];
	int BitsPerComponent;
	float Decode[2*4];
	const float *pDecode;
	bool Interpolate;
} image_params;

/* Common code for unpacking an image dictionary. */
/* Assume *op is a dictionary. */
private int
image_dict_unpack(os_ptr op, image_params *pip, int max_bits_per_component)
{	int code;
	int num_components;
	int decode_size;
	ref *pds;
	check_dict_read(*op);
	num_components = gs_currentcolorspace(igs)->type->num_components;
	if ( num_components < 1 )
		return_error(e_rangecheck);
	if ( max_bits_per_component == 1 )	/* imagemask */
		num_components = 1;		/* for Decode */
	if ( (code = dict_int_param(op, "ImageType", 1, 1, 1,
				    &code)) < 0 ||
	     (code = dict_int_param(op, "Width", 0, 0x7fff, -1,
				    &pip->Width)) < 0 ||
	     (code = dict_int_param(op, "Height", 0, 0x7fff, -1,
				    &pip->Height)) < 0 ||
	     (code = dict_matrix_param(op, "ImageMatrix",
				    &pip->ImageMatrix)) < 0 ||
	     (code = dict_bool_param(op, "MultipleDataSources", false,
				    &pip->MultipleDataSources)) < 0 ||
	     (code = dict_int_param(op, "BitsPerComponent", 0,
				    max_bits_per_component, -1,
				    &pip->BitsPerComponent)) < 0 ||
	     (code = decode_size = dict_float_array_param(op, "Decode",
				    num_components * 2,
				    &pip->Decode[0], NULL)) < 0 ||
	     (code = dict_bool_param(op, "Interpolate", false,
				    &pip->Interpolate)) < 0
	   )
		return code;
	if ( decode_size == 0 )
		pip->pDecode = 0;
	else if ( decode_size != num_components * 2 )
		return_error(e_rangecheck);
	else
		pip->pDecode = &pip->Decode[0];
	/* Extract and check the data sources. */
	if ( (code = dict_find_string(op, "DataSource", &pds)) < 0 )
		return code;
	if ( pip->MultipleDataSources )
	{	check_type(*pds, t_array);
		if ( r_size(pds) != num_components )
			return_error(e_rangecheck);
		memcpy(&pip->DataSource[0], pds->value.refs, sizeof(ref) * num_components);
	}
	else
		pip->DataSource[0] = *pds;
	return 0;
}

/* (<width> <height> <bits/sample> <matrix> <datasrc> image -) */
/* <dict> image - */
int
z2image(register os_ptr op)
{	if ( level2_enabled && r_has_type(op, t_dictionary) )
	{	image_params ip;
		int code = image_dict_unpack(op, &ip, 12);
		if ( code < 0 )
			return code;
		return zimage_setup(ip.Width, ip.Height, &ip.ImageMatrix,
			&ip.DataSource[0], ip.BitsPerComponent,
			ip.MultipleDataSources, gs_currentcolorspace(igs),
			0, ip.pDecode, 1);
	}
	/* Level 1 image operator */
	check_op(5);
	return zimage(op);
}

/* (<width> <height> <paint_1s> <matrix> <datasrc> imagemask -) */
/* <dict> imagemask - */
int
z2imagemask(register os_ptr op)
{	if ( level2_enabled && r_has_type(op, t_dictionary) )
	{	image_params ip;
		gs_color_space cs;
		int code = image_dict_unpack(op, &ip, 1);
		if ( code < 0 )
			return code;
		if ( ip.MultipleDataSources )
			return_error(e_rangecheck);
		cs.type = &gs_color_space_type_DeviceGray;
		return zimage_setup(ip.Width, ip.Height, &ip.ImageMatrix,
			&ip.DataSource[0], 1, false, &cs, 1, ip.pDecode, 1);
	}
	/* Level 1 imagemask operator */
	check_op(5);
	return zimagemask(op);
}

/* ------ Experimental code ------ */

#include "ialloc.h"

/*
 *	Image rescaling code, based on public domain code from
 *	Graphics Gems III (pp. 414-424), Academic Press, 1992.
 */

/* Mitchell filter definition */
#define Mitchell_support 2.0
#define B (1.0 / 3.0)
#define C (1.0 / 3.0)
private double
Mitchell_filter(double t)
{	double t2 = t * t;

	if ( t < 0 )
	  t = -t;

	if ( t < 1 )
	  return
	    ((12 - 9 * B - 6 * C) * (t * t2) +
	     (-18 + 12 * B + 6 * C) * t2 +
	     (6 - 2 * B)) / 6;
	else if ( t < 2 )
	  return
	    ((-1 * B - 6 * C) * (t * t2) +
	     (6 * B + 30 * C) * t2 +
	     (-12 * B - 48 * C) * t +
	     (8 * B + 24 * C)) / 6;
	else
	  return 0;
}

#define filter_support Mitchell_support
#define filter_proc Mitchell_filter

typedef byte PixelValue;
#define min_PixelValue 0
#define max_PixelValue 255
#define fproc(t) filter_proc(t)
#define fwidth filter_support

typedef struct {
	int	pixel;
	double	weight;
} CONTRIB;

typedef struct {
	int	n;		/* number of contributors */
	CONTRIB	*p;		/* pointer to list of contributions */
} CLIST;

/*
 * The environment provides the following macros:
 *	typedef PixelValue
 *	double fproc(double t)
 *	double fwidth
 *	PixelValue min_PixelValue, max_PixelValue
 */
#define CLAMP(v, mn, mx)\
  (v < mn ? mn : v > mx ? mx : v)

/* Pre-calculate filter contributions for a row or a column. */
private CLIST *
calculate_contrib(double scale, int size, int limit, int stride,
  gs_memory_t *mem)
{
	CLIST *contrib = (CLIST *)gs_alloc_byte_array(mem, size, sizeof(CLIST),
						      "contrib");
	double width, fscale;
	bool squeeze;
	int i, j;

	if ( scale < 1.0 )
	  {	width = fwidth / scale;
		fscale = 1.0 / scale;
		squeeze = true;
	  }
	else
	  {	width = fwidth;
		fscale = 1.0;
		squeeze = false;
	  }
	for ( i = 0; i < size; ++i )
	  {	double center = i / scale;
		int left = (int)ceil(center - width);
		int right = (int)floor(center + width);
		contrib[i].n = 0;
		contrib[i].p = (CONTRIB *)gs_alloc_byte_array(mem, (int) (width * 2 + 1),
				sizeof(CONTRIB), "contrib[i]");
		if ( squeeze )
		  {	for ( j = left; j <= right; ++j )
			  {	double weight =
				  fproc((center - j) / fscale) / fscale;
				int n =
				  (j < 0 ? -j :
				   j >= limit ? (limit - j) + limit - 1 :
				   j);
				int k = contrib[i].n++;
				contrib[i].p[k].pixel = n * stride;
				contrib[i].p[k].weight = weight;
			  }
		  }
		else
		  {	for ( j = left; j <= right; ++j )
			  {	double weight = fproc(center - j);
				int n =
				  (j < 0 ? -j :
				   j >= limit ? (limit - j) + limit - 1 :
				   j);
				int k = contrib[i].n++;
				contrib[i].p[k].pixel = n * stride;
				contrib[i].p[k].weight = weight;
			  }
		  }
	  }
	return contrib;
}


private int
zoom(PixelValue *dst, const PixelValue *src, int dst_width, int dst_height,
  int src_width, int src_height, int values_per_pixel, gs_memory_t *mem)
{
	double				/* zoom scale factors */
	  xscale = (double)dst_width / (double)src_width,
	  yscale = (double)dst_height / (double)src_height;
	PixelValue *tmp;			/* intermediate image */
	int c, i, j, k;			/* loop variables */
	CLIST *contrib = 0;		/* array of contribution lists */
	int code = 0;

	/* create intermediate image to hold horizontal zoom */
	tmp = (PixelValue *)gs_alloc_byte_array(mem,
				dst_width * src_height * values_per_pixel,
				sizeof(PixelValue), "tmp");
	if ( tmp == 0 )
	  {	code = gs_note_error(e_VMerror);
		goto out;
	  }
#define tmp_width dst_width
#define tmp_height src_height

	/* pre-calculate filter contributions for a row */
	contrib = calculate_contrib(xscale, dst_width, src_width,
				    values_per_pixel, mem);
	if ( contrib == 0 )
	  {	code = gs_note_error(e_VMerror);
		goto out;
	  }

	/* apply filter to zoom horizontally from src to tmp */
	for ( c = 0; c < values_per_pixel; ++c )
	  for ( k = 0; k < tmp_height; ++k )
	  {	const PixelValue *raster =
		  &src[k * src_width * values_per_pixel + c];
		for ( i = 0; i < tmp_width; ++i )
		  {	double weight = 0.0;
			for ( j = 0; j < contrib[i].n; ++j )
				weight +=
				  raster[contrib[i].p[j].pixel]
				    * contrib[i].p[j].weight;
			tmp[(k * tmp_width + i) * values_per_pixel + c] =
			  (PixelValue)CLAMP(weight, min_PixelValue, max_PixelValue);
		  }
	  }

	/* free the memory allocated for horizontal filter weights */
	for ( i = dst_width - 1; i >= 0; --i )
	  gs_free_object(mem, contrib[i].p, "contrib[i](x)");
	gs_free_object(mem, contrib, "contrib(x)");

	/* pre-calculate filter contributions for a column */
	contrib = calculate_contrib(yscale, dst_height, tmp_height,
				    dst_width * values_per_pixel, mem);
	if ( contrib == 0 )
	  {	code = gs_note_error(e_VMerror);
		goto out;
	  }

	/* apply filter to zoom vertically from tmp to dst */
	for ( c = 0; c < values_per_pixel; ++c )
	  for ( k = 0; k < dst_width; ++k )
	  {	const PixelValue *raster =
		  &tmp[k * values_per_pixel + c];
		for ( i = 0; i < dst_height; ++i )
		  {	double weight = 0.0;
			for ( j = 0; j < contrib[i].n; ++j )
				weight +=
				  raster[contrib[i].p[j].pixel]
				    * contrib[i].p[j].weight;
			dst[(i * dst_width + k) * values_per_pixel + c] =
			  (PixelValue)CLAMP(weight, min_PixelValue, max_PixelValue);
		  }
	  }

	/* free the memory allocated for vertical filter weights */
out:
	if ( contrib != 0 )
	  {	for ( i = dst_height - 1; i >= 0; --i )
		  gs_free_object(mem, contrib[i].p, "contrib[i](x)");
		gs_free_object(mem, contrib, "contrib(x)");
	  }

	gs_free_object(mem, tmp, "tmp");
	return code;
}

/* <indata> <w> <h> <mat> <outdata> .scaleimage - */
/* Rescale an image represented as an array of bytes (representing values */
/* in the range [0..1]) using filtering. */
private int
zscaleimage(os_ptr op)
{	const ref *rin = op - 4;
	const ref *rout = op;
	int w, h, dw, dh;
	gs_matrix mat;
	int code;
	check_read_type(*rin, t_string);
	check_type(op[-3], t_integer);
	check_type(op[-2], t_integer);
	check_write_type(*rout, t_string);
	if ( op[-3].value.intval <= 0 || op[-2].value.intval <= 0 ||
	     r_size(rin) != op[-3].value.intval * op[-2].value.intval
	   )
	  return_error(e_rangecheck);
	if ( (code = read_matrix(op - 1, &mat)) < 0 )
	  return code;
	if ( mat.xx < 1 || mat.xy != 0 || mat.yx != 0 || mat.yy < 1 )
	  return_error(e_rangecheck);
	w = op[-3].value.intval;
	h = op[-2].value.intval;
	dw = (int)ceil(w * mat.xx);
	dh = (int)ceil(h * mat.yy);
	if ( r_size(rout) != dw * dh )
	  return_error(e_rangecheck);
	code = zoom(rout->value.bytes, rin->value.bytes,
		    dw, dh, w, h, 1, imemory);
	pop(5);
	return 0;
}

/* ------ Initialization procedure ------ */

/* Note that these override the definitions in zpaint.c. */
op_def zimage2_l2_op_defs[] = {
		op_def_begin_level2(),
	{"1image", z2image},
	{"1imagemask", z2imagemask},
	{"5.scaleimage", zscaleimage},
	op_def_end(0)
};
