/* Copyright (C) 1994 Aladdin Enterprises.  All rights reserved.
  
  This file is part of Aladdin Ghostscript.
  
  Aladdin Ghostscript is distributed with NO WARRANTY OF ANY KIND.  No author
  or distributor accepts any responsibility for the consequences of using it,
  or for whether it serves any particular purpose or works at all, unless he
  or she says so in writing.  Refer to the Aladdin Ghostscript Free Public
  License (the "License") for full details.
  
  Every copy of Aladdin Ghostscript must include a copy of the License,
  normally in a plain ASCII text file named PUBLIC.  The License grants you
  the right to copy, modify and redistribute Aladdin Ghostscript, but only
  under certain conditions described in the License.  Among other things, the
  License requires that the copyright notice and this notice be preserved on
  all copies.
*/

/* sjpeg.c */
/* Ghostscript interface routines for IJG code. */
#include "stdio_.h"
#include "string_.h"
#include "jpeglib.h"
#include "jerror.h"
#include "gx.h"
#include "gserrors.h"
#include "strimpl.h"
#include "sdct.h"
#include "sjpeg.h"

/*
 * Error handling routines (these replace corresponding IJG routines from
 * jpeg/jerror.c).  These are used for both compression and decompression.
 * We assume
 * offset_of(jpeg_compress_data, cinfo)==offset_of(jpeg_decompress_data, dinfo)
 */

private void
gs_jpeg_error_exit (j_common_ptr cinfo)
{	jpeg_stream_data *jcomdp =
	  (jpeg_stream_data *)((char *)cinfo -
			       offset_of(jpeg_compress_data, cinfo));
	longjmp(jcomdp->exit_jmpbuf, 1);
}

private void
gs_jpeg_emit_message (j_common_ptr cinfo, int msg_level)
{	if ( msg_level < 0 )
	{	/* GS policy is to ignore IJG warnings when Picky=0,
		 * treat them as errors when Picky=1.
		 */
		jpeg_stream_data *jcomdp =
		  (jpeg_stream_data *)((char *)cinfo -
				       offset_of(jpeg_compress_data, cinfo));
		if ( jcomdp->Picky )
			gs_jpeg_error_exit(cinfo);
	}
	/* Trace messages are always ignored. */
}


/* Stuff the IJG error message into errorinfo after an error exit. */

private int
gs_jpeg_log_error (stream_DCT_state *st)
{	j_common_ptr cinfo = (j_common_ptr) &st->data.compress->cinfo;
	char buffer[JMSG_LENGTH_MAX];
	/* Format the error message */
	(*cinfo->err->format_message)(cinfo, buffer);
	(*st->report_error)((stream_state *)st, buffer);
	return gs_error_ioerror;	/* caller will do return_error() */
}


/*
 * Interface routines.  This layer of routines exists solely to limit
 * side-effects from using setjmp.
 */

int
gs_jpeg_create_compress (stream_DCT_state *st)
{	/* Initialize error handling */
	st->data.compress->cinfo.err = jpeg_std_error(&st->data.common->err);
	st->data.common->err.error_exit = gs_jpeg_error_exit;
	st->data.common->err.emit_message = gs_jpeg_emit_message;
	/* Establish the setjmp return context for gs_jpeg_error_exit to use. */
	if (setjmp(st->data.common->exit_jmpbuf))
		return_error(gs_jpeg_log_error(st));

	jpeg_create_compress(&st->data.compress->cinfo);
	return 0;
}

int
gs_jpeg_create_decompress (stream_DCT_state *st)
{	/* Initialize error handling */
	st->data.decompress->dinfo.err = jpeg_std_error(&st->data.common->err);
	st->data.common->err.error_exit = gs_jpeg_error_exit;
	st->data.common->err.emit_message = gs_jpeg_emit_message;
	/* Establish the setjmp return context for gs_jpeg_error_exit to use. */
	if (setjmp(st->data.common->exit_jmpbuf))
		return_error(gs_jpeg_log_error(st));

	jpeg_create_decompress(&st->data.decompress->dinfo);
	return 0;
}

int
gs_jpeg_set_defaults (stream_DCT_state *st)
{	if (setjmp(st->data.common->exit_jmpbuf))
		return_error(gs_jpeg_log_error(st));
	jpeg_set_defaults(&st->data.compress->cinfo);
	return 0;
}

int
gs_jpeg_set_colorspace (stream_DCT_state *st,
			J_COLOR_SPACE colorspace)
{	if (setjmp(st->data.common->exit_jmpbuf))
		return_error(gs_jpeg_log_error(st));
	jpeg_set_colorspace(&st->data.compress->cinfo, colorspace);
	return 0;
}

int
gs_jpeg_set_linear_quality (stream_DCT_state *st,
			    int scale_factor, boolean force_baseline)
{	if (setjmp(st->data.common->exit_jmpbuf))
		return_error(gs_jpeg_log_error(st));
	jpeg_set_linear_quality(&st->data.compress->cinfo,
				scale_factor, force_baseline);
	return 0;
}

JQUANT_TBL *
gs_jpeg_alloc_quant_table (stream_DCT_state *st)
{	if (setjmp(st->data.common->exit_jmpbuf))
	{	gs_jpeg_log_error(st);
		return NULL;
	}
	return jpeg_alloc_quant_table((j_common_ptr)
				      &st->data.compress->cinfo);
}

JHUFF_TBL *
gs_jpeg_alloc_huff_table (stream_DCT_state *st)
{	if (setjmp(st->data.common->exit_jmpbuf))
	{	gs_jpeg_log_error(st);
		return NULL;
	}
	return jpeg_alloc_huff_table((j_common_ptr)
				     &st->data.compress->cinfo);
}

int
gs_jpeg_start_compress (stream_DCT_state *st,
			boolean write_all_tables)
{	if (setjmp(st->data.common->exit_jmpbuf))
		return_error(gs_jpeg_log_error(st));
	jpeg_start_compress(&st->data.compress->cinfo, write_all_tables);
	return 0;
}

int
gs_jpeg_write_scanlines (stream_DCT_state *st,
			 JSAMPARRAY scanlines,
			 int num_lines)
{	if (setjmp(st->data.common->exit_jmpbuf))
		return_error(gs_jpeg_log_error(st));
	return (int) jpeg_write_scanlines(&st->data.compress->cinfo,
					  scanlines, (JDIMENSION) num_lines);
}

int
gs_jpeg_finish_compress (stream_DCT_state *st)
{	if (setjmp(st->data.common->exit_jmpbuf))
		return_error(gs_jpeg_log_error(st));
	jpeg_finish_compress(&st->data.compress->cinfo);
	return 0;
}

int
gs_jpeg_read_header (stream_DCT_state *st,
		     boolean require_image)
{	if (setjmp(st->data.common->exit_jmpbuf))
		return_error(gs_jpeg_log_error(st));
	return jpeg_read_header(&st->data.decompress->dinfo, require_image);
}

int
gs_jpeg_start_decompress (stream_DCT_state *st)
{	if (setjmp(st->data.common->exit_jmpbuf))
		return_error(gs_jpeg_log_error(st));
	jpeg_start_decompress(&st->data.decompress->dinfo);
	return 0;
}

int
gs_jpeg_read_scanlines (stream_DCT_state *st,
			JSAMPARRAY scanlines,
			int max_lines)
{	if (setjmp(st->data.common->exit_jmpbuf))
		return_error(gs_jpeg_log_error(st));
	return (int) jpeg_read_scanlines(&st->data.decompress->dinfo,
					 scanlines, (JDIMENSION) max_lines);
}

int
gs_jpeg_finish_decompress (stream_DCT_state *st)
{	if (setjmp(st->data.common->exit_jmpbuf))
		return_error(gs_jpeg_log_error(st));
	return (int) jpeg_finish_decompress(&st->data.decompress->dinfo);
}

int
gs_jpeg_destroy (stream_DCT_state *st)
{	if (setjmp(st->data.common->exit_jmpbuf))
		return_error(gs_jpeg_log_error(st));
	jpeg_destroy((j_common_ptr) &st->data.compress->cinfo);
	return 0;
}


/*
 * These routines replace the low-level memory manager of the IJG library.
 * They pass malloc/free calls to the Ghostscript memory manager.
 * Note we do not need these to be declared in any GS header file.
 */

void *
jpeg_get_small (j_common_ptr cinfo, size_t sizeofobject)
{
  return gs_malloc(1, sizeofobject, "JPEG small internal data allocation");
}

void
jpeg_free_small (j_common_ptr cinfo, void * object, size_t sizeofobject)
{
  gs_free(object, 1, sizeofobject, "Freeing JPEG small internal data");
}

void FAR *
jpeg_get_large (j_common_ptr cinfo, size_t sizeofobject)
{
  return gs_malloc(1, sizeofobject, "JPEG large internal data allocation");
}

void
jpeg_free_large (j_common_ptr cinfo, void FAR * object, size_t sizeofobject)
{
  gs_free(object, 1, sizeofobject, "Freeing JPEG large internal data");
}

long
jpeg_mem_available (j_common_ptr cinfo, long min_bytes_needed,
		    long max_bytes_needed, long already_allocated)
{
  return max_bytes_needed;
}

void
jpeg_open_backing_store (j_common_ptr cinfo, void * info,
			 long total_bytes_needed)
{
  ERREXIT(cinfo, JERR_NO_BACKING_STORE);
}

long
jpeg_mem_init (j_common_ptr cinfo)
{
  return 0;			/* just set max_memory_to_use to 0 */
}

void
jpeg_mem_term (j_common_ptr cinfo)
{
  /* no work */
}
