/* Copyright (C) 1989, 1992, 1993 Aladdin Enterprises.  All rights reserved.
  
  This file is part of Aladdin Ghostscript.
  
  Aladdin Ghostscript is distributed with NO WARRANTY OF ANY KIND.  No author
  or distributor accepts any responsibility for the consequences of using it,
  or for whether it serves any particular purpose or works at all, unless he
  or she says so in writing.  Refer to the Aladdin Ghostscript Free Public
  License (the "License") for full details.
  
  Every copy of Aladdin Ghostscript must include a copy of the License,
  normally in a plain ASCII text file named PUBLIC.  The License grants you
  the right to copy, modify and redistribute Aladdin Ghostscript, but only
  under certain conditions described in the License.  Among other things, the
  License requires that the copyright notice and this notice be preserved on
  all copies.
*/

/* gspaint.c */
/* Painting procedures for Ghostscript library */
#include "gx.h"
#include "gpcheck.h"
#include "gserrors.h"
#include "gxfixed.h"
#include "gxmatrix.h"			/* for gs_state */
#include "gspaint.h"
#include "gzpath.h"
#include "gxpaint.h"
#include "gzstate.h"
#include "gxcpath.h"
#include "gxdevmem.h"
#include "gximage.h"

/* Erase the page */
int
gs_erasepage(gs_state *pgs)
{	/* We can't just fill with device white; we must take the */
	/* transfer function into account. */
	int code;
	if ( (code = gs_gsave(pgs)) < 0 )
		return code;
	if ( (code = gs_setgray(pgs, 1.0)) >= 0 )
	{	/* Fill the page directly, ignoring clipping. */
		code = gs_fillpage(pgs);
	}
	gs_grestore(pgs);
	return code;
}

/* Fill the page with the current color. */
int
gs_fillpage(gs_state *pgs)
{	gx_device *dev;
	int code;
	gx_set_dev_color(pgs);
	dev = gs_currentdevice(pgs);
	/* Fill the page directly, ignoring clipping. */
	code = gx_fill_rectangle(0, 0, dev->width, dev->height,
				 pgs->dev_color, pgs);
	if ( code < 0 )
		return code;
	return (*dev_proc(dev, sync_output))(dev);
}

/* Fill using the winding number rule */
int
gs_fill(gs_state *pgs)
{	int code;
	/* If we're inside a charpath, just merge the current path */
	/* into the parent's path. */
	if ( pgs->in_charpath )
		code = gx_path_add_path(pgs->show_gstate->path, pgs->path);
	else
	{	gx_set_dev_color(pgs);
		code = gx_color_load(pgs->dev_color, pgs);
		if ( code < 0 )
			return code;
		code = gx_fill_path(pgs->path, pgs->dev_color, pgs,
				    gx_rule_winding_number, fixed_0);
		if ( !code ) gs_newpath(pgs);
	}
	return code;
}

/* Fill using the even/odd rule */
int
gs_eofill(gs_state *pgs)
{	int code;
	/* If we're inside a charpath, just merge the current path */
	/* into the parent's path. */
	if ( pgs->in_charpath )
		code = gx_path_add_path(pgs->show_gstate->path, pgs->path);
	else
	{	gx_set_dev_color(pgs);
		code = gx_color_load(pgs->dev_color, pgs);
		if ( code < 0 )
			return code;
		code = gx_fill_path(pgs->path, pgs->dev_color, pgs,
				    gx_rule_even_odd, fixed_0);
		if ( !code ) gs_newpath(pgs);
	}
	return code;
}

/* Stroke the current path */
int
gs_stroke(gs_state *pgs)
{	int code;
	/* If we're inside a charpath, just merge the current path */
	/* into the parent's path. */
	if ( pgs->in_charpath )
		code = gx_path_add_path(pgs->show_gstate->path, pgs->path);
	else
	{	gx_set_dev_color(pgs);
		code = gx_color_load(pgs->dev_color, pgs);
		if ( code < 0 )
			return code;
		code = gx_stroke_fill(pgs->path, pgs);
		if ( !code ) gs_newpath(pgs);
	}
	return code;
}

/* Compute the stroked outline of the current path */
int
gs_strokepath(gs_state *pgs)
{	gx_path spath;
	int code;
	gx_path_init(&spath, pgs->memory);
	code = gx_stroke_add(pgs->path, &spath, pgs);
	if ( code < 0 ) return code;
	gx_path_release(pgs->path);
	*pgs->path = spath;
	return 0;
}
