/* Copyright (C) 1989, 1992 Aladdin Enterprises.  All rights reserved.
  
  This file is part of Aladdin Ghostscript.
  
  Aladdin Ghostscript is distributed with NO WARRANTY OF ANY KIND.  No author
  or distributor accepts any responsibility for the consequences of using it,
  or for whether it serves any particular purpose or works at all, unless he
  or she says so in writing.  Refer to the Aladdin Ghostscript Free Public
  License (the "License") for full details.
  
  Every copy of Aladdin Ghostscript must include a copy of the License,
  normally in a plain ASCII text file named PUBLIC.  The License grants you
  the right to copy, modify and redistribute Aladdin Ghostscript, but only
  under certain conditions described in the License.  Among other things, the
  License requires that the copyright notice and this notice be preserved on
  all copies.
*/

/* gdevmem1.c */
/* Generic and monobit "memory" (stored bitmap) device */
/* for Ghostscript library. */
#include "memory_.h"
#include "gx.h"
#include "gserrors.h"
#include "gsstruct.h"
#include "gxdevice.h"
#include "gxdevmem.h"			/* semi-public definitions */
#include "gdevmem.h"			/* private definitions */

/* Define masks for little-endian operation. */
#if !arch_is_big_endian
const bits16 gdev_mem_swapped_left_masks[17] = {
	0xffff, 0xff7f, 0xff3f, 0xff1f, 0xff0f, 0xff07, 0xff03, 0xff01,
	0xff00, 0x7f00, 0x3f00, 0x1f00, 0x0f00, 0x0700, 0x0300, 0x0100,
	0x0000
};
#endif

/* Structure descriptor */
public_st_device_memory();

/* GC procedures */
#define mptr ((gx_device_memory *)vptr)
private ENUM_PTRS_BEGIN(device_memory_enum_ptrs) {
	return (*st_device_forward.enum_ptrs)(vptr, sizeof(gx_device_forward), index-2, pep);
	}
	case 0:
	  *pep = (mptr->foreign_bits ? NULL : (void *)mptr->base);
	  break;
	ENUM_STRING_PTR(1, gx_device_memory, palette);
ENUM_PTRS_END
private RELOC_PTRS_BEGIN(device_memory_reloc_ptrs) {
	if ( !mptr->foreign_bits )
	{	byte *base_old = mptr->base;
		uint reloc;
		int y;
		RELOC_PTR(gx_device_memory, base);
		reloc = base_old - mptr->base;
		for ( y = 0; y < mptr->height; y++ )
		  mptr->line_ptrs[y] -= reloc;
	}
	RELOC_STRING_PTR(gx_device_memory, palette);
	(*st_device_forward.reloc_ptrs)(vptr, sizeof(gx_device_forward), gcst);
} RELOC_PTRS_END
#undef mptr

/* ------ Generic code ------ */

/* Return the appropriate memory device for a given */
/* number of bits per pixel (0 if none suitable). */
const gx_device_memory *
gdev_mem_device_for_bits(int bits_per_pixel)
{	switch ( bits_per_pixel )
	   {
	case 1: return &mem_mono_device;
	case 2: return &mem_mapped2_color_device;
	case 4: return &mem_mapped4_color_device;
	case 8: return &mem_mapped8_color_device;
	case 16: return &mem_true16_color_device;
	case 24: return &mem_true24_color_device;
	case 32: return &mem_true32_color_device;
	default: return 0;
	   }
}

/* Make a memory device. */
void
gs_make_mem_device(gx_device_memory *dev, const gx_device_memory *cdev,
  gs_memory_t *mem, int page_device)
{	*dev = *cdev;
	dev->memory = mem;
	dev->procs = &dev->std_procs;
	switch ( page_device )
	  {
	  case -1:
	    dev->std_procs.get_page_device = gx_default_get_page_device;
	    break;
	  case 1:
	    dev->std_procs.get_page_device = gx_page_device_get_page_device;
	    break;
	  }
}
/* Make a monobit memory device.  This is never a page device. */
void
gs_make_mem_mono_device(gx_device_memory *dev, gs_memory_t *mem)
{	gs_make_mem_device(dev, &mem_mono_device, mem, -1);
}

/* Compute the size of the bitmap storage, */
/* including the space for the scan line pointer table. */
/* Note that scan lines are padded to a multiple of align_bitmap_mod bytes, */
/* and additional padding may be needed if the pointer table */
/* must be aligned to an even larger modulus. */
private ulong
mem_bitmap_bits_size(const gx_device_memory *dev)
{	return round_up((ulong)dev->height * gdev_mem_raster(dev),
			max(align_bitmap_mod, arch_align_ptr_mod));
}
ulong
gdev_mem_bitmap_size(const gx_device_memory *dev)
{        return mem_bitmap_bits_size(dev) +
		(ulong)dev->height * sizeof(byte *);
}

/* Open a memory device, allocating the data area if appropriate, */
/* and create the scan line table. */
int
mem_open(gx_device *dev)
{	byte *scan_line;
	uint raster = mdev->raster = gdev_mem_raster(mdev);
	byte **pptr;
	byte **pend;
	if ( mdev->bitmap_memory != 0 )
	{	/* Allocate the data now. */
		ulong size = gdev_mem_bitmap_size(mdev);
		if ( (uint)size != size )
			return_error(gs_error_limitcheck);
		mdev->base = gs_alloc_bytes(mdev->bitmap_memory, (uint)size,
					    "mem_open");
		if ( mdev->base == 0 )
			return_error(gs_error_VMerror);
		mdev->foreign_bits = false;
	}
	scan_line = mdev->base;
	pptr = (byte **)byte_ptr_add(scan_line, mem_bitmap_bits_size(mdev));
	pend = pptr + dev->height;
	mdev->line_ptrs = pptr;
	while ( pptr < pend )
	   {	*pptr++ = scan_line;
		scan_line = byte_ptr_add(scan_line, raster);
	   }
	return 0;
}

/* Return the initial transformation matrix */
void
mem_get_initial_matrix(gx_device *dev, gs_matrix *pmat)
{	pmat->xx = mdev->initial_matrix.xx;
	pmat->xy = mdev->initial_matrix.xy;
	pmat->yx = mdev->initial_matrix.yx;
	pmat->yy = mdev->initial_matrix.yy;
	pmat->tx = mdev->initial_matrix.tx;
	pmat->ty = mdev->initial_matrix.ty;
}

/* Test whether a device is a memory device */
bool
gs_device_is_memory(const gx_device *dev)
{	/* We can't just compare the procs, or even an individual proc, */
	/* because we might be tracing.  Compare the device name, */
	/* and hope for the best. */
	const char *name = dev->dname;
	int i;
	for ( i = 0; i < 6; i++ )
	  if ( name[i] != "image("[i] ) return false;
	return true;
}

/* Close a memory device, freeing the data area if appropriate. */
int
mem_close(gx_device *dev)
{	if ( mdev->bitmap_memory != 0 )
	  gs_free_object(mdev->bitmap_memory, mdev->base, "mem_close");
	return 0;
}

/* Copy a scan line to a client. */
#undef chunk
#define chunk byte
int
mem_get_bits(gx_device *dev, int y, byte *str, byte **actual_data)
{	byte *src;
	if ( y < 0 || y >= dev->height )
		return_error(gs_error_rangecheck);
	src = scan_line_base(mdev, y);
	if ( actual_data == 0 )
		memcpy(str, src, gx_device_raster(dev, 0));
	else
		*actual_data = src;
	return 0;
}

/* ------ Monochrome ------ */

/* Procedures */
private dev_proc_copy_mono(mem_mono_copy_mono);
private dev_proc_fill_rectangle(mem_mono_fill_rectangle);

/* The device descriptor. */
/* The instance is public. */
const gx_device_memory far_data mem_mono_device =
  mem_device("image(mono)", 0, 1,
    gx_default_map_rgb_color, gx_default_map_color_rgb,
    mem_mono_copy_mono, gx_default_copy_color, mem_mono_fill_rectangle);

/* Convert x coordinate to byte offset in scan line. */
#define x_to_byte(x) ((x) >> 3)

/* Fill a rectangle with a color. */
#undef chunk
#define chunk mono_chunk
private int
mem_mono_fill_rectangle(gx_device *dev, int x, int y, int w, int h,
  gx_color_index color)
{	uint bit;
	chunk right_mask;
	byte fill;
	declare_scan_ptr(dest);
	fit_fill(dev, x, y, w, h);
	setup_rect(dest);
#define write_loop(stat)\
 { int line_count = h;\
   chunk *ptr = dest;\
   do { stat; inc_chunk_ptr(ptr, draster); }\
   while ( --line_count );\
 }
#define write_partial(msk)\
   if ( fill ) write_loop(*ptr |= msk)\
   else write_loop(*ptr &= ~msk)
	switch ( color )
	   {
	case 0: fill = mdev->invert; break;
	case 1: fill = ~mdev->invert; break;
	case gx_no_color_index: return 0;		/* transparent */
	default: return -1;		/* invalid */
	   }
	bit = x & chunk_align_bit_mask;
	if ( bit + w <= chunk_bits )
	   {	/* Only one word. */
		set_mono_thin_mask(right_mask, w, bit);
	   }
	else
	   {	int byte_count;
		if ( bit )
		   {	chunk mask;
			set_mono_left_mask(mask, bit);
			write_partial(mask);
			dest++;
			w += bit - chunk_bits;
		   }
		set_mono_right_mask(right_mask, w & chunk_bit_mask);
		if ( (byte_count = (w >> 3) & -chunk_bytes) != 0 )
		  switch ( byte_count )
		{
		case chunk_bytes:
		{	chunk cfill = -(chunk)(fill & 1);
			write_loop(*ptr = cfill);
			inc_chunk_ptr(dest, chunk_bytes);
		}	break;
		case chunk_bytes * 2:
		{	chunk cfill = -(chunk)(fill & 1);
#define set2(p, v) (*p = v, p[1] = v)
			write_loop(set2(ptr, cfill));
#undef set2
			inc_chunk_ptr(dest, chunk_bytes * 2);
		}	break;
		default:
			write_loop(memset(ptr, fill, byte_count));
			inc_chunk_ptr(dest, byte_count);
			break;
		}
	   }
	if ( right_mask )
		write_partial(right_mask);
	return 0;
}

/* Copy a monochrome bitmap. */

/* Fetch a chunk from the source. */
/* The source data are always stored big-endian. */
/* Note that the macros always cast cptr, */
/* so it doesn't matter what the type of cptr is. */
/* cshift = chunk_bits - shift. */
#undef chunk
#if arch_is_big_endian
#  define chunk uint
#  define cfetch_right(cptr, shift, cshift)\
	(cfetch_aligned(cptr) >> shift)
#  define cfetch_left(cptr, shift, cshift)\
	(cfetch_aligned(cptr) << shift)
/* Fetch a chunk that straddles a chunk boundary. */
#  define cfetch2(cptr, cskew, skew)\
    (cfetch_left(cptr, cskew, skew) +\
     cfetch_right((chunk *)(cptr) + 1, skew, cskew))
#else				/* little-endian */
#  define chunk bits16
private const bits16 right_masks2[9] = {
	0xffff, 0x7f7f, 0x3f3f, 0x1f1f, 0x0f0f, 0x0707, 0x0303, 0x0101, 0x0000
};
private const bits16 left_masks2[9] = {
	0xffff, 0xfefe, 0xfcfc, 0xf8f8, 0xf0f0, 0xe0e0, 0xc0c0, 0x8080, 0x0000
};
#  define ccont(cptr, off) (((chunk *)(cptr))[off])
#  define cfetch_right(cptr, shift, cshift)\
	((shift) < 8 ?\
	 ((ccont(cptr, 0) >> (shift)) & right_masks2[shift]) +\
	  (ccont(cptr, 0) << (cshift)) :\
	 ((chunk)*(byte *)(cptr) << (cshift)) & 0xff00)
#  define cfetch_left(cptr, shift, cshift)\
	((shift) < 8 ?\
	 ((ccont(cptr, 0) << (shift)) & left_masks2[shift]) +\
	  (ccont(cptr, 0) >> (cshift)) :\
	 ((ccont(cptr, 0) & 0xff00) >> (cshift)) & 0xff)
/* Fetch a chunk that straddles a chunk boundary. */
/* We can avoid testing the shift amount twice */
/* by expanding the cfetch_left/right macros in-line. */
#  define cfetch2(cptr, cskew, skew)\
	((cskew) < 8 ?\
	 ((ccont(cptr, 0) << (cskew)) & left_masks2[cskew]) +\
	  (ccont(cptr, 0) >> (skew)) +\
	  (((chunk)(((byte *)(cptr))[2]) << (cskew)) & 0xff00) :\
	 (((ccont(cptr, 0) & 0xff00) >> (skew)) & 0xff) +\
	  ((ccont(cptr, 1) >> (skew)) & right_masks2[skew]) +\
	   (ccont(cptr, 1) << (cskew)))
#endif
/* Since source and destination are both always big-endian, */
/* fetching an aligned chunk never requires byte swapping. */
#  define cfetch_aligned(cptr)\
	(*(chunk *)(cptr))

/* copy_function and copy_shift get added together for dispatch */
typedef enum {
	copy_or = 0, copy_store, copy_and, copy_funny
} copy_function;
/* copy_right/left is not an enum, because compilers complain about */
/* an enumeration clash when these are added to a copy_function. */
#define copy_right ((copy_function)0)
#define copy_left ((copy_function)4)
typedef struct {
	short invert;
	ushort op;			/* copy_function */
} copy_mode;
/* Map from <c0,c1,invert> to copy_mode. */
#define cm(i,op) { i, (ushort)op }
private copy_mode copy_modes[9*2] = {
	cm(-1, copy_funny),		/* NN */
	cm(-1, copy_and),		/* N0 */
	cm(0, copy_or),			/* N1 */
	cm(0, copy_and),		/* 0N */
	cm(0, copy_funny),		/* 00 */
	cm(0, copy_store),		/* 01 */
	cm(-1, copy_or),		/* 1N */
	cm(-1, copy_store),		/* 10 */
	cm(0, copy_funny),		/* 11 */
	cm(-1, copy_funny),		/* NNi */
	cm(0, copy_or),			/* N1i */
	cm(-1, copy_and),		/* N0i */
	cm(-1, copy_or),		/* 1Ni */
	cm(0, copy_funny),		/* 11i */
	cm(-1, copy_store),		/* 10i */
	cm(0, copy_and),		/* 0Ni */
	cm(0, copy_store),		/* 01i */
	cm(0, copy_funny)		/* 00i */
};
private int
mem_mono_copy_mono(gx_device *dev,
  const byte *base, int sourcex, int sraster, gx_bitmap_id id,
  int x, int y, int w, int h, gx_color_index zero, gx_color_index one)
{	register const byte *bptr;		/* actually chunk * */
	int dbit, wleft;
	uint mask;
	copy_mode mode;
#define function (copy_function)(mode.op)
	declare_scan_ptr_as(dbptr, byte *);
#define optr ((chunk *)dbptr)
	register int skew;
	register uint invert;
	fit_copy(dev, base, sourcex, sraster, id, x, y, w, h);
#if gx_no_color_index_value != -1		/* hokey! */
	if ( zero == gx_no_color_index ) zero = -1;
	if ( one == gx_no_color_index ) one = -1;
#endif
#define izero (int)zero
#define ione (int)one
	mode =
	  copy_modes[(mdev->invert & 9) + izero + izero + izero + ione + 4];
#undef izero
#undef ione
	invert = (uint)(int)mode.invert;	/* load register */
	setup_rect_as(dbptr, byte *);
	bptr = base + ((sourcex & ~chunk_align_bit_mask) >> 3);
	dbit = x & chunk_align_bit_mask;
	skew = dbit - (sourcex & chunk_align_bit_mask);
/* Macros for writing partial chunks. */
/* The destination pointer is always named optr, */
/* and must be declared as chunk *. */
/* cinvert may be temporarily redefined. */
#define cinvert(bits) ((bits) ^ invert)
#define write_or_masked(bits, mask, off)\
  optr[off] |= (cinvert(bits) & mask)
#define write_store_masked(bits, mask, off)\
  optr[off] = ((optr[off] & ~mask) | (cinvert(bits) & mask))
#define write_and_masked(bits, mask, off)\
  optr[off] &= (cinvert(bits) | ~mask)
/* Macros for writing full chunks. */
#define write_or(bits)  *optr |= cinvert(bits)
#define write_store(bits) *optr = cinvert(bits)
#define write_and(bits) *optr &= cinvert(bits)
/* Macro for incrementing to next chunk. */
#define next_x_chunk\
  bptr += chunk_bytes; dbptr += chunk_bytes
/* Common macro for the end of each scan line. */
#define end_y_loop(sdelta, ddelta)\
  if ( --h == 0 ) break;\
  bptr += sdelta; dbptr += ddelta
	if ( (wleft = w + dbit - chunk_bits) <= 0 )
	   {	/* The entire operation fits in one (destination) chunk. */
		set_mono_thin_mask(mask, w, dbit);
#define write_single(wr_op, src)\
  for ( ; ; )\
   { wr_op(src, mask, 0);\
     end_y_loop(sraster, draster);\
   }
#define write1_loop(src)\
  switch ( function ) {\
    case copy_or: write_single(write_or_masked, src); break;\
    case copy_store: write_single(write_store_masked, src); break;\
    case copy_and: write_single(write_and_masked, src); break;\
    default: goto funny;\
  }
		if ( skew >= 0 )	/* single -> single, right/no shift */
		{	if ( skew == 0 )	/* no shift */
			{	write1_loop(cfetch_aligned(bptr));
			}
			else			/* right shift */
			{	int cskew = chunk_bits - skew;
				write1_loop(cfetch_right(bptr, skew, cskew));
			}
		}
		else if ( wleft <= skew )	/* single -> single, left shift */
		{	int cskew = chunk_bits + skew;
			skew = -skew;
			write1_loop(cfetch_left(bptr, skew, cskew));
		}
		else			/* double -> single */
		{	int cskew = -skew;
			skew += chunk_bits;
			write1_loop(cfetch2(bptr, cskew, skew));
		}
#undef write1_loop
#undef write_single
	   }
	else if ( wleft <= skew )
	   {	/* 1 source chunk -> 2 destination chunks. */
		/* This is an important special case for */
		/* both characters and halftone tiles. */
		register uint bits;
		uint rmask;
		int cskew = chunk_bits - skew;
		set_mono_left_mask(mask, dbit);
		set_mono_right_mask(rmask, wleft);
#undef cinvert
#define cinvert(bits) (bits)		/* pre-inverted here */
#if arch_is_big_endian			/* no byte swapping */
#  define write_1to2(wr_op)\
  for ( ; ; )\
   { bits = cfetch_aligned(bptr) ^ invert;\
     wr_op(bits >> skew, mask, 0);\
     wr_op(bits << cskew, rmask, 1);\
     end_y_loop(sraster, draster);\
   }
#else					/* byte swapping */
#  define write_1to2(wr_op)\
  for ( ; ; )\
   { wr_op(cfetch_right(bptr, skew, cskew) ^ invert, mask, 0);\
     wr_op(cfetch_left(bptr, cskew, skew) ^ invert, rmask, 1);\
     end_y_loop(sraster, draster);\
   }
#endif
		switch ( function )
		   {
		case copy_or: write_1to2(write_or_masked); break;
		case copy_store: write_1to2(write_store_masked); break;
		case copy_and: write_1to2(write_and_masked); break;
		default: goto funny;
		   }
#undef cinvert
#define cinvert(bits) ((bits) ^ invert)
#undef write_1to2
	   }
	else
	   {	/* More than one source chunk and more than one */
		/* destination chunk are involved. */
		uint rmask;
		int words = (wleft & ~chunk_bit_mask) >> 3;
		uint sskip = sraster - words;
		uint dskip = draster - words;
		register uint bits;
		set_mono_left_mask(mask, dbit);
		set_mono_right_mask(rmask, wleft & chunk_bit_mask);
		if ( skew == 0 )	/* optimize the aligned case */
		   {
#define write_aligned(wr_op, wr_op_masked)\
  for ( ; ; )\
   { int count = wleft;\
     /* Do first partial chunk. */\
     wr_op_masked(cfetch_aligned(bptr), mask, 0);\
     /* Do full chunks. */\
     while ( (count -= chunk_bits) >= 0 )\
      { next_x_chunk; wr_op(cfetch_aligned(bptr)); }\
     /* Do last chunk */\
     if ( count > -chunk_bits )\
      { wr_op_masked(cfetch_aligned(bptr + chunk_bytes), rmask, 1); }\
     end_y_loop(sskip, dskip);\
   }
			switch ( function )
			  {
			  case copy_or:
			    write_aligned(write_or, write_or_masked);
			    break;
			  case copy_store:
			    write_aligned(write_store, write_store_masked);
			    break;
			  case copy_and:
			    write_aligned(write_and, write_and_masked);
			    break;
			  default:
			    goto funny;
			  }
#undef write_aligned
		   }
		else			/* not aligned */
		   {	int ccase =
			  (skew >= 0 ? copy_right :
			   ((bptr += chunk_bytes), copy_left))
			  + (int)function;
			int cskew = -skew & chunk_bit_mask;
			skew &= chunk_bit_mask;
			for ( ; ; )
			   {	int count = wleft;
#define prefetch_right\
  bits = cfetch_right(bptr, skew, cskew)
#define prefetch_left\
  bits = cfetch2(bptr - chunk_bytes, cskew, skew)
#define write_unaligned(wr_op, wr_op_masked)\
  wr_op_masked(bits, mask, 0);\
  /* Do full chunks. */\
  while ( count >= chunk_bits )\
    { bits = cfetch2(bptr, cskew, skew);\
      next_x_chunk; wr_op(bits); count -= chunk_bits;\
    }\
  /* Do last chunk */\
  if ( count > 0 )\
    { bits = cfetch_left(bptr, cskew, skew);\
      if ( count > skew ) bits += cfetch_right(bptr + chunk_bytes, skew, cskew);\
      wr_op_masked(bits, rmask, 1);\
    }
				switch ( ccase )
				  {
				  case copy_or + copy_left:
				    prefetch_left; goto uor;
				  case copy_or + copy_right:
				    prefetch_right;
uor:				    write_unaligned(write_or, write_or_masked);
				    break;
				  case copy_store + copy_left:
				    prefetch_left; goto ustore;
				  case copy_store + copy_right:
				    prefetch_right;
ustore:				    write_unaligned(write_store, write_store_masked);
				    break;
				  case copy_and + copy_left:
				    prefetch_left; goto uand;
				  case copy_and + copy_right:
				    prefetch_right;
uand:				    write_unaligned(write_and, write_and_masked);
				    break;
				  default:
				    goto funny;
				  }
				end_y_loop(sskip, dskip);
#undef write_unaligned
#undef prefetch_left
#undef prefetch_right
			   }
		   }
	   }
#undef end_y_loop
#undef next_x_chunk
	return 0;
	/* Handle the funny cases that aren't supposed to happen. */
funny:	return (invert ? -1 : mem_mono_fill_rectangle(dev, x, y, w, h, zero));
#undef optr
}
