
{*******************************************************}
{                                                       }
{       WinG Planet Sample                              }
{                                                       }
{       Written by Mike Scott, CIS 100140,2420          }
{       You'll need to install WinG to run this         }
{       It's available on CompuServe, GO WINMM          }
{                                                       }
{*******************************************************}

program Planet ;

uses WinTypes, WinProcs, Objects, OWindows, ODialogs, Strings, WinG ;

{$R planet.res}

const
  xSize = 15 ;
  ySize = 15 ;
  nEntries = xSize * ySize ;
  StepSize = ySize ;
  cm_about = 100 ;

type
  PBytes = ^TBytes ;
  TBytes = array[ 0..65520 ] of byte ;
  TPalEntries = array[ 0..nEntries - 1 ] of TPaletteEntry ;


{ TMainWindow }

type
  PMainWindow = ^TMainWindow ;
  TMainWindow = object( TWindow )
    constructor Init( AParent : PWindowsObject ;
                      AName   : PChar ) ;
    destructor Done ; virtual ;
    function  GetClassName : PChar ; virtual ;
    procedure GetWindowClass( var WndClass : TWndClass ) ; virtual ;
    procedure SetupWindow ; virtual ;
    procedure wmQueryNewPalette( var Msg : TMessage ) ; virtual wm_First + wm_QueryNewPalette ;
    procedure wmActivateApp( var Msg : TMessage ) ; virtual wm_First + wm_ActivateApp ;
    procedure DefCommandProc( var Msg : TMessage ) ; virtual ;
    procedure PaintPlanet( DC    : HDC ;
                           ARect : TRect ) ;
    procedure Paint( DC : HDC ;
                     var PaintInfo : TPaintStruct ) ; virtual ;
    procedure IdleAction ;
    procedure wmTimer( var Msg : TMessage ) ; virtual wm_First + wm_Timer ;
    procedure cmAbout( var Msg : TMessage ) ; virtual cm_First + cm_About ;
  private
    Palette      : HPalette ;
    PaletteEntries : TPalEntries ;
    ColourCycle    : TPalEntries ;
    Direction      : array[ 0..nEntries - 1 ] of boolean ;
    CurrentRGB     : array[ 0..nEntries ] of 1..3 ;
    WinGBitmap     : HBitmap ;
    BitmapBits     : PBytes ;
    Width          : integer ;
    ScanWidth      : integer ;
    Height         : integer ;
    xMessagePos    : integer ;
    AppIsActive    : boolean ;
    PlanetsAcross  : byte ;
    FontNumber     : word ;
    procedure GetScrollingBits ;
    function  PixelAt( x, y : word ) : byte ;
    procedure SetPaletteEntry( x, y   : word ;
                               R, G, B : byte ) ;
    procedure ScrollMessage ;
  end ;


constructor TMainWindow.Init( AParent : PWindowsObject ;
                              AName   : PChar ) ;

var PaletteBytes : word ;
    LogPalette   : PLogPalette ;
    i            : integer ;
    n, StepCount : byte ;

begin
  inherited Init( AParent, AName ) ;
  WinGBitmap := 0 ;
  BitmapBits := nil ;
  Width := 0 ;
  ScanWidth := 0 ;
  Height := 0 ;
  xMessagePos := -xSize ;
  AppIsActive := false ;
  PlanetsAcross := 1 ;
  FontNumber := 201 ;
  with Attr do begin
    x := 8 ;
    y := 300 ;
    w := 240 ;
    h := 240 ;
    Menu := LoadMenu( HInstance, 'AppMenu' ) ;
    CheckMenuItem( Menu, PlanetsAcross, MF_BYCOMMAND or MF_CHECKED ) ;
    CheckMenuItem( Attr.Menu, FontNumber, MF_BYCOMMAND or MF_CHECKED ) ;
  end ;

  { create a palette }
  Palette := 0 ;
  PaletteBytes := sizeof( TLogPalette ) +
                  sizeof( TPaletteEntry ) * ( nEntries - 1 ) ;
  GetMem( LogPalette, PaletteBytes ) ;
  if LogPalette <> nil then with LogPalette^ do begin
    palVersion := $300 ;
    palNumEntries := nEntries ;
    StepCount := StepSize ;
    n := StepSize ;
    for i := 0 to nEntries - 1 do with palPalEntry[ i ] do begin
      peRed := n ;
      dec( StepCount ) ;
      if StepCount = 0 then begin
        inc( n, StepSize ) ;
        StepCount := StepSize ;
      end ;
      peGreen := 0 ;
      peBlue := 0 ;
      peFlags := PC_RESERVED ;

      { initialse the increment direction and colour component }
      Direction[ i ] := false ;
      CurrentRGB[ i ] := 1 ;
    end ;
    Move( palPalEntry, ColourCycle, sizeof( TPaletteEntry ) * nEntries ) ;
    Palette := CreatePalette( LogPalette^ ) ;
    FreeMem( LogPalette, PaletteBytes ) ;
  end ;
end ;


destructor TMainWindow.Done ;

begin
  inherited Done ;
  if Palette <> 0 then DeleteObject( Palette ) ;
  if WinGBitmap <> 0 then DeleteObject( WinGBitmap ) ;
  KillTimer( HWindow, 1 ) ;
end ;


function  TMainWindow.GetClassName : PChar ;

begin
  GetClassName := 'WinGPlanet' ;
end ;


procedure TMainWindow.GetWindowClass( var WndClass : TWndClass ) ;

begin
  inherited GetWindowClass( WndClass ) ;
  WndClass.hbrBackground := GetStockObject( BLACK_BRUSH ) ;
  WndClass.hIcon := 0 ;
end ;


procedure TMainWindow.SetupWindow ;

begin
  inherited SetupWindow ;
  GetScrollingBits ;
end ;


procedure TMainWindow.GetScrollingBits ;

const AMessage = 'Welcome to the world of WinG...    ' +
                 'demo written by Mike Scott    ' +
                 'CIS 100140,2420...    in Borland Pascal...          ' ;

var BitmapInfo : PBitmapInfo ;
    Len : integer ;
    WinGDC  : HDC ;
    DC      : HDC ;
    Extent : longint ;
    OldBitmap : HBitmap ;
    i         : integer ;
    TempValue : byte ;
    Font      : HFont ;
    OldFont   : HFont ;
    LogFont   : TLogFont ;
    Name      : PChar ;
    HeaderAndPalette : record
      Header : TBitmapInfoHeader ;
      aColorTable : array[ 0..255 ] of TRGBQuad ;
    end ;

begin
  { dispose of any previous bits }
  if WinGBitmap <> 0 then begin
    BitmapBits := nil ;
    DeleteObject( WinGBitmap ) ;
    WinGBitmap := 0 ;
  end ;

  { get a WinG DC which we'll use to get the pixel bitmap for 'scrolling' }
  WinGDC := WinGCreateDC ;
  if WinGDC <> 0 then begin

    { fill in the DIB header with the recommended format }
    WinGRecommendDIBFormat( PBitmapInfo( @HeaderAndPalette.Header ) ) ;

    { create a nice font }
    FillChar( LogFont, sizeof( LogFont ), 0 ) ;
    with LogFont do begin
      lfHeight := -16 ;
      case FontNumber of
        202 : Name := 'System' ;
        203 : Name := 'Times New Roman' ;
        204 : Name := 'Courier New' ;
        else Name := 'Arial' ;
      end ;
      StrCopy( lfFaceName, Name ) ;
    end ;
    OldFont := SelectObject( WinGDC, CreateFontIndirect( LogFont ) ) ;

    { get the width & height of the text }
    Len := StrLen( AMessage ) ;
    Extent := GetTextExtent( WinGDC, AMessage, Len ) ;

    { move into the header and get a WinG bitmap of this size }
    with HeaderAndPalette.Header do begin
      Width := LoWord( Extent ) ;
      Height := HiWord( Extent ) ;
      if biWidth < 0 then biWidth := -Width else biWidth := Width ;
      biHeight := Height ;

      { round up scan width to a multiple of 4 bytes as per DIB spec }
      ScanWidth := ( Width + 3 ) and not 3 ;
    end ;

    { create a colour table from the system's }
    DC := GetDC( HWindow ) ;
    GetSystemPaletteEntries( DC, 0, 256, HeaderAndPalette.aColorTable ) ;
    for i := 0 to 255 do with HeaderAndPalette.aColorTable[ i ] do begin
      TempValue := rgbBlue ;
      rgbBlue := rgbRed ;
      rgbRed := TempValue ;
    end ;
    ReleaseDC( HWindow, DC ) ;

    { create the bitmap }
    WinGBitmap := WinGCreateBitmap( WinGDC,
                                    PBitmapInfo( @HeaderAndPalette.Header ),
                                    @pointer( BitmapBits ) ) ;

    { if the bitmap was OK then select it and paint the text }
    if WinGBitmap <> 0 then begin
      OldBitmap := SelectObject( WinGDC, WingBitmap ) ;
      SetTextColor( WinGDC, $01000001 ) ;
      SetBkColor( WinGDC, $01000000 ) ;
      TextOut( WinGDC, 0, 0, AMessage, Len ) ;
      SelectObject( WinGDC, OldBitmap ) ;
    end else MessageBox( HWindow, 'Unable to create a WinG bitmap!',
                         Application^.Name, mb_IconStop or mb_OK ) ;

    { we don't need the WinGDC any more }
    DeleteObject( SelectObject( WinGDC, OldFont ) ) ;
    DeleteDC( WinGDC ) ;
  end else MessageBox( HWindow, 'Unable to get a WinG DC!', Application^.Name,
                       mb_IconStop or mb_OK ) ;

  { if we got a bitmap OK then set the scroll timer else shut down }
  if WinGBitmap <> 0 then SetTimer( HWindow, 1, 20, nil ) else CloseWindow ;
end ;


procedure TMainWindow.wmQueryNewPalette( var Msg : TMessage ) ;

begin
  DefWndProc( Msg ) ;
end ;


procedure TMainWindow.wmActivateApp( var Msg : TMessage ) ;

begin
  DefWndProc( Msg ) ;
  AppIsActive := Msg.wParam <> 0 ;
end ;


procedure TMainWindow.DefCommandProc( var Msg : TMessage ) ;

begin
  inherited DefCommandProc( Msg ) ;

  { check if command was from the planet menu }
  with Msg do if LParam = 0 then begin
    if ( 1 <= WParam ) and ( WParam <= 99 ) then begin
      CheckMenuItem( Attr.Menu, PlanetsAcross, MF_BYCOMMAND or MF_UNCHECKED ) ;
      PlanetsAcross := WParam ;
      CheckMenuItem( Attr.Menu, PlanetsAcross, MF_BYCOMMAND or MF_CHECKED ) ;
      InvalidateRect( HWindow, nil, true ) ;
    end else
    if ( 201 <= WParam ) and ( WParam <= 299 ) then begin
      CheckMenuItem( Attr.Menu, FontNumber, MF_BYCOMMAND or MF_UNCHECKED ) ;
      FontNumber := WParam ;
      CheckMenuItem( Attr.Menu, FontNumber, MF_BYCOMMAND or MF_CHECKED ) ;
      GetScrollingBits ;
      InvalidateRect( HWindow, nil, true ) ;
    end ;
  end ;
end ;


procedure TMainWindow.PaintPlanet( DC    : HDC ;
                                   ARect : TRect ) ;

const Angle = PI / xSize ;

var w, h            : integer ;
    OldBrush        : HBrush ;
    i, j            : integer ;
    ABrush          : HBrush ;
    AResult         : integer ;
    Elliptic1       : HRgn ;
    Elliptic2       : HRgn ;
    Slice           : HRgn ;
    EllipticSlice1  : HRgn ;
    EllipticSlice2  : HRgn ;
    ResultRgn       : HRgn ;
    x1, xinc        : integer ;
    Adjacent        : integer ;
    OldPalette      : HPalette ;

begin
  InflateRect( ARect, -4, -4 ) ;
  with ARect do begin
    w := Right - Left ;
    h := Bottom - Top ;

    { fill lines of 'longitude' using ellipses }
    xinc := w div xSize ;
    x1 := xinc ;
    for i := 0 to xSize - 1 do begin
      Adjacent := round( cos( i * Angle ) * w / 2 ) ;
      Elliptic1 := CreateEllipticRgn( Left + w div 2 - Adjacent, Top,
                                      Right - w div 2 + Adjacent, Bottom ) ;
      if not Odd( xSize ) or ( i <> xSize div 2 ) then begin
        Adjacent := round( cos( ( i + 1 ) * Angle ) * w / 2 ) ;
        Elliptic2 := CreateEllipticRgn( Left + w div 2 - Adjacent, Top,
                                        Right - w div 2 + Adjacent, Bottom ) ;
      end else Elliptic2 := CreateRectRgn( 0, 0, 0, 0 ) ;

      { paint 'slices' across the sphere }
      for j := 0 to ySize - 1 do begin
        if not Odd( xSize ) or ( i <> xSize div 2 ) then begin
          if i < xSize div 2 then
            Slice := CreateRectRgn( Left, Top + j * h div xSize,
                                    Right - w div 2, Top + ( j + 1 ) * h div ySize )
          else
            Slice := CreateRectRgn( Right - w div 2, Top + j * h div xSize,
                                    Right, Top + ( j + 1 ) * h div ySize ) ;
        end else
          Slice := CreateRectRgn( Left, Top + j * h div xSize,
                                  Right, Top + ( j + 1 ) * h div xSize ) ;

        { create the intersection of the elliptic regions with the slice }
        EllipticSlice1 := CreateRectRgn( 0, 0, 0, 0 ) ;
        CombineRgn( EllipticSlice1, Elliptic1, Slice, RGN_AND ) ;
        EllipticSlice2 := CreateRectRgn( 0, 0, 0, 0 ) ;
        CombineRgn( EllipticSlice2, Elliptic2, Slice, RGN_AND ) ;

        { xor the elliptic slices }
        ResultRgn := CreateRectRgn( 0, 0, 0, 0 ) ;
        CombineRgn( ResultRgn, EllipticSlice1, EllipticSlice2, RGN_XOR ) ;

        { fill the result }
        ABrush := CreateSolidBrush( $01000000 + i * xSize + j ) ;
        FillRgn( DC, ResultRgn, ABrush ) ;
        DeleteObject( ABrush ) ;

        { tidy up regions }
        DeleteObject( ResultRgn ) ;
        DeleteObject( EllipticSlice1 ) ;
        DeleteObject( EllipticSlice2 ) ;
        DeleteObject( Slice ) ;
      end ;
      inc( x1, xinc ) ;

      { tidy up }
      DeleteObject( Elliptic1 ) ;
      DeleteObject( Elliptic2 ) ;
    end ;
  end ;
end ;


procedure TMainWindow.Paint( DC : HDC ;
                             var PaintInfo : TPaintStruct ) ;

var ARect           : TRect ;
    OldPalette      : HPalette ;
    w, h, x, y      : integer ;
    FirstOne        : boolean ;

begin
  { set up the palette }
  if Palette <> 0 then begin
    OldPalette := SelectPalette( DC, Palette, false ) ;
    RealizePalette( DC ) ;
  end ;

  { get the client width and height }
  GetClientRect( HWindow, ARect ) ;
  if IsIconic( HWindow ) then PaintPlanet( DC, ARect ) else
  begin
    with ARect do begin
      w := Right - Left ;
      h := Bottom - Top ;
    end ;

    { draw the little planets }
    FirstOne := true ;
    with ARect do for y := 0 to PlanetsAcross - 1 do begin
      Top := h * y div PlanetsAcross ;
      Bottom := h * ( y + 1 ) div PlanetsAcross ;
      for x := 0 to PlanetsAcross - 1 do begin
        Left := w * x div PlanetsAcross ;
        Right := w * ( x + 1 ) div PlanetsAcross ;
        if FirstOne then begin
          PaintPlanet( DC, ARect ) ;
          FirstOne := false ;
        end else with ARect do
          BitBlt( DC, Left, Top, Right - Left, Bottom - Top,
                  DC, 0, 0, SRCCOPY ) ;
      end ;
    end ;
  end ;

  { reset palette }
  if Palette <> 0 then begin
    SelectPalette( DC, OldPalette, true ) ;
    RealizePalette( DC ) ;
  end ;
end ;


function  TMainWindow.PixelAt( x, y : word ) : byte ;

begin
  if ( x <= Width ) and ( y <= Height ) then
    PixelAt := BitmapBits^[ y * ScanWidth + x ] else
    PixelAt := 0 ;
end ;


procedure TMainWindow.SetPaletteEntry( x, y   : word ;
                                       R, G, B : byte ) ;

begin
  { calculate the offset into the palette and set the entry }
  if ( x < xSize ) and ( y < ySize ) then with PaletteEntries[ y * xSize + x ] do begin
    peRed := R ;
    peGreen := G ;
    peBlue := B ;
  end ;
end ;


procedure TMainWindow.ScrollMessage ;

var xMax, yMax : integer ;
    x, y       : integer ;
    i          : integer ;
    xStart     : integer ;

begin
  { set the palette entries to white where there is a bit of the message }
  xMax := XMessagePos + xSize ;
  if xMax > Width then xMax := Width ;
  if Height > ySize then yMax := ySize else yMax := Height ;
  if xMessagePos >= 0 then xStart := xMessagePos else xStart := 0 ;
  for x := xStart to xMax - 1 do begin
    for y := 0 to yMax - 1 do begin
      if PixelAt( x, y ) = 1 then
        SetPaletteEntry( ySize - y, x - xMessagePos, 255, 255, 255 ) ;
    end ;
  end ;
  inc( xMessagePos ) ;
  if xMessagePos >= Width then xMessagePos := -xSize ;
end ;


procedure TMainWindow.IdleAction ;

var i : integer ;
    DC : HDC ;
    OldPalette : HPalette ;

  procedure Change( var AComponent : byte ) ;

  begin
    if ( Direction[ i ] and ( integer( AComponent ) + StepSize > 255 ) ) or
       ( not Direction[ i ] and ( integer( AComponent ) - StepSize < 0 ) ) then
      Direction[ i ] := not Direction[ i ] ;
    if Direction[ i ] then inc( AComponent, StepSize div 2 ) else
    begin
      dec( AComponent, StepSize div 2 ) ;
      if AComponent < StepSize then begin
        AComponent := 0 ;
        if CurrentRGB[ i ] < 3 then inc( CurrentRGB[ i ] ) else CurrentRGB[ i ] := 1 ;
      end ;
    end ;
  end ;

begin
  { only do something if we're an icon or if we're the active app. }
  if not IsIconic( HWindow ) and not AppIsActive then exit ;

  { cycle the palette }
  for i := 0 to nEntries - 1 do with ColourCycle[ i ] do begin
    case CurrentRGB[ i ] of
      1 : Change( peRed ) ;
      2 : Change( peGreen ) ;
      3 : Change( peBlue ) ;
    end ;
  end ;
  PaletteEntries := ColourCycle ;
  ScrollMessage ;
  DC := GetDC( HWindow ) ;
  OldPalette := SelectPalette( DC, Palette, false ) ;
  AnimatePalette( Palette, 0, nEntries, PaletteEntries[ 0 ] ) ;
  SelectPalette( DC, OldPalette, true ) ;
  ReleaseDC( HWindow, DC ) ;
end ;


procedure TMainWindow.wmTimer( var Msg : TMessage ) ;

begin
  IdleAction ;
end ;


procedure TMainWindow.cmAbout( var Msg : TMessage ) ;

begin
  Application^.ExecDialog( new( PDialog, Init( @Self, 'About' ) ) ) ;
end ;


{ TApp }

type
  PApp = ^TApp ;
  TApp = object( TApplication )
    procedure InitMainWindow ; virtual ;
  end ;

procedure TApp.InitMainWindow ;

begin
  MainWindow := new( PMainWindow, Init( nil, 'WinG Pascal Demo' ) ) ;
end ;


var App : TApp ;

begin
  with App do begin
    Init( 'Planet' ) ;
    Run ;
    Done ;
  end ;
end.
