{OWL Custom Control: Divided List Box with Headers}
{Copyright  1992, 1994 Andrew J Wozniewicz}
  {
  I am hereby making this unit available to you, but retain the copyright to same.
  You are free to use this code in any way you like, except for publishing in whole
  or portions thereof in a source code format.

  This source code is provided AS IS. Please use it at your own risk.
  The author gives no warranty, express or implied, as to suitability of this
  code or the processes it describes for any particular purpose. The Author
  assumes no liability whatsoever for any damages that may result, either directly
  or indirectly, from the use of this source code, or any derivatives thereof,
  including compiled object code.

  Please, let me know if you find any bugs or a way to improve it.

  Enjoy,

  Andrew J. Wozniewicz
  CompuServe: 75020,3617
  July 14, 1994
  }

unit DivLstBx;

interface

  uses
    WinTypes,
    WinProcs,
    Win31,
    Objects,
    OWindows,
    ODialogs;

  type {SplitterCollection}
    PSplitterCollection = ^TSplitterCollection;
    TSplitterCollection = object(TCollection)
      public
        procedure FreeItem(p: Pointer); virtual;
      end;

  const
    id_slVertical = 101;

  type {DividedListBox!!!}
    PDividedListBox = ^TDividedListBox;
    TDividedListBox = object(TListBox)
      {A divided list box object.}
        {
        Has a vertical splitter dividing it into two panes.
        Splitter can be dragged with the mouse  to resize the two panes
        within the confines of the standard list box.
        }
      private
        mySplitters: PSplitterCollection;
          {
          Holds a list of split positions, rather than a list
          of pointers to objects. The positions are stored in LoWord
          of each element/"pointer".
          }
        meCapture: Bool;
          {
          Flag indicating whether we have mouse capture.
          It is on (True) only when user currently drags the mouse
          (a splitter) within the window.
          }
        myHExtent: Integer;
          {
          The range of horizontal scrolling.
          }
        myHScrollPos: Integer;
          {
          Current horizontal scroll position in device pixels.
          }
        myClientRect: TRect;
          {
          }
        mySplitColor: TColorRef;
          {
          Color of the dividing splitters/sliders.
          }
      private
        procedure IncSplitPos(anIndex, delta: Integer);
          {Change the specified splitter's position by delta.}
      public
        constructor Init(AParent: PWindowsObject; AnId: Integer; X,Y,W,H: Integer);
          {Initializes an instance.}
        constructor InitResource(AParent: PWindowsObject; ResourceID: Word);
          {Initializes an instance from a resource.}
        destructor Done; virtual;
          {Destroys the instance.}
        procedure Initialize; virtual;
          {Initializes this object's data members.}
        function HitTest(x: Integer): Integer;
          {Returns the index of the splitter at or near position x, or -1 if none.}
        procedure WMSetCursor(var msg: TMessage); virtual wm_First + wm_SetCursor;
          {Changes cursor shape if over a splitter.}
        procedure WMPaint(var msg: TMessage); virtual wm_First + wm_Paint;
          {Enables painting of dividers in addition to standard list box painting.}
        procedure WMSize(var msg: TMessage); virtual wm_First + wm_Size;
          {Remembers client size}
        procedure WMLButtonDown(var Msg: TMessage); virtual wm_First+wm_LButtonDown;
          {Responds to left mouse button down.}
        procedure WMLButtonUp(var Msg: TMessage); virtual wm_First+wm_LButtonUp;
          {Responds to left muse button up by releasing the capture, if any.}
        procedure WMMouseMove(var Msg: TMessage); virtual wm_First+wm_MouseMove;
          {Drags the splitter, if we have the capture.}
        procedure WMHScroll(var Msg: TMessage); virtual wm_First+wm_HScroll;
          {}
        procedure LBSetHorizontalExtent(var Msg: TMessage); virtual wm_First+lb_SetHorizontalExtent;
          {Remember the range of the horizontal scrollbar}
        procedure Paint(DC: HDC; var PaintInfo: TPaintStruct); virtual;
          {Paints the splitters on the list box's background.}
        function GetItemPtr(i: Integer): Pointer; virtual;
          {Returns the pointer associated with a list box entry.}
        procedure CalculateLimits(SplitPos: Integer); virtual;
          {Determines the limits for dragging the splitter.}
        procedure MoveSplitter(startIndex, delta: Integer);
          {Change the position of the specified splitter's and
          all splitters to the right of it  by delta}
        function GetSplitPos(SplitNo: Integer): Integer; virtual;
          {Returns the absolute position of the splitter with the SplitNo.}
        function PaintSplitPos(SplitNo: Integer): Integer; virtual;
          {Returns the scroll-relative position of the splitter with the SplitNo.}
        procedure SetSplitPos(splitNo, x: Integer); virtual;
          {Explicitly sets the position of the splitter with SplitNo.}
        procedure AddSplitter(x: Integer);
          {Adds another splitter at position x.}
        procedure DeleteSplitter(SplitNo: Integer);
          {Deletes the splitter with the SplitNo.}
        procedure SetHorizontalExtent(cx: Integer);
          {Determines the horizontal scrolling range}
        procedure SetSplitColor(aColor: TColorRef);
          {Set the color of the splitter}
        function HScrollPos: Integer;
          {Return the horizontal scroll position in pixels}
        function Splitters: PSplitterCollection;
          {Returns the collection of splitter objects}
      end;


    PDataListBox = ^TDataListBox;
    TDataListBox = object(TDividedListBox)
      myItemHt: Integer;
    public
      procedure SetupWindow; virtual;
      procedure WMNCCalcSize(var Msg : TMessage); virtual wm_First+wm_NCCalcSize;
      procedure WMEraseBkgnd(var msg: TMessage); virtual wm_First + wm_EraseBkgnd;
        {Paints the dividers}
      procedure PaintHeader(DC: HDC; var PaintInfo: TPaintStruct); virtual;
        {Paints the splitters on the list box's background.}
      procedure CalculateLimits(SplitPos: Integer); virtual;
        {Determines the limits for dragging the splitter.}
    end;



implementation

  {$R DIVLSTBX.RES}

  {Module Global Variables}

    var
      cursorN,
        {
        Normal cursor (presumably idc_Arrow).
        This variable is initialized by the unit startup code and
        stores the cursor handle for subsequent use.
        }
      cursorV: HCursor;
        {
        Vertical splitter cursor (custom built).
        This variable is initialized by the unit startup code and
        stores the cursor handle for subsequent use.
        }
      unitDC: HDC;
        {
        Client area display context. The value is meaningful only
        when meCapture of the currently active list box is True, i.e.
        the list box has the focus and the mouse (a splitter) is being
        dragged.
        }
      unitOldROP: Integer;
        {
        Used to store the ROP2 of the device context before the mouse-
        dragging started. Value is meaningful only if meCapture of the
        currently active list box is True.
        }
      unitCR: TRect;
        {
        Rectangle coordinates of the splitter. These determine the vertical
        dimensions of the active splitter and its horizontal movement limits.
        }
      unitSplitIndex: Integer;
        {
        The index of the active splitter.
        The value is meaningful only if the myCapture member of the
        currently active list box object is True.
        }
      unitSplitPos: Integer;
        {
        The position of the active splitter.
        The value is meaningful only if the myCapture member of the
        currently active list box object is True.
        This variable is also used as a scratch temporary storage
        for retrieved splitter positions by many methods. Its value
        is then meaningful only to the routine that sets it.
        }
      unitOldSplitPos: Integer;
        {
        Stores the initial position of the just activated splitter.
        This variable is set by the WMLButtonDown handler of the active
        List Box, and used by the WMLButtonUp handler to adjust the positions
        of all the splitters to the rigth of the activile dragged one.
        }
      unitOffsetPos: Integer;
        {
        }


  {Utility Routines}

    function Within(left, right, number: Integer): Bool;
      {
      Determines if the "number" is within given bounds, i.e.
      whether (left <= number <= right).
      }
      begin
        Within := (left <= number) and (number <= right);
      end;


  {TDividedListBox!!!}

    constructor TDividedListBox.Init(AParent: PWindowsObject; AnId: Integer; X,Y,W,H: Integer);
      {
      Initializes an instance of a TDividedListBox.
      }
      begin
        inherited Init(aParent,anID,x,y,w,h);
        Initialize;
      end;


    constructor TDividedListBox.InitResource(AParent: PWindowsObject; ResourceID: Word);
      {
      Initializes an instance of a TDividedListBox from a resource.
      }
      begin
        inherited InitResource(aParent,resourceID);
        Initialize;
      end;


    procedure TDividedListBox.Initialize;
      {
      Performs initialization of the instance's data members.
      }
      begin
        attr.style := attr.style or LBS_OWNERDRAWFIXED;
        New(mySplitters,Init(8,8));
        meCapture := False;
        myHScrollPos := 0;
        myHExtent := attr.w;
        mySplitColor := RGB(0,0,0);
      end;


    destructor TDividedListBox.Done;
      {
      Destroys the instance of a TDividedListBox by
      deallocating the list of splitters.
      }
      begin
        if Assigned(mySplitters) then
          Dispose(mySplitters,Done);
        inherited Done;
      end;


    procedure TDividedListBox.WMSetCursor(var msg: TMessage);
      {
      Changes the mouse cursor to a vertical split-drag shape
      if it is over (or close to, within one pixel) one of the splitters.
      }
      var
        i: Integer;
        pt: TPoint;
      begin
        if meCapture then
          Exit;
        GetCursorPos(pt);
        ScreenToClient(HWindow,pt);
        i := HitTest(pt.x);
        if ( i >= 0) and PtInRect(myClientRect,pt) then
          SetCursor(cursorV)
        else
          SetCursor(cursorN);
        msg.result := 1;
      end;


    procedure TDividedListBox.WMPaint;
      {
        Supplements the standard WMPaint response (i.e. "do nothing, Windows
        takes care of painting") for built-in controls with our own Paint method.
      }
      var
        info: TPaintStruct;
        DC: HDC;
      begin
        (*
        BeginPaint(HWindow,info);
        Paint(info.hdc,info);
        EndPaint(HWindow,info);
        msg.result := 1;
        *)
        DefWndProc(msg);
        DC := GetDC(HWindow);
        Paint(DC,info);
        ReleaseDC(HWindow,DC);
      end;



    procedure TDividedListBox.WMSize;
      {
      }
      begin
        GetClientRect(HWindow,myClientRect);
        inherited WMSize(msg);
      end;



    procedure TDividedListBox.LBSetHorizontalExtent(var Msg: TMessage);
      begin
        myHExtent := msg.wParam;
        DefWndProc(msg);
        SetScrollRange(HWindow,SB_HORZ,0,myHExtent,FALSE);
        (* if myHScrollPos >= myClientRect.right then
          myHScrollPos := 0; *)
        SetScrollPos(HWindow,SB_HORZ,myHScrollPos,TRUE);
      end;


    procedure TDividedListBox.WMHScroll;

      procedure ScrollBottom;
        begin
          myHScrollPos := myHExtent;
        end;

      procedure ScrollTop;
        begin
          myHScrollPos := 0;
        end;

      procedure ScrollLineDown;
        begin
          Inc(myHScrollPos,10);
        end;

      procedure ScrollLineUp;
        begin
          Dec(myHScrollPos,10);
        end;

      procedure ScrollPageDown;
        begin
          Inc(myHScrollPos,25);
        end;

      procedure ScrollPageUp;
        begin
          Dec(myHScrollPos,25);
        end;

      procedure ScrollThumbTrack;
        begin
          myHScrollPos := msg.lParamLo;
        end;

      begin
        case msg.wParam of
          SB_BOTTOM       : ScrollBottom;
          SB_TOP          : ScrollTop;
          SB_ENDSCROLL    : ;
          SB_LINEDOWN     : ScrollLineDown;
          SB_LINEUP       : ScrollLineUp;
          SB_PAGEDOWN     : ScrollPageDown;
          SB_PAGEUP       : ScrollPageUp;
          SB_THUMBPOSITION: ;
          SB_THUMBTRACK   : ScrollThumbTrack;
        end;
        if myHScrollPos < 0 then
          myHSCrollPos := 0;
        if myHSCrollPos > myHExtent then
          myHScrollPos := myHExtent;
        SetScrollPos(HWindow,SB_HORZ,myHScrollPos,True);
        InvalidateRect(HWindow,nil,True);
      end;


    procedure TDividedListBox.WMLButtonDown(var Msg : TMessage);
      {
      Respond to the Left Mouse Button being pressed.
      }
      var
        i: Integer;
        pt: TPoint;
      begin
        GetCursorPos(Pt);
        ScreenToClient(hWindow, Pt);
        i := HitTest(pt.x);
        if i < 0 then
          begin
            DefWndProc(msg);
            Exit;
          end;
        unitSplitIndex := i;
        unitOldSplitPos := GetSplitPos(i) - myHScrollPos;
        unitSplitPos := pt.x;
        unitOffsetPos := unitOldSplitPos - unitSplitPos;
        meCapture := True;
        SetCapture(hWindow);
        CalculateLimits(i);
        unitDC := GetWindowDC(HWindow);
        unitOldROP := SetROP2(unitDC,R2_NOT);
        MoveTo(unitDC,pt.x,unitCR.top);
        LineTo(unitDC,pt.x,unitCR.bottom);
      end;


    procedure TDividedListBox.WMMouseMove(var Msg : TMessage);
      {
      Respond to mouse moves within the client area.
      }
      var
        r: TRect;
        pt: TPoint;
      begin
        if not meCapture then begin
          DefWndProc(msg);
          Exit;
        end;
        GetCursorPos(Pt);
        ScreenToClient(hWindow, Pt);
        with msg do begin
          if pt.x < Succ(unitCR.left) then
            pt.x := Succ(unitCR.left);
          if pt.x > Pred(unitCR.right-1) then
            pt.x := Pred(unitCR.right-1);
          MoveTo(unitDC,unitSplitPos,unitCR.top);
          LineTo(unitDC,unitSplitPos,unitCR.bottom);
          MoveTo(unitDC,pt.x,unitCR.top);
          LineTo(unitDC,pt.x,unitCR.bottom);
          unitSplitPos := pt.x;
        end;
      end;


    procedure TDividedListBox.WMLButtonUp(var Msg : TMessage);
      {
      Respond to the Left Mouse Button being released.
      }
      var
        swpFlags: Word;
      begin
        if not meCapture then begin
          DefWndProc(msg);
          Exit;
        end;
        MoveTo(unitDC,unitSplitPos,unitCR.top);
        LineTo(unitDC,unitSplitPos,unitCR.bottom);
        SetROP2(unitDC,unitOldROP);
        ReleaseDC(HWindow,unitDC);
        ReleaseCapture;
        meCapture := False;
        unitSplitPos := unitSplitPos;
        MoveSplitter(unitSplitIndex,unitSplitPos - unitOldSplitPos);
        InvalidateRect(HWindow,nil,True);
      end;



    procedure TDividedListBox.MoveSplitter(startIndex, delta: Integer);
      {
      Change the position of the splitter designated with splitIndex, and
      adjust the positions of the splitters to the right of the specified one
      by delta.
      }
      var
        splitIndex: Integer;
      begin
        for SplitIndex := startIndex to Pred(mySplitters^.Count) do begin
          IncSplitPos(SplitIndex,delta);
        end;
        SendMessage(HWindow,LB_SETHORIZONTALEXTENT,myHExtent+delta,0);
      end;


    procedure TDividedListBox.IncSplitPos(anIndex, delta: Integer);
      {
      Change the position of the individual splitter designated by anIndex.
      Does NOT adjust any other peer splitters.
      }
      begin
        SetSplitPos(anIndex,GetSplitPos(anIndex)+delta);
      end;


    procedure TDividedListBox.Paint;
      {
      Paint the splitters.
      }
      var
        OldPen,
        ThePen: HPen;
        SCrollPos: Integer;

      procedure PaintSplitter(p: Pointer); far;
        begin
          unitSplitPos := Integer(LongInt(p));
          if (unitSplitPos-HScrollPos) > myClientRect.right then
            Exit;
          MoveTo(DC,unitSplitPos-ScrollPos,0);
          LineTo(DC,unitSplitPos-ScrollPos,myClientRect.bottom);
        end;

      begin
        GetClientRect(HWindow,myClientRect);
        ThePen := CreatePen(PS_SOLID,0,mySplitColor);
        OldPen := SelectObject(DC,ThePen);
        ScrollPos := HScrollPos;
        mySplitters^.ForEach(@PaintSplitter);
        SelectObject(DC,OldPen);
        DeleteObject(ThePen);
      end;


    procedure TDividedListBox.CalculateLimits(SplitPos: Integer);
      { Calculates the position, size, and movement bounds for the splitters. }
        {
        Builds unitCR rectangle that stores the size of the currently active splitter
        and the limits for its movement. By default, the splitter divides
        the list box vertically top to bottom, and moves within the area delimited by
        the adjacent splitters or within the bounds of the ListBox's client rectangle.
        Override this method to change the default limits.
        }
      begin
        GetClientRect(HWindow,unitCR);
        {If there are other splitters around, adjust the movement boundaries.}
        with unitCR do begin
          if SplitPos > 0 then
            left := PaintSplitPos(Pred(SplitPos))+1;
        end;
      end;


    function TDividedListBox.GetItemPtr;
      {
      Returns the pointer associated with a list box entry.
      }
      begin
        GetItemPtr := Pointer(SendMessage(HWindow,LB_GETITEMDATA,Word(i),0));
      end;


    procedure TDividedListBox.AddSplitter
      (
      x: Integer
        {Initial position for the splitter.}
      );
      {
      Adds a new splitter at a given position x.
      }
      begin
        mySplitters^.Insert(Pointer(MakeLong(x,0)));
      end;


    procedure TDividedListBox.DeleteSplitter
      (
      SplitNo: Integer
        {Index number of the splitter in question.}
      );
      {
      Deletes a splitter identified by SplitNo.
      The remaining splitters, if any, may be renumbered
      as a result of this action.
      }
      begin
        mySplitters^.AtDelete(SplitNo);
      end;


    function TDividedListBox.GetSplitPos(SplitNo: Integer): Integer;
      {
      Returns the position of a given splitter.
      }
      var
        p: Pointer;
      begin
        p := mySplitters^.At(SplitNo);
        GetSplitPos := Integer(LongInt(p));
      end;



    function TDividedListBox.PaintSplitPos(SplitNo: Integer): Integer;
      {
      Returns the scroll-relative position of a given splitter.
      }
      var
        ScrollPos: Integer;
      begin
        ScrollPos := HScrollPos;
        PaintSplitPos := GetSplitPos(SplitNo) - ScrollPos;
      end;



    procedure TDividedListBox.SetSplitPos
      (
      splitNo,
        {Index number of the splitter in question.}
      x: Integer
        {New position for the splitter.}
      );
      {
      Sets the position of a given splitter explicitly.
      }
      begin
        mySplitters^.AtPut(SplitNo,Pointer(MakeLong(x,0)));
      end;


    function TDividedListBox.HitTest(x: Integer): Integer;
      {
      Returns the zero-based index of the splitter at or near position x.
      }
      var
        P: Pointer;
        index: Integer;
        ScrollPos: Integer;

      function IsCloseToX(p: Pointer): Boolean; far;
        begin
          Dec(index);
          unitSplitPos := Integer(LongInt(p));
          IsCloseToX := Within(unitSplitPos-hScrollPos,unitSplitPos-HScrollPos+1,x);
        end;

      begin
        ScrollPos := HScrollPos;
        index := mySplitters^.Count;
        p := mySplitters^.LastThat(@IsCloseToX);
        if p <> nil then
          HitTest := index
        else
          HitTest := -1;
      end;


   procedure TDividedListBox.SetHorizontalExtent(cx: Integer);
     begin
       SendMessage(HWindow,LB_SETHORIZONTALEXTENT,cx,0);
     end;


   procedure TDividedListBox.SetSplitColor(aColor: TColorRef);
     begin
       mySplitColor := aColor;
       if HWindow <> 0 then
         InvalidateRect(HWindow,nil,False);
     end;


   function TDividedListBox.HScrollPos;
     begin
       HScrollPos := myHScrollPos;
     end;


   function TDividedListBox.Splitters;
     begin
       Splitters := mySplitters;
     end;


  {TDataListBox}

    procedure TDataListBox.SetupWindow;
      begin
        inherited SetupWindow;
        myItemHt := SendMessage(HWindow,LB_GETITEMHEIGHT,0,0);
      end;


    procedure TDataListBox.WMNCCalcSize(var Msg : TMessage);
      var
        lpnCSP: PNCCalcSize_Params;
      begin
        lpnCSP := PNCCalcSize_Params(msg.lParam);
        DefWndProc(msg);
        Inc(lpnCSP^.rgrc[0].top,myItemHt);
      end;


    procedure TDataListBox.WMEraseBkgnd;
      var
        info: TPaintStruct;
        DC: HDC;
      begin
        GetClientRect(HWindow,myClientRect);
        DC := GetWindowDC(HWindow);
        with info do begin
          hDC := DC;
          fErase := True;
          SetRect(rcPaint,0,0,myClientRect.right,myItemHt);
        end;
        PaintHeader(DC,info);
        ReleaseDC(HWindow,DC);
        msg.result := 1;
      end;


    procedure TDataListBox.PaintHeader;
      {
      Paint the header.
      }

      var
        wRect: TRect;
        off: Integer;
        oldPen: HPen;
        DrawStruct: TDrawItemStruct;

      procedure PaintSplitter(p: Pointer); far;
        begin
          unitSplitPos := Integer(LongInt(p)) + off;
          MoveTo(DC,unitSplitPos-myHScrollPos,0);
          LineTo(DC,unitSplitPos-myHScrollPos,myItemHt);
        end;

      begin
        GetWindowRect(HWindow,wRect);
        {Paint Gray Headers}
        with PaintInfo do begin
          wRect.left := 0;
          wRect.top := rcPaint.top;
          wRect.bottom := rcPaint.bottom;
          FillRect(DC,wRect,GetStockObject(LTGRAY_BRUSH));
          MoveTo(DC,rcPaint.left,rcPaint.bottom);
          LineTo(DC,rcPaint.right,rcPaint.bottom);
          off := 0;
          mySplitters^.ForEach(@PaintSplitter);
          off := 1;
          oldPen := SelectObject(DC,GetStockObject(WHITE_PEN));
          mySplitters^.ForEach(@PaintSplitter);
          SelectObject(DC,oldPen);
        end;
        {Owner Draws The Header Text}
        with DrawStruct do begin
          CtlType := ODT_ListBox;
          CtlID := $FFFF;
          itemID := $FFFF;
          itemAction := ODA_DRAWENTIRE;
          itemState := 0;
          hWndItem := HWindow;
          hDC := DC;
          rcItem := PaintInfo.rcPaint;
          itemData := -1;
        end;
        SendMessage(Parent^.HWindow,WM_DRAWITEM,GetID,LongInt(@DrawStruct));
      end;


    procedure TDataListBox.CalculateLimits(SplitPos: Integer);
      { Calculates the position, size, and movement bounds for the splitters. }
      begin
        inherited CalculateLimits(SplitPos);
        Inc(unitCR.bottom,myItemHt);
      end;



  {TSplitterCollection}

    procedure TSplitterCollection.FreeItem(p: Pointer);
      {
      Overrides an inherited method with that name to disable
      attempts to free the list elements.
      }
      begin
        {
        Do nothing. The pointer has a splitter position in its
        LoWord. It does NOT point to any objects.
        }
      end;


  {Module Maintenance}

    procedure UnitExit; far;
      begin
        DestroyCursor(cursorV);
      end;


  var
    OldExitProc: Pointer;
  begin
    cursorN := LoadCursor(0,IDC_ARROW);
    cursorV := LoadCursor(System.hInstance,'VSPLIT_DIVLSTBX');
    OldExitProc := ExitProc;
    ExitProc := @UnitExit;
  end.
