//----------------------------------------------------------------------------
//           Name: Msg()
//        Purpose: Display a message in a pop-up window
//
//         Author: Wilson H. Yuen
//   Date created: 12-12-92
//   Time created: 12:37:04pm
//      Copyright: Wilson H. Yuen
//
//         Syntax: Msg( cMsg_, cBoxHeader, lWait, nRow, nDelay ) --> cRetVal
//
//       Argument: cMsg_      - message to display in array or character
//                 cBoxHeader - box header.
//                               E - Error
//                               W - Warning
//                               P - Press any key
//                              "" - nothing
//                              Len() > 0 - cBoxHeader is user as header
//                 lWait      - wait for a key to be press. default to .t.
//                 nRow       - row to start message. default to 14
//                 nDelay     - number of sec. to delay between beeps
//                              default to 1 if cBoxHeader = "E"
//                              default to 1 if nDelay = NIL
//
//   Return Value: cRetVal    - The character pressed if lWait = .t.
//
//        Example: Msg("Record not found?")
//
//----------------------------------------------------------------------------
#include "setcurs.ch"
#include "Backup.ch"

FUNCTION Msg( cMsg_, cBoxHeader, lWait, nRow, nDelay )
    local nMsgRow                                                 //-- the starting row to display the message on
    local lNotify                                                 //-- logical.  Whether or not to beep when a process is done.
    local cMsgColor                                               //-- the color to display the message in
    local cMsgColor2                                              //-- the color to display the message in
    local nCounter                                                //-- used in for/next loops
    local nArrayLen                                               //-- length of the array of messages
    local nArrayLen2                                              //-- length of the array of messages
    local cCurrScrn                                               //-- save old screen so it can be restored later
    local nStartCol                                               //-- starting column position of the message
    local cHeader                                                 //-- text to appear on top of the box
    local cMsg2_     := {}                                        //-- the array of messages to display
    local nWidth     := 0                                         //-- stores the width of the longest message
    local nCurCursor := SetCursor( SC_NONE )                      //-- cursor status
    local cRetVal    := ""                                        //-- return value
    local cMsg       := "***  Press any key to continue  ***"     //-- default message
    local nMsgLen    := 35                                        //-- default message length
    local cOldColor  := SetColor()                                //-- current color settings


    //-- default values
    DEFAULT cBoxHeader TO "",;
            lWait      TO pTRUE


    *-----------------------------------------------------------------------
    *  start by making sure all values have been passed.  If not, assign   |
    *  defaults.  message array.  If the message passed is character type, |
    *  then convert it to an array.                                        |
    *-----------------------------------------------------------------------
    do case
    case ValType(cMsg_) == "C"                   //  character type message was passed
        aadd(cMsg2_, cMsg_)
    case ValType(cMsg_) == "A"                   //  array type message was passed
        cMsg2_ := cMsg_
    endcase

    *------------------------------------------------------
    *  define the row to start printing the message on    |
    *  determine which row to print the message on        |
    *------------------------------------------------------
    if nROW == NIL                    // if a row parameter was not passed
      nMsgRow := nROW := 14
    else
      nMsgRow := nROW
    endif


    *------------------------------------------------------
    *  determine if you should call the notify function.  |
    *  Yes, if nDELAY was passed                          |
    *------------------------------------------------------
    lNotify := if( ValType( nDELAY ) == "N", .t., .f.)

    *----------------------------------------
    *  determine length of longest message  |
    *----------------------------------------
    nArrayLen := len( cMsg2_ )
    for nCounter := 1 to nArrayLen
        nWidth := max( nWidth, len( cMsg2_[nCounter] ) )
    next

    *---------------------------------------------------------------
    *  make sure the length is at least 35 characters long so the  |
    *  "Press any key" message will fit                            |
    *---------------------------------------------------------------
    if ValType( cBoxHeader ) == "C"               // a header was passed
       if cBoxHeader == "P"
           nMsgLen := 23
       endif
    endif
    nWidth := max( nMsgLen, nWidth )

    *----------------------------------------------------------
    *  calculate the left column of the start of the message  |
    *----------------------------------------------------------
    nStartCol := round( (lastcol()-nWidth) /2,0 )



    *----------------------------------------------------------
    *  save the screen where the message will be displayed    |
    *----------------------------------------------------------
    nArrayLen2 := if( lWait, 2, 1 )
    cCurrScrn := SaveScreen( nMsgRow-1, nStartCol-4, nMsgRow+nArrayLen+nArrayLen2+1, nStartCol+nWidth+2 )


    *----------------------------------------------------------
    *  see if there is a header to print                      |
    *----------------------------------------------------------
    cMsgColor2 := "GR+/W"
    if ValType( cBoxHeader ) == "C"               // a header was passed
        do case
        case cBoxHeader == "E"
            cHeader := "< ERROR >"
            cMsgColor  := pERR
            cMsgColor2 := "GR+/R"

        case cBoxHeader == "W"
            cHeader := "< WARNING >"
            cMsgColor := pHELP

        case cBoxHeader == "P"
            cMsg:="***  Press any key  ***"

        case len(cBoxHeader) > 0
            cHeader := "< " + cBoxHeader + " >"
            cMsgColor := pHELP

        otherwise
            cHeader := ""
            cMsgColor := pHELP
        endcase
    else
        cHeader := ""
        cMsgColor := pHELP
    endif

    *--------------------
    *  draw the box     |
    *--------------------
    nArrayLen2 := if( lNotify .or. lWait , 1, 0 )
    box( nMsgRow-1, nStartCol-2, nMsgRow+nArrayLen+nArrayLen2 , nStartCol+nWidth+1, , atoattr(cMsgColor), 1 )


    *---------------------------
    *  display the message     |
    *---------------------------
    center( nMsgRow-1, cHeader, cMsgColor )
    for nCounter := 1 to nArrayLen
        CENTER( nMsgRow-1+nCounter, cMsg2_[nCounter], cMsgColor )
    next

    *----------------------------------
    * beep, if it's an error message  |
    *----------------------------------
    if cBoxHeader == "E"
        beep()
    endif


    *-------------------------------------
    * wait for a keypress (if necessary) |
    *-------------------------------------
    if lWait
        CENTER( nMsgRow+nArrayLen, cMsg, cMsgColor2 )
        clear typeahead
        cRetVal := UPPER(CHR(INKEY(0)))
    else
        if lNotify
            CENTER( nMsgRow+nArrayLen, cMsg, cMsgColor )
            Notify(nDELAY)
        endif
    endif

    if lWait
       RestScreen( nMsgRow-1, nStartCol-4, nMsgRow+nArrayLen+nArrayLen2+1, nStartCol+nWidth+2, cCurrScrn )
    endif

    SetCursor( nCurCursor )                         // reset the old cursor shape
    SetColor( cOldColor )

return( cRetVal )
* end msg()
*
*
//----------------------------------------------------------------------------
//           Name: Notify()
//        Purpose: Notify the user by until the user press a key
//
//         Author: Wilson H. Yuen
//   Date created: 03-21-93
//   Time created: 05:37:24pm
//      Copyright: Softsystems, Inc.  1988-1992
//
//         Syntax: notify( nDelay, nNumTimes ) --> NIL
//
//       Argument: nDelay    - number of seconds between beeps
//                 nNumTImes - number of times to beep
//                             if 0 the system will beep until a key is
//                             pressed.
//
//   Return Value: NIL
//
//        Example: Notify(4,3)
//
//----------------------------------------------------------------------------
Function Notify( nDelay, nNumTimes )
    local nCounter := 0                          //  used for checking the number of loops that have executed

    DEFAULT nDelay    TO 1,;
            nNumTimes TO 1

    do while .t.

       AEval( NOTIFY_SOUND, { | aArray |  tone( aArray, 1 ) } )

       if inkey(nDelay) > 0
          exit
       endif


        *------------------------------------------------------------------
        *  check to see if the maximum number of loops has been exceeded  |
        *------------------------------------------------------------------
        nCounter++
        if  (nCounter == nNumTimes)  .and.  (! nNumTimes == 0)
            exit
        endif
    enddo

return(NIL)
* eof Notify()
*
*
