/////////////////////////////////////////////////
//  DEBUG DYNAMIC MEMORY ALLOCATION 
//      newdebug.h
//
//      Implements a debugging replacement for 
//      the standard C++ memory allocation 
//      operators.
//
//      Tested with:
//          Microsoft C/C++ 7.00
//          Borland C++ 3.x
//
//      Copyright 1992 Scott Robert Ladd
//      All Rights Reserved
/////////////////////////////////////////////////

#ifndef NEWDEBUG_H
#define NEWDEBUG_H

#include "stddef.h"

//-----------------------------------------------
// NewDbg
//      This class encapsulates the tools used in 
//      tracking and verifying dynamic memory 
//      allocations. "new" and "delete" are 
//      friends to provide access to the 
//      protected members of NewDbg.
//-----------------------------------------------

class NewDbg
    {
    friend void * operator new (size_t size);
    friend void operator delete (void * ptr);

    public:
        // error codes
        enum Errors
            {
            EF_OKAY,
            EF_ZEROSIZE,
            EF_NOTALLOC,
            EF_INVALID,
            EF_UNDERFLOW,
            EF_TOOMANY
            };

        // returns number of active allocations
        static long GetCount();

        // returns number of bytes allocated
        static long GetBytes();

        // returns non-zero if ptr
        // was allocated with new
        static int IsDynamic(void * ptr);

        // returns the current error code
        static Errors GetError();

        // sets the error code to EF_OKAY
        static void ClearError();

        // sets the function called on error
        static void InstallHandler(
                     void (* func)(Errors flag));

        // called when an error occurs
        static void ReportError(Errors err);

    protected:

        // header data for tracking allocations
        struct AllocHeader
            {
            unsigned short Marker;
            size_t         Bytes;
            };

        // number of items allocated
        static long Count;

        // number of bytes allocated
        static long Bytes;

        // current error code
        static Errors ErrorFlag;

        // allocation marker
        static const unsigned short CorrectMark;

        // pointer to function called on error
        static void (* ErrorFunc)(Errors flag);
    };

// pointer to C++ error handler
extern void (* _new_handler)();

// function to set _new_handler
void (* set_new_handler(void (* handler)()))();

// synonym for set_new_handler
#define _set_new_handler(ptr) set_new_handler(ptr)

// debugging implementations of new and delete
void * operator new (size_t size);
void operator delete (void * ptr);

#endif
