UNIT FormatOutput ;
(****************************************************************************)
(* FORMATOUTPUT.PAS (TURBO PASCAL 6.0 UNIT)                                 *)
(* Formatted numeric output                                                 *)
(****************************************************************************)
(* Revision history:                                                        *)
(* 02-Jun-1989  written by hs                                               *)
(* 11-Sep-1989  Change FO_DTOut to return current date/time if argument     *)
(*              equals zero                                                 *)
(* 30-Dec-1989  Add Real-type raw formatting routines; change return        *)
(*              string-type names                                           *)
(* 31-Dec-1989  add FO_FormatReal                                           *)
(* 06-Jan-1990  add FO_OctByte                                              *)
(* 06-Feb-1990  change USES Mathematics into USES Math                      *)
(* 30-Apr-1991  rewrite FO_HexDigit, FO_HexByte, FO_HexWord, FO_HexLong,    *)
(*              and FO_HexAddress in TURBO-6 style; Add Digit and HexConv   *)
(*              internal routines.                                          *)
(* 28-Jan-1992  add comments; rewrite HexConv, thereby eliminating bug (in  *)
(*              not-286 compile mode, every other digit was wrong)          *)
(* 06-Mar-1992  add more comments; rewrite binary, octal, and date/time     *)
(*              conversion routines; add BinConv and OctConv                *)
(*              introduce FO_NowTime constant (equals -1) which replaces    *)
(*              the 0 as an argument to FO_DTOut meaning "current time."    *)
(*              Note that this is an INCOMPATIBLE change.                   *)
(* 09-Mar-1992  add conversion of real types (TP's 6-byte real and 8087     *)
(*              types) along with HexXConv, BinXConv, and OctXConv routines.*)
(*              Small optimization in HexConv. This makes use of the un-    *)
(*              documented 80X86 feature that the opcode 0D4h will do a     *)
(*              divide-and-modulo operation on AL, with the second operand  *)
(*              in the byte following the opcode. (0D4h,0Ah is the normal   *)
(*              coding for the AAM opcode.)                                 *)
(*  10-Mar-1992 Bugfix in HexConv                                           *)
(*  12-May-1992 Rewrite FO_HexOut and FO_BinOut in BASM                     *)
(*  13-May-1992 Change all FO_XXXX names into foXXXX. remove real output    *)
(*              and real output service routines. Make entries for          *)
(*              foOctXXXX routines in INTERFACE section. Add foOctOut.      *)
(*              Remove {$N-} switch: unit compiles in any state.            *)
(****************************************************************************)
(* This unit was written by Hans Schleichert, Marburg and London.           *)
(****************************************************************************)

(****************************************************************************)
(*************************************) INTERFACE (**************************)
(****************************************************************************)

    TYPE                            (****************************************)
                                    (* formatted-output string types        *)
                                    (****************************************)
        foHByteStr  = String [2] ;  (* hex byte                             *)
        foHWordStr  = String [4] ;  (* hex word                             *)
        foHLongStr  = String [8] ;  (* hex longword                         *)
        foHSnglStr  = String [8] ;  (* hex 4-byte real                      *)
        foHRealStr  = String [12] ; (* hex 6-byte real                      *)
        foHDbleStr  = String [16] ; (* hex 8-byte real                      *)
        foHCompStr  = String [16] ; (* hex 8-byte integer                   *)
        foHExtStr   = String [20] ; (* hex 10-byte real                     *)
        foHAddrStr  = String [9] ;  (* hex address in seg:ofs form          *)
        foBByteStr  = String [8] ;  (* bin bytes                            *)
        foBWordStr  = String [16] ; (* bin words                            *)
        foBLongStr  = String [32] ; (* bin longwords                        *)
        foBSnglStr  = String [32] ; (* hex 4-byte real                      *)
        foBRealStr  = String [48] ; (* hex 6-byte real                      *)
        foBDbleStr  = String [64] ; (* hex 8-byte real                      *)
        foBCompStr  = String [64] ; (* hex 8-byte integer                   *)
        foBExtStr   = String [80] ; (* hex 10-byte real                     *)
        foOByteStr  = String [3] ;  (* oct bytes                            *)
        foOWordStr  = String [6] ;  (* oct words                            *)
        foOLongStr  = String [11] ; (* oct longwords                        *)
        foOSnglStr  = String [11] ; (* hex 4-byte real                      *)
        foORealStr  = String [16] ; (* hex 6-byte real                      *)
        foODbleStr  = String [22] ; (* hex 8-byte real                      *)
        foOCompStr  = String [22] ; (* hex 8-byte integer                   *)
        foOExtStr   = String [27] ; (* hex 10-byte real                     *)
        foDTStr     = String [20] ; (* date & time                          *)
        foBoolStr   = String [5] ;  (* boolean values                       *)
        foOutStr    = String [127] ;(* string type for foxxxOut functions   *)
        {MantStr        = String [35] ; {mantissas}

    (************************************************************************)
    (* hexadecimal conversion routines                                      *)
    (************************************************************************)
    FUNCTION foHexDigit (Data : Byte) : Char ;
    FUNCTION foHexByte (Data : Byte) : foHByteStr ;
    FUNCTION foHexWord (Data : Word) : foHWordStr ;
    FUNCTION foHexLong (Data : LongInt) : foHLongStr ;
    FUNCTION foHexSingle (Data : Single) : foHSnglStr   ;
    FUNCTION foHexReal (Data : Real) : foHRealStr ;
    FUNCTION foHexDouble (Data : Double) : foHDbleStr   ;
    FUNCTION foHexComp (Data : Comp) : foHCompStr ;
    FUNCTION foHexExtended (Data : Extended) : foHExtStr ;
    FUNCTION foHexAddress (Data : Pointer) : foHAddrStr ;
    FUNCTION foHexOut (
        Data : LongInt ; Width, Digits : ShortInt
    ) : foOutStr ;

    (************************************************************************)
    (* binary conversion routines                                           *)
    (************************************************************************)
    FUNCTION foBinByte (Data : Byte) : foBByteStr ;
    FUNCTION foBinWord (Data : Word) : foBWordStr ;
    FUNCTION foBinLong (Data : LongInt) : foBLongStr ;
    FUNCTION foBinSingle (Data : Single) : foBSnglStr   ;
    FUNCTION foBinReal (Data : Real) : foBRealStr ;
    FUNCTION foBinDouble (Data : Double) : foBDbleStr   ;
    FUNCTION foBinComp (Data : Comp) : foBCompStr ;
    FUNCTION foBinExtended (Data : Extended) : foBExtStr ;
    FUNCTION foBinOut (
        Data : LongInt ; Width, Digits : ShortInt
    ) : foOutStr ;

    (************************************************************************)
    (* octal conversion routines                                            *)
    (************************************************************************)
    FUNCTION foOctByte (Data : Byte) : foOByteStr ;
    FUNCTION foOctWord (Data : Word) : foOWordStr ;
    FUNCTION foOctLong (Data : LongInt) : foOLongStr ;
    FUNCTION foOctSingle (Data : Single) : foOSnglStr ;
    FUNCTION foOctReal (Data : Real) : foORealStr ;
    FUNCTION foOctDouble (Data : Double) : foODbleStr ;
    FUNCTION foOctComp (Data : Comp) : foOCompStr ;
    FUNCTION foOctExtended (Data : Extended) : foOExtStr ;
    FUNCTION foOctOut (
        Data : LongInt ; Width, Digits : ShortInt
    ) : foOutStr ;

    (************************************************************************)
    (* decimal conversion routines                                          *)
    (************************************************************************)
    FUNCTION foDecOut (OutLong : LongInt ; Size, Digits : Byte) : String ;

    (************************************************************************)
    (* miscellaneous conversion routines                                    *)
    (************************************************************************)
    CONST
        foNowTime   =   -1 ;            (* return current date and time     *)

    FUNCTION foDTOut (Data : LongInt) : foDTStr ; (**********************)
    (* convert a packed date-and-time record into its ASCIC             *)
    (* representation                                                   *)
    (* Input:                                                           *)
    (*  Data    packed date-and-time record to be converted             *)
    (*          Data=foNowTime (defined above) means current date and   *)
    (*          time                                                    *)
    (* Returns:                                                         *)
    (*  The date and time in the format dd-mmm-yyyy hh:mm:ss where mmm  *)
    (*  is one of these strings: Jan, Feb, Mar, Apr, May, Jun, Jul,     *)
    (*  Aug, Sep, Oct, Nov, Dec, and hh is the 24-hour time.            *)
    (********************************************************************)

    FUNCTION foBoolOut (Data : Boolean) : foBoolStr ; (******************)
    (* convert a boolean value into an ASCIC string                     *)
    (* Input:                                                           *)
    (*  Data    a boolean value                                         *)
    (* Returns:                                                         *)
    (*  'False' or 'True' depending on the value of Data                *)
    (********************************************************************)

    {FUNCTION foRoundStr (VAR S : String ; Length : Byte) : Boolean ;}
    (*********************************************************************)
    (* Rounds a character string that solely consists of digits to a     *)
    (* given length.                                                     *)
    (* Input:                                                            *)
    (*  S       - the string to be processed                             *)
    (*  Length  - the length to which S should be rounded                *)
    (* Output:                                                           *)
    (*  S       - the rounded string                                     *)
    (* Returns:                                                          *)
    (*  True is the rounding caused an overflow; False otherwise         *)
    (* Algorithm:                                                        *)
    (*  No action occurs if the S is not longer than Length characters.  *)
    (*  If S is longer, it is truncated to the given length. If the      *)
    (*  leftmost cut-off digit is greater than or equal to 5, the        *)
    (*  rightmost not discarded digit is incremented. If this leads to   *)
    (*  an ASCII value greater that '9', it is set to '0'and the digit   *)
    (*  to its left is incremented, etc., until the leftmost (first)     *)
    (*  digit is reached. If incrementing the first digit results in an  *)
    (*  ASCII value greater than '9', the first digit is set to '0', and *)
    (*  True is returned.                                                *)
    (*********************************************************************)

    {PROCEDURE foRawRealOut (
        X : Real ;
        VAR Sign : Boolean ;
        VAR DExpo : Integer ;
        VAR DMant : MantStr
    ) ;}
    (*********************************************************************)
    (* Raw Real-type binary-to-decimal conversion                        *)
    (* Input:                                                            *)
    (*  X       - a real-type number                                     *)
    (* Output:                                                           *)
    (*  Sign    - True is X was negative, False otherwise                *)
    (*  DExpo   - X's decimal exponent                                   *)
    (*  DMant   - X's decimal mantissa as a string of 13 digits          *)
    (* Note:                                                             *)
    (*  Use foRoundStr to round DMant. The least-significant digit of    *)
    (*  DMant is not very meaningful.                                    *)
    (*  If X was a NaN ("Not a Number"), Sign is set to False, DExpo is  *)
    (*  set to zero, and DMant is set to 'NaN'.                          *)
    (*********************************************************************)

    {CONST Flags for foFormatReal
        foStd       = $01 ;
        foFix       = $02 ;
        foSci       = $04 ;
        foEng       = $08 ;
        foBlankNaN  = $40 ;
        foPlus      = $80 ;}

    {FUNCTION foFormatReal (
        RealVal : Real ;
        FieldWidth : Byte ;
        Flags : Byte ;
        Digits : Byte
    ) : String ;}
    (*********************************************************************)
    (* Real number formatting for output                                 *)
    (* Input:                                                            *)
    (*  RealVal     - a Real-type number                                 *)
    (*  FieldWidth  - the width of the output field                      *)
    (*  Flags       - a combination of the above defined flag bits       *)
    (*  Digits      - the number of decimal digits                       *)
    (* Returns:                                                          *)
    (*  an ASCII string of length FieldWidth that is the decimal repre-  *)
    (*  sentation of RealVal                                             *)
    (* Notes:                                                            *)
    (*  1 - NaNs can be processed; Infs cannot.                          *)
    (*  2 - How to use the flags:                                        *)
    (*      foStd overrides foFix; foEng overrides foSci;            *)
    (*      foFix formatting tries to fit the output string with Digits *)
    (*      decimals into the output field. If necessary, least signi-   *)
    (*      ficant zeroes are truncated. If this is not possible, foSci *)
    (*      formatting is used unless foEng is explicitly requested.     *)
    (*      foStd formatting works like foFix but tries to use as few    *)
    (*      decimals as possible.                                        *)
    (*      foSci formatting generates the mantissa/exponent form, with *)
    (*      the exponent displayed as "Enn". The decimal point is to    *)
    (*      the right of the most significant digit.                     *)
    (*      foEng works like foSci, but the exponent is always an        *)
    (*      integer multiple of 3. If necessery, the decimal point is    *)
    (*      shifted right so that the mantissa/exponent combination is   *)
    (*      always correct.                                              *)
    (*      The foBlankNaN flag replaces the default 'NaN' output by     *)
    (*      a blank field if a RealVal is an NaN ("not a number").       *)
    (*      The foPlus flag forces a plus ("+") sign before every not-   *)
    (*      negative number. By default, only a minus ("-") sign is put  *)
    (*      before negative numbers.                                     *)
    (*  3 - It is the programmer's responsiblity to set Digits and       *)
    (*      FieldWidth to reasonable values. If either of them is too    *)
    (*      small for the output field to hold the string, a string of   *)
    (*      FieldWidth asterisks is returned.                            *)
    (*********************************************************************)

(****************************************************************************)
(*************************************) IMPLEMENTATION (*********************)
(****************************************************************************)

    USES
        Dos{, Math} ;

(*************************************** Local procedures *******************)

    PROCEDURE Digit ; NEAR ; ASSEMBLER ; (*******************************)
    (* Convert a nybble into its hexadezimal ASCII representation       *)
    (* Input:                                                           *)
    (*  AL      nybble to convert (only lower 4 bits are significant)   *)
    (* Output:                                                          *)
    (*  AL      ASCII representation of nybble                          *)
    (*  Flags   undefined                                               *)
    (* Note: Works for values from 00h to 0Fh.                          *)
    (********************************************************************)

        ASM     (************************** PROCEDURE Digit *****************)
            AND     AL,0Fh
            SUB     AL,10
            JL      @L1
            ADD     AL,'A'-('9'+1)
@L1:        ADD     AL,'0' + 10
        END ;   (********************** PROCEDURE Digit *********************)

    PROCEDURE HexConv ; NEAR ; ASSEMBLER ; (*****************************)
    (* convert a number into a hexadecimal ASCIC string                 *)
    (* Input:                                                           *)
    (*  CL      number of digits to generate (string length)            *)
    (*  BX,DX   value to convert                                        *)
    (*  ES:DI   pointer to ASCIC string area                            *)
    (* Output:                                                          *)
    (*  [ES:old DI] filled with ASCIC string                            *)
    (*  AX,DI   undefined                                               *)
    (*  BX,CX,DX    zero                                                *)
    (*  Flags   DF=1; ZF,OF,PF,SF undefined                             *)
    (********************************************************************)

        ASM     (********************** PROCEDURE HexConv *******************)
            MOV     [ES:DI],CL
            MOV     CH,0
            ADD     DI,CX
            DEC     DI
            STD
            SHR     CX,1
            PUSHF
            JCXZ    @PastLoop
@Loop:      MOV     AL,BL
            DB      0D4h,10h            (* get high and low nybble          *)
            CALL    Digit
            XCHG    AH,AL
            CALL    Digit
            STOSW
            MOV     BL,BH
            MOV     BH,DL
            MOV     DL,DH
            MOV     DH,0
            LOOP    @Loop
@PastLoop:  POPF
            JNC     @Done
            MOV     AL,BL
            AND     AL,0Fh
            CALL    Digit
            STOSB
@Done:  END ; (************************ PROCEDURE HexConv *******************)

    PROCEDURE HexXConv ; NEAR ; ASSEMBLER ; (****************************)
    (* convert a number into a hexadecimal ASCIC string                 *)
    (* Input:                                                           *)
    (*  CL      number of characters to generate (string length)        *)
    (*  DS:SI   pointer to value to convert                             *)
    (*  ES:DI   pointer to ASCIC string area                            *)
    (* Output:                                                          *)
    (*  [ES:old DI] filled with ASCIC string                            *)
    (*  AX      undefined                                               *)
    (*  CX      zero                                                    *)
    (*  SI,DI   undefined                                               *)
    (*  Flags   DF=1; ZF,OF,PF,SF undefined                             *)
    (* Note:                                                            *)
    (*  This procedure will generate up to 255 characters.              *)
    (********************************************************************)

        ASM     (********************** PROCEDURE HexXConv ******************)
            MOV     CH,0
            MOV     AX,CX
            STD
            STOSB
            ADD     DI,AX
            SHR     CX,1
            PUSHF
            JCXZ    @@2
@Loop:      CLD
            LODSB
            DB      0D4h,10h            (* get high and low nybble          *)
            CALL    Digit
            XCHG    AL,AH
            CALL    Digit
            STD
            STOSW
            LOOP    @Loop
@@2:        POPF
            JNC     @Done
            LODSB
            AND     AL,0Fh
            CALL    Digit
            INC     DI
            STOSB
@Done:  END ; (************************ PROCEDURE HexXConv ******************)

    PROCEDURE BinConv ; NEAR ; ASSEMBLER ; (*****************************)
    (* convert a number into a binary ASCIC string                      *)
    (* Input:                                                           *)
    (*  CL      number of digits to generate (string length)            *)
    (*  BX,DX   value to convert                                        *)
    (*  ES:DI   pointer to ASCIC string area                            *)
    (* Output:                                                          *)
    (*  [ES:DI] filled with ASCIC string                                *)
    (*  AX      undefined                                               *)
    (*  BX,CX,DX    zero                                                *)
    (*  Flags   DF=1; ZF,OF,PF,SF undefined                             *)
    (********************************************************************)

        ASM (************************** PROCEDURE BinConv *******************)
            MOV     [ES:DI],CL
            MOV     CH,0
            JCXZ    @Done
            ADD     DI,CX
            STD
@Loop:      SHR     DX,1
            RCR     BX,1
            MOV     AL,'0'
            ADC     AL,0
            STOSB
            LOOP    @Loop
@Done:  END ; (************************ PROCEDURE BinConv *******************)

    PROCEDURE BinXConv ; NEAR ; ASSEMBLER ; (****************************)
    (* convert a number into a binary ASCIC string                      *)
    (* Input:                                                           *)
    (*  CL      number of characters to generate (string length)        *)
    (*  DS:SI   pointer to value to convert                             *)
    (*  ES:DI   pointer to ASCIC string area                            *)
    (* Output:                                                          *)
    (*  [ES:old DI] filled with ASCIC string                            *)
    (*  AX      undefined                                               *)
    (*  CX      zero                                                    *)
    (*  SI,DI   undefined                                               *)
    (*  Flags   DF=1; ZF,OF,PF,SF undefined                             *)
    (* Note:                                                            *)
    (*  This procedure will generate up to 255 characters.              *)
    (********************************************************************)

        ASM
            MOV     CH,0
            MOV     AL,CL
            STD
            STOSB
            JCXZ    @Done
            ADD     DI,CX
            INC     DI
            SUB     BL,BL
@Loop:      JNZ     @@1
            CLD
            LODSB
            MOV     AH,AL
            MOV     BL,8
            STD
@@1:        SHR     AH,1
            MOV     AL,0
            ADC     AL,'0'
            STOSB
            DEC     BL
            LOOP    @Loop
@Done:  END ;

    PROCEDURE OctConv ; NEAR ; ASSEMBLER ; (*****************************)
    (* convert a number into a octal ASCIC string                       *)
    (* Input:                                                           *)
    (*  CL      number of digits to generate (string length)            *)
    (*  BX,DX   value to convert                                        *)
    (*  ES:DI   pointer to ASCIC string area                            *)
    (* Output:                                                          *)
    (*  [ES:DI] filled with ASCIC string                                *)
    (*  AX      undefined                                               *)
    (*  BX,CX,DX    zero                                                *)
    (*  Flags   DF=1; ZF,OF,PF,SF undefined                             *)
    (********************************************************************)

        ASM (************************** PROCEDURE OctConv *******************)
            MOV     [ES:DI],CL
            MOV     CH,0
            JCXZ    @Done
            ADD     DI,CX
            STD
@Loop:      MOV     AL,BL
            AND     AL,07h
            ADD     AL,'0'
            STOSB
            SHR     DX,1
            RCR     BX,1
            SHR     DX,1
            RCR     BX,1
            SHR     DX,1
            RCR     BX,1
            LOOP    @Loop
@Done:  END ; (************************ PROCEDURE OctConv *******************)

    PROCEDURE OctXConv ; NEAR ; ASSEMBLER ; (****************************)
    (* convert a number into an octal ASCIC string                      *)
    (* Input:                                                           *)
    (*  CL      number of digits to generate (string length)            *)
    (*  DS:SI   pointer to value to convert                             *)
    (*  ES:DI   pointer to ASCIC string area                            *)
    (* Output:                                                          *)
    (*  [ES:old DI] filled with ASCIC string                            *)
    (*  AX      undefined                                               *)
    (*  CX      zero                                                    *)
    (*  SI,DI   undefined                                               *)
    (*  Flags   DF=1; ZF,OF,PF,SF undefined                             *)
    (* Note:                                                            *)
    (*  This procedure will produce up to 255 characters.               *)
    (********************************************************************)

        ASM     (********************** PROCEDURE OctXConv ******************)
            MOV     CH,0
            MOV     AL,CL
            STD
            STOSB
            JCXZ    @Done
            ADD     DI,CX
            INC     DI
            XOR     BX,BX
@Loop:      MOV     AL,[SI]
            JMP     [Word PTR CS:@Dispatch+BX]
@Dispatch:  DW      @B1, @B2, @B3, @B4, @B5, @B6, @B7, @B8
@B8:        CLD
            LODSB
            JMP     @Shift5
@B3:        CLD
            LODSB
            MOV     AH,[SI]
            JMP     @Shift6
@B6:        CLD
            LODSB
            MOV     AH,[SI]
@Shift7:    SHR     AX,1
@Shift6:    SHR     AX,1
@Shift5:    SHR     AX,1
            STD
@B5:        SHR     AX,1
@B2:        SHR     AX,1
@B7:        SHR     AX,1
@B4:        SHR     AX,1
@B1:        AND     AL,07h
            OR      AL,'0'
            STOSB
            ADD     BL,2
            AND     BL,0Eh
            LOOP    @Loop
@Done:  END ;   (********************** PROCEDURE OctXConv ******************)

(************************************** hexadecimal conversion procedures ***)

    FUNCTION foHexDigit (Data : Byte) : Char ; ASSEMBLER ;

        ASM     (********************** FUNCTION foHexDigit *****************)
            MOV     AL,[Data]
            CALL    Digit
        END ;   (********************** FUNCTION foHexDigit *****************)

    FUNCTION foHexByte (Data : Byte) : foHByteStr ; ASSEMBLER ;

        ASM     (********************** FUNCTION foHexByte ******************)
            LES     DI,[@Result]
            MOV     CL,2
            MOV     BL,[Data]
            CALL    HexConv
        END ;   (********************** FUNCTION foHexByte ******************)

    FUNCTION foHexWord (Data : Word) : foHWordStr ; ASSEMBLER ;

        ASM     (********************** FUNCTION foHexWord ******************)
            LES     DI,[@Result]
            MOV     CL,4
            MOV     BX,[Data]
            CALL    HexConv
        END ;   (********************** FUNCTION foHexWord ******************)

    FUNCTION foHexLong (Data : LongInt) : foHLongStr ; ASSEMBLER ;

        ASM     (********************** FUNCTION foHexLong ******************)
            LES     BX,[Data]
            MOV     DX,ES
            LES     DI,[@Result]
            MOV     CL,8
            CALL    HexConv
        END ;   (********************** FUNCTION foHexLong ******************)

    FUNCTION foHexSingle (Data : Single) : foHSnglStr   ; ASSEMBLER ;

        ASM     (********************** FUNCTION foHexSingle ****************)
            LES     BX,[Data]
            MOV     DX,ES
            LES     DI,[@Result]
            MOV     CL,8
            CALL    HexConv
        END ;   (********************** FUNCTION foHexSingle ****************)

    FUNCTION foHexReal (Data : Real) : foHRealStr ; ASSEMBLER ;

        ASM     (********************** FUNCTION foHexReal ******************)
            PUSH    DS
            LEA     SI,[Data]
            PUSH    SS
            POP     DS
            LES     DI,[@Result]
            MOV     CL,12
            CALL    HexXConv
            POP     DS
        END ;   (********************** FUNCTION foHexReal ******************)

    FUNCTION foHexDouble (Data : Double) : foHDbleStr   ; ASSEMBLER ;

        ASM     (********************** FUNCTION foHexDouble ****************)
            PUSH    DS
            LEA     SI,[Data]
            PUSH    SS
            POP     DS
            LES     DI,[@Result]
            MOV     CL,16
            CALL    HexXConv
            POP     DS
        END ;   (********************** FUNCTION foHexDouble ****************)

    FUNCTION foHexComp (Data : Comp) : foHCompStr ; ASSEMBLER ;

        ASM     (********************** FUNCTION foHexComp ******************)
            PUSH    DS
            LEA     SI,[Data]
            PUSH    SS
            POP     DS
            LES     DI,[@Result]
            MOV     CL,16
            CALL    HexXConv
            POP     DS
        END ;   (********************** FUNCTION foHexComp ******************)

    FUNCTION foHexExtended (Data : Extended) : foHExtStr ; ASSEMBLER ;

        ASM     (********************** FUNCTION foHexExtended **************)
            PUSH    DS
            LEA     SI,[Data]
            PUSH    SS
            POP     DS
            LES     DI,[@Result]
            MOV     CL,20
            CALL    HexXConv
            POP     DS
        END ;   (********************** FUNCTION foHexExtended **************)

    FUNCTION foHexAddress (Data : Pointer) : foHAddrStr ; ASSEMBLER ;

        ASM     (********************** FUNCTION foHexAddress ***************)
            LES     BX,[Data]
            MOV     DX,ES
            LES     DI,[@Result]
            CLD
            MOV     AL,9
            STOSB
            MOV     CX,2
            CALL    @L1
            MOV     AL,':'
            STOSB
            MOV     CL,2
            CALL    @L1
            JMP     @Done
@L1:        MOV     AL,DH
            MOV     DH,DL
            MOV     DL,BH
            MOV     BH,BL
            MOV     AH,AL
{$IFOPT G+}
            SHR     AH,4
{$ELSE}
            SHR     AL,1
            SHR     AL,1
            SHR     AL,1
            SHR     AL,1
{$ENDIF}
            CALL    Digit
            XCHG    AL,AH
            CALL    Digit
            STOSW
            LOOP    @L1
            RETN    0
@Done:  END ;   (********************** FUNCTION foHexAddress ***************)

    FUNCTION foHexOut (
        Data : LongInt ; Width, Digits : ShortInt
    ) : foOutStr ; ASSEMBLER ;

        ASM     (********************** FUNCTION foHexOut *******************)
            LES     BX,[Data]
            MOV     DX,ES
            LES     DI,[@Result]
            LEA     DI,[ES:DI+(TYPE foOutStr-1)]
            STD
            MOV     CH,[Digits]
            OR      CH,CH
            PUSHF
            JG      @@2
            JE      @@1
            NOT     CH
@@1:        INC     CH
@@2:        MOV     AH,CH
            MOV     CL,4

@@3:        CALL    @Digit
            DEC     CH
            JNZ     @@3

            POPF
            JG      @@5
@@4:        MOV     SI,DX
            OR      SI,BX
            JZ      @@5
            CALL    @Digit
            INC     AH
            JMP     @@4

@Digit:     MOV     AL,BL
            AND     AL,0Fh
            SUB     AL,10
            JL      @@D1
            ADD     AL,'A'-('9'+1)
@@D1:       ADD     AL,'0'+10
            STOSB
            SHR     BX,CL
            MOV     AL,DL
            SHL     AL,CL
            OR      BH,AL
            SHR     DX,CL
            RETN    0

@@5:        MOV     CH,[Width]
            OR      CH,CH
            PUSHF
            POP     SI
            JGE     @@6
            NEG     CH

@@6:        CMP     AH,CH
            JL      @@7
            JE      @@9
            PUSH    SI
            POPF
            JG      @@9
            MOV     CH,AH
            JMP     @@9
@@7:        MOV     AL,' '
            SUB     AH,CH
@@8:        STOSB
            INC     AH
            JNZ     @@8

@@9:        PUSH    DS
            LDS     DI,[@Result]
            LEA     SI,[DS:DI+TYPE foOutStr]
            MOV     CL,CH
            MOV     CH,0
            SUB     SI,CX
            CLD
            MOV     AL,CL
            STOSB
            SHR     CX,1
            JNC     @@10
            MOVSB
@@10:       REP     MOVSW
            POP     DS
        END ;   (********************** FUNCTION foHexOut *******************)

(************************************** binary conversion procedures ********)

    FUNCTION foBinLong (Data : LongInt) : foBLongStr ; ASSEMBLER ;

        ASM     (********************** FUNCTION foBinLong ******************)
            LES     BX,[Data]
            MOV     DX,ES
            LES     DI,[@Result]
            MOV     CL,32
            CALL    BinConv
        END ;   (********************** FUNCTION foBinLong ******************)

    FUNCTION foBinWord (Data : Word) : foBWordStr ; ASSEMBLER ;

        ASM     (********************** FUNCTION foBinWord ******************)
            MOV     BX,[Data]
            LES     DI,[@Result]
            MOV     CL,16
            CALL    BinConv
        END ;   (********************** FUNCTION foBinWord ******************)

    FUNCTION foBinByte (Data : Byte) : foBByteStr ; ASSEMBLER ;

        ASM     (********************** FUNCTION foBinByte ******************)
            MOV     BL,[Data]
            LES     DI,[@Result]
            MOV     CL,8
            CALL    BinConv
        END ;   (********************** FUNCTION foBinByte ******************)

    FUNCTION foBinSingle (Data : Single) : foBSnglStr   ; ASSEMBLER ;

        ASM     (********************** FUNCTION foBinSingle ****************)
            PUSH    DS
            LEA     SI,[Data]
            PUSH    SS
            POP     DS
            LES     DI,[@Result]
            MOV     CL,32
            CALL    BinXConv
            POP     DS
        END ;   (********************** FUNCTION foBinSingle ****************)

    FUNCTION foBinReal (Data : Real) : foBRealStr ; ASSEMBLER ;

        ASM     (********************** FUNCTION foBinReal ******************)
            PUSH    DS
            LEA     SI,[Data]
            PUSH    SS
            POP     DS
            LES     DI,[@Result]
            MOV     CL,48
            CALL    BinXConv
            POP     DS
        END ;   (********************** FUNCTION foBinReal ******************)

    FUNCTION foBinDouble (Data : Double) : foBDbleStr   ; ASSEMBLER ;

        ASM     (********************** FUNCTION foBinDouble ****************)
            PUSH    DS
            LEA     SI,[Data]
            PUSH    SS
            POP     DS
            LES     DI,[@Result]
            MOV     CL,64
            CALL    BinXConv
            POP     DS
        END ;   (********************** FUNCTION foBinDouble ****************)

    FUNCTION foBinComp (Data : Comp) : foBCompStr ; ASSEMBLER ;

        ASM     (********************** FUNCTION foBinComp ******************)
            PUSH    DS
            LEA     SI,[Data]
            PUSH    SS
            POP     DS
            LES     DI,[@Result]
            MOV     CL,64
            CALL    BinXConv
            POP     DS
        END ;   (********************** FUNCTION foBinComp ******************)

    FUNCTION foBinExtended (Data : Extended) : foBExtStr ; ASSEMBLER ;

        ASM     (********************** FUNCTION foBinExtended **************)
            PUSH    DS
            LEA     SI,[Data]
            PUSH    SS
            POP     DS
            LES     DI,[@Result]
            MOV     CL,80
            CALL    BinXConv
            POP     DS
        END ;   (********************** FUNCTION foBinExtended **************)

    FUNCTION foBinOut (
        Data : LongInt ; Width, Digits : ShortInt
    ) : foOutStr ; ASSEMBLER ;

        ASM     (********************** FUNCTION foBinOut *******************)
            LES     BX,[Data]
            MOV     DX,ES
            LES     DI,[@Result]
            LEA     DI,[ES:DI+(TYPE foOutStr-1)]
            STD
            MOV     CH,[Digits]
            OR      CH,CH
            PUSHF
            JG      @@2
            JE      @@1
            NOT     CH
@@1:        INC     CH
@@2:        MOV     AH,CH
            MOV     CL,4

@@3:        CALL    @Digit
            DEC     CH
            JNZ     @@3

            POPF
            JG      @@5
@@4:        MOV     SI,DX
            OR      SI,BX
            JZ      @@5
            CALL    @Digit
            INC     AH
            JMP     @@4

@Digit:     SHR     DX,1
            RCR     BX,1
            MOV     AL,'0'/2
            RCL     AL,1
            STOSB
            RETN    0

@@5:        MOV     CH,[Width]
            OR      CH,CH
            PUSHF
            POP     SI
            JGE     @@6
            NEG     CH

@@6:        CMP     AH,CH
            JL      @@7
            JE      @@9
            PUSH    SI
            POPF
            JG      @@9
            MOV     CH,AH
            JMP     @@9
@@7:        MOV     AL,' '
            SUB     AH,CH
@@8:        STOSB
            INC     AH
            JNZ     @@8

@@9:        PUSH    DS
            LDS     DI,[@Result]
            LEA     SI,[DS:DI+TYPE foOutStr]
            MOV     CL,CH
            MOV     CH,0
            SUB     SI,CX
            CLD
            MOV     AL,CL
            STOSB
            SHR     CX,1
            JNC     @@10
            MOVSB
@@10:       REP     MOVSW
            POP     DS
        END ;   (********************** FUNCTION foBinOut *******************)

(************************************** octal conversion procedures *********)

    FUNCTION foOctByte (Data : Byte) : foOByteStr ; ASSEMBLER ;          

        ASM     (********************** FUNCTION foOctByte ******************)
            MOV     BL,[Data]
            MOV     BH,0
            LES     DI,[@Result]
            MOV     CL,3
            CALL    OctConv
        END ;   (********************** FUNCTION foOctByte ******************)

    FUNCTION foOctWord (Data : Word) : foOWordStr ; ASSEMBLER ;

        ASM     (********************** FUNCTION foOctWord ******************)
            MOV     BX,[Data]
            MOV     DL,0
            LES     DI,[@Result]
            MOV     CL,6
            CALL    OctConv
        END ;   (********************** FUNCTION foOctWord ******************)

    FUNCTION foOctLong (Data : LongInt) : foOLongStr ; ASSEMBLER ;

        ASM     (********************** FUNCTION foOctLong ******************)
            LES     BX,[Data]
            MOV     DX,ES
            LES     DI,[@Result]
            MOV     CL,11
            CALL    OctConv
        END ;   (********************** FUNCTION foOctLong ******************)

    FUNCTION foOctSingle (Data : Single) : foOSnglStr ; ASSEMBLER ;

        ASM     (********************** FUNCTION foOctSingle ****************)
            PUSH    DS
            LEA     SI,[Data]
            PUSH    SS
            POP     DS
            LES     DI,[@Result]
            MOV     CL,11
            CALL    OctXConv
            POP     DS
        END ;   (********************** FUNCTION foOctSingle ****************)

    FUNCTION foOctReal (Data : Real) : foORealStr ; ASSEMBLER ;

        ASM     (********************** FUNCTION foOctReal ******************)
            PUSH    DS
            LEA     SI,[Data]
            PUSH    SS
            POP     DS
            LES     DI,[@Result]
            MOV     CL,16
            CALL    OctXConv
            POP     DS
        END ;   (********************** FUNCTION foOctReal ******************)

    FUNCTION foOctDouble (Data : Double) : foODbleStr ; ASSEMBLER ;

        ASM     (********************** FUNCTION foOctDouble ****************)
            PUSH    DS
            LEA     SI,[Data]
            PUSH    SS
            POP     DS
            LES     DI,[@Result]
            MOV     CL,22
            CALL    OctXConv
            POP     DS
        END ;   (********************** FUNCTION foOctDouble ****************)

    FUNCTION foOctComp (Data : Comp) : foOCompStr ; ASSEMBLER ;

        ASM     (********************** FUNCTION foOctComp ******************)
            PUSH    DS
            LEA     SI,[Data]
            PUSH    SS
            POP     DS
            LES     DI,[@Result]
            MOV     CL,22
            CALL    OctXConv
            POP     DS
        END ;   (********************** FUNCTION foOctComp ******************)

    FUNCTION foOctExtended (Data : Extended) : foOExtStr ; ASSEMBLER ;

        ASM     (********************** FUNCTION foOctExtended **************)
            PUSH    DS
            LEA     SI,[Data]
            PUSH    SS
            POP     DS
            LES     DI,[@Result]
            MOV     CL,27
            CALL    OctXConv
            POP     DS
        END ;   (********************** FUNCTION foOctExtended **************)

    FUNCTION foOctOut (
        Data : LongInt ; Width, Digits : ShortInt
    ) : foOutStr ; ASSEMBLER ;

        ASM     (********************** FUNCTION foOctOut *******************)
            LES     BX,[Data]
            MOV     DX,ES
            LES     DI,[@Result]
            LEA     DI,[ES:DI+(TYPE foOutStr-1)]
            STD
            MOV     CH,[Digits]
            OR      CH,CH
            PUSHF
            JG      @@2
            JE      @@1
            NOT     CH
@@1:        INC     CH
@@2:        MOV     AH,CH
            MOV     CL,4

@@3:        CALL    @Digit
            DEC     CH
            JNZ     @@3

            POPF
            JG      @@5
@@4:        MOV     SI,DX
            OR      SI,BX
            JZ      @@5
            CALL    @Digit
            INC     AH
            JMP     @@4

@Digit:     MOV     AL,BL
            AND     AL,07h
            ADD     AL,'0'
            STOSB
            SHR     DX,1
            RCR     BX,1
            SHR     DX,1
            RCR     BX,1
            SHR     DX,1
            RCR     BX,1
            RETN    0

@@5:        MOV     CH,[Width]
            OR      CH,CH
            PUSHF
            POP     SI
            JGE     @@6
            NEG     CH

@@6:        CMP     AH,CH
            JL      @@7
            JE      @@9
            PUSH    SI
            POPF
            JG      @@9
            MOV     CH,AH
            JMP     @@9
@@7:        MOV     AL,' '
            SUB     AH,CH
@@8:        STOSB
            INC     AH
            JNZ     @@8

@@9:        PUSH    DS
            LDS     DI,[@Result]
            LEA     SI,[DS:DI+TYPE foOutStr]
            MOV     CL,CH
            MOV     CH,0
            SUB     SI,CX
            CLD
            MOV     AL,CL
            STOSB
            SHR     CX,1
            JNC     @@10
            MOVSB
@@10:       REP     MOVSW
            POP     DS
        END ;   (********************** FUNCTION foOctOut *******************)

(************************************** decimal conversion procedures *******)

    FUNCTION foDecOut (OutLong : LongInt ; Size, Digits : Byte) : String ;

        VAR
            i : Byte ;
            Buffer : String ;
            Sign : Boolean ;

        BEGIN
            Buffer := '' ;
            IF OutLong < 0 THEN
            BEGIN
                Sign := True ; OutLong := -OutLong
            END
            ELSE Sign := False ;
            WHILE OutLong <> 0 DO
            BEGIN
                Buffer := Chr (OutLong MOD 10 + Ord ('0')) + Buffer ;
                OutLong := OutLong DIV 10
            END ;
            IF Digits = 0 THEN
            BEGIN
                Digits := Length (Buffer) ;
                IF Buffer = '' THEN Inc (Digits)
            END ;
            IF Size = 0 THEN
            BEGIN
                Size := Digits ;
                IF Sign THEN Inc (Size)
            END ;
            IF Size < Length (Buffer) THEN
            Buffer := Copy (Buffer, Length (Buffer) + 1 - Size, Size)
            ELSE FOR i := Length (Buffer) + 1 TO Size DO
            Insert (' ', Buffer, 1) ;
            IF Digits > Size THEN Digits := Size ;
            IF (Digits = Size) AND Sign THEN Dec (Digits) ;
            FOR i := 1 TO Size - Digits DO Buffer [i] := ' ' ;
            FOR i := Size - Digits + 1 TO Size DO
            IF Buffer [i] = ' ' THEN Buffer [i] := '0' ;
            IF Sign THEN Buffer [Length (Buffer) - Digits] := '-' ;
            foDecOut := Buffer
        END {FUNCTION foDecOut} ;

(************************************** miscellaneous conversion procedures *)

    FUNCTION foDTOut (Data : LongInt) : foDTStr ; ASSEMBLER ;

        ASM     (********************** FUNCTION foDTOut ********************)
            LES     BX,[Data]
            MOV     DX,ES
            MOV     AX,-1
            CMP     BX,AX
            JNE     @@1
            CMP     DX,AX
            JNE     @@1
            MOV     AH,2Ch
            INT     21h
            PUSH    DX
            PUSH    CX
            MOV     AH,2Ah
            INT     21h
            PUSH    CX
            PUSH    DX
            JMP     @@2

@Conv2:     PUSH    AX
            CALL    @Conv1
            POP     AX
            XCHG    AH,AL
            XCHG    BH,BL
            CALL    @Conv1
            RETN    0

@Conv1:     CMP     AL,0
            JE      @Conv1a
            STOSB
@Conv1a:    MOV     AL,BL
            AAM
            OR      AX,'00'
            XCHG    AH,AL
            STOSW
            RETN    0

@MonthNames:
            DB      'Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun'
            DB      'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec'

@OneHundred:
            DB      100

@@1:        MOV     AH,BL
            SHL     AH,1
            AND     AH,3Eh
            PUSH    AX
{$IFOPT G+}
            SHR     BX,3
            SHR     BL,2
{$ELSE}
            SHR     BX,1
            SHR     BX,1
            SHR     BX,1
            SHR     BL,1
            SHR     BL,1
{$ENDIF}
            PUSH    BX
            MOV     AL,DH
            SHR     AL,1
            AND     AX,007Fh
            ADD     AX,1980
            PUSH    AX
            MOV     AX,DX
{$IFOPT G+}
            SHL     AX,3
{$ELSE}
            SHL     AX,1
            SHL     AX,1
            SHL     AX,1
{$ENDIF}
            MOV     AL,DL
            AND     AX,0F1Fh
            PUSH    AX

@@2:        LES     DI,[@Result]
            CLD
            MOV     AL,TYPE foDTStr-1
            POP     BX
            CALL    @Conv1
            MOV     BL,BH
            MOV     BH,0
            MOV     SI,BX
            SHL     BX,1
            MOV     AH,[Byte PTR CS:@MonthNames+BX+SI-3]
            MOV     AL,'-'
            STOSW
            MOV     AX,[Word PTR CS:@MonthNames+BX+SI-2]
            STOSW
            POP     AX
            DIV     [Byte PTR CS:@OneHundred]
            MOV     BX,AX
            MOV     AX,'-'
            CALL    @Conv2
            MOV     AX,': '
            POP     BX
            XCHG    BH,BL
            CALL    @Conv2
            MOV     AL,':'
            POP     BX
            MOV     BL,BH
            CALL    @Conv1
        END ;   (********************** FUNCTION foDTOut ********************)

    FUNCTION foBoolOut (Data : Boolean) : foBoolStr ; ASSEMBLER ;

        ASM
            PUSH    DS
            LES     DI,[@Result]
            PUSH    CS
            POP     DS
            LEA     SI,[@Strings]
            MOV     CX,6
            CMP     [Data],0
            JE      @@1
            ADD     SI,CX
@@1:        CLD
            REP     MOVSB
            POP     DS
            JMP     @Done
@Strings:   DB      5, 'False', 4, 'True'
@Done:  END ;

    {FUNCTION EstimDExp (BExp : Integer) : Integer ;}
    (*********************************************************************)
    (* Estimates the decimal exponent of a floating-point number         *)
    (* Input:                                                            *)
    (*  BExp    - the binary exponent of the normalized floating-point   *)
    (*            number                                                 *)
    (* Returns:                                                          *)
    (*  An integer value that is not too far from the actual decimal     *)
    (*  exponent. In fact the returned value is between the decimal      *)
    (*  exponent minus one and the deminal exponent.                     *)
    (* Algorithm:                                                        *)
    (*  The input value is integer multiplied by 2466, which is about    *)
    (*  Log(2)*2^13, and then shifted left by 3 bits; only the high-     *)
    (*  order word is returned, thus giving BExp*Log(2).                 *)
    (*********************************************************************)

        {INLINE (
            {0000   $58/                {POP AX
            {0001   $BA/2466/           {MOV DX,2466;  Log(2)*2^13
            {0004   $F7/$EA/            {IMUL DX
            {0006   $B9/>3/             {MOV CX,3
            {0009                   shift:
            {0009   $D1/$D0/            {RCL AX,1
            {000B   $D1/$D2/            {RCL DX,1
            {000D   $E2/<$0009-$000F/   {LOOP shift
            {000F   $89/$D0             {MOV AX,DX
            {0011
        ) ;}

    {FUNCTION RealAbsSign (VAR X : Real) : Boolean ;}
    (*********************************************************************)
    (* Returns the sign of a Real-type number and resets it.             *)
    (* Input:                                                            *)
    (*  X       - a real number                                          *)
    (* Output:                                                           *)
    (*  X       - the absolute value of X's input value                  *)
    (* Returns:                                                          *)
    (*  True if X was less than zero; False if X was greater than or     *)
    (*  equal to zero.                                                   *)
    (*********************************************************************)

        {INLINE (
            {0000   $5E/            {POP DI}
            {0001   $07/            {POP ES}
            {0002   $83/$C7/5/      {ADD DI,5}
            {0005   $26/$8A/$05/    {MOV AL,ES:[DI]}
            {0008   $88/$C4/        {MOV AH,AL}
            {000A   $24/$7F/        {AND AL,$7F}
            {000C   $AA/            {STOSB}
            {000D   $30/$C0/        {XOR AL,AL}
            {000F   $D1/$C0         {ROL AX,1}
            {0011
        ) ;}

    {FUNCTION RealExp (X : Real) : Integer ;}
    (*********************************************************************)
    (* Returns the binary exponent of a Real-type number.                *)
    (* Input:                                                            *)
    (*  X       - a Real-type number                                     *)
    (* Returns:                                                          *)
    (*  an Integer-type value between -129 and +126 that is the binary   *)
    (*  exponent of X                                                    *)
    (*********************************************************************)

        {INLINE (
            {0000   $58/        {POP AX}
            {0001   $5B/        {POP BX}
            {0002   $5B/        {POP BX}
            {0003   $30/$E4/    {XOR AH,AH}
            {0005   $2D/>129    {SUB AX,129}
            {0008
        ) ;}

    {PROCEDURE Norm6B (VAR X : Real) ;}
    (*********************************************************************)
    (* Normalizes the special 6-bit unsigned fixed-point representation  *)
    (* used for Real-type floating-point binary-to-decimal conversion    *)
    (* Input:                                                            *)
    (*  X       - a Real-type value; 0 <= X < 10                         *)
    (* Output:                                                           *)
    (*  X       - the special 6-bit unsigned fixed-point form of X's     *)
    (*            input value                                            *)
    (*********************************************************************)
    (* The special 6-bit unsigned fixed-point integer is defined as      *)
    (* follows:                                                          *)
    (* bits  0 thru 40 contain the fractional part                       *)
    (* bits 41 thru 44 contain the integer part                          *)
    (* bits 45 thru 47 must be zero                                      *)
    (*********************************************************************)

        {INLINE (
            {0000   $5E/                {POP SI}
            {0001   $5B/                {POP BX}
            {0002   $1E/                {PUSH DS}
            {0003   $89/$F7/            {MOV DI,SI}
            {0005   $8E/$C3/            {MOV ES,BX}
            {0007   $8E/$DB/            {MOV DS,BX}
            {0009   $31/$C9/            {XOR CX,CX}
            {000B   $FC/                {CLD}
            {000C   $AD/                {LODSW}
            {000D   $86/$C1/            {XCHG AL,CL}
            {000F   $81/$E9/>129/       {SUB CX,129}
            {0013   $89/$C2/            {MOV DX,AX}
            {0015   $AD/                {LODSW}
            {0016   $89/$C3/            {MOV BX,AX}
            {0018   $AD/                {LODSW}
            {0019   $80/$CC/$80/        {OR AH,$80}
            {001C   $F7/$D9/            {NEG CX}
            {001E   $83/$C1/8/          {ADD CX,8}
            {0020                   shift:}
            {0020   $D1/$E8/            {SHR AX,1}
            {0022   $D1/$DB/            {RCR BX,1}
            {0024   $D1/$DA/            {RCR DX,1}
            {0026   $E2/<$0019-$0021/   {LOOP shift}
            {0028   $92/                {XCHG AX,DX}
            {0029   $AB/                {STOSW}
            {002A   $89/$D8/            {MOV AX,BX}
            {002C   $AB/                {STOSW}
            {002D   $89/$D0/            {MOV AX,DX}
            {002F   $AB/                {STOSW}
            {0030   $1F                 {POP DS}
            {0031
        ) ;}

    {FUNCTION DecDigit (VAR X : Real) : Char ;}
    (*********************************************************************)
    (* Gets the next decimal digit from the special 6-byte unsigned      *)
    (* fixed-point representation defined in the description to Norm6B   *)
    (* Input:                                                            *)
    (*  X       - a 6-byte unsigned fixed-point number                   *)
    (* Output:                                                           *)
    (*  X       - the fractional part of X's input value times 10        *)
    (* Returns:                                                          *)
    (*  a character that is the ASCII representation of the integer part *)
    (*  of X's input value                                               *)
    (*********************************************************************)


        {INLINE (
            {0000   $5E/            {POP SI}
            {0001   $5B/            {POP BX}
            {0002   $1E/            {PUSH DS}
            {0003   $89/$F7/        {MOV DI,SI}
            {0005   $8E/$C3/        {MOV ES,BX}
            {0007   $8E/$DB/        {MOV DS,BX}
            {0009   $FC/            {CLD}
            {000A   $AD/            {LODSW}
            {000B   $89/$C2/        {MOV DX,AX}
            {000D   $AD/            {LODSW}
            {000E   $89/$C3/        {MOV BX,AX}
            {0010   $AD/            {LODSW}
            {0011   $89/$C1/        {MOV CX,AX}
            {0013   $24/$80/        {AND AL,$80}
            {0015   $50/            {PUSH AX}
            {0016   $31/$C8/        {XOR AX,CX}
            {0018   $D1/$E2/        {SHL DX,1}
            {001A   $D1/$D3/        {RCL BX,1}
            {001C   $D1/$D0/        {RCL AX,1}
            {001E   $50/            {PUSH AX}
            {001F   $53/            {PUSH BX}
            {0020   $52/            {PUSH DX}
            {0021   $D1/$E2/        {SHL DX,1}
            {0023   $D1/$D3/        {RCL BX,1}
            {0025   $D1/$D0/        {RCL AX,1}
            {0027   $D1/$E2/        {SHL DX,1}
            {0029   $D1/$D3/        {RCL BX,1}
            {002B   $D1/$D0/        {RCL AX,1}
            {002D   $59/            {POP CX}
            {002E   $01/$CA/        {ADD DX,CX}
            {0030   $59/            {POP CX}
            {0031   $11/$CB/        {ADC BX,CX}
            {0033   $59/            {POP CX}
            {0034   $11/$C8/        {ADC AX,CX}
            {0036   $92/            {XCHG AX,DX}
            {0037   $AB/            {STOSW}
            {0038   $89/$D8/        {MOV AX,BX}
            {003A   $AB/            {STOSW}
            {003B   $89/$D0/        {MOV AX,DX}
            {003D   $80/$34/$03/    {AND AH,$03}
            {0040   $AB/            {STOSW}
            {0041   $58/            {POP AX}
            {0042   $D1/$E0/        {SHL AX,1}
            {0044   $88/$E0/        {MOV AL,AH}
            {0046   $0C/Ord ('0')/  {OR AL,'0'}
            {0048   $1F             {POP DS}
            {0049
        ) ;}

    {FUNCTION RoundStr (VAR S : String ; Length : Byte) : Boolean ;}
    (*********************************************************************)
    (* Permforms what is described with foRoundStr.                  *)
    (* Usage of this INLINE routine is discouraged.                      *)
    (*********************************************************************)

        {INLINE (
            {0000   $59/                {POP CX}
            {0001   $5E/                {POP SI}
            {0002   $5B/                {POP BX}
            {0003   $8C/$DA/            {MOV DX,DS}
            {0005   $8E/$DB/            {MOV DS,BX}
            {0007   $8E/$C3/            {MOV ES,BX}
            {0009   $3A/$0C/            {CMP CL,[SI]}
            {000B   $73/<$002E-$000D/   {JAE done}
            {000D   $30/$ED/            {XOR CH,CH}
            {000F   $88/$0C/            {MOV [SI],CL}
            {0011   $01/$CE/            {ADD SI,CX}
            {0013   $89/$F7/            {MOV DI,SI}
            {0015   $80/$7C/$01/$35/    {CMP Byte Ptr [SI+1],'5'}
            {0019   $72/<$002E-$001B/   {JB done}
            {001B   $FD/                {STD}
            {001C               incloop:}
            {001C   $AC/                {LODSB}
            {001D   $FE/$C0/            {INC AL}
            {001F   $3C/$3A/            {CMP AL,'0'+10}
            {0021   $72/<$0025-$0023/   {JB nocarry}
            {0023   $B0/$30/            {MOV AL,'0'}
            {0025               nocarry:}
            {0025   $AA/                {STOSB}
            {0026   $72/<$002A-$0028/   {JB exitloop}
            {0028   $E2/<$001C-$002A/   {LOOP incloop}
            {002A               exitloop:}
            {002A   $B0/$01/            {MOV AL,1}
            {002C   $73/<$0030-$002E/   {JNB exit}
            {002E               done:}
            {002E   $B0/$00/            {MOV AL,0}
            {0030               exit:}
            {0030   $8E/$DA             {MOV DS,DX}
            {0032
        ) ;}

    {FUNCTION Pow10 (Pow : Integer) : Real ;}
    (*********************************************************************)
    (* Calculates a power of 10.                                         *)
    (* Input:                                                            *)
    (*  Pow     - an integer between -38 and +38                         *)
    (* Returns:                                                          *)
    (*  a Real-type number that is 10^Pow                                *)
    (*********************************************************************)

        {CONST
            F10Tab : ARRAY [-6..6] OF Real = (
                1E-32, 1E-16, 1E-8, 1E-4, 1E-2, 1E-1,
                1E0, 1E1, 1E2, 1E4, 1E8, 1E16, 1E32
            ) ;}

        {VAR
            Count : ShortInt ;
            P10 : Real ;}

        {BEGIN {FUNCTION Pow10
            IF Pow >= 0 THEN
            BEGIN
                P10 := 1 ;
                FOR Count := 1 TO 6 DO
                IF Pow AND (1 SHL (Count-1)) <> 0 THEN
                P10 := P10 * F10Tab [Count]
            END
            ELSE
            BEGIN
                P10 := 0.1 ;
                FOR Count := 1 TO 6 DO
                IF Pow AND (1 SHL (Count-1)) = 0 THEN
                P10 := P10 * F10Tab [-Count]
            END ;
            Pow10 := P10
        END {FUNCTION Pow10 ;}

    {FUNCTION foRoundStr (VAR S : String ; Length : Byte) : Boolean ;

        BEGIN {FUNCTION foRoundStr
            foRoundStr := RoundStr (S, Length)
        END {FUNCTION foRoundStr ;}

    {PROCEDURE foRawRealOut (
        X : Real ;
        VAR Sign : Boolean ;
        VAR DExpo : Integer ;
        VAR DMant : MantStr
    ) ;

        VAR
            i : Byte ;
            Count : Integer ;
            Y, P10 : Real ;

        BEGIN {PROCEDURE foRawRealOut
            Sign := RealAbsSign (X) ;
            DExpo := EstimDExp (RealExp (X)) ;
            P10 := Pow10 (DExpo) ;
            IF TestRealNaN (X) THEN
            BEGIN
                Sign := False ;
                DExpo := 0 ;
                DMant := 'NaN'
            END ELSE IF (X = 0) OR (P10 = 0) THEN
            BEGIN
                Sign := False ;
                DExpo := 0 ;
                DMant := '0000000000000'
            END
            ELSE
            BEGIN
                Y := X / P10 ;
                IF Y >= 10 THEN
                BEGIN
                    Inc (DExpo) ;
                    P10 := Pow10 (DExpo) ;
                    Y := X / P10 ;
                END ;
                Norm6B (Y) ;
                DMant [0] := #13 ;
                FOR i := 1 TO 13 DO DMant [i] := DecDigit (Y)
            END
        END {PROCEDURE foRawRealOut ;}

    {CONST
        foFixStd    = foFix OR foStd ;
        foSciEng    = foSci OR foEng ;}

    {FUNCTION FixFormat (
        VAR Buf : String ;
        Mant : MantStr ;
        Expo, Digits : ShortInt
    ) : Boolean ;

        VAR
            i : ShortInt ;

        BEGIN {PROCEDURE FixFormat
            IF Expo > 12 THEN FixFormat := False
            ELSE
            BEGIN
                FixFormat := True ;
                Buf := Mant ;
                WHILE Length (Buf) <= Expo DO Insert ('0', Buf, 255) ;
                FOR i := -1 DOWNTO Expo DO Insert ('0', Buf, 1) ;
                i := IMax (0, Expo+1) + Digits ;
                IF foRoundStr (Buf, i) THEN
                Buf := Copy ('1000000000000', 1, i) ;
                Insert ('.', Buf, IMax (Expo, 0) + 2)
            END
        END {PROCEDURE FixFormat ;}

    {PROCEDURE SciFormat (
        VAR Buf : String ;
        Mant : MantStr ;
        Expo, Digits, DecimalPoint : ShortInt
    ) ;

        BEGIN {PROCEDURE SciFormat
            Buf := Mant ;
            IF foRoundStr (Buf, Digits+1) THEN
            BEGIN
                Buf := Copy ('1000000000000', 1, Digits+1) ;
                Inc (Expo)
            END ;
            IF DecimalPoint + 2 <= Length (Buf) THEN
            Insert ('.', Buf, DecimalPoint + 2) ;
            Insert ('E' + Chr (Ord ('+') + (Ord ('-') - Ord ('+')) *
                Ord (Expo < 0)) + foDecOut (Abs (Expo), 2, 2), Buf, 255)
        END {PROCEDURE SciFormat ;}


    {FUNCTION foFormatReal (
        RealVal : Real ;
        FieldWidth : Byte ;
        Flags : Byte ;
        Digits : Byte
    ) : String ;

        VAR
            Expo : Integer ;
            Digits2, E, SignDigits : ShortInt ;
            Done, Sign : Boolean ;
            Mant : MantStr ;

        VAR
            Buf : String ;

        BEGIN {FUNCTION foFormatReal
            IF TestRealNaN (RealVal) THEN
            BEGIN
                IF Flags AND foBlankNaN <> 0 THEN
                Buf := ' '
                ELSE Buf := 'NaN'
            END
            ELSE
            BEGIN
                foRawRealOut (RealVal, Sign, Expo, Mant) ;
                IF foRoundStr (Mant, Imin (12, FieldWidth)) THEN
                BEGIN
                    Mant := '100000000000' ; Inc (Expo)
                END ;
                SignDigits := Ord (Sign OR (Flags AND foPlus <> 0)) ;
                Dec (FieldWidth, SignDigits) ;
                IF Digits > 12 THEN Digits := 12 ;
                IF Flags AND foFixStd <> 0 THEN
                BEGIN
                    Done := True ;
                    IF Flags AND foStd <> 0 THEN Digits2 := 12
                    ELSE Digits2 := Digits ;
                    IF (Expo < Length (Mant)) AND
                        FixFormat (Buf, Mant, Expo, Digits2) THEN
                    BEGIN
                        IF (Flags AND foStd <> 0) AND (Pos ('.', Buf) > 0)
                            THEN
                        WHILE Buf [Ord (Buf [0])] = '0' DO Dec (Buf [0]) ;
                        IF Buf [Length (Buf)] = '.' THEN Dec (Buf [0]) ;
                        IF (Length (Buf) = FieldWidth + 1) AND (Buf [1] = '0')
                            THEN
                        Delete (Buf, 1, 1) ;
                        WHILE (Length (Buf) > FieldWidth) AND
                            (Buf [Ord (Buf [0])] = '0') DO Dec (Buf [0]) ;
                        IF Length (Buf) > FieldWidth THEN
                        BEGIN
                            IF Buf [1] = '0' THEN Delete (Buf, 1, 1) ;
                            Delete (Buf, Pos ('.', Buf), 1) ;
                            IF foRoundStr (Buf, FieldWidth-1) THEN
                            BEGIN
                                IF FixFormat (Buf, '1000000000000', Expo+1,
                                    Digits2) THEN
                                BEGIN
                                    IF (Flags AND foStd <> 0) AND
                                        (Pos ('.', Buf) > 0) THEN
                                    WHILE Buf [Ord (Buf [0])] = '0' DO
                                    Dec (Buf [0]) ;
                                    IF Buf [Length (Buf)] = '.' THEN
                                    Dec (Buf [0]) ;
                                END
                                ELSE Done := False
                            END
                            ELSE Done := False
                        END
                    END
                    ELSE Done := False ;
                    Digits := IMin (Digits, FieldWidth - 6) ;
                    IF Done THEN
                    BEGIN
                        E := Length (Buf) ;
                        WHILE (E > 0) AND ((Buf [E] = '0') OR (Buf [E] = '.'))
                            DO Dec (E) ;
                        Done := E > 0
                    END ;
                    IF Done THEN Flags := Flags AND NOT foSciEng
                    ELSE Flags := Flags OR foSci
                END ;
                IF Flags AND foEng = 0 THEN E := 0
                ELSE
                BEGIN
                    E := ((Expo MOD 3) + 3) MOD 3 ;
                    Dec (Expo, E) ;
                    IF foRoundStr (Mant, Digits+1) THEN
                    BEGIN
                        Mant := Copy ('100000000000', 1, Digits) ;
                        IF E = 2 THEN
                        BEGIN
                            Inc (Expo, 3) ; E := 0
                        END
                        ELSE Inc (E)
                    END ;
                    Digits := IMax (Digits, E) ;
                    WHILE Length (Mant) <= E DO Insert ('0', Mant, 255) ;
                    Flags := Flags OR foSci
                END ;
                IF Flags AND foSci <> 0 THEN
                BEGIN
                    SciFormat (Buf, Mant, Expo, Digits, E) ;
                    IF Length (Buf) > FieldWidth THEN
                    Buf := Copy ('*******************', 1,
                        FieldWidth + SignDigits)
                END ;
                IF (Buf [1] <> '*') THEN
                IF Sign THEN Insert ('-', Buf, 1)
                ELSE IF Flags AND foPlus <> 0 THEN Insert ('+', Buf, 1) ;
                Inc (FieldWidth, SignDigits)
            END ;
            WHILE Length (Buf) < FieldWidth DO Insert (' ', Buf, 1) ;
            foFormatReal := Buf
        END {FUNCTION foFormatReal ;}

END . (******************************** UNIT FormatOutput *******************)