Unit DialogWn;
{ Unit:      DialogWn
  Version:   1.05
  Purpose:   make a descendant of tWindow named tDialogWindow that behaves like
             a modeless or modal dialog.
  Features:  - tDialogWindow descends from tWindow
             - tDialogWindow and descendants may be used as MDI childs
             - support for calculated resources is included e.g. a dialog
               childs class & style may be changed on-the-fly (see GetChildClass)
               tJanusDialogWindow object is an example for this: it decides at
               runtime whether to uses BorDlg's or standard dialogs
  Date:      30.08.1992

  Developer: Peter Sawatzki (PS)
             Buchenhof 3, D-5800 Hagen 1, Germany
 CompuServe: 100031,3002
       FIDO: 2:245/5800.17

  Date:     Author:
  22.04.92  PS       intial release by PS
  25.07.92  PS/jwp   added Scroller support
  01.08.92  PS       added RunModal and modal support
  12.08.92  PS       removed SetClassName and NewClass, fixed bug in MDI support
  14.08.92  PS       fixed Focus problems in MDI, give focus to first ws_TabStop child
  30.08.92  PS       fixed more focus problems in MDI, added SysModal support

  Contributing: Jeroen W. Pluimers (jwp)

  Copyright (c) 1992 Peter Sawatzki. All Rights Reserved.

}
Interface
Uses
  WinTypes,
  WObjects;
Type
  tChildClass = Record
    wX, wY, wCX, wCY, wID: Integer;
    dwStyle: LongInt;
    szClass: Array[0..63] Of Char;
    szTitle: Array[0..131] Of Char;
    CtlDataSize: Byte;
    CtlData: Array[0..255] Of Byte;
  End;

  tDialogWindowAttr = Record
    Name: pChar;
    ItemCount: Integer;
    MenuName,
    ClassName,
    FontName: pChar;
    Font: hFont;
    PointSize: Integer;
    DlgItems: Pointer;
    ResW,
    ResH: Integer;
    wUnitsX,
    wUnitsY: Word
  End;

  pDialogWindow = ^tDialogWindow;
  tDialogWindow = Object(tWindow)
    DialogAttr: tDialogWindowAttr;
    ModalCode: pInteger;
    Constructor Init (aParent: pWindowsObject; aName: pChar);
    Destructor Done;                   Virtual;
    Function  Create: Boolean;         Virtual;
    Procedure Destroy;                 Virtual;
    Procedure SetupWindow;             Virtual;
    Function  GetWindowProc: tFarProc; Virtual;
    Function  GetClassName: pChar;     Virtual;
    Procedure GetChildClass (Var aChildClass: tChildClass); Virtual;
    Function  CreateDialogChild (Var aChildClass: tChildClass): hWnd; Virtual;
    Procedure CreateDialogChildren;
    Procedure CreateDialogFont;
    Procedure GetDialogInfo (aPtr: Pointer);
    Procedure UpdateDialog;
    Function  RunModal: Integer; Virtual;
    Procedure EndDlg (aRetValue: Integer); Virtual;
    Procedure Ok (Var Msg: tMessage); Virtual id_First+id_Ok;
    Procedure Cancel (Var Msg: tMessage); Virtual id_First+id_Cancel;
    Procedure SaveFocus;
    Procedure wmClose (Var Msg: tMessage); Virtual wm_First+wm_Close;
    Procedure wmNCActivate (Var Msg: tMessage); Virtual wm_First+wm_NCActivate;
    Procedure wmSetFocus (Var Msg:  tMessage); Virtual wm_First+wm_SetFocus;
    Procedure wmKillFocus (Var Msg:  tMessage); Virtual wm_First+wm_KillFocus;
    Procedure wmSysCommand (Var Msg:  tMessage); Virtual wm_First+wm_SysCommand;
    Procedure wmSize(Var Msg: tMessage); Virtual wm_First+wm_Size;
  End;

  pJanusDialogWindow = ^tJanusDialogWindow;
  tJanusDialogWindow = Object(tDialogWindow)
    useBWCC: Boolean;
    Constructor Init (aParent: pWindowsObject; aName: pChar; BorStyle: Boolean);
    Function  GetWindowProc: tFarProc; Virtual;
    Procedure GetChildClass (Var aChildClass: tChildClass); Virtual;
  End;

  Function ExecDialogWindow (aDialogWindow: pDialogWindow): Integer;

Implementation
Uses
  WinProcs,
  Strings;
Const
  sztDialogWindow = 'tDialogWindow';

  BorDialog = 'BorDlg';
  BorButton = 'BorBtn';
  BorRadio  = 'BorRadio';
  BorCheck  = 'BorCheck';
  BorShade  = 'BorShade';

  bss_Group = 1; {group box}
  bss_Hdip  = 2; {horizontal border}
  bss_Vdip  = 3; {hertical border}
  bss_Hbump = 4; {horizontal speed bump}
  bss_Vbump = 5; {vertical speed bump}

Const
  BWCCInst: tHandle = tHandle(0);

Function DlgToClientX (x, Units: Integer): Integer;
{DlgToClientX:= x*Units Div 4}
Inline($59/$58/    {Pop Cx Ax}
       $F7/$E1/    {Mul Cx}
       $D1/$E8/    {Shr Ax,1}
       $D1/$E8);   {Shr Ax,1}

Function DlgToClientY (y, Units: Integer): Integer;
{DlgToClientY:= y*Units Div 8}
Inline($59/$58/    {Pop Cx Ax}
       $F7/$E1/    {Mul Cx}
       $D1/$E8/    {Shr Ax,1}
       $D1/$E8/    {Shr Ax,1}
       $D1/$E8);   {Shr Ax,1}

Constructor tDialogWindow.Init (aParent: pWindowsObject; aName: pChar);
Begin
  tWindow.Init(aParent,sztDialogWindow); {fake title}
  FillChar(DialogAttr,SizeOf(DialogAttr),0);
  ModalCode:= Nil;                       {assume modeless window}
  With DialogAttr Do
    If PtrRec(aName).Seg=0 Then Name:= aName Else Name:= StrNew(aName);
  DefaultProc:= GetWindowProc
End;

Destructor tDialogWindow.Done;
Begin
  With DialogAttr Do Begin
    If PtrRec(Name).Seg<>0 Then StrDispose(Name);
    StrDispose(MenuName);
    StrDispose(ClassName);
    If FontName<>Nil Then
      StrDispose(FontName)
  End;
  tWindow.Done
End;

Function tDialogWindow.Create: Boolean;
Var
  aRes: tHandle;
Begin
  Create:= False;
  If (Status<>0) Or (DialogAttr.Name=Nil) Then
    Exit;
  aRes:= LoadResource(hInstance,
                      FindResource(hInstance, DialogAttr.Name, rt_Dialog));
  If aRes=0 Then
    Status:= em_InvalidWindow
  Else Begin
    If ModalCode<>Nil Then
      If Parent=Nil Then Begin
        Status:= em_InvalidWindow;
        Exit
      End Else Begin
        EnableWindow(Parent^.hWindow,False); {disable Parent}
        ModalCode^:= 0                       {begin modal state}
      End;
    GetDialogInfo(LockResource(aRes));
    CreateDialogFont;
    UpdateDialog;
    DisableAutoCreate;
    EnableKBHandler;
    Create:= tWindow.Create;
    UnlockResource(aRes);
    FreeResource(aRes)
  End
End;

Procedure tDialogWindow.Destroy;
Begin
  If (ModalCode<>Nil) And (Parent<>Nil) Then Begin
    EnableWindow(Parent^.hWindow,True); {enable Parent}
    If ModalCode^=0 Then {terminate modal window if not already terminated}
      ModalCode^:= id_Cancel
  End;

  If DialogAttr.FontName<>Nil Then
    DeleteObject(DialogAttr.Font);
  tWindow.Destroy
End;

Procedure tDialogWindow.SetupWindow;
Begin
  tWindow.SetupWindow;
  SendMessage(hWindow,wm_SetFont,DialogAttr.Font,0);
  CreateDialogChildren;
  If  (Scroller<>Nil) Then
  With Scroller^ Do Begin
    {fix BWCC background quirk}
    XUnit:= (XUnit+1) And Not 1; { make even }
    YUnit:= (YUnit+1) And Not 1
  End
End;

Function tDialogWindow.GetWindowProc: tFarProc;
Begin
  If IsFlagSet(wb_MDIChild) Then
    GetWindowProc:= @DefMdiChildProc
  Else
    GetWindowProc:= @DefWindowProc
End;

Function tDialogWindow.GetClassName: pChar;
Begin
  If ModalCode<>Nil Then
    GetClassName:= pChar(32770)
  Else
    GetClassName:= sztDialogWindow;
End;

Procedure tDialogWindow.GetChildClass (Var aChildClass: tChildClass);
{-change a childs window class. Standard windows behaviour is simulated here:
  change special resource shortcuts (#$80..#$85) to their appropriate class names}
Const
  PreDefClasses: Array[#$80..#$85] Of pChar =
    ('BUTTON','EDIT','STATIC','LISTBOX','SCROLLBAR','COMBOBOX');
Begin
  With aChildClass Do
    Case szClass[0] Of
      #$80..#$85: StrCopy(szClass,PreDefClasses[szClass[0]])
    End
End;

Function tDialogWindow.CreateDialogChild (Var aChildClass: tChildClass): hWnd;
Var
  aCtl: hWnd;
  lpDlgItemInfo: Pointer;
Begin
  With DialogAttr, aChildClass Do Begin
    If CtlDataSize=0 Then
      lpDlgItemInfo:= Nil
    Else
      lpDlgItemInfo:= @CtlData;
    aCtl:= CreateWindowEx(ws_Ex_NoParentNotify, szClass, szTitle, dwStyle,
                          DlgToClientX(wX,wUnitsX),  DlgToClientY(wY,wUnitsY),
                          DlgToClientX(wCX,wUnitsX), DlgToClientY(wCY,wUnitsY),
                          hWindow, wID, System.hInstance,
                          lpDlgItemInfo);
    If aCtl<>0 Then
      SendMessage(aCtl, wm_SetFont, Font, 0)
  End;
  CreateDialogChild:= aCtl
End;

Procedure tDialogWindow.CreateDialogChildren;
Var
  i: Integer;
  sp: Pointer;
  anItem: tChildClass;
  aCtl: hWnd;
Begin
  sp:= DialogAttr.DlgItems;
  With DialogAttr,anItem Do
  For i:= 1 To DialogAttr.ItemCount Do Begin
    {-copy fixed header and first byte of szClass}
    Move(sp^,anItem,15); Inc(Word(sp),15);
    Case szClass[0] Of
      #$80..#$85: szClass[1]:= #0;   {be safe}
    Else
      StrCopy(szClass+1,sp);       {copy rest of classname}
      Inc(Word(sp),StrLen(sp)+1)
    End;
    StrCopy(szTitle,sp); Inc(Word(sp),StrLen(sp)+1);
    Move(sp^,CtlDataSize,Byte(sp^)+1);
    Inc(Word(sp),CtlDataSize+1);
    {-maybe a descendant class wants to change child class :-) }
    GetChildClass(anItem);
    aCtl:= CreateDialogChild(anItem);
    If aCtl=0 Then Begin
      Status:= em_InvalidChild;
      Exit
    End Else
      If (FocusChildHandle=0) And (anItem.dwStyle And ws_TabStop>0) Then
        FocusChildHandle:= aCtl
  End
End;

Procedure tDialogWindow.GetDialogInfo (aPtr: Pointer);
Begin
  With Attr,DialogAttr Do Begin
    Style:= LongInt(aPtr^);   Inc(Word(aPtr),SizeOf(LongInt));
    ItemCount:= Byte(aPtr^);  Inc(Word(aPtr),SizeOf(Byte));
    If Not IsFlagSet(wb_MdiChild) Then
      X:= Integer(aPtr^);     Inc(Word(aPtr),SizeOf(Integer));
    Y:= Integer(aPtr^);       Inc(Word(aPtr),SizeOf(Integer));
    W:= Integer(aPtr^);       Inc(Word(aPtr),SizeOf(Integer));
    H:= Integer(aPtr^);       Inc(Word(aPtr),SizeOf(Integer));
    MenuName:= StrNew(aPtr);  Inc(Word(aPtr),StrLen(aPtr)+1);
    ClassName:= StrNew(aPtr); Inc(Word(aPtr),StrLen(aPtr)+1);
    Title:= StrNew(aPtr);     Inc(Word(aPtr),StrLen(aPtr)+1);
    If Style And ds_SetFont>0 Then Begin
      PointSize:= Integer(aPtr^); Inc(Word(aPtr),SizeOf(Integer));
      FontName:= StrNew(aPtr); Inc(Word(aPtr),StrLen(aPtr)+1)
    End Else Begin
      PointSize:= 0;
      FontName:= Nil
    End;
    DlgItems:= aPtr
  End
End;

Procedure tDialogWindow.UpdateDialog;
{-update and resize dialog window according to its style}
Var
  TheMDIClient: pMdiClient;
  aRect: tRect;
Begin With Attr, DialogAttr Do Begin
  {-update style bits for MDI}
  If isFlagSet(wb_MdiChild) Then Begin
    {-reject use of ws_PopUp for a MDI child!}
    If Style And ws_PopUp<>0 Then
      Style:= (Style Or ws_Child) And Not ws_PopUp;
    TheMDIClient:= Parent^.GetClient;
    {-check if the Client window has the MDIs_allChildStyles bit set}
    If (TheMDIClient=Nil)
    Or (GetWindowLong(TheMDIClient^.hWindow,gwl_Style) And 1=0) Then
      Style:= Style Or ws_Child Or ws_ClipSiblings Or ws_ClipChildren
                    Or ws_SysMenu Or ws_Caption Or ws_ThickFrame
                    Or ws_MinimizeBox Or ws_MaximizeBox
  End;

  {-resize the window according to its style and size}
  With aRect Do Begin
    left:= 0;
    top:= 0;
    right:=  DlgToClientX(w, wUnitsX);
    bottom:= DlgToClientY(h, wUnitsY);
    AdjustWindowRect(aRect, Style, Menu<>0);
    w:= right-left;
    h:= bottom-top;
    ResW:= w;
    ResH:= h;
  End
End End;

Procedure tDialogWindow.CreateDialogFont;
{-create the dialog font and calculate dialog units based on font}
Const
  aWidthString = 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz';
Var
  aDC: hDC;
  anOldFont: hFont;
  aLogFont: tLogFont;
  aTextMetric: tTextMetric;
Begin With DialogAttr Do Begin
  aDC:= GetDC(0);
  If FontName=Nil Then
    Font:= GetStockObject(System_Font)
  Else Begin
    FillChar(aLogFont,SizeOf(aLogFont),0);
    With aLogFont Do Begin
      StrCopy(lfFaceName,FontName);
      lfHeight:= -MulDiv(DialogAttr.PointSize,GetDeviceCaps(aDC, LogPixelsY),72);
      lfWeight:= FW_BOLD
    End;
    Font:= CreateFontIndirect(aLogFont)
  End;
  anOldFont:= SelectObject(aDC, Font);
  GetTextMetrics(aDC, aTextMetric);
  {-use the Microsoft recommended way to retrieve average width}
  wUnitsX:= Word(GetTextExtent(aDC, aWidthString, Length(aWidthString))) Div Length(aWidthString);
  wUnitsY:= aTextMetric.tmHeight;
  SelectObject(aDC, anOldFont);
  ReleaseDC(0, aDC)
End End;

Function tDialogWindow.RunModal: Integer;
Var
  aMsg: tMsg;
  ReturnCode: Integer;
Begin
  ReturnCode:= 0;
  ModalCode:= @ReturnCode;  {Trick OWL}
  SetFlags(wb_MDIChild, False);
  DefaultProc:= GetWindowProc;
  Create;

  If Status<>0 Then Begin
    RunModal:= Status;
    Exit
  End;

  Attr.Style:= Attr.Style Or ws_Visible;
  If Attr.Style And ds_SysModal>0 Then
    SetSysModalWindow(hWindow); {support SysModal dialogs as well}
  Repeat
    If PeekMessage(aMsg, 0, 0, 0, pm_Remove) Then
    If Not Application^.ProcessDlgMsg(aMsg) Then Begin
      TranslateMessage(aMsg);
      DispatchMessage(aMsg)
    End
  Until ReturnCode<>0; {until window is no longer modal}

  RunModal:= ReturnCode
End;

Procedure tDialogWindow.EndDlg (aRetValue: Integer);
Begin
  If ModalCode<>Nil Then {set return code if it's a modal window}
    ModalCode^:= aRetValue;
  CloseWindow
End;

Procedure tDialogWindow.Ok (Var Msg: tMessage);
Begin
  If ModalCode=Nil Then
    CloseWindow
  Else
    If CanClose Then Begin
      TransferData(tf_GetData);
      EndDlg(id_Ok)
    End
End;

Procedure tDialogWindow.Cancel (Var Msg: tMessage);
Begin
  EndDlg(id_Cancel)
End;

Procedure tDialogWindow.wmClose (Var Msg:  tMessage);
Begin
  Cancel(Msg)
End;

Procedure tDialogWindow.SaveFocus;
Var
  aFocus: hWnd;
Begin
  aFocus:= GetFocus;
  If (aFocus<>0) And IsChild(hWindow, aFocus) Then
    FocusChildHandle:= aFocus
End;

Procedure tDialogWindow.wmNCActivate (Var Msg: tMessage);
{-save the Focus when we switch to another app}
Begin
  If Msg.wParam=0 Then
    SaveFocus;
  DefWndProc(Msg)
End;

Procedure tDialogWindow.wmSetFocus (Var Msg: tMessage);
Begin
  If IsFlagSet(wb_KBHandler) And Not IsIconic(hWindow) Then Begin
    Application^.SetKBHandler(@Self);
    If (FocusChildHandle<>0) And IsWindow(FocusChildHandle) Then Begin
      SetFocus(FocusChildHandle);
      Exit
    End
  End Else
    Application^.SetKBHandler(Nil);
  DefWndProc(Msg)
End;

Procedure tDialogWindow.wmKillFocus (Var Msg: tMessage);
{-save the focus even when another MDI child is created}
Begin
  SaveFocus;
  DefWndProc(Msg)
End;

Procedure tDialogWindow.wmSysCommand (Var Msg: tMessage);
{-save the focus even when the MDI child is minimized}
Begin
  SaveFocus;
  DefWndProc(Msg)
End;

Procedure tDialogWindow.wmSize (Var Msg: TMessage);
Begin
  TWindow.WMSize(Msg);
  If Scroller<>Nil Then With Scroller^ Do Begin
    AutoOrg:= Msg.wParam<>sizeIconic;
    If AutoOrg Then Begin
      With DialogAttr, Attr Do
        SetRange(ResW-W, ResH-H);
      ScrollTo(0, 0);
      InvalidateRect(HWindow, nil, True)
    End
  End
End;

Function ExecDialogWindow (aDialogWindow: pDialogWindow): Integer;
Var
  ExecReturn: Integer;
Begin
  ExecDialogWindow:= id_Cancel;
  If Application^.ValidWindow(aDialogWindow)<>Nil Then Begin
    ExecReturn:= aDialogWindow^.RunModal;
    If ExecReturn<0 Then
      Application^.Error(ExecReturn)
    Else
      ExecDialogWindow:= ExecReturn;
  End
End;

Function LoadBWCC: Bool;
Var
  aWndClass: tWndClass;
  prevMode: Word;
Begin
  If BWCCInst=0 Then Begin
    prevMode:= SetErrorMode($8000); {SEM_NoOpenFileErrorBox}
    BWCCInst:= LoadLibrary('BWCC.DLL');
    SetErrorMode(prevMode);
    If BWCCInst<32 Then Begin
      LoadBWCC:= False;
      BWCCInst:= 0;
      Exit
    End
  End;
  LoadBWCC:= GetClassInfo(System.hInstance,BorButton,aWndClass)
End;

Procedure UnLoadBWCC;
Begin
  If BWCCInst<>0 Then Begin
    FreeLibrary(BWCCInst);
    BWCCInst:= 0
  End
End;

Constructor tJanusDialogWindow.Init (aParent: pWindowsObject; aName: pChar; BorStyle: Boolean);
Begin
  useBWCC:= BorStyle And LoadBWCC;
  tDialogWindow.Init(aParent, aName)
End;

Function tJanusDialogWindow.GetWindowProc: tFarProc;
Begin
  If useBWCC Then
    If IsFlagSet(wb_MDIChild) Then
      GetWindowProc:= GetProcAddress(BWCCInst,'BWCCDefMdiChildProc')
    Else
      GetWindowProc:= GetProcAddress(BWCCInst,'BWCCDefWindowProc')
  Else
    GetWindowProc:= tDialogWindow.GetWindowProc
End;

Procedure tJanusDialogWindow.GetChildClass (Var aChildClass: tChildClass);
Begin With aChildClass Do
  If useBWCC Then
    If szClass[0]=#$80 Then
      Case dwStyle And $F Of
        bs_CheckBox,
        bs_AutoCheckBox:        StrCopy(szClass,BorCheck);
        bs_RadioButton..bs_Auto3State,
        bs_AutoRadioButton:     StrCopy(szClass,BorRadio);
        bs_GroupBox:            StrCopy(szClass,BorShade);
      Else
        tDialogWindow.GetChildClass(aChildClass)
      End
    Else
      tDialogWindow.GetChildClass(aChildClass)
  Else
    If      (StrIComp(szClass,BorCheck)=0)
    Or      (StrIComp(szClass,BorRadio)=0)
    Or      (StrIComp(szClass,BorButton)=0) Then szClass[0]:= #$80
    Else If (StrIComp(szClass,BorShade)=0)  Then
      Case dwStyle And $F Of
        bss_Group: Begin szClass[0]:= #$80; dwStyle:= (dwStyle And $FFFFFFF0) Or bs_GroupBox End;
        bss_Hdip,
        bss_Hbump,
        bss_Vdip,
        bss_Vbump: Begin szClass[0]:= #$82; dwStyle:= (dwStyle And $FFFFFFF0) Or ss_BlackRect End;
      End;
    tDialogWindow.GetChildClass(aChildClass)
End;

Var
  PrevExit: Pointer;
Procedure DialogWnExit; Far;
Begin
  ExitProc:= PrevExit;
  UnLoadBWCC
End;

Begin
  PrevExit:= ExitProc;
  ExitProc:= @DialogWnExit
End.
