/* device.c:
 *
 * Handler for ISO-9660 (+ Rock Ridge) CDROM filing system.
 * Based on DOSDEV V1.10 (2 Nov 87) by Matthew Dillon.
 *
 * ----------------------------------------------------------------------
 * This code is (C) Copyright 1993 by Frank Munkert.
 * All rights reserved.
 * This software may be freely distributed and redistributed for
 * non-commercial purposes, provided this notice is included.
 */

/*
 *  Debugging routines are disabled by simply attempting to open the
 *  file "debugoff", turned on again with "debugon".  No prefix may be
 *  attached to these names (you must be CD'd to TEST:).
 *
 *  See Documentation for a detailed discussion.
 */
 
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <stdio.h>
#include <ctype.h>

#include "device.h"
#include "cdrom.h"
#include "iso9660.h"
#include "intui.h"

/*
 *  Since this code might be called several times in a row without being
 *  unloaded, you CANNOT ASSUME GLOBALS HAVE BEEN ZERO'D!!  This also goes
 *  for any global/static assignments that might be changed by running the
 *  code.
 */

#define CHECK_INTERVAL 3   /* in seconds */

PROC	*DosProc;   /*	Our Process				    */
DEVNODE *DosNode;   /*	Our DOS node.. created by DOS for us	    */
DEVLIST *DevList;   /*	Device List structure for our volume node   */

void	*SysBase;   /*	EXEC library base			*/
DOSLIB	*DOSBase;   /*	DOS library base for debug process	*/

CDROM   *g_cd;
VOLUME  *g_volume;
CDROM_OBJ *g_top_level_obj;
char	*g_vol_name;

PORT *g_timer_mp;   		/*  timer message port		*/
struct timerequest *g_timer_io; /*  timer i/o request		*/
ULONG g_timer_sigbit;
ULONG g_dos_sigbit;

char	g_device[80];		/* SCSI device name */
short	g_unit;			/* SCSI unit */
short	g_use_rock_ridge;	/* Use Rock Ridge flag 'R' */
short	g_map_to_lowercase;	/* Map to lower case flag 'L' */

#ifndef NDEBUG
		    /*	DEBUGGING			*/
PORT *Dbport;	    /*	owned by the debug process	*/
PORT *Dback;	    /*	owned by the DOS device driver	*/
short DBDisable;
MSG DummyMsg;	    /*	Dummy message that debug proc can use	*/
#endif

void *dosalloc(ulong);
void dosfree (ulong *);
void btos(LONG, char *);
void *NextNode (NODE *);
void *GetHead (LIST *);
LOCK *cdlock(CDROM_OBJ *, int);
void cdunlock (LOCK *);
CDROM_OBJ *getlockfile (LONG);
char *typetostr (int);
void returnpacket(struct DosPacket *);
int packetsqueued (void);
int Check_For_Volume_Name_Prefix (char *);
void Fill_FileInfoBlock (FIB *, CDROM_INFO *, VOLUME *);
void Copy_ISO_Identifier (char *, char *, int);
void Mount (void);
void Unmount (void);
int Mount_Check (void);
void Check_Disk (void);
void Send_Timer_Request (void);
void Cleanup_Timer_Device (void);
int Open_Timer_Device (void);
int Get_Startup (LONG);
#ifndef NDEBUG
void dbinit (void);
void dbuninit (void);
void dbprintf (char *, ...);
#endif

/*
 *  Don't call the entry point main().  This way, if you make a mistake
 *  with the compile options you'll get a link error.
 */

void noname (void)
{
    register PACKET *packet;
    register short   error;
    MSG     *msg;
    ubyte   notdone = 1;
    char    buf[256];
    void    *tmp;
    ULONG   signals;

    /*
     *	Initialize all global variables.  SysBase MUST be initialized before
     *	we can make Exec calls.  AbsExecBase is a library symbol
     *	referencing absolute memory location 4.  The DOS library is openned
     *	for the debug process only.
     */

    BUG(DBDisable = 0;)				/*  Init. globals	*/
    BUG(Dbport = Dback = NULL;)
    SysBase = AbsExecBase;
    DOSBase = (DOSLIB *) OpenLibrary ((UBYTE *) "dos.library",0);
    DosProc = (PROC *) FindTask(NULL);
    DevList = NULL;
    {
	WaitPort(&DosProc->pr_MsgPort); 	/*  Get Startup Packet	*/
	msg = GetMsg(&DosProc->pr_MsgPort);
	packet = (PACKET *)msg->mn_Node.ln_Name;

	/*
	 *  Loading DosNode->dn_Task causes DOS *NOT* to startup a new
	 *  instance of the device driver for every reference.	E.G. if
	 *  you were writing a CON device you would want this field to
	 *  be NULL.
	 */

        DosNode = BTOC(packet->dp_Arg3);

	Init_Intui ();

	if (DOSBase && Get_Startup (packet->dp_Arg2)) {
	    /*
	     *	Set dn_Task field which tells DOS not to startup a new
	     *	process on every reference.
	     */

  	    DosNode->dn_Task = &DosProc->pr_MsgPort;
	    packet->dp_Res1 = DOS_TRUE;
 	    packet->dp_Res2 = 0;

	} else {			    /*	couldn't open dos.library   */
	    packet->dp_Res1 = DOS_FALSE;
	    returnpacket(packet);
	    return;			    /*	exit process		    */
	}
	returnpacket(packet);
    }

    /*
     *	Initialize debugging code
     */

    BUG(dbinit();)

    /* Initialize timer: */
    if (Open_Timer_Device ())
      Send_Timer_Request ();
    else
      g_timer_sigbit = 0;

    /* Mount volume (if any disk is inserted): */
    Mount ();

    g_dos_sigbit = 1L << DosProc->pr_MsgPort.mp_SigBit;

    /*
     *	Here begins the endless loop, waiting for requests over our
     *	message port and executing them.  Since requests are sent over
     *	our message port, this precludes being able to call DOS functions
     *	ourselves (that is why the debugging routines are a separate process)
     */

top:
    for (; notdone;) {
	signals = Wait(g_dos_sigbit | g_timer_sigbit);
	if (signals & g_timer_sigbit) {
	  GetMsg (g_timer_mp);
	  Check_Disk ();
	  Send_Timer_Request ();
	}
	if (!(signals & g_dos_sigbit))
	  continue;
	while (msg = GetMsg(&DosProc->pr_MsgPort)) {
	    register ubyte *ptr;
	    packet = (PACKET *)msg->mn_Node.ln_Name;
	    packet->dp_Res1 = DOS_TRUE;
	    packet->dp_Res2 = 0;
	    error = 0;
#ifndef NDEBUG
	    dbprintf("Packet: %3ld %08lx %08lx %08lx %10s ",
		packet->dp_Type,
		packet->dp_Arg1, packet->dp_Arg2,
		packet->dp_Arg3,
		typetostr(packet->dp_Type)
	    );
#endif


	    switch(packet->dp_Type) {
	    case ACTION_DIE:	    /*	attempt to die? 		    */
		notdone = 0;	    /*	try to die			    */
		break;
	    case ACTION_FINDINPUT:  /*  FileHandle,Lock,Name        Bool    */
		{
		    if (Mount_Check ()) {
		      CDROM_OBJ *obj;
		      CDROM_OBJ *parentdir = getlockfile(packet->dp_Arg2);
		      char      *ptr;
		      int       offs;

		      btos(packet->dp_Arg3,buf);
		      BUG(dbprintf("'%s' ", buf);)
		      offs = Check_For_Volume_Name_Prefix (buf);
		      if (obj = Open_Object (parentdir, buf + offs)) {
			if (obj->directory_f) {
			    error = ERROR_OBJECT_WRONG_TYPE;
			    goto openbreak;
			}
		      } else {
			if (iso_errno == ISOERR_ILLEGAL_NAME) {
			    error = ERROR_INVALID_COMPONENT_NAME;
			    goto openbreak;
			} else if (iso_errno == ISOERR_NOT_FOUND)
		          error = ERROR_OBJECT_NOT_FOUND;
			else if (iso_errno == ISOERR_NO_MEMORY) {
			  error = ERROR_NO_FREE_STORE;
			  goto openbreak;
			} else {
			  error = 333;
			  goto openbreak;
			}
		      }
		      if (!error) {
		  	((FH *)BTOC(packet->dp_Arg1))->fh_Arg1 = (long) obj;
		      }
		    } else
		      error = ERROR_NO_DISK;
		}
	      openbreak:
		break;
	    case ACTION_READ:	    /*	 FHArg1,CPTRBuffer,Length   ActLength  */
		{
		    CDROM_OBJ *obj = (CDROM_OBJ *) packet->dp_Arg1;
		    char      *ptr = (char *) packet->dp_Arg2;
		    long    length = packet->dp_Arg3;
		    int     actual;

		    actual = Read_From_File (obj, ptr, length);
		    packet->dp_Res1 = actual;
		}
		break;
	    case ACTION_END:	    /*	 FHArg1 		    Bool:TRUE  */
		{
		    CDROM_OBJ *obj = (CDROM_OBJ *) packet->dp_Arg1;

		    Close_Object (obj);
		}
		break;
	    case ACTION_SEEK:	    /*	 FHArg1,Position,Mode	    OldPosition*/
	        {
		    CDROM_OBJ *obj = (CDROM_OBJ *) packet->dp_Arg1;
		    long offset = packet->dp_Arg2;
		    int mode = packet->dp_Arg3;
		    
		    packet->dp_Res1 = obj->pos;
		    if (!Seek_Position (obj, offset, mode)) {
		      error = ERROR_SEEK_ERROR;
		      packet->dp_Res1 = -1;
		    }
		}
		break;
	    case ACTION_EXAMINE_NEXT: /*   Lock,Fib		      Bool	 */
		{
		    FIB       *fib = BTOC (packet->dp_Arg2);
		    CDROM_OBJ *dir = getlockfile (packet->dp_Arg1);
		    CDROM_INFO info;

		    if (!dir->directory_f) {
			error = ERROR_OBJECT_WRONG_TYPE;
			break;
		    }
		    if (Examine_Next (dir, &info,
		    		      (unsigned long *) &fib->fib_DiskKey)) {
		      error = 0;
		      Fill_FileInfoBlock (fib, &info, dir->volume);
		    } else {
		      error = ERROR_NO_MORE_ENTRIES;
		    }
		    break;
		}
	    case ACTION_EXAMINE_OBJECT: /*   Lock,Fib			Bool	   */
		{
		    FIB *fib = BTOC (packet->dp_Arg2);
                    CDROM_OBJ *obj = getlockfile (packet->dp_Arg1);
		    CDROM_INFO info;

		    fib->fib_DiskKey = 0;
		    error = 0;
		    if (!CDROM_Info (obj, &info))
		      error = -1;
		    else
		      Fill_FileInfoBlock (fib, &info, obj->volume);
		}
		break;
	    case ACTION_INFO:	    /*	Lock, InfoData	  Bool:TRUE    */
		tmp = BTOC(packet->dp_Arg2);
		error = -1;
		/*  fall through    */
	    case ACTION_DISK_INFO:  /*	InfoData	  Bool:TRUE    */
		{
		  if (Mount_Check ()) {
		    register INFODATA *id;

		    /*
		     *	Note:	id_NumBlocks is never 0, but only to get
		     *	around a bug I found in my shell (where I divide
		     *	by id_NumBlocks).  Other programs probably break
		     *	as well.
		     */

		    (error) ? (id = tmp) : (id = BTOC (packet->dp_Arg1));
		    error = 0;
		    memset (id, 0, sizeof(*id));
		    id->id_DiskState = ID_WRITE_PROTECTED;
		    id->id_NumBlocks	 = g_volume->pvd.space_size_m;
		    id->id_NumBlocksUsed = g_volume->pvd.space_size_m;
		    id->id_BytesPerBlock = 2048;
		    id->id_DiskType = ID_DOS_DISK;
		    id->id_VolumeNode = (long) CTOB (DosNode);
		    id->id_InUse = 0;
		  }
		}
		break;
	    case ACTION_PARENT:     /*	 Lock			    ParentLock */
	        {
		  if (Mount_Check ()) {
		    if (packet->dp_Arg1) {
		      CDROM_OBJ *obj = getlockfile (packet->dp_Arg1);
		      CDROM_OBJ *parent;
		  
		      if (Is_Top_Level_Object (obj)) {
		        packet->dp_Res1 = packet->dp_Res2 = 0;
		      } else {
		        parent = Find_Parent (obj);
		        if (!parent) {
		          if (iso_errno == ISOERR_NO_MEMORY)
		            error = ERROR_NO_FREE_STORE;
		          else
		            error = ERROR_OBJECT_NOT_FOUND;
		        } else {
		          packet->dp_Res1 = (long)
			    CTOB (cdlock (parent, ACCESS_READ));
		        }
		      }
		    } else
		      error = ERROR_OBJECT_NOT_FOUND;
		  } else
		    error = ERROR_NO_DISK;
		}
		break;
	    case ACTION_LOCATE_OBJECT:	/*   Lock,Name,Mode		Lock	   */
		{
		  if (Mount_Check ()) {
		    CDROM_OBJ *parentdir = getlockfile (packet->dp_Arg1);
                    CDROM_OBJ *obj;
		    int offs;

		    btos (packet->dp_Arg2, buf);
#ifndef NDEBUG
		    dbprintf ("'%s' %ld ", buf, packet->dp_Arg3);
		    if (strcmp(buf,"debugoff") == 0)
			DBDisable = 1;
		    if (strcmp(buf,"debugon") == 0)
			DBDisable = 0;
#endif

		    offs = Check_For_Volume_Name_Prefix (buf);
		    if (buf[offs]==0) {
		      if (parentdir)
		        obj = Clone_Object (parentdir);
		      else
		        obj = Open_Top_Level_Directory (g_volume);
		    } else
		      obj = Open_Object (parentdir, buf + offs);
		    
		    if (obj) {
		      packet->dp_Res1 = (long) CTOB (cdlock (obj, packet->dp_Arg3));
		    } else {
		      if (iso_errno == ISOERR_SCSI_ERROR) {
		        error = ERROR_OBJECT_NOT_FOUND;
			Unmount ();
		      } else if (iso_errno == ISOERR_ILLEGAL_NAME)
			error = ERROR_INVALID_COMPONENT_NAME;
		      else if (iso_errno == ISOERR_NOT_FOUND)
			error = ERROR_OBJECT_NOT_FOUND;
		      else if (iso_errno == ISOERR_NO_MEMORY)
		        error = ERROR_NO_FREE_STORE;
		      else
		        error = 333;
		    }
		  } else
		    error = ERROR_NO_DISK;
		}
		break;
	    case ACTION_COPY_DIR:   /*	 Lock,			    Lock       */
	    	{
		  if (packet->dp_Arg1) {
		    CDROM_OBJ *obj = getlockfile (packet->dp_Arg1);
		    CDROM_OBJ *new = Clone_Object (obj);
		  
		    if (!new)
		      error = ERROR_NO_FREE_STORE;
		    else
		      packet->dp_Res1 = (long) CTOB (cdlock (new, ACCESS_READ));
		  } else
		    packet->dp_Res1 = 0;
		}
		break;
	    case ACTION_FREE_LOCK:  /*	 Lock,			    Bool       */
		if (packet->dp_Arg1);
		    cdunlock (BTOC(packet->dp_Arg1));
		break;
	    /*
	     *	A few other packet types which we do not support
	     */
	    case ACTION_INHIBIT:    /*	 Bool			    Bool       */
		/*  Return success for the hell of it	*/
		break;
	    case ACTION_RENAME_DISK:/*	 BSTR:NewName		    Bool       */
	    case ACTION_MORE_CACHE: /*	 #BufsToAdd		    Bool       */
	    case ACTION_WAIT_CHAR:  /*	 Timeout, ticks 	    Bool       */
	    case ACTION_FLUSH:	    /*	 writeout bufs, disk motor off	       */
	    case ACTION_SCREEN_MODE:/*	 Bool(-1:RAW 0:CON)	    OldState   */
	    default:
		error = ERROR_ACTION_NOT_KNOWN;
		break;
	    }
	    if (packet) {
		if (error) {
		    BUG(dbprintf("ERR=%ld\n", error);)
		    packet->dp_Res1 = DOS_FALSE;
		    packet->dp_Res2 = error;
		} else {
		    BUG(dbprintf("RES=%06lx\n", packet->dp_Res1));
		}
		returnpacket(packet);
	    }
	}
    }
    BUG(dbprintf("Can we remove ourselves? ");)
    Delay(100);	    /*	I wanna even see the debug message! */
    Forbid();
    if (packetsqueued()) {
	Permit();
	BUG(dbprintf(" ..  not yet!\n");)
	goto top;		/*  sorry... can't exit     */
    }

    /*
     *	Causes a new process to be created on next reference
     */

    DosNode->dn_Task = FALSE;

    Unmount ();
    
    Cleanup_CDROM (g_cd);

    /* remove timer device and any pending timer requests: */
    if (g_timer_sigbit)
      Cleanup_Timer_Device ();

    Close_Intui ();

    /*
     *	Remove debug process, closedown, fall of the end of the world
     *	(which is how you kill yourself if a PROCESS.  A TASK would have
     *	had to RemTask(NULL) itself).
     */

    BUG(dbuninit();)
    CloseLibrary ((struct Library *) DOSBase);
}


/*
 *  PACKET ROUTINES.	Dos Packets are in a rather strange format as you
 *  can see by this and how the PACKET structure is extracted in the
 *  GetMsg() of the main routine.
 */

void returnpacket(struct DosPacket *packet)
{
    register struct Message *mess;
    register struct MsgPort *replyport;

    replyport		     = packet->dp_Port;
    mess		     = packet->dp_Link;
    packet->dp_Port	     = &DosProc->pr_MsgPort;
    mess->mn_Node.ln_Name    = (char *)packet;
    mess->mn_Node.ln_Succ    = NULL;
    mess->mn_Node.ln_Pred    = NULL;
    PutMsg(replyport, mess);
}

/*
 *  Are there any packets queued to our device?
 */

int packetsqueued (void)
{
    return ((void *)DosProc->pr_MsgPort.mp_MsgList.lh_Head !=
	    (void *)&DosProc->pr_MsgPort.mp_MsgList.lh_Tail);
}

/*
 *  DOS MEMORY ROUTINES
 *
 *  DOS makes certain assumptions about LOCKS.	A lock must minimally be
 *  a FileLock structure, with additional private information after the
 *  FileLock structure.  The longword before the beginning of the structure
 *  must contain the length of structure + 4.
 *
 *  NOTE!!!!! The workbench does not follow the rules and assumes it can
 *  copy lock structures.  This means that if you want to be workbench
 *  compatible, your lock structures must be EXACTLY sizeof(struct FileLock).
 */

void *dosalloc(ulong bytes)
{
    register ulong *ptr;

    bytes += 4;
    ptr = AllocMem(bytes, MEMF_PUBLIC|MEMF_CLEAR);
    *ptr = bytes;
    return(ptr+1);
}

void dosfree (ulong *ptr)
{
    --ptr;
    FreeMem(ptr, *ptr);
}

/*
 *  Convert a BSTR into a normal string.. copying the string into buf.
 *  I use normal strings for internal storage, and convert back and forth
 *  when required.
 */

void btos(LONG bstr, char *buf)
{
    unsigned char *str = BTOC(bstr);
    bmov((char *) str+1, buf, *str);
    buf[*str] = 0;
}

/*
 *  Some EXEC list handling routines not found in the EXEC library.
 */

void *NextNode (NODE *node)
{
    node = node->mln_Succ;
    if (node->mln_Succ == NULL)
	return(NULL);
    return(node);
}

void *GetHead (LIST *list)
{
    if ((void *)list->mlh_Head != (void *)&list->mlh_Tail)
	return(list->mlh_Head);
    return(NULL);
}

/*
 *  The lock function.	The file has already been checked to see if it
 *  is lockable given the mode.
 */

LOCK *cdlock(CDROM_OBJ *cdfile, int mode)
{
  LOCK *lock = dosalloc (sizeof(LOCK));

  cdfile->volume->locks++;
  lock->fl_Key = (long) cdfile;
  lock->fl_Access = ACCESS_READ;
  lock->fl_Task = &DosProc->pr_MsgPort;
  lock->fl_Volume = (BPTR) CTOB (DosNode);
  return(lock);
}

void cdunlock (LOCK *lock)
{
  CDROM_OBJ *obj = (CDROM_OBJ *) lock->fl_Key;

  if (--obj->volume->locks == 0 && !obj->volume->valid) {
    Close_Volume (obj->volume);
  }
  Close_Object (obj);
  dosfree ((ulong *) lock);				/* free lock	    */
}

/*
 *  GETLOCKFILE(bptrlock)
 *
 *  Return the CDROM_OBJ (file or directory) associated with the
 *  given lock, which is passed as a BPTR.
 *
 *  According to the DOS spec, the only way a NULL lock will ever be
 *  passed to you is if the DosNode->dn_Lock is NULL, but I'm not sure.
 *  In anycase, If a NULL lock is passed to me I simply assume it means
 *  the root directory of the CDROM.
 */

CDROM_OBJ *getlockfile (LONG lock)
{
  LOCK *rl = BTOC (lock);

  if (rl)
    return (CDROM_OBJ *) rl->fl_Key;
  return g_top_level_obj;
}

char *typetostr (int ty)
{
    switch(ty) {
    case ACTION_DIE:		return("DIE");
    case ACTION_FINDUPDATE: 	return("OPEN-RW");
    case ACTION_FINDINPUT:	return("OPEN-OLD");
    case ACTION_FINDOUTPUT:	return("OPEN-NEW");
    case ACTION_READ:		return("READ");
    case ACTION_WRITE:		return("WRITE");
    case ACTION_END:		return("CLOSE");
    case ACTION_SEEK:		return("SEEK");
    case ACTION_EXAMINE_NEXT:	return("EXAMINE NEXT");
    case ACTION_EXAMINE_OBJECT: return("EXAMINE OBJ");
    case ACTION_INFO:		return("INFO");
    case ACTION_DISK_INFO:	return("DISK INFO");
    case ACTION_PARENT: 	return("PARENTDIR");
    case ACTION_DELETE_OBJECT:	return("DELETE");
    case ACTION_CREATE_DIR:	return("CREATEDIR");
    case ACTION_LOCATE_OBJECT:	return("LOCK");
    case ACTION_COPY_DIR:	return("DUPLOCK");
    case ACTION_FREE_LOCK:	return("FREELOCK");
    case ACTION_SET_PROTECT:	return("SETPROTECT");
    case ACTION_SET_COMMENT:	return("SETCOMMENT");
    case ACTION_RENAME_OBJECT:	return("RENAME");
    case ACTION_INHIBIT:	return("INHIBIT");
    case ACTION_RENAME_DISK:	return("RENAME DISK");
    case ACTION_MORE_CACHE:	return("MORE CACHE");
    case ACTION_WAIT_CHAR:	return("WAIT FOR CHAR");
    case ACTION_FLUSH:		return("FLUSH");
    case ACTION_SCREEN_MODE:	return("SCREENMODE");
    default:			return("---------UNKNOWN-------");
    }
}

#ifndef NDEBUG

/*
 *  DEBUGGING CODE.	You cannot make DOS library calls that access other
 *  devices from within a DOS device driver because they use the same
 *  message port as the driver.  If you need to make such calls you must
 *  create a port and construct the DOS messages yourself.  I do not
 *  do this.  To get debugging info out another PROCESS is created to which
 *  debugging messages can be sent.
 *
 *  You want the priority of the debug process to be larger than the
 *  priority of your DOS handler.  This is so if your DOS handler crashes
 *  you have a better idea of where it died from the debugging messages
 *  (remember that the two processes are asyncronous from each other).
 */

extern void debugproc();

char *TheVersion = "$VER: CDROM-Handler " VERSION;

void dbinit (void)
{
    TASK *task = FindTask(NULL);

    Dback = CreatePort(NULL, 0);
    CreateProc((UBYTE *) "DEV_DB", task->tc_Node.ln_Pri+1, (BPTR) (CTOB(debugproc)),
    	       4096);
    WaitPort(Dback);				    /* handshake startup    */
    GetMsg(Dback);				    /* remove dummy msg     */
    dbprintf("Debugger running: %s, %s\n", TheVersion+6, __TIME__);
}

void dbuninit (void)
{
    MSG killmsg;

    if (Dbport) {
	killmsg.mn_Length = 0;	    /*	0 means die	    */
	PutMsg(Dbport,&killmsg);
	WaitPort(Dback);	    /*	He's dead jim!      */
	GetMsg(Dback);
	DeletePort(Dback);

	/*
	 *  Since the debug process is running at a greater priority, I
	 *  am pretty sure that it is guarenteed to be completely removed
	 *  before this task gets control again.  Still, it doesn't hurt...
	 */

	Delay(50);		    /*	ensure he's dead    */
    }
}

void dbprintf (char *format, ...)
{
    va_list arg;
    char buf[256];
    MSG *msg;

    va_start (arg, format);
    if (Dbport && !DBDisable) {
	vsprintf (buf, format, arg);
	msg = AllocMem(sizeof(MSG)+strlen(buf)+1, MEMF_PUBLIC|MEMF_CLEAR);
	msg->mn_Length = strlen(buf)+1;     /*	Length NEVER 0	*/
	strcpy((char *) (msg+1), buf);
	PutMsg(Dbport,msg);
    }
    va_end (arg);
}

/*
 *  BTW, the DOS library used by debugmain() was actually openned by
 *  the device driver.	Note: DummyMsg cannot be on debugmain()'s stack
 *  since debugmain() goes away on the final handshake.
 */

void debugmain (void)
{
    MSG *msg;
    short len;
    void *fh;

    Dbport = CreatePort(NULL, 0);
    fh = (void *) Open ((UBYTE *) "con:0/0/640/100/debugwindow", 1006);
    PutMsg(Dback, &DummyMsg);
    for (;;) {
	WaitPort(Dbport);
	msg = GetMsg(Dbport);
	len = msg->mn_Length;
	if (len == 0)
	    break;
	--len;			      /*  Fix length up   */
	Write((BPTR) fh, msg+1, len);
	FreeMem(msg,sizeof(MSG)+len+1);
    }
    Close ((BPTR) fh);
    DeletePort(Dbport);
    PutMsg(Dback,&DummyMsg);	      /*  Kill handshake  */
}

#endif /* !NDEBUG */

/*
 * If p_pathname contains a ':' character, return the position of the first
 * character after ':'
 * Otherwise, return 0.
 */

int Check_For_Volume_Name_Prefix (char *p_pathname)
{
  char *pos = strchr (p_pathname, ':');
  
  return pos ? (pos - p_pathname) + 1 : 0;
}

/*
 * Fills a FileInfoBlock with the information contained in the
 * directory record of a CD-ROM directory or file.
 */

void Fill_FileInfoBlock (FIB *p_fib, CDROM_INFO *p_info, VOLUME *p_volume)
{
  char *src = p_info->name;
  char *dest = p_fib->fib_FileName+1;
  int len = p_info->name_length;
  struct DateTime dt;
  char date[9], time[9];
  directory_record *dir = &p_info->dir_record;
  
  p_fib->fib_DirEntryType = (dir->flags & 2)?2:-3;
  
  /* I don't know exactly why I have to set fib_EntryType, but other
   * handlers (e.g. DiskHandler by J Toebes et.al.) also do this.
   */
   
  p_fib->fib_EntryType = p_fib->fib_DirEntryType;
  
  if (len == 1 && *src == ':') {
    /* root of file system: */
    p_fib->fib_DirEntryType = 1;
    /* file name == volume name: */
    memcpy (p_fib->fib_FileName, g_vol_name, g_vol_name[0]+1);
  } else {
    /* copy file name (without version number): */
    for (; len && *src != ';'; len--)
      *dest++ = *src++;
    p_fib->fib_FileName[0] = p_info->name_length-len;
  }
  p_fib->fib_Protection = 0;
  p_fib->fib_Size = dir->data_length_m;
  p_fib->fib_NumBlocks = p_fib->fib_Size >> 11;
  p_fib->fib_Comment[0] = 0;

  /* convert ISO date to Amiga date: */
  dt.dat_Format = FORMAT_CDN;
  dt.dat_Flags = 0;
  sprintf (date, "%02d-%02d-%02d",
  	   (int) dir->day, (int) dir->month, (int) dir->year);
  dt.dat_StrDate = (UBYTE *) date;
  sprintf (time, "%02d:%02d:%02d",
  	   (int) dir->hour, (int) dir->minute, (int) dir->second);
  dt.dat_StrTime = (UBYTE *) time;
  if (StrToDate (&dt))
    p_fib->fib_Date = dt.dat_Stamp;
  else {
    p_fib->fib_Date.ds_Days = 0;
    p_fib->fib_Date.ds_Minute = 0;
    p_fib->fib_Date.ds_Tick = 0;
  }
  
  if (g_map_to_lowercase && !p_volume->use_rock_ridge) {
    /* convert ISO filename to lowercase: */
    int i, len = p_fib->fib_FileName[0];
    char *cp = p_fib->fib_FileName + 1;
    
    for (i=0; i<len; i++, cp++)
      *cp = tolower (*cp);
  }
}

/*
 *  Copy the ISO identifier p_ident with maximum length p_length
 *  into the BCPL string buffer p_buf. Trailing blanks of p_ident
 *  will be stripped.
 */

void Copy_ISO_Identifier (char *p_buf, char *p_ident, int p_length)
{
  int len = p_length;
  
  while (len && p_ident[len-1] == ' ')
    len--;
  
  if (len)
    memcpy (p_buf+1, p_ident, len);
  
  p_buf[0] = len;
}

/*
 * Mount a volume.
 */

void Mount (void)
{
  DOSINFO *di = BTOC(((ROOTNODE *)DOSBase->dl_Root)->rn_Info);
  DEVLIST *dl;

  g_volume = Open_Volume (g_cd, g_use_rock_ridge);
  if (!g_volume) {
    BUG(dbprintf ("!!! cannot open VOLUME !!!\n");)
    return;
  } else {
    g_top_level_obj = Open_Top_Level_Directory (g_volume);
    if (!g_top_level_obj) {
      BUG(dbprintf ("!!! cannot open top level directory !!!\n");)
      return;
    }
  }
  
  BUG(dbprintf ("***mounting*** ");)
  
  g_vol_name = dosalloc (128);
  Copy_ISO_Identifier (g_vol_name, g_volume->pvd.volume_id, 32);

  /*
   *	Create Volume node and add to the device list.	This will
   *	cause the WORKBENCH to recognize us as a disk.	If we don't
   *	create a Volume node, Wb will not recognize us.
   */

  DevList = dl = dosalloc(sizeof(DEVLIST));
  dl->dl_Type = DLT_VOLUME;
  dl->dl_Task = &DosProc->pr_MsgPort;
  dl->dl_DiskType = ID_DOS_DISK;
  /* dl->dl_Name = (int)DosNode->dn_Name; */
  dl->dl_Name = (BSTR) CTOB (g_vol_name);
  dl->dl_Next = di->di_DevInfo;
  di->di_DevInfo = (long)CTOB(dl);
}

/*
 *	Remove Volume entry.  Since DOS uses singly linked lists, we
 *	must (ugg) search it manually to find the link before our
 *	Volume entry.
 */

void Unmount (void)
{
  DOSINFO *di = BTOC(((ROOTNODE *)DOSBase->dl_Root)->rn_Info);
  DEVLIST *dl;
  void *dlp;

  if (DevList) {
    dlp = &di->di_DevInfo;
    for (dl = BTOC(di->di_DevInfo); dl && dl != DevList; dl = BTOC(dl->dl_Next))
      dlp = &dl->dl_Next;
    if (dl == DevList) {
      *(BPTR *)dlp = dl->dl_Next;
      dosfree((ulong *) dl);
    } else {
      BUG(dbprintf("****PANIC: Unable to find volume node\n");)
    }
    DevList = NULL;

    Close_Object (g_top_level_obj);
    if (g_volume->locks == 0)
      Close_Volume (g_volume);
    else {
      g_volume->valid = 0;
    }
    dosfree ((ulong *) g_vol_name);
  }
}

/*
 * Mount_Check returns 1 if a disk is inserted in the drive. A check is
 * only performed if previously the drive was empty.
 */

int Mount_Check (void)
{
  if (DevList == NULL) {
   /*
    * No disk was inserted up to now: we will check whether
    * a disk has been inserted by sending the test unit ready
    * command. We have to send the command twice because
    * the first SCSI command after inserting a new disk is
    * always rejected.
    */
    if (Test_Unit_Ready (g_cd) ||
        Test_Unit_Ready (g_cd)) {
      Mount ();
    } else {
      return 0;
    }
    if (DevList)
      return 1;
    else {
      /* Mount() did not succeed: */
      return 0;
    }
  }
  return 1;
}

/*
 *  Open timer device structures:
 */

int Open_Timer_Device (void)
{
  if (!(g_timer_mp = CreatePort (NULL, 0))) {
    BUG(dbprintf ("cannot create timer message port!\n");)
    return 0;
  }
  if (!(g_timer_io = (struct timerequest *)
  	 CreateExtIO (g_timer_mp, sizeof (struct timerequest)))) {
    BUG(dbprintf ("cannot create timer i/o structure!\n");)
    DeletePort (g_timer_mp);
    return 0;
  }
  if (OpenDevice ((UBYTE *) TIMERNAME, UNIT_VBLANK,
  		  (struct IORequest *) g_timer_io, 0)) {
    BUG(dbprintf ("cannot open timer device!\n");)
    DeleteExtIO ((struct IORequest *) g_timer_io);
    DeletePort (g_timer_mp);
    return 0;
  }
  g_timer_sigbit = 1L << g_timer_mp->mp_SigBit;
  return 1;
}

/*
 *  Remove timer device structures:
 */

void Cleanup_Timer_Device (void)
{
  /* remove any pending requests: */
  if (!CheckIO ((struct IORequest *) g_timer_io))
    AbortIO ((struct IORequest *) g_timer_io);
  WaitIO ((struct IORequest *) g_timer_io);
  
  CloseDevice ((struct IORequest *) g_timer_io);
  DeleteExtIO ((struct IORequest *) g_timer_io);
  DeletePort (g_timer_mp);
}

/*
 *  Send timer request
 */

void Send_Timer_Request (void)
{
  g_timer_io->tr_node.io_Command = TR_ADDREQUEST;
  g_timer_io->tr_time.tv_secs = CHECK_INTERVAL;
  g_timer_io->tr_time.tv_micro = 0;
  SendIO ((struct IORequest *) g_timer_io);
}

/*
 *  Check whether the disk has been removed or inserted.
 */

void Check_Disk (void)
{
  BUG(dbprintf ("Checking Disk... ");)
  if (DevList) {
    if (Test_Unit_Ready (g_cd)) {
      BUG(dbprintf ("no disk change  "));
    } else {
      BUG(dbprintf ("disk has been removed  ");)
      if (DevList)
        Unmount ();
    }
  }
  if (DevList == NULL) {
    if (Test_Unit_Ready (g_cd) ||
        Test_Unit_Ready (g_cd)) {
      BUG(dbprintf ("disk has been inserted");)
      Mount ();
    }
  }
  BUG(dbprintf ("\n");)
}

/*
 * Extract information from Mountlist "Startup" field.
 */

int Get_Startup (LONG p_startup)
{
  char buf[100];
  char *cp;
  int i;

  btos (p_startup, buf);
  
  cp = buf;
  if (*cp == '"')  /* skip quote */
    cp++;

  for (i=0; *cp && *cp != ':'; ) {	/* get SCSI device name */
    g_device[i++] = *cp++;
    if (i == sizeof (g_device)) {
      Display_Error ("Mountlist field 'Startup':\n"
      		     "SCSI device name too long");
      return 0;
    }
  }
  g_device[i] = 0;
  if (*cp != ':' || !isdigit (cp[1])) {
    Display_Error ("Mountlist field 'Startup':\n"
    		   "SCSI unit number is missing");
    return 0;
  }
  for (cp++, i=0; *cp && isdigit (*cp); cp++)	/* get SCSI unit number */
    i = 10 * i + *cp - '0';
  g_unit = i;

  /* Initialize CDROM: */
  g_cd = Open_CDROM (g_device, g_unit);
  if (!g_cd) {
    Display_Error ("Cannot open %s unit %d", g_device, g_unit);
    return 0;
  }

  g_use_rock_ridge = 0;
  g_map_to_lowercase = 0;

  if (*cp == ':') {
    /* parse options: */
    for (cp++; isalpha (*cp); cp++) {
      switch (*cp) {
        case 'r':
        case 'R':
	  g_use_rock_ridge = 1;
	  break;
        case 'l':
        case 'L':
	  g_map_to_lowercase = 1;
	  break;
	default:
	  Display_Error ("Mountlist field 'Startup':\n"
	  		 "Illegal option character '%c'", *cp);
	  return 0;
      }
    }
  }

  return 1;
}

#ifndef NDEBUG

/*
 *  The assembly tag for the DOS process:  CNOP causes alignment problems
 *  with the Aztec assembler for some reason.  I assume then, that the
 *  alignment is unknown.  Since the BCPL conversion basically zero's the
 *  lower two bits of the address the actual code may start anywhere around
 *  the label....  Sigh....  (see CreatProc() above).
 */

#asm
	public	_debugproc
	public	_debugmain

	cseg
	nop
	nop
	nop
_debugproc:
	nop
	nop
	movem.l D2-D7/A2-A6,-(sp)
	jsr	_debugmain
	movem.l (sp)+,D2-D7/A2-A6
	rts
#endasm


#endif /* !NDEBUG */
