//----------------------------------------------------------------------------
//   Application: NPEdit()
//     File Name: NotePad Edit
//
//        Author: Wilson H. Yuen
//  Date created: 04-02-93
//  Time created: 10:04:53pm
//     Make File: Note.RMK
//     Exec File: Note.EXE
//     Copyright: Wilson H. Yuen
//
//   Description: Word processor using the new Funcky II Notepad Class.
//
//                Features:
//
//                ENTER       - Move cursor to the next line.  Insert line
//                              if insert mode is on.
//                SPACE       - Delete character under cursor and move
//                              cursor one character to the right.  Insert
//                              one character if insert mode is on.
//                ESC         - Exit edit mode.  If changes were made the
//                              system will prompt to save their changes.
//                DEL         - Delete character under cursor, or current
//                              blocked text.
//                INS         - Toggle insert mode.
//                BACKSPACE   - Delete previous character.
//                LEFT ARROW  - Move cursor one character to the left.
//                RIGHT ARROW - Move cursor one character to the right.
//                UP ARROW    - Move cusror up one line.
//                DOWN ARROW  - Move cursor down one line.
//                PGUP        - Move cursor one page up.
//                PGDN        - Move cursor one page down.
//                HOME        - Move cursor to the beginning of the line.
//                END         - Move cursor to the end of the line.
//                TAB         - Tab four characters to the right.
//
//                CTRL-LEFT   - Move cursor to the first character of the
//                              previous word.
//                CTRL-RIGHT  - Move cursor to the first character of the
//                              next word.
//                CTRL-PGUP   - Move cursor to the top of the text.
//                CTRL-PGDN   - Move cursor to the end of the text.
//                CTRL-HOME   - Move cursor to the top of the window.
//                CTRL-END    - Move cursor to the bottom of the window.
//                CTRL-BS     - Delete previous word.
//                CTRL-F5     - Center a line or block of text.
//                CTRL-F6     - Left justify a line of text.
//                CTRL-F7     - Right justify a line of text.
//                CTRL-F8     - Justify the current paragraph.
//                CTRL-Y      - Delete to end of line, starting at cursor
//                              position.
//
//                ALT-C       - BLOCK COPY: Copy a block of text.
//                              Press Alt-C to start blocking.  Use arrow
//                              keys to block text.  Press Enter to end
//                              blocking.  Move cursor to the target
//                              location.  Press Enter again to copy text
//                              to new location.  Press + to copy the block
//                              to the scratch pad. Press Esc or Alt-C before
//                              Enter to cancel copying.
//
//                ALT-D       - DELETE BLOCK: Delete a block of text.
//                              Press Alt-D to start blocking.  Use arrow
//                              keys to block text.  Press Enter to delete
//                              blocked text.  Press + to copy the block to
//                              the scratch pad. Press Esc or Alt-D before
//                              Enter to cancel deletion.
//
//                ALT-F       - SEARCH: Non-case sensitive search forward
//                              or backward.
//
//                ALT-M       - MOVE BLOCK: Move a block of text.
//                              Press Alt-M to start blocking.  Use arrow
//                              keys to block text.  Press Enter to end
//                              blocking.  Move cursor to the target
//                              location.  Press Enter again to move text
//                              to new location.  Press + to copy the block
//                              to the scratch pad.  Press Esc or Alt-M before
//                              Enter to cancel moving process.
//
//                ALT-P       - Insert text that was last blocked into
//                              buffer starting at cursor position.
//
//                ALT-R       - Read an ASCII file into current buffer.
//                ALT-T       - Search and replace.
//                ALT-S       - Save text and exit edit mode.
//                ALT-U       - Undelete previous deleted text.
//                ALT-W       - WRITE BLOCK: Write the text or a block of
//                              text to file. Press Alt-W to start blocking.
//                              Use arrow keys to block text.  Press Enter
//                              to end block.  Enter name of the file to
//                              write the text to.  Press + to copy the block
//                              to the scratch pad.  Press Esc or Alt-W before
//                              Enter to cancel process.
//
//     Revisions:
//
//----------------------------------------------------------------------------
#include "FUNCky.ch"
#include "Inkey.ch"
#include "Setcurs.ch"
#include "Note.ch"

#define K_SPACE                 32  //-- this should be added to Inkey.ch


#define pBLOCK_ON             "Block on"
#define pBLOCK_PROMPT         "Move cursor; press Enter to retrieve"



//-- lBlock_
#define pCOPY1  1              //-- copy block phase I flag
#define pCOPY2  2              //-- copy or move block phase II flag
#define pDELETE 3              //-- delete block flag
#define pMOVE1  4              //-- move block phase I flag
#define pWRITE  5              //-- write block to a file


Static lBlock_[5]              //-- block flags
Static nMaxRow                 //-- maximum screen row
Static nMaxCol                 //-- maximum screen column
Static cWPStatusLine           //-- current status line
Static lOK2Edit                //-- TRUE memo can be edited
memvar getlist



//----------------------------------------------------------------------------
//           Name: NPEdit()
//        Purpose: Edit text
//
//         Author: Wilson H. Yuen
//   Date created: 04-02-93
//   Time created: 11:44:54pm
//      Copyright: Wilson H. Yuen
//
//         Syntax: NPEdit( o ) --> NIL
//
//       Argument: o - edit object
//
//   Return Value: NIL
//
//        Example: NPEdit(oNotePad)
//
//----------------------------------------------------------------------------
FUNCTION NPEdit( o )
    local InsertMode := FALSE           //-- flag for insert mode
    local cOrgBuffer := o:buffer        //-- original text buffer
    local nKey                          //-- key input
    local cLine                         //-- line of text at cursor
    local nChoice                       //-- choice
    local nCurRow                       //-- current cursor row
    local nRetVal                       //-- return value
    local nEle                          //-- element
    local x,y

    nMaxRow  := MaxRow()                //-- maximum screen row
    nMaxCol  := MaxCol()                //-- maximum screen column
    lOK2Edit := OK2Edit()               //-- can the text be edited


    SetCursor(If(lOK2Edit , SC_NORMAL, SC_NONE))



    //-------------------------------------------------------------
    //  Reset delete buffer
    //-------------------------------------------------------------
    DeleteBuffer("RESET")

    //----------------------------------------------------
    // Execute define special functions or functions to
    // override predefined function.  The aFunc array is
    // two dimensional.  The 1st element is the inkey code
    // to seek and the 2nd is the code block to execute
    // for the corresponding inkey.  For example, the
    // Alt-R is pre-programed to read an ASCII file into
    // the current edit buffer.  You can re-program this
    // key for the current edit object by adding it to
    // the aFunc array.  For example you want to reporgram
    // this key to restore the current buffer to it's
    // original state.
    //
    //         Local cOrgBuffer := oEdit:Buffer
    //         Local aFunc      := { K_ALT_R,;
    //                             {|oEdit| oEdit:Buffer := cOrgBuffer }
    //
    // The Edit() will check this array to see if it needs
    // to execute a function.  In the above example the
    // current edit buffer will be restored to it's
    // original form if Alt-R is pressed.
    //
    //----------------------------------------------------
    If valtype(o:Cargo) == "A" .and.;
       ( nEle := ascan(o:Cargo,{|aCargo| aCargo[1] == "STATUSLINE" }) ) > 0
       DispMsg(nMaxRow, o:Cargo[nEle,2] )
    Endif


    cWPStatusLine := WPSaveStatusLine() //-- save current status line


    //-- initialize logic variables
    if lBlock_[ pCOPY1 ] == NIL
       AFILL(lBlock_, .F.)
    endif

    o:BlockColor := pUNS                                    //-- set block color
    csrtype( 0 )                                            //-- Initial editing cursor

    //-------------------------------------------
    //   Infinite loop to process keystrokes
    //-------------------------------------------
    DO WHILE ( TRUE )

        //-------------------------------------------------
        //   Display the editing buffer and position the
        //   cursor to its proper screen position
        //-------------------------------------------------
        o:Display()                                                            //-- dispaly text
        csrput( o:ScrRow, o:ScrCol )                                           //

        nKey := inkey(0)                                    //-- Get a keystroke

        //-------------------------------------------------
        //  Process the keystrokes any way we see fit
        //-------------------------------------------------
        DO CASE

            //-------------------------------------------------
            //  Display help text
            //-------------------------------------------------
            case nKey == K_F1
                DispHelp()

            //-------------------------------------------------
            //  Evaluate special functions if needed.
            //-------------------------------------------------
            Case valtype(o:Cargo) == "A" .and.;
                 ! empty(o:Cargo)        .and.;
                 ( nEle := ascan(o:Cargo,{|aCargo| valtype(aCargo[1]) == "N" .and. aCargo[1] == nKey }) ) > 0
                 // ( nEle := ascan(o:Cargo,{|aCargo| aCargo[1] == nKey }) ) > 0

                 Eval( o:Cargo[nEle,2], o )
                 csrtype( IIF(InsertMode, 1, 0) )

            //=================================================
            //  BLOCK MODE ON
            //=================================================
            CASE ( nKey == K_ENTER .or. nKey == ESC .or. chr(nKey) == "+" ) .and.;
                 ( lBlock_[ pCOPY1  ] .or.;
                   lBlock_[ pCOPY2  ] .or.;
                   lBlock_[ pDELETE ] .or.;
                   lBlock_[ pMOVE1  ] .or.;
                   lBlock_[ pWRITE  ]  ) .and.;
                   lOK2Edit

                 If nKey == K_ENTER
                    BlockEnter(lBlock_,o)                   //-- process request
                 else
                    if chr(nKey) == "+"
                       BlockBuffer(o:Block)
                    endif
                    AFILL(lBlock_, .F.)
                    o:UnMark()                              //-- unblock
                    WPRestStatusLine(cWPStatusLine)
                 endif


            //-------------------------------------------------
            //  All the rest is the minimal needed for editing.
            //-------------------------------------------------
            CASE ( nKey > K_SPACE AND nKey LE 255 ) .and.;
                 ! nKey == K_CTRL_BS                .and.;
                 lOK2Edit

                IF( InsertMode )
                    o:Insert( nKey )
                ELSE
                    DeleteBufffer(chr(o:Char))              //-- store character to delete buffer
                    o:OverWrite( nKey )
                ENDIF


            //=================================================
            //  KEYS
            //=================================================
            CASE( nKey EQ ENTER ) .and.;                    //-- ENTER
                lOK2Edit

                IF( InsertMode )
                    nCurRow := o:csrrow
                    o:Insert( nKey )
                    If o:csrrow == nCurRow
                       o:Insert( nKey )
                    Endif

                ELSE
                    nCurRow := o:CsrRow                     //-- current cursor row
                    o:CsrRow++                              //-- move down one row
                    If nCurRow == o:CsrRow                  //-- if that didn't work, maybe because of OS/2
                       npCsrEnd(o)                          //-- move cursor to end of line
                       o:Insert( nKey )                     //-- insert carriage return
                    Endif
                    o:CsrCol := 1                           //-- move cursor to 1st column
                ENDIF  // InsertMode


            case nKey == K_SPACE .and.;                     //-- SPACE
                 lOK2Edit

                DeleteBufffer(chr(o:Char))                  //-- store character to delete buffer
                if InsertMode
                    o:InsertText(" ")                       //-- insert space
                else
                    o:Char := K_SPACE                       //-- repalce character under cursor
                endif

            CASE( nKey EQ ESC )                             //-- ESC
                If ! cOrgBuffer == o:buffer                 //-- buffer was changed
                   nChoice := ALERT("Save changes?", {"Yes","No","Continue"})
                   If nChoice == 3                          //-- continue
                      loop
                   Endif
                   If nChoice == 2                          //-- do not save changes
                      o:buffer := cOrgBuffer
                   Endif
                Endif
                Exit

            case nKey == K_DEL .and.;
                 lOK2Edit

                if ! empty(o:Block)
                   If Alert("Delete Block?",{"Yes","No"}) == 1
                      DeleteBuffer(o:Block)                 //-- store block to delete buffer
                      o:Block := NIL                        //-- delete block
                      AFILL(lBlock_, .F.)                   //-- reset block flags
                      WPRestStatusLine(cWPStatusLine)
                   Endif
                else
                   DeleteBuffer(o:Char)                     //-- store character to delete buffer
                   o:Char := NIL                            //-- delete character
                endif

            CASE( nKey EQ INS ) .and.;                      //-- INSERT
                lOK2Edit

                InsertMode := (! InsertMode)
                csrtype( IIF(InsertMode, 1, 0) )

            CASE( nKey EQ BACKSPACE ) .and.;                //-- BACKSPACE
                lOK2Edit

                o:BackSpace()

            CASE( nKey EQ LEFTARROW )                       //-- LEFT
                o:CsrIndex--

            CASE( nKey EQ RIGHTARROW )                      //-- RIGHT
                o:CsrIndex++

            CASE( nKey EQ UPARROW )                         //-- UP
                o:CsrRow--

            CASE( nKey EQ DOWNARROW )                       //-- DOWN
                o:CsrRow++

            CASE( nKey EQ PGUP )                            //-- PGUP
                o:PageUp()

            CASE( nKey EQ PGDN )                            //-- PGDN
                o:PageDown()

            case nKey == K_HOME                             //-- HOME
                npCsrHome(o)                                //-- beginning of line

            case nKey == K_END                              //-- END
                npCsrEnd(o)                                 //-- end of line

            case nKey == K_TAB                              //-- TAB
                o:InsertText(space(o:TabSize))              //-- tab right



            //=================================================
            //  CONTROL
            //=================================================
            case nKey == K_CTRL_F5 .and. lOK2Edit           //-- CTRL-F5 Center a line or block of text.

                 If empty(o:Block)                          //-- no text blocked
                    CenterLine(o)                           //-- center current line
                 Else
                    If Alert("Justify entire block?",{ "Yes","No"}) == 1
                       CenterBlock(o)                       //-- Center every line that is blocked
                    Else
                       o:UnMark()
                    Endif
                    AFILL(lBlock_, .F.)
                    WPRestStatusLine(cWPStatusLine)
                 Endif

            case nKey == K_CTRL_F6 .and. lOK2Edit           //-- CTRL-F6 Left justify a line of text
                 LeftJustify(o)

            case nKey == K_CTRL_F7 .and. lOK2Edit           //-- CTRL-F7 Right justify a line of text
                 RightJustify(o)

            case nKey == K_CTRL_F8 .and. lOK2Edit           //-- CTRL-F8 Justify the current paragraph
                 o:Justify()

            case nKey == K_CTRL_BS .and. lOK2Edit           //-- delete word at cursor
                                                            //-- if the character to the left list not a space
                                                            //-- move cursor to 1st character of the word
                 o:CsrIndex--                               //-- move left one character
                 if chr(o:Char) == " "                      //-- is blank
                    o:CsrIndex++                            //-- move right one character
                 else
                    o:WordLeft()                            //-- move cursor to the beginning of the word
                 endif
                 DeleteBuffer(o:Word)                       //-- store word to detele buffer
                 o:Word := NIL                              //-- delete word

            case nKey == K_CTRL_LEFT .and. lOK2Edit         //-- CTRL-LEFT
                o:WordLeft()                                //-- left one word

            case nKey == K_CTRL_RIGHT .and. lOK2Edit        //-- CTRL-RIGHT
                o:WordRight()                               //-- right one word

            case nKey == K_CTRL_PGUP                        //-- CTRL-PGUP
                npHome(o)                                   //-- beginning of page

            case nKey == K_CTRL_PGDN                        //-- CTRL-PGDN
                npEnd(o)                                    //-- end of page

            case nKey == K_CTRL_HOME                        //-- CTRL-HOME
                npCsrTop(o)                                 //-- top of window

            case nKey == K_CTRL_END                         //-- CTRL-END
                npCsrBottom(o)                              //-- bottom of window

            case nKey == K_CTRL_Y    .and. lOK2Edit         //-- CTRL-Y
                 Delete2EOL(o)                              //-- delete to end of line


            //=================================================
            //  ALT KEYS
            //-------------------------------------------------
            //  BLOCK MODE
            //=================================================
            case (nKey == K_ALT_C .or.;                     //-- ALT-C block copy
                  nKey == K_ALT_D .or.;                     //-- ALT-D block delete
                  nKey == K_ALT_M .or.;                     //-- ALT-M block move
                  nKey == K_ALT_W ) .and. lOK2Edit          //-- ALT-W save blocked text to a file

                 If lBlock_[ pCOPY1 ]  .or.;
                    lBlock_[ pDELETE ] .or.;
                    lBlock_[ pMOVE1 ]  .or.;
                    lBlock_[ pWRITE ]
                    AFILL(lBlock_, .F.)                     //-- reset block mode flag
                    o:Unmark()                              //-- unmark block
                    WPRestStatusLine(cWPStatusLine)
                 Else
                         do case
                            case nKey == K_ALT_C
                                 lBlock_[ pCOPY1 ]  := .t.
                            case nKey == K_ALT_D
                                 lBlock_[ pDELETE ] := .t.
                            case nKey == K_ALT_M
                                 lBlock_[ pMOVE1 ]  := .t.
                            case nKey == K_ALT_W
                                 lBlock_[ pWRITE ]  := .t.
                         endcase
                         BeginMark(o)                       //-- start marking
                         VScroll(nMaxRow,0,nMaxRow,nMaxCol,1,pSTD)
                         Print(nMaxRow,0,pBLOCK_ON,pFLASH)
                 Endif

            case nKey == K_ALT_P .and. lOK2Edit             //-- ALT-P
                 nRetVal := o:InsertText(BlockBuffer())     //-- paste last blocked text to current buffer
                 If nRetVal == 1 .or. nRetval == 2
                    ErrMsg("Insertion failed. The editing buffer would overflow.")
                 Endif

            case nKey == K_ALT_F                            //-- search forward/backward
                 nKey := Alert("Search",{ "Forward","Backward"})
                 WPSearch(o,If(nKey==1,"F","B"))

            case nKey == K_ALT_R .and. lOK2Edit             //-- ALT-R read a text file into current buffer
                 ReadInFile(o)

            case nKey == K_ALT_S .and. lOK2Edit             //-- ALT-S save text and exit edit mode
                 exit

            case nKey == K_ALT_T .and. lOK2Edit             //-- search and replace
                 Replace(o)

            case nKey == K_ALT_U .and. lOK2Edit             //-- ALT-U undelete previously deleted text
                 Undelete(o)


            //=================================================
            //  SHIFT KEYS
            //=================================================
            case nKey == K_SH_TAB                           //-- SHIFT-TAB
                TabLeft(o)                                  //-- tab left


        ENDCASE
    ENDDO


return NIL
* eof NPEdit()
*
*
//----------------------------------------------------------------------------
//           Name: WP5Edit()
//        Purpose: Process key strokes
//
//         Author: Wilson H. Yuen
//   Date created: 04-13-93
//   Time created: 02:29:47pm
//      Copyright: Wilson H. Yuen
//
//         Syntax: WP5Edit(o) --> NIL
//
//       Argument: o - edit object
//
//   Return Value: NIL
//
//    Description: Process key strokes
//
//                 WordPerfect Type Functionality
//                 
//
//
//                 NAVIGATION METHODS:
//                 
//
//                   Top of Document     HOME, HOME, UP
//                   Bottom of Document  HOME, HOME, DOWN
//                   Word Right          CTRL-RIGHT
//                   Word Left           CTRL-LEFT
//                   End of Line         End or HOME, RIGHT
//                   Beginning of Line   HOME, LEFT
//                   Top of window       HOME, UP
//                   Bottom of window    HOME, DOWN
//
//                   END                 Move to the end of the line
//                   CTRL-LEFT           Move to the first character of the
//                                        previous word
//                   CTRL-RIGHT          Move to the first character of the
//                                        next word
//                   CTRL-Y              Delete to end of line, starting from
//                                        the cursor position
//                   TAB                 Tab four space to the right
//                   SHIFT-TAB           Tab four space to the left
//                   DEL                 Delete character under the cursor or
//                                        blocked text
//                   INS                 Toggle insert mode
//                   ESC or F7           Exit edit mode
//
//
//                 NOTE: Check out some of the "Hot Keys" at the end of this
//                       document for some helpful short cuts.
//
//
//                 FUNCTION KEYS
//                 
//
//                 ESC        - UNDELETE
//                                Press Esc to undelete
//                                Display string to undelete
//                                Press 1 or R to restore the string currently being displayed.
//                                Press 2 or P to display other deleted stings in buffer.
//
//                 F2         - SEARCH FORWARD
//                                Press F2 search for a string from cursor on down.
//                                Enter string to search for.
//                                Press F2,F7 or Enter to start search.
//                                Non-case sensitive
//
//                 SH-F2      - SEARCH BACKWARD
//                                Press Shift-F2 search for a string from cursor on up.
//                                Enter string to search for.
//                                Press Shift-F2,F7 or Enter to start search.
//                                Non-case sensitive
//
//                 ALT-F2     - REPLACE
//                                Press Alt-F2 to replace a string.
//                                Enter string to search for.
//                                Press Alt-F2, F2, F7 or Enter to start search.
//                                System will ask if you want to confirm each change.
//                                  If yes, then prompt user before each replacement.
//                                  If no, replace all occurrences and move cursor to the
//                                   second character of the last replaced string.
//                                Press Esc to cancel search and replace.
//                                Non-case sensitive
//
//                 F7         - EXIT
//                                Press F7 to exit notepad.
//                                Press Esc to cancel and return to edit mode
//
//                 CTRL-F5    - TEXT IN/OUT
//                                Press Ctrl-F5 to save memo to text or retrieve a file into
//                                 the current text.
//                                Press Ctrl-F5 or Esc to exit.
//                                Press 1 or S to save the text to a DOS text file.
//                                Press 2 or R to retrieve a DOS text file.
//                                If you try to retrieve a file that does not exist an error
//                                 message will display.
//
//                 CTRL-END   - DELETE TO END OF LINE
//                                Press Ctrl-End to delete to end of line from current cursor
//                                  position.
//
//                 CTRL-PGDN  - DELETE TO END OF FILE
//                                Press Ctrl-PgDn to delete to end of page from current cursor
//                                  position.
//                                Press Esc or F7 to cancel
//                                Default is no
//
//                 CTRL-BS    - DELETE WORD
//                                Press Ctrl-BKSP to delete current word at cursor.
//                                Move cursor to the first character of the next word on the
//                                 right.
//
//                 ALT-F4     - BLOCK OPTIONS
//                                Press Alt-F4 to turn block mode on.
//                                Press Alt-F4 or Esc to turn block mode off.
//                                Use cursor to block text
//                                Press Ctrl-F4 to display block option.
//                                  Press 1 or M to move block.
//                                  Press 2 or C to copy block.
//                                  Press 3 or D or DEL to delete block.
//                                  Press + to copy the block to the scratch pad.
//
//                 F12        - BLOCK OPTIONS
//                                Press F12 to turn block mode on.
//
//                 ALT-F6     - FLUSH RIGHT
//                                Press Alt-F6 to right justify text on current line.
//
//                 SH-F6      - CENTER TEXT
//                                Press Sh-F6 to center current line of text.
//                                Press Sh-F6 or Esc to cancel.
//                                If the text is blocked the system will ask if you want to
//                                 "Just:Center" the text.
//
//
//
//                 HOT KEYS:
//                 
//
//                 ALT-C        - BLOCK COPY
//                                  Press Alt-C to start blocking.
//                                  User arrow keys to block text.
//                                  Press Enter to end blocking.
//                                  Move cursor to the target location.
//                                  Press Enter to copy text.
//                                  Press DEL to delete the block.
//                                  Press + to copy the block to the scratch pad.
//                                  Press Esc or Alt-C before Enter to cancel.
//
//                 ALT-D        - BLOCK DELETE
//                                  Press Alt-D to start blocking.
//                                  User arrow keys to block text.
//                                  Press Enter to delete block.
//                                  Press + to copy the block to the scratch pad.
//                                  Press Esc or Alt-D before Enter to cancel.
//
//                 ALT-M        - BLOCK MOVE
//                                  Press Alt-M to start blocking.
//                                  User arrow keys to block text.
//                                  Press Enter to end blocking.
//                                  Move cursor to the target location.
//                                  Press Enter to move text.
//                                  Press DEL to delete the block.
//                                  Press + to copy the block to the scratch pad.
//                                  Press Esc or Alt-M before Enter to cancel.
//
//                 ALT-P       - PASTE FROM SCRATCH PAD
//                                  Paste text from previously blocked text.
//                                  Note: Use the ESC key to retrieve a previously deleted
//                                        block of text.
//
//                 ALT-R        - RETRIEVE A FILE
//                                  Retrieve an ASCII file into current text.
//
//                 ALT-S        - SAVE AND EXIT
//                                  Save and exit edit mode.
//
//                 ALT-W        - WRITE TO A FILE
//                                  Press Alt-W to start blocking.
//                                  User arrow keys to block text.
//                                  Press Enter to end blocking.
//                                  Enter the name of the file to save the block of text to.
//                                  Press DEL to delete the block.
//                                  Press + to copy the block to the scratch pad.
//                                  Press Esc or Alt-W before Enter to cancel.
//
//        Example: o := NotePadNew(0,0,23,79, pSTD, 4000, 80)
//                 WP5Edit(o)
//
//----------------------------------------------------------------------------
FUNCTION WP5Edit( o )
    local nKey                       //-- key input
    local cLine                      //-- line of text at cursor
    local nChoice                    //-- choice
    local nCurRow                    //-- current cursor row
    local nRetVal                    //-- return value
    local nEle                       //-- element
    local cKey                       //-- chr pressed
    local InsertMode   := FALSE      //-- flag for insert mode
    local cOrgBuffer   := o:buffer   //-- original text buffer
    local lBlockModeOn := FALSE      //-- flag for block mode
    local x


    nMaxRow            := MaxRow()   //-- maximum screen row
    nMaxCol            := MaxCol()   //-- maximum screen column
    lOK2Edit           := OK2Edit()  //-- can the text be edited


    SetCursor(If(lOK2Edit , SC_NORMAL, SC_NONE))



    //-------------------------------------------------------------
    //  Reset delete buffer
    //-------------------------------------------------------------
    DeleteBuffer("RESET")

    //----------------------------------------------------
    // Execute define special functions or functions to
    // override predefined function.  The aFunc array is
    // two dimensional.  The 1st element is the inkey code
    // to seek and the 2nd is the code block to execute
    // for the corresponding inkey.  For example, the
    // Alt-R is pre-programed to read an ASCII file into
    // the current edit buffer.  You can re-program this
    // key for the current edit object by adding it to
    // the aFunc array.  For example you want to reporgram
    // this key to restore the current buffer to it's
    // original state.
    //
    //         Local cOrgBuffer := oEdit:Buffer
    //         Local aFunc      := { K_ALT_R,;
    //                             {|oEdit| oEdit:Buffer := cOrgBuffer }
    //
    // The Edit() will check this array to see if it needs
    // to execute a function.  In the above example the
    // current edit buffer will be restored to it's
    // original form if Alt-R is pressed.
    //
    //----------------------------------------------------
    If valtype(o:Cargo) == "A" .and.;
       ( nEle := ascan(o:Cargo,{|aCargo| aCargo[1] == "STATUSLINE" }) ) > 0
       DispMsg(nMaxRow, o:Cargo[nEle,2] )
    Endif


    cWPStatusLine := WPSaveStatusLine() //-- save current status line


    //-- initialize logic variables
    if lBlock_[ pCOPY1 ] == NIL
       AFILL(lBlock_, .F.)
    endif

    o:BlockColor := pUNS                                    //-- set block color
    csrtype( 0 )                                            //-- Initial editing cursor

    //-------------------------------------------
    //   Infinite loop to process keystrokes
    //-------------------------------------------
    DO WHILE ( TRUE )

        //-------------------------------------------------
        //   Display the editing buffer and position the
        //   cursor to its proper screen position
        //-------------------------------------------------
        o:Display()                                                            //-- dispaly text
        csrput( o:ScrRow, o:ScrCol )                                           //-- move cursor to its proper screen position

        /*
        *   Get a keystroke
        */
        nKey := inkey(0)

        /*
        *   Process the keystrokes any way we see fit
        */
        DO CASE

            //-- evaluate custom functions it exist
            Case valtype(o:Cargo) == "A" .and.;
                 ! empty(o:Cargo)        .and.;
                 ( nEle := ascan(o:Cargo,{|x| x[1] == nKey },2) ) > 0

                 Eval( o:Cargo[nEle,2], o )
                 csrtype( IIF(InsertMode, 1, 0) )

           /*
            *   All the rest is the minimal needed for editing.
            */
            CASE ( nKey > K_SPACE AND nKey LE 255 ) .and.;
                 ! nKey == K_CTRL_BS .and.;
                 lOK2Edit
                IF( InsertMode )
                    o:Insert( nKey )
                ELSE
                    o:OverWrite( nKey )
                ENDIF

            CASE ( nKey == K_ENTER .or. nKey == K_ESC ) .and.;
                 ( lBlock_[ pCOPY1  ] .or.;
                   lBlock_[ pCOPY2  ] .or.;
                   lBlock_[ pDELETE ] .or.;
                   lBlock_[ pMOVE1  ] .or.;
                   lBlock_[ pWRITE  ]  ) .and.;
                   ! lBlockModeOn                           //-- not in block mode

                 WPRestStatusLine(cWPStatusLine)
                 If nKey == K_ENTER
                    BlockEnter(lBlock_,o)
                 else
                    AFILL(lBlock_, .F.)
                    o:UnMark()                              //-- unblock
                 endif

            CASE( nKey EQ ENTER ) .and.;                    //-- enter
                 lBlockModeOn     .and.;                    //-- block mode on by pressing F4
                 lBlock_[ pCOPY2 ]                          //-- copy phase II
                 BlockEnter(lBlock_,o)                      //-- process request
                 lBlockModeOn := .f.                        //-- set block mode off
                 AFILL(lBlock_, .F.)                        //-- reset block flags


            CASE( nKey EQ ENTER )  .and.;                   //-- enter
                 lOK2Edit

                IF( InsertMode )
                    nCurRow := o:csrrow
                    o:Insert( nKey )
                    If o:csrrow == nCurRow
                       o:Insert( nKey )
                    Endif

                ELSE
                    nCurRow := o:CsrRow
                    o:CsrRow++
                    If nCurRow == o:CsrRow
                       npCsrEnd(o)                          //-- move cursor to end of line
                       o:Insert( nKey )                     //-- insert carriage return
                    Endif
                    o:CsrCol := 1
                ENDIF  // InsertMode


            CASE( nKey EQ BACKSPACE ) .and.;                //-- backspace
                 lOK2Edit
                o:BackSpace()

            CASE( nKey EQ INS )  .and.;                     //-- insert
                 lOK2Edit
                InsertMode := (! InsertMode)
                csrtype( IIF(InsertMode, 1, 0) )

            CASE( nKey EQ LEFTARROW )                       //-- left arrow
                o:CsrIndex--
                If HomeCounter() > 0                        //-- home key was pressed
                  npCsrHome(o)                              //-- move to beginning of line
                  HomeCounter(0)                            //-- reset home counter
                Endif

            CASE( nKey EQ RIGHTARROW )                      //-- right arrow
                o:CsrIndex++
                If HomeCounter() > 0                        //-- home key was pressed
                  npCsrEnd(o)                               //-- move to end of line
                  HomeCounter(0)                            //-- reset home counter
                Endif

            CASE( nKey EQ UPARROW )                         //-- up arrow
                o:CsrRow--
                If HomeCounter() == 1
                  x := o:CsrCol                             //-- current cursor column
                  npCsrTop(o)                               //-- move to top of window
                  HomeCounter(0)                            //-- reset home counter
                  csrput( o:CsrRow, x )                     //-- restore cursor column
                ElseIf HomeCounter() > 1                    //-- home key was pressed
                  npHome(o)                                 //-- move to beginning of text
                  HomeCounter(0)                            //-- reset home counter
                Endif

            CASE( nKey EQ DOWNARROW )                       //-- down arrow
                o:CsrRow++
                If HomeCounter() == 1
                  x := o:CsrCol                             //-- current cursor column
                  npCsrBottom(o)                            //-- cmoe to bottom of window
                  HomeCounter(0)                            //-- reset home counter
                  csrput( o:CsrRow, x )                     //-- restore cursor column

                ElseIf HomeCounter() > 1                    //-- home key was pressed
                  npEnd(o)                                  //-- move to end of text
                  HomeCounter(0)                            //-- reset home counter
                Endif

            CASE( nKey EQ PGUP )                            //-- page up
                o:PageUp()

            CASE( nKey EQ PGDN )                            //-- page down
                o:PageDown()

            case nKey == K_HOME                             //-- increment home counter
                 HomeCounter(HomeCounter()+1)

            case nKey == K_END                              //-- end of line
                npCsrEnd(o)

            case nKey == K_DEL  .and.;                      //-- delete letter
                 ! lBlockModeOn .and.;
                 lOK2Edit
                if ! empty(o:Block)
                   If WPGetChoice("Delete Block: ~No (~Yes)","N") == "Y"
                      DeleteBuffer(o:Block)                 //-- store block to delete buffer
                      o:Block := NIL                        //-- delete block
                      AFILL(lBlock_, .F.)                   //-- reset block flags
                      WPRestStatusLine(cWPStatusLine)    //-- restore status line
                   Endif
                else
                   DeleteBuffer(o:Char)                     //-- store character to delete buffer
                   o:Char := NIL                            //-- delete character
                endif

            case nKey == K_TAB  .and.;                      //-- tab right
                 lOK2Edit
                o:InsertText(space(o:TabSize))

            case nKey == K_SH_TAB .and.;                    //-- tab left
                 lOK2Edit
                TabLeft(o)

            case nKey == K_SPACE  .and.;                    //-- space bar
                 lOK2Edit
                if InsertMode
                    o:InsertText(" ")
                else
                    If o:Char == K_RETURN
                       o:InsertText(" ")
                    else
                       o:Char := K_SPACE
                    Endif
                endif

            case nKey == K_ESC .and. lBlockModeOn           //-- ESC and block mode is on
                lBlockModeOn := .f.
                o:UnMark()
                WPRestStatusLine(cWPStatusLine)

            case nKey == K_ESC .and.;                       //-- undelete previous deletion
                 lOK2Edit
                Undelete(o)



            //-------------------------------------------------
            //  FUNCTION KEYS
            //-------------------------------------------------
            case nKey == K_F1                               //-- help screen
                 DispHelp()

            case nKey == K_F2                               //-- search forward
                 WPSearch(o,"F")

            CASE( nKey == K_F7 ) .or.;                      //-- Exit
                ( nKey == K_ESC .and. ! lOK2Edit )          //-- ESC and not in edit mode

                If ! cOrgBuffer == o:buffer                 //-- buffer has been changed
                  cKey := WPGetChoice("Save docuemnt? ~Yes (~No)","Y")

                  If cKey == chr(K_ESC)                     //-- ESC
                     loop
                  Elseif cKey == "N"
                     o:buffer := cOrgBuffer
                  Endif
                Endif
                Exit


            //-------------------------------------------------
            //  SHIFT FUNCTION KEYS
            //-------------------------------------------------
            case nKey == K_SH_F2                             //-- search backward
                 WPSearch(o,"B")

            case nKey == K_SH_F6  .and.;                     //-- Center text
                 lOK2Edit
                 If empty(o:Block)                           //-- no text blocked
                    CenterLine(o)                            //-- center current line
                 Else
                    If WPGetChoice("[Just:Center]? ~No (~Yes)","N") == "Y"
                       CenterBlock(o)     //-- Center every line that is blocked
                    Else
                       o:UnMark()
                    Endif
                    WPRestStatusLine(cWPStatusLine)
                    AFILL(lBlock_, .F.)                 //-- reset block flags
                    lBlockModeOn := .f.
                 Endif


            //-------------------------------------------------
            //  CONTROL KEYS
            //-------------------------------------------------
            case nKey == K_CTRL_F5 .and.;                    //-- Text in/out
                 lOK2Edit

                 cKey := WPGetChoice("~1 ~Save ~2 ~Retrieve: ~0","0")
                 If cKey $ "1~S"
                    Write2File(o)
                 Elseif cKey $ "2~R"
                    ReadInFile(o)
                 Endif

            case nKey == K_CTRL_BS  .and.;                  //-- delete word at cursor
                 lOK2Edit                                   //-- if the character to the left list not a space
                                                            //-- move cursor to 1st character of the word
                 o:CsrIndex--                               //-- move left one character
                 if chr(o:Char) == " "                      //-- is blank
                    o:CsrIndex++                            //-- move right one character
                 else
                    o:WordLeft()                            //-- move cursor to the beginning of the word
                 endif
                 DeleteBuffer(o:Word)                       //-- store word to detele buffer

                 o:Word := NIL

            case nKey == K_CTRL_END  .and.;                 //-- delete to end of line
                 lOK2Edit
                 BeginMark(o)
                 npCsrEnd(o)
                 o:EndMark()
                 DeleteBuffer(o:Block)                      //-- store blocked text to delete buffer
                 o:Block := NIL                             //-- delete block

            case nKey == K_CTRL_LEFT                        //-- left one word
                o:WordLeft()

            case nKey == K_CTRL_RIGHT                       //-- right one word
                o:WordRight()

            case nKey == K_CTRL_PGDN .and.;                 //-- delete remainder of page
                 lOK2Edit

                 If WPGetChoice("Delete Remainder of page? ~No (~Yes)","N") == "Y"
                    BeginMark(o)
                    npEnd(o)
                    o:EndMark()
                    DeleteBuffer(o:Block)                   //-- store blocked text to delete buffer
                    o:Block := NIL                          //-- delete block

                 Endif


            //-------------------------------------------------
            //  ALT KEYS
            //-------------------------------------------------
            case nKey == K_ALT_F2 .and.;                    //-- search and replace
                 lOK2Edit
                 Replace(o)

            case nKey == K_ALT_F4 .and.;                    //-- block
                 ! lBlockModeOn   .and.;
                 lOK2Edit
                 lBlockModeON := .t.                        //-- set block mode on
                 BeginMark(o)
                 WPClrStatusLine()                          //-- clear status line
                 Print(nMaxRow,0,pBLOCK_ON,pFLASH)

            case nKey == K_ALT_F6 .and.;                    //-- flush right
                 lOK2Edit
                 RightJustify(o)

            case nKey == K_ALT_P  .and.;                    //-- past text from current buffer
                 lOK2Edit
                 nRetVal := o:InsertText(BlockBuffer())
                 If nRetVal == 1 .or. nRetval == 2
                    ErrMsg("Insertion failed. The editing buffer would overflow.")
                 Endif



            //-------------------------------------------------
            //  BLOCK FUNCTIONS
            //-------------------------------------------------
            case lBlockModeOn   //-- Block mode is on
                do case
                   case nKey == K_DEL                       //-- delete block
                        If WPGetChoice("Delete Block: ~No (~Yes)","N") == "Y"
                           DeleteBuffer(o:Block)            //-- store block to delete buffer
                           lBlockModeOn := .f.              //-- turn block mode off
                           o:Block := NIL                   //-- delete block
                           WPRestStatusLine(cWPStatusLine)
                        Endif

                   case nKey == K_ALT_F4 .or. nKey == K_ESC //-- cancel block
                        lBlockModeOn := .f.                 //-- set block mode off
                        AFILL(lBlock_, .F.)                 //-- reset block flags
                        o:UnMark()
                        WPRestStatusLine(cWPStatusLine)

                   case nKey == K_CTRL_F4 .or.;             //-- display block options
                        nKey == K_F12
                        cKey := WPGetChoice("~1 ~Move; ~2 ~Copy; ~3 ~Delete: ~0","0")
                        do case
                           case cKey $ "1~M"                //-- move block
                                lBlock_[pMOVE1] := .t.
                                BlockEnter(lBlock_,o)

                           case cKey $ "2~C"                //-- copy block
                                lBlock_[pCOPY1] := .t.
                                BlockEnter(lBlock_,o)       //-- process request

                           case cKey $ "3~D"                //-- delete block
                                DeleteBuffer(o:Block)       //-- store block to delete buffer
                                lBlockModeOn := .f.         //-- turn block mode off
                                o:Block := NIL              //-- delete block
                                WPRestStatusLine(cWPStatusLine)
                        endcase


                   case nKey == K_SH_F6                     //-- center text
                        If WPGetChoice("[Just:Center]? ~No (~Yes)","N") == "Y"
                           CenterBlock(o)                   //-- Center every line that is blocked
                        Else
                           o:UnMark()
                        Endif
                        WPRestStatusLine(cWPStatusLine)
                        AFILL(lBlock_, .F.)                 //-- reset block flags
                        lBlockModeOn := .f.

                endcase





            //-------------------------------------------------
            //  Quick Keys
            //-------------------------------------------------
            case ( nKey == K_ALT_C .or.;                      //-- block copy
                   nKey == K_ALT_D .or.;                      //-- block delete
                   nKey == K_ALT_M .or.;                      //-- block move
                   nKey == K_ALT_W  ) .and.;                  //-- save blocked text to a file
                   lOK2Edit

                 do case
                    case lBlock_[ pCOPY1 ]
                         lBlock_[ pCOPY1 ]  := .f.
                    case lBlock_[ pDELETE ]
                         lBlock_[ pDELETE ] := .f.
                    case lBlock_[ pMOVE1 ]
                         lBlock_[ pMOVE1 ]  := .f.
                    case lBlock_[ pWRITE ]
                         lBlock_[ pWRITE ]  := .f.
                    otherwise
                         do case
                            case nKey == K_ALT_C
                                 lBlock_[ pCOPY1 ]  := .t.
                            case nKey == K_ALT_D
                                 lBlock_[ pDELETE ] := .t.
                            case nKey == K_ALT_M
                                 lBlock_[ pMOVE1 ]  := .t.
                            case nKey == K_ALT_W
                                 lBlock_[ pWRITE ]  := .t.
                         endcase
                         BeginMark(o)
                 endcase

                 If ! lBlock_[ pCOPY1  ] .and.;
                    ! lBlock_[ pDELETE ] .and.;
                    ! lBlock_[ pMOVE1  ] .and.;
                    ! lBlock_[ pWRITE  ]
                    WPRestStatusLine(cWPStatusLine)
                    o:Unmark()                              //-- unmark block
                 else
                    WPClrStatusLine()                       //-- clear status line
                    Print(nMaxRow,0,pBLOCK_ON,pFLASH)
                 Endif


            case nKey == K_ALT_R  .and. lOK2Edit            //-- read a text file into current buffer
                 ReadInFile(o)

            case nKey == K_ALT_S  .and. lOK2Edit            //-- save text and exit edit mode
                 exit

            case nKey == K_CTRL_Y .and. lOK2Edit            //-- delete to end of line
                 Delete2EOL(o)                              //-- delete to end of line

        ENDCASE
    ENDDO


return NIL
* eof WP5Edit()
*
*
//----------------------------------------------------------------------------
//           Name: BeginMark()
//        Purpose: Turn blocking on and set BlockStartRow()
//
//         Author: Wilson H. Yuen
//   Date created: 04-04-93
//   Time created: 07:35:08pm
//      Copyright: Wilson H. Yuen
//
//         Syntax: BeginMark( o ) --> NIL
//
//       Argument: o - edit object
//
//   Return Value: NIL
//
//    Description: Turn blocking on and store the current row to the static
//                 variable in BlockStartRow().
//
//        Example: BlockStartRow(o:CsrRow)
//
//----------------------------------------------------------------------------
Static Function BeginMark(o)
  o:BeginMark()
  BlockStartRow(o:CsrRow)
Return(NIL)
* eof BeginMark()
*
*
//----------------------------------------------------------------------------
//           Name: BlockBuffer()
//        Purpose: Get/Set last blocked text
//
//         Author: Wilson H. Yuen
//   Date created: 04-04-93
//   Time created: 00:39:59am
//      Copyright: Wilson H. Yuen
//
//         Syntax: BlockBuffer( cNewVal ) --> cRetVal
//
//       Argument: cNewVal - current locked text
//
//   Return Value: cRetVal - the last blocked text
//
//    Description: Used to store the last blocked text.  Press Alt-P to
//                 paste previously blocked text.
//
//        Example: BlockBuffer(o:Block)       //-- store blocked text
//                 ? BlockBuffer()            //-- retrieve previously
//                                                 blocked text
//
//----------------------------------------------------------------------------
Static Function BlockBuffer(cNewVal)
    Static cRetVal := ""
    If ! cNewVal == NIL
        cRetVal := cNewVal
    Endif
Return(cRetVal)
* eof BlockBuffer()
*
*
//----------------------------------------------------------------------------
//           Name: BlockEnter()
//        Purpose: Process block function
//
//         Author: Wilson H. Yuen
//   Date created: 04-04-93
//   Time created: 00:58:11am
//      Copyright: Wilson H. Yuen
//
//         Syntax: BlockEnter( lBlock_, o ) --> NIL
//
//       Argument: lBlock_ - used to determine what block function was
//                           executed.
//
//                           1 - block copy phase 1, turn block on
//                           2 - block copy phase 2, turn block off and
//                               insert text
//                           3 - delete block
//                           4 - write block to file
//
//                 o       - edit object
//
//   Return Value: NIL
//
//    Description: After pressing Alt-C, Alt-D or Alt-W, the system will
//                 wait for the user to press Esc or one of the Alt keys
//                 to cancel blocking, or Enter to execute the function.
//
//----------------------------------------------------------------------------
Static Function BlockEnter(lBlock_,o)
   local  nRetVal

   do case
      case lBlock_[ pCOPY1  ]  .or. ;                //-- copy or move block
           lBlock_[ pMOVE1  ]
           //------------------------------------------------
           // COPY or MOVE BLOCK - Phase I
           // Prompt user to move cursor to desire position
           // Press Enter to copy block
           //------------------------------------------------
           BlockBuffer(o:Block)                      //-- store blocked text to block buffer
           If lBlock_[ pCOPY1 ]
              o:UnMark()                             //-- block copy
           Else
              o:Block := NIL                         //-- move block
           Endif

           lBlock_[ pCOPY1 ] := .f.                  //-- reset block flags
           lBlock_[ pMOVE1 ] := .f.
           lBlock_[ pCOPY2 ] := .t.
           VScroll(nMaxRow,0,nMaxRow,nMaxCol,1,pSTD)
           Print(nMaxRow,0,pBLOCK_PROMPT,pSTD)


      case lBlock_[ pCOPY2 ]                         //-- paste from scratchpad
           //----------------------------------------
           // COPY or MOVE BLOCK - Phase II
           //----------------------------------------
           nRetVal := o:InsertText(BlockBuffer())    //-- insert text from block buffer
           If nRetVal == 1 .or. nRetval == 2
              ErrMsg({"Block copy/move failed.",;
                      "The editing buffer would overflow."})
           Endif
           WPRestStatusLine(cWPStatusLine)
           lBlock_[ pCOPY2 ] := .f.


      case lBlock_[ pDELETE ]  //-- delete block
           //----------------------------------------
           // DELETE BLOCK
           //----------------------------------------
           DeleteBuffer(o:Block)                     //-- store blocked text to delete buffer
           o:Block := NIL                            //-- delete block
           lBlock_[ pDELETE ] := .f.


      case lBlock_[ pWRITE  ]  //-- write block to a file
           //----------------------------------------
           // WRITE BLOCK TO A FILE
           //----------------------------------------
           Write2File(o)                             //-- write block to file
           AFILL(lBlock_, .F.)                       //-- reset block flags
           o:UnMark()                                //-- unblock
           WPRestStatusLine(cWPStatusLine)

   endcase

Return(NIL)
* eof BlockEnter()
*
*
//----------------------------------------------------------------------------
//           Name: BlockStartRow()
//        Purpose: Get/set the starting block row
//
//         Author: Wilson H. Yuen
//   Date created: 04-04-93
//   Time created: 06:30:47pm
//      Copyright: Wilson H. Yuen
//
//         Syntax: BlockStartRow( nNewVal ) --> nRetVal
//
//       Argument: nNewVal - the starting block row
//
//   Return Value: nRetVal - the starting block row
//
//    Description: Get and set the row that the block started on.
//
//        Example: ? BlockStart(o:CsrRow)      //-- store the current cursor
//                                             //-- row
//                 ? BlockStart()              //-- Get starting block row
//
//----------------------------------------------------------------------------
Static Function BlockStartRow(nNewVal)
   static nRetVal
   If ! nNewVal == NIL
      nRetVal := nNewVal
   Endif
Return(nRetVal)
* eof BlockStartRow()
*
*
//----------------------------------------------------------------------------
//           Name: CenterBlock()
//        Purpose: Center a block of text
//
//         Author: Wilson H. Yuen
//   Date created: 04-04-93
//   Time created: 06:35:53pm
//      Copyright: Wilson H. Yuen
//
//         Syntax: CenterBlock( o ) --> NIL
//
//       Argument: o - edit object
//
//   Return Value: NIL
//
//    Description: Center a block of text.  Use any of the hot keys to block
//                 one or move lines of text.  Press Ctrl-C to center the
//                 block of text
//
//        Example: NIL
//
//----------------------------------------------------------------------------
Static Function CenterBlock(o)
   local nStart    := BlockStartRow()    //-- get the row the block started on
   local nEnd      := o:CsrRow           //-- get the row the block ended on
   local nCurRow   := o:CsrRow           //-- current row
   local nCurCol   := o:CsrCol           //-- current column
   local nStartRow
   local nEndRow
   local n                               //-- FOR/NEXT counter

   DispMsg(nMaxRow, "Centering text ... Please wait ..." )
   o:Unmark()                            //-- unmark block
   nStartRow := min(nStart, nEnd)
   nEndRow   := max(nStart, nEnd)
   o:CsrRow  := nStartRow                //-- return move cursor to the starting row

   for n := nStartRow to nEndRow         //-- start with 1st line
       CenterLine(o)                     //-- center line
       o:CsrRow++                        //-- move to next line
   next

   o:CsrRow := nCurRow                   //-- return to cursor postion
   o:CsrCol := nCurCol

Return(NIL)
* eof CenterBlock()
*
*
//----------------------------------------------------------------------------
//           Name: CenterLine()
//        Purpose: Center current line of text
//
//         Author: Wilson H. Yuen
//   Date created: 04-04-93
//   Time created: 07:45:33pm
//      Copyright: Wilson H. Yuen
//
//         Syntax: CenterLine( o ) --> NIL
//
//       Argument: o - edit object
//
//   Return Value: NIL
//
//    Description: Center current line of text.  Note, only lines that
//                 start and end in a hard carriage return can be centered,
//                 and only when the target line is not part of a word
//                 wrapped paragraph.
//
//                 The function HardCRLine() will make sure the current
//                 line starts and ends in a hard carriage return.
//
//        Example: CenterLine(o)
//
//----------------------------------------------------------------------------
Static Function CenterLine(o)
     HardCRLine(o)                 //-- string to be center must start and end with a hard CR
     o:Line := alltrim(o:Line)     //-- trim the current line
     npCenter(o)                   //-- center current line at cursor
Return(NIL)
* eof CenterLine()
*
*
//----------------------------------------------------------------------------
//           Name: Delete2EOL()
//        Purpose: Delete to end of line
//
//         Author: Wilson H. Yuen
//   Date created: 04-18-93
//   Time created: 04:25:52pm
//      Copyright: Wilson H. Yuen
//
//         Syntax: Delete2EOL(o) --> NIL
//
//       Argument: o - edit object
//
//   Return Value: NIL
//
//    Description: Delelte to end of line.
//
//        Example: Delete2EOL(oEdit)
//
//----------------------------------------------------------------------------
Static Function  Delete2EOL(o)
  local nCurCol := o:CsrCol       //-- current column

  o:BeginMark()                   //-- begin marking
  npCsrEnd(o)                     //-- block to end of line
  o:EndMark()                     //-- end marking

  If ! nCurCol == o:CsrCol        //-- delete only if there is something to delete
     DeleteBuffer(o:Block)        //-- store block to delete buffer
     o:Block := NIL               //-- delete block
  Else
     o:UnMark()                   //-- unmark block
     DeleteBuffer(o:Char)         //-- store block to delete buffer
     o:Char := NIL                //-- delete block
  Endif

Return(NIL)
* eof Delete2EOL()
*
*
//----------------------------------------------------------------------------
//           Name: DeleteBuffer()
//        Purpose: Get/set the last block of text that was deleted
//
//         Author: Wilson H. Yuen
//   Date created: 04-04-93
//   Time created: 07:56:37pm
//      Copyright: Wilson H. Yuen
//
//         Syntax: DeleteBuffer( cNewVal ) --> cRetVal
//
//       Argument: cNewVal - block of text to be deleted
//
//   Return Value: cRetVal - last block of text that was deleted
//
//    Description: Store deleted text to a static array.  Use Undelete()
//                 to undelete previous deleted text.
//
//        Example: DeleteBuffer(o:Block)    //-- store current block of text
//                 ? DeleteBuffer()         //-- get the last block of text
//                                          //-- that was deleted
//
//----------------------------------------------------------------------------
Static Function DeleteBuffer(cNewVal)
   Static cRetVal := {}

   If ! cNewVal == NIL
     cNewVal := If( valtype(cNewVal)=="N",chr(cNewVal),cNewVal)
     If upper(cNewVal) == "RESET"                                  //-- reset deletion buffer
        cNewVal := {}
     Else
        aadd( cRetVal, cNewVal )                                   //-- store to deletion buffer
     Endif
   Endif

Return(cRetVal)
* eof DeleteBuffer()
*
*
//----------------------------------------------------------------------------
//           Name: DispHelp()
//        Purpose: Display help text
//
//         Author: Wilson H. Yuen
//   Date created: 04-15-93
//   Time created: 06:54:53pm
//      Copyright: Wilson H. Yuen
//
//         Syntax: DispHelp() --> NIL
//
//       Argument: None
//
//   Return Value: NIL
//
//    Description: Display help text
//
//        Example: DispHelp()
//
//----------------------------------------------------------------------------
Static Function DispHelp()
    Static cText
    local cWPSaveScreen := WPSaveScreen()
    local nCurCursor    := SetCursor(SC_NONE)
    local cCurColor     := SetColor("W+/BR")

    If  cText == NIL .or.;
      ! cText == HelpText()    //-- interface was changed

       cText := HelpText()     //-- load new text file
       If ! file(cText)
          ErrMsg("The help text, "+cText+" cannot be found")
       else
          cText := MemoRead(cText)
       Endif
    Endif

    If ! cText == NIL
        VScroll(nMaxRow,0,nMaxRow,nMaxCol,,pSTD)
        Print(nMaxRow,0,"Use key pad to move around",pSTD)
        MemoEdit(cText,0,0,nMaxRow-1,nMaxCol,.f.)
        WPRestScreen(cWPSaveScreen)
        SetCursor(nCurCursor)
        SetColor(cCurColor )
    Endif

Return(NIL)
* eof DispHelp()
*
*
//----------------------------------------------------------------------------
//           Name: DispMsg()
//        Purpose: Display a message at a specific row with hi-lights
//
//         Author: Wilson H. Yuen
//   Date created: 04-08-93
//   Time created: 09:33:14pm
//      Copyright: Wilson H. Yuen
//
//         Syntax: DispMsg( nMaxRow, cMsg ) --> NIL
//
//       Argument: nMaxRow - the row to display the message on
//                 cMsg    - the message to display
//
//   Return Value: NIL
//
//    Description: Display a message at a specific row with the option to
//                 hi-light the hot keys.
//
//        Example: The example below will display the message at line 23,
//                 and will hi-light the letters next to the "~" marks.
//
//                  DispMsg(23,"~F~1=Help"+space(60)+"~E~s~c=Exit")
//
//----------------------------------------------------------------------------
Function DispMsg(nRow, cMsg)
   local cMsg2     := strtran(cMsg,"~")       //-- create new message string
   local nHotKeys  := strcount("~",cMsg)      //-- count the number of ~
   local cHotKey_[ nHotKeys ]                 //-- store the letter to be hi-lighted
   local x                                    //-- FOR/NEXT loop

   //-- store the letter to be hi-lighted
   For x := 1 To nHotKeys
       cHotKey_[x] := substr(cMsg,stratnext("~",cMsg,x)+1,1)
   Next

   //-- clear the line and print message
   vscroll(nMaxRow,0,nMaxRow,nMaxCol, 1, pSTD)
   print(nMaxRow,0,cMsg2,pSTD)

   //-- hi-light the letter
   For x := 1 To nHotKeys
      Print(nMaxRow,at(cHotKey_[x],cMsg2)-1,cHotKey_[x], pHOT1 )
   Next

Return(NIL)
* eof DispMsg()
*
*
//----------------------------------------------------------------------------
//           Name: HelpText()
//        Purpose: Get/set the help text file name
//
//         Author: Wilson H. Yuen
//   Date created: 04-18-93
//   Time created: 03:29:34pm
//      Copyright: Wilson H. Yuen
//
//         Syntax: HelpText(cNewVal) --> cRetVal
//
//       Argument: cNewVal - the file name to use
//
//   Return Value: cRetVal - the file name
//
//    Description: Get and set the help text file name.
//
//        Example: HelpText("NPHelp.txt")
//
//----------------------------------------------------------------------------
Function HelpText(cNewVal)
   Static cRetVal := ""
   If ! cNewVal == NIL
       cRetVal := cNewVal
   Endif
Return(cRetVal)
* eof HelpText()
*
*
//----------------------------------------------------------------------------
//           Name: HardCRLine()
//        Purpose: Make sure the current line starts and ends with a hard
//                 carriage return
//
//         Author: Wilson H. Yuen
//   Date created: 04-04-93
//   Time created: 08:22:24pm
//      Copyright: Wilson H. Yuen
//
//         Syntax: HardCRLine( o ) --> NIL
//
//       Argument: o - edit object
//
//   Return Value: NIL
//
//        Example: HardCRLine(o)
//
//----------------------------------------------------------------------------
Static Function HardCRLine(o)
   local lInsertOn  := ReadInsert(.t.)     //-- store current insert mode

     //----------------------------------------------------
     // Steps:
     //   1. Turn insert mode on
     //   2. move cursor to the beginning of the line
     //   3. add a hard carriage return
     //   4. move up one row
     //   5. if line is empty then delete it, else move
     //      the cursor to the end of the line and add
     //      a hard carriage return.  If line is empty
     //      then delete it.
     //   6. move down one line
     //   7. add a hard carriage return
     //   8. move up one line
     //   9. if line is empty then delete it and
     //      move up one line
     //  10. reset original insert mode
     //  11. center text
     //
     // Steps 1-5 will make sure that the current line
     //       is not part of a word wrapped paragraph.
     //
     // Steps 6-9 will make sure that the current line
     //       ends in a had carriage return.
     //
     //---------------------------------------------------
     npCsrHome(o)                  //-- move cursor to beginning of line
     o:Insert(13)                  //-- insert a hard carriage return
     o:CsrRow--                    //-- move cursor up one row

     If empty(o:line)              //-- if line is empty
        o:Line := NIL              //-- delete line
     else
        npCsrEnd(o)                //-- move cursor to the end of the line
        o:Insert(13)               //-- insert a hard carriage return
        If empty(o:line)           //-- if line is empty
           o:Line := NIL           //-- delete line
        Endif
     Endif

     //---------------------------------------------
     //  The cursor should now be on the line
     //  that is to be centered, or justified
     //---------------------------------------------
     o:CsrRow++                    //-- move cursor down one row
     o:Insert(13)                  //-- insert a hard carriage return
     o:CsrRow--                    //-- move up one row
     If empty(o:line)              //-- if the line is empty
        o:Line := NIL              //-- delete line
        o:CsrRow--                 //-- back to the line to be centered
     Endif

   ReadInsert(lInsertOn)           //-- reset insert mode

Return(NIL)
* eof HardCRLine()
*
*
//----------------------------------------------------------------------------
//           Name: HomeCounter()
//        Purpose: Get/set the number of time the HOME key was pressed
//
//         Author: Wilson H. Yuen
//   Date created: 04-13-93
//   Time created: 02:44:14pm
//      Copyright: Wilson H. Yuen
//
//         Syntax: HomeCounter(nNewVal) --> nRetVal
//
//       Argument: nNewVal - the number to set
//
//   Return Value: nRetVal - the number of times the HOME key was pressed
//
//    Description: This function is used to emulate WordPerfect HOME key
//                 function.
//
//        Example: If lastkey() == K_HOME
//                    HomeCounter( HomeCounter()+1 )
//                 Endif
//
//----------------------------------------------------------------------------
Static Function HomeCounter(nNewVal)
   Static nRetVal := 0
   If ! nNewVal == NIL
      nRetVal := nNewVal
   Endif
Return(nRetVal)
* eof HomeCounter()
*
*
//----------------------------------------------------------------------------
//           Name: LeftJustify()
//        Purpose: Left justify current line
//
//         Author: Wilson H. Yuen
//   Date created: 04-04-93
//   Time created: 08:27:28pm
//      Copyright: Wilson H. Yuen
//
//         Syntax: LeftJustify( o ) --> NIL
//
//       Argument: o - edit object
//
//   Return Value: NIL
//
//    Description: Left justify current line.  Like the Center() function
//                 only lines that start and end in a hard carriage return
//                 can be centered, and only when the target line is not
//                 part of a word wrapped paragraph.
//
//                 The function HardCRLine() will make sure the current
//                 line starts and ends in a hard carriage return.
//
//        Example: LeftJustify(o)
//
//----------------------------------------------------------------------------
Static Function LeftJustify(o)
     HardCRLine(o)                 //-- string to be justified must start and end with a hard CR
     o:Line := alltrim(o:Line)     //-- trim the current line
     npLeftJustify(o)              //-- left justify current line
Return(NIL)
* eof LeftJust()
*
*
//----------------------------------------------------------------------------
//           Name: NPInterface()
//        Purpose: Get/set flag to determine which interface to use
//
//         Author: Wilson H. Yuen
//   Date created: 04-16-93
//   Time created: 11:22:57pm
//      Copyright: Wilson H. Yuen
//
//         Syntax: NPInterface(lNewVal) --> lRetVal
//
//       Argument: lNewVal - new value to set
//
//   Return Value: lRetVal - TRUE for NotePad interface
//
//    Description: Get and set flag to determine which interface to use.
//
//        Example: NPInterface(.t.)
//
//----------------------------------------------------------------------------
Function NPInterFace( lNewVal )
  Static lRetVal := .f.
  if ! lNewVal == NIL
     lRetVal := lNewVal
  endif
Return(lRetVal)
* eof NPInterFace()
*
*
//----------------------------------------------------------------------------
//           Name: OK2Edit()
//        Purpose: Get/Set flag to determine if memo can be edited
//
//         Author: Wilson H. Yuen
//   Date created: 04-04-93
//   Time created: 00:39:59am
//      Copyright: Wilson H. Yuen
//
//         Syntax: OK2Edit( lNewVal ) --> lRetVal
//
//       Argument: lNewVal - value to set
//
//   Return Value: lRetVal - TRUE OK to edit
//
//    Description: Use to determine if the memo can be edited.
//
//        Example: ? OK2Edit()
//
//----------------------------------------------------------------------------
Function OK2Edit(lNewVal)
    Static lRetVal
    If ! lNewVal == NIL
        lRetVal := lNewVal
    Endif
Return(lRetVal)
* eof OK2Edit()
*
*
//----------------------------------------------------------------------------
//           Name: ReadInFile()
//        Purpose: Read an ASCII file into the current buffer
//
//         Author: Wilson H. Yuen
//   Date created: 04-04-93
//   Time created: 08:30:19pm
//      Copyright: Wilson H. Yuen
//
//         Syntax: ReadInFile( o ) --> NIL
//
//       Argument: o - edit object
//
//   Return Value: NIL
//
//    Description: Prompt user for the pathname of the file they want to
//                 read into the current edit buffer.
//
//        Example: ReadInFile(o)
//
//----------------------------------------------------------------------------
Static Function ReadInFile(o)
   local cWPSaveScreen := WPSaveScreen()        //-- save current screen
   local cFileName     := space(45)             //-- file name
   local cFile                                  //-- file name trimed
   local cFileText                              //-- buffer to read in
   local nFileSize                              //-- size of file to read in
   local nHandle                                //-- handle of the file to read in
   local cErrMsg                                //-- error message
   local nRetVal                                //-- result of reading in the file

   //------------------------------------------
   // Prompt user for file name
   // Check if file exist
   // Read file
   // Insert file into buffer
   //------------------------------------------
   vscroll(nMaxRow,0,nMaxRow,nMaxCol, 1, pSTD)
   @ nMaxRow,0 say "Enter the name of the file to read in: "
   Do while .t.
       cErrMsg := NIL
       @ nMaxRow,39 get cFileName PICTURE "@!"
       read

       cFile := alltrim(cFileName)
       do case
          case lastkey() == K_ESC                                    //-- pressed ESC
               exit
          case empty(cFile)
               cErrMsg := "A file name must be entered"              //-- file name was not entered
          case ! file(cFile)
               cErrMsg := "File "+cFile+" is not found."             //-- file was not found
          otherwise
               nFileSize := filesize(cfile)                          //-- size of file
               cFileText := space(nFileSize)                         //-- size for read buffer
               if ( nHandle := fopen(cFile) )== -1                   //-- open file
                  cErrMsg := "Unable to read "+cFile
               else
                  fread(nHandle,@cFileText,nFileSize)                //-- read file
                  cFileText := StrTran( cFileText, chr(10), "" )     //-- strip all line feeds
                  nRetVal   := o:InsertText(cFileText)               //-- insert text into edit buffer

                  If nRetVal == 1 .or. nRetval == 2
                     ErrMsg({"Block copy/move failed.",;
                             "The editing buffer would overflow."})
                  Endif
                  fclose(nHandle)                                    //-- close file

               endif
       endcase

       if ! cErrMsg == NIL                                           //-- display error message if any
          ErrMsg(cErrMsg)
          loop
       endif
      exit
   enddo

   WPRestScreen( cWPSaveScreen )                                     //-- restore screen

Return(NIL)
* eof ReadInFile()
*
*
//----------------------------------------------------------------------------
//           Name: Replace()
//        Purpose: Replace a character string
//
//         Author: Wilson H. Yuen
//   Date created: 04-07-93
//   Time created: 10:35:27pm
//      Copyright: Wilson H. Yuen
//
//         Syntax: Replace(o) --> NIL
//
//       Argument: o - edit object
//
//   Return Value: NIL
//
//    Description: Replace a character string.  Non-case sensitive.
//
//        Example: Replace(o)
//
//----------------------------------------------------------------------------
Static Function Replace(o)

   local cWPSaveStatusLine := WPSaveStatusLine()             //-- save status line
   local nStartCol         := o:CsrCol                       //-- current column
   local nStartRow         := o:CsrRow                       //-- current row
   local nLastRow          := o:LineCount                    //-- number of lines in text
   local nFoundRow                                           //-- row where string is found
   local nFoundCol                                           //-- column where string is found
   local nLastFoundCol                                       //-- last column where string was found
   local lFound            := .f.                            //-- TRUE string was found
   local lConfirm                                            //-- TRUE confirm replacement
   local x                                                   //-- FOR/NEXT counter
   local y                                                   //-- FOR/NEXT counter
   local cChoice                                             //-- choice
   local cNewStr           := space(45)                      //-- repalce string
   local cNewStr2
   local cOldStr2
   static cOldStr                                            //-- search string

   If cOldStr == NIL
      cOldStr := space(45)
   Endif


   Begin sequence
      cChoice := WPGetChoice("w/Confirm? ~No (~Yes)","N")
      If cChoice == chr(K_ESC)
         break
      Endif

      lConfirm := If( cChoice == "Y", .t., .f. )

      //----------------------------------------------------
      // Prompt user for the search and replace string
      //----------------------------------------------------
      print(nMaxRow,0,"-> Srch:",pSTD)
      @ nMaxRow,9 get cOldStr
      read
      If lastkey() == K_ESC .or. empty(cOldStr)
         break
      Endif

      print(nMaxRow,0,"Replace with: ",pSTD)
      @ nMaxRow,14 get cNewStr
      read
      If lastkey() == K_ESC
         break
      Endif


      cOldStr2 := alltrim(cOldStr)
      cNewStr2 := alltrim(cNewStr)

      //----------------------------------------------------
      //  Start from the top and do not prmopt to confirm
      //----------------------------------------------------
      If !lConfirm
         o:BeginMark()
         npEnd(o)
         o:EndMark()
         o:Block := strtrani(o:Block,cOldStr2,cNewStr2)
         o:UnMark()
      Else

      //-----------------------------------------------------
      //   confirm each change
      //-----------------------------------------------------
      For x := nStartRow To nLastRow                               //-- search each row for the string

          y := 1                                                   //-- start search from the 1st column
          do while .t.

             if (nFoundCol:=stratnexti(cOldStr2,o:Line,y)) == 0    //-- with the maximum occurrence equal to
                exit                                               //-- line length. If not found exit.
             endif

             If nFoundCol > 0
                nLastFoundCol := nFoundCol                         //-- save column where the string was last found
             Endif

             o:CsrRow := nFoundRow := x                            //-- move cursor to row and column to the
             o:CsrCol := nFoundCol                                 //-- place where the string was found

             WPClrStatusLine()
             cChoice := WPGetChoice("Confirm? ~No (~Yes)","N",o)   //-- prompt change and redisplay screen
             If cChoice == chr(K_ESC)
                break
             Elseif cChoice == "Y"
                o:Word   := cNewStr2                               //-- replace old string with new string
                loop                                               //-- re-search for 1st occurrence in current line
             Endif
             y++                                                   //-- search next occurence in current line


             //---------------------------------------------------------
             // Reason for checking if Y is greater than o:LineLength
             // is because the maximum occurence of a the search string
             // on the same line is equal to the length of the line.
             //
             // Example: If the line length is 80 and the search string
             //          is the character "A", the maximum occurece of
             //          this character is 80.  So we want to prompt
             //          the user 80 times to confirm the change.
             //
             //---------------------------------------------------------
             If y > o:LineLength                                   //-- if end of line exit
                exit
             Endif
          enddo

          o:CsrRow++                                               //-- move down one row

      Next

      Endif

   End sequence

   //-- restore cursor position
   If lConfirm
      o:CsrRow := nFoundRow                                        //-- move the cursor to the last place
      o:CsrCol := nLastFoundCol                                    //-- place where the string was found
   Else
      o:CsrRow := nStartRow                                        //-- move the cursor to the starting
      o:CsrCol := nStartCol                                        //-- position
   Endif
   WPRestStatusLine(cWPSaveStatusLine)

Return(NIL)
* eof Replace()
*
*
//----------------------------------------------------------------------------
//           Name: RightJustify()
//        Purpose: Right justify current line
//
//         Author: Wilson H. Yuen
//   Date created: 04-04-93
//   Time created: 08:45:40pm
//      Copyright: Wilson H. Yuen
//
//         Syntax: RightJustify( o ) --> NIL
//
//       Argument: o - edit buffer
//
//   Return Value: NIL
//
//    Description: Right justify current line.  Like the Center() function
//                 only lines that start and end in a hard carriage return
//                 can be centered, and only when the target line is not
//                 part of a word wrapped paragraph.
//
//                 The function HardCRLine() will make sure the current
//                 line starts and ends in a hard carriage return.
//
//        Example:
//
//----------------------------------------------------------------------------
Static Function RightJustify(o)
     HardCRLine(o)                 //-- string to be justified must start and end with a hard CR
     o:Line := alltrim(o:Line)     //-- trim the current line
     npRightJustify(o)             //-- right justify current line
Return(NIL)
* eof RightJust()
*
*
//----------------------------------------------------------------------------
//           Name: TabLeft()
//        Purpose: Tab left
//
//         Author: Wilson H. Yuen
//   Date created: 04-04-93
//   Time created: 08:48:05pm
//      Copyright: Wilson H. Yuen
//
//         Syntax: TabLeft(o) --> NIL
//
//       Argument: o - edit object
//
//   Return Value: NIL
//
//    Description: Tab to the left.  If the number of spaces to the left
//                 equals the the tab size, delete the spaces, otherwise
//                 move cursor to the left.
//
//        Example: TabLeft(o)
//
//----------------------------------------------------------------------------
Static Function TabLeft(o)
   local lOK2TabLeft := .t.           //-- flag to determine if it's
   local cCurCol     := o:CsrCol      //-- current column position
   local n                            //-- FOR/NEXT loop

   for n := 1 to o:TabSize            //-- move cursor left up to tab size
       o:CsrCol--                     //-- move one column to the left
       If ! chr(o:Char) == " "        //-- is the space empty
          lOK2TabLeft := .f.
          Exit
       Endif
   next

   If lOK2TabLeft                     //-- OK to delete (tabsize) spaces
      o:Delete(o:TabSize)             //-- left.
   Else
      o:CsrCol := o:CsrCol + n -1     //-- move cursor to the left
   Endif

Return(NIL)
* eof TabLeft()
*
*
//----------------------------------------------------------------------------
//           Name: Undelete()
//        Purpose: Undelete previous deleted text
//
//         Author: Wilson H. Yuen
//   Date created: 04-06-93
//   Time created: 08:52:23pm
//      Copyright: Wilson H. Yuen
//
//         Syntax: Undelete( o ) --> NIL
//
//       Argument: o - edit object
//
//   Return Value: NIL
//
//    Description: Display deleted text at cursor position and prompt user
//                 to undelte or to select another previously deleted text.
//                 This function calls DeleteBuffer() to get an array list
//                 of deleted text to display.
//
//        Example: Undelete(o)
//
//----------------------------------------------------------------------------
Static Function Undelete(o)
    local cDelBlocks_ := DeleteBuffer()                            //-- create array of deleted string
    local nLastBlock  := len(cDelBlocks_)                          //-- number of deletions
    local nElement    := nLastBlock                                //-- element of the array, start with the last deletion
    local cSaveScrn   := WPSaveStatusLine()                        //-- save status line
    local cChoice                                                  //-- user's choice
    local nLenDelText                                              //-- length of the deleted text
    local x                                                        //-- FOR/NEXT loop counter

    If ! empty(cDelBlocks_)
        do while .t.

           nLenDelText := len(cDelBlocks_[nElement]) -1                //-- get the length of the deleted text
           o:InsertText(cDelBlocks_[nElement])                         //-- insert the deleted text
           o:CsrIndex--                                                //-- move the cursor
           o:BeginMark()                                               //-- begin blocking
           For x := 1 To nLenDelText                                   //-- block deleted text
              o:CsrIndex--
           Next
           o:EndMark()

           cChoice := WPGetChoice("Undelte: ~1 ~Restore; ~2 ~Previous Deletion: ~0","0",o)   //-- prompt change and redisplay screen
           do case
              case cChoice $ "1~R"                                     //-- restore
                   o:UnMark()
                   exit

              case cChoice $ "2~P"                                     //-- display previous deletion
                   o:Block := NIL                                      //-- delete block
                   nElement--                                          //-- get previous deletion
                   If nElement == 0
                      nElement := nLastBlock                           //-- cycle back to the last deletion
                   Endif

              otherwise
                   o:Block := NIL                                      //-- delete block
                   exit

           endcase

        enddo  // .t.

    EndIf ! empty(cDelBlocks_)

    WPRestStatusLine(cSaveScrn)                                       //-- restore status line

Return(NIL)
* eof Undelete()
*
*
//----------------------------------------------------------------------------
//           Name: WPClrStatusLine()
//        Purpose: Clear the status line
//
//         Author: Wilson H. Yuen
//   Date created: 04-13-93
//   Time created: 02:44:14pm
//      Copyright: Wilson H. Yuen
//
//         Syntax: WPClrStatusLine() --> NIL
//
//       Argument: None
//
//   Return Value: NIL
//
//    Description: Clear the last row on the screen.
//
//        Example: WPClrStatusLine()
//
//----------------------------------------------------------------------------
Static Function WPClrStatusLine()
Return(VScroll(nMaxRow,0,nMaxRow,nMaxCol,pSTD))
* eof WPClrStatusLine()
*
*
//----------------------------------------------------------------------------
//           Name: WPGetChoice()
//        Purpose: Display list of options and get reponse
//
//         Author: Wilson H. Yuen
//   Date created: 04-06-93
//   Time created: 09:12:03pm
//      Copyright: Wilson H. Yuen
//
//         Syntax: WPGetChoice( cChoice, cDefault, o ) --> character
//
//       Argument: cChoice   - list of choices
//                 cDefault  - default choice
//                 o         - edit object
//
//   Return Value: character - ASCII character of key pressed
//
//    Description: Display a list of options at MaxRow().  Place the
//                 character "~" before the defined hot-letter.
//
//        Example: The example below will dislplay the message:
//
//                            Delete Exit
//
//                 at line MaxRow().  The default return value is "E".
//                 The highlighted hot-keys are "D" and "E".
//
//                 If WPGetChoice("~Delete ~Exit","E",o) == "D"
//                    ... << delete >> ...
//                 Endif
//
//----------------------------------------------------------------------------
Static Function WPGetChoice( cChoice, cDefault, o )
   local nKey      := cDefault                     //-- set default key
   local cChoice2  := strtran(cChoice,"~")         //-- create new choice string
   local nHotKeys  := strcount("~",cChoice)        //-- count the number of ~
   local nCurCsr   := SetCursor()                  //-- save current cursor setting
   local cSaveScrn := WPSaveStatusLine()           //-- save current status line
   local cHotKey_[ nHotKeys ]                      //-- store the letter to be hi-lighted
   local x                                         //-- FOR/NEXT loop


   setcursor(SC_NORMAL)
   //-- store the letter to be hi-lighted
   For x := 1 To nHotKeys
       cHotKey_[x] := substr(cChoice,stratnext("~",cChoice,x)+1,1)
   Next

   //-- clear the line and print message
   vscroll(nMaxRow,0,nMaxRow,nMaxCol, 1, pSTD)
   print(nMaxRow,0,cChoice2,pSTD)

   //-- hi-light the letter
   For x := 1 To nHotKeys
      Print(nMaxRow,at(cHotKey_[x],cChoice2)-1,cHotKey_[x], pHOT1 )
   Next

   //-- put cursor on default letter
   csrput(nMaxRow,at(cDefault,cChoice2)-1)
   If ! o == NIL
      o:display()
      csrput( o:WinCsrRow+o:WinTop, o:WinCsrCol+o:WinLeft )
   Endif

   //-- wait for a key to be pressed
   nKey := Inkey(0)
   If nKey == K_ENTER
      nKey := asc(cDefault)       //-- default
   Endif


   WPRestStatusLine(cSaveScrn)    //-- restore status line
   SetCursor(nCurCsr)             //-- restore cursor

Return(upper(chr(nKey)))
* eof WPGetChoice()
*
*
//----------------------------------------------------------------------------
//           Name: WPRestScreen()
//        Purpose: Restore a previously saved screen
//
//         Author: Wilson H. Yuen
//   Date created: 04-06-93
//   Time created: 08:30:37pm
//      Copyright: Wilson H. Yuen
//
//         Syntax: WPRestScreen( cScrn ) --> NIL
//
//       Argument: cScrn - screen variable
//
//   Return Value: NIL
//
//    Description: Restore screen.  Used with WPSaveScreen().
//
//        Example: Local cSaveScrn := WPSaveScreen()
//
//                   .... << code >> ....
//
//                 WPRestScreen(cSaveScrn)
//
//----------------------------------------------------------------------------
Static Function WPRestScreen(cScrn)
Return( RestVideo(0,0,nMaxRow,nMaxCol,cScrn) )
* eof WPRestScreen()
*
*
//----------------------------------------------------------------------------
//           Name: WPRestStatusLine()
//        Purpose: Restore a previous saved status line
//
//         Author: Wilson H. Yuen
//   Date created: 04-06-93
//   Time created: 09:26:25pm
//      Copyright: Wilson H. Yuen
//
//         Syntax: WPRestStatusLine() --> NIL
//
//       Argument: cSaveScrn - screen variable
//
//   Return Value: NIL
//
//    Description: Restore a previously saved screen.  Used with the
//                 WPSaveStatusLine() function.
//
//        Example: Local cSaveScrn := WPSaveStatusLine()
//
//                   .... << code >> ....
//
//                 WPRestStatusLine(cSaveScrn)
//
//----------------------------------------------------------------------------
Static Function WPRestStatusLine(cSaveScrn)
Return(RestScreen(nMaxRow,0,nMaxRow,nMaxCol,cSaveScrn))
* eof WPRestStatusLine()
*
*
//----------------------------------------------------------------------------
//           Name: WPSaveScreen()
//        Purpose: Save screen
//
//         Author: Wilson H. Yuen
//   Date created: 04-06-93
//   Time created: 08:27:51pm
//      Copyright: Wilson H. Yuen
//
//         Syntax: WPSaveScreen() --> NIL
//
//   Return Value: NIL
//
//    Description: Save screen.
//
//        Example: Local cSaveScrn := WPSaveScreen()
//
//----------------------------------------------------------------------------
Static Function WPSaveScreen()
Return( SaveVideo(0,0,nMaxRow, nMaxCol ) )
* eof WPSaveScreen()
*
*
//----------------------------------------------------------------------------
//           Name: WPSaveStatusLine()
//        Purpose: Save status line
//
//         Author: Wilson H. Yuen
//   Date created: 04-06-93
//   Time created: 09:24:50pm
//      Copyright: Wilson H. Yuen
//
//         Syntax: WPSaveStatusLine() --> NIL
//
//   Return Value: NIL
//
//    Description: Save status line at MaxRow().
//
//        Example: Local cStatLine := WPSaveStatusLine()
//
//----------------------------------------------------------------------------
Static Function WPSaveStatusLine()
Return(SaveScreen(nMaxRow,0,nMaxRow,nMaxCol))
* eof WPSaveStatusLine()
*
*
//----------------------------------------------------------------------------
//           Name: WPSearch()
//        Purpose: Non-case sensitive search forward or backward
//
//         Author: Wilson H. Yuen
//   Date created: 04-06-93
//   Time created: 10:23:57pm
//      Copyright: Wilson H. Yuen
//
//         Syntax: WPSearch( o, cDirection ) --> NIL
//
//       Argument: o          - edit object
//                 cDirection - search direction
//
//   Return Value: NIL
//
//    Description: Non-case sensitive search.
//
//        Example: WPSearch(o,"F")   //-- search forward
//
//----------------------------------------------------------------------------
Static Function WPSearch(o,cDirection)
   local cWPSaveStatusLine := WPSaveStatusLine()             //-- save status line
   local nStartCol         := o:CsrCol                       //-- current column
   local nStartRow         := o:CsrRow                       //-- current row
   local nLastRow          := o:LineCount                    //-- number of lines in text
   local lFound            := .f.                            //-- text found
   local cMsg              := If(cDirection=="F","->","<-")  //-- search direction
   local nStartCol2                                          //-- new start column
   local cText2                                              //-- text to search
   local cLine                                               //-- the line to search
   local nCol                                                //-- column where string is found
   local x                                                   //-- FOR/NEXT counter
   local nStep                                               //-- the amount changed for each iteration of the FOR/NEXT loop
   static cText                                              //-- input field

   If cText == NIL
      cText := space(45)                                     //-- input field
   Endif

   if cDirection == "F"                                      //-- search forward
      nStartRow := o:CsrRow                                  //-- current row
      nLastRow  := o:LineCount                               //-- number of lines in text
      nStep     := 1                                         //-- FOR/NEXT STEP
   else
      nStartRow := o:CsrRow                                  //-- current row
      nLastRow  := 1                                         //-- first row
      nStep     := -1                                        //-- FOR/NEXT STEP
   endif


   //------------------------------
   //  Main processing loop
   //------------------------------
   vscroll(nMaxRow,0,nMaxRow,nMaxCol, 1, pSTD)
   Do While .t.
      print(nMaxRow,0,cMsg+" Srch:",pSTD)
      @ nMaxRow,9 get cText
      read

      if lastkey() == K_ESC .or. empty(cText)
         exit
      endif

      cText2 := alltrim(cText)                               //-- trim search sctring

      For x := nStartRow to nLastRow STEP nStep              //-- starting from the original row

          If nStartRow == x                                  //-- we are evaluating the starting line.
              nStartCol2 := nStartCol
              if upper(o:word) == upper(ctext2)              //-- if the word at the cursor is the same
                 If cDirection == "F"                        //-- move cursor so we could find the next
                    nStartCol2 := nStartCol + 1
                 else
                    nStartCol2 := nStartCol - 1
                 Endif
              endif

              If cDirection == "F"                         //-- search forward
                 cLine := space(nStartCol2-nStartCol)+substr(o:line,nStartCol2)         //-- start search from original column position
              else
                 cLine := substr(o:line,1,nStartCol2)      //-- start search from original column position
              endif
          Else
              cLine := o:Line                              //-- search the whole line
          Endif


          if cDirection == "F"
             nCol := ati( cText2, cLine )                  //-- search forward
          else
             nCol := rati( cText2, cLine )                 //-- search backward
          endif

          If nCol > 0                                      //-- found
             lFound := .t.
             exit
          Endif

          If cDirection == "F"
             o:CsrRow++                                    //-- skip to next row
          Else
             o:CsrRow--                                    //-- skip to back a row
          Endif

      Next


          If lFound
             If nStartRow == o:CsrRow
                if cDirection == "F"
                   o:CsrCol := nStartCol + nCol - 1        //-- move cursor to string
                else                                       //-- search backward
                   o:CsrCol := nCol                        //-- move cursor to string
                endif
             Else
                npCsrHome(o)                               //-- move cursor to beginning of line
                o:CsrCol := nCol                           //-- move cursor to string
             Endif

          else
             //---------------------------------
             //  reset cursor position
             //---------------------------------
             ErrMsg("* Not found *")
             o:CsrRow := nStartRow                         //-- return cursor to starting row
             o:CsrCol := nStartCol                         //-- return cursor to starting column
             loop
          Endif


      Exit

   Enddo


   WPRestStatusLine(cWPSaveStatusLine)

Return(NIL)
* eof WPSearch()
*
*
//----------------------------------------------------------------------------
//           Name: Write2File()
//        Purpose: Write entire text buffer or a block of text to a file
//
//         Author: Wilson H. Yuen
//   Date created: 04-06-93
//   Time created: 08:32:19pm
//      Copyright: Wilson H. Yuen
//
//         Syntax: Write2File( o ) --> NIL
//
//       Argument: o - edit object
//
//   Return Value: NIL
//
//    Description: Write entire text buffer or a block of text to a file.
//
//        Example: Write2File(o)
//
//----------------------------------------------------------------------------
Static Function Write2File(o)
   local cWPSaveScreen := WPSaveScreen()         //-- save current screen
   local cFileName     := space(45)              //-- file name
   local cFile                                   //-- trimed file name
   local nHandle                                 //-- file handle
   local cErrMsg                                 //-- error message


    //------------------------------------------
    // Prompt user for file name
    // Check if file exist
    // If it exist ask if it's OK to overwrite
    // Write block to file
    //------------------------------------------
    vscroll(nMaxRow,0,nMaxRow,nMaxCol, 1, pSTD)
    @ nMaxRow,0 say "Enter name of the file: "
    Do while .t.
        cErrMsg := NIL
        @ nMaxRow,24 get cFileName PICTURE "@!"
        read

        cFile := alltrim(cFileName)
        do case                                               //-- ESC
           case lastkey() == K_ESC
                exit                                          //-- no file name given
           case empty(cFile)
                cErrMsg := "A file name must be entered"
           case file(cFile)                                   //-- file already exist
                If Alert({"File "+cFile+" already exist.",;
                          "Do you want to overwrite this file?"},;
                         {"Yes","No"}) == 2
                   loop
                Endif
        endcase


        If ! empty(cFile)
             //--------------------------------------------
             //  WRITE BLOCK TO FILE
             //--------------------------------------------
             if ( nHandle := fcreate(cFile) )== -1                 //-- create the file
                cErrMsg := "Unable to create the "+cFile
             else

                //------------------------------------------------
                //  If no blocked text, then write entire buffer
                //  to the file, else write the block to file.
                //------------------------------------------------
                BlockBuffer(o:Block)                               //-- store block to buffer
                fwrite(nHandle,o:Block)                            //-- write buffer/block to the file
                fclose(nHandle)                                    //-- close the file
             endif
        Endif

        if ! cErrMsg == NIL
           ErrMsg(cErrMsg)                                    //-- display error message
           loop
        endif
       exit
    enddo

    WPRestScreen( cWPSaveScreen )                             //-- restore screen

Return(NIL)
* eof Write2File()
*
*
