/*
 * This file is part of PB-Lib v3.0 C++ Programming Library
 *
 * Copyright (c) 1995, 1997 by Branislav L. Slantchev
 * A fine product of Silicon Creations, Inc. (gargoyle)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the License which accompanies this
 * software. This library is distributed in the hope that it will
 * be useful, but without any warranty; without even the implied
 * warranty of merchantability or fitness for a particular purpose.
 *
 * You should have received a copy of the License along with this
 * library, in the file LICENSE.DOC; if not, write to the address
 * below to receive a copy via electronic mail.
 *
 * You can reach Branislav L. Slantchev (Silicon Creations, Inc.)
 * at bslantch@cs.angelo.edu. The file SUPPORT.DOC has the current
 * telephone numbers and the postal address for contacts.
*/

#include "pbuser.h"
#include "pbnames.h"
#include "file.h"
#include "utils.h"
#include "proboard.h"

#ifndef PB_SDK
	#include <stdio.h>
#else
	#include "pblibc.h"
#endif

zProUser::zProUser()
{
}

/*
 * load a USER_REC (m_rec) from the userbase - this involves some work
 * we need to load the record from USERSPB.BBS, then from the USERS.BBS
 * (which needs to be converted to C/PB format), and construct the m_rec
*/
Boolean
zProUser::Load(long recno)
{
	memset(&m_rec, 0, sizeof(m_rec));
#ifdef PB_SDK
	return Boolean( TRUE == ReadUser(&m_rec, recno) );
#else

	// read in the data from USERS.BBS
	FILE *fp = pb_fopen(fnUSERS, "rb", PBDIR_USER);
	if( !fp || !file_read(fp, &m_ra, recno))
	{
		fclose(fp);
		return False;
	}
	fclose(fp);

	// read in the data from USERSPB.BBS
	fp = pb_fopen(fnUSERSPB, "rb", PBDIR_USER);
	if( !fp || !file_read(fp, &m_pb, recno) )
	{
		fclose(fp);
		return False;
	}
	fclose(fp);
	// convert to the USER_REC structure
	m_rec.record = recno;
	ConvertToRecord();

	return True;
#endif
}

// backup all userbase files
void
zProUser::Backup()
{
	file_backup(pb_getpath(fnUSERS, PBDIR_USER));
	file_backup(pb_getpath(fnUSERSPB, PBDIR_USER));
	file_backup(pb_getpath(fnUSERSIDX, PBDIR_USER));
	file_backup(pb_getpath(fnUSERSXI, PBDIR_USER));
}

#ifndef PB_SDK
// convert the internal record to the two external records
void
zProUser::ConvertToFiles()
{
	// convert to the PB structure for the USERSPB.BBS file
	strcpy(m_pb.name, m_rec.name);
	strcpy(m_pb.country, m_rec.country);
	strcpy(m_pb.state, m_rec.state);
	strcpy(m_pb.faxPhone, m_rec.faxPhone);
	strcpy(m_pb.passWord, m_rec.passWord);
	strcpy(m_pb.language, m_rec.language);
	memcpy(m_pb.lastPwdChange, m_rec.lastPwdChange, sizeof(DateType));
	memcpy(m_pb.lastNewFilesCheck, m_rec.lastNewFilesCheck, sizeof(DateType));
	memcpy(m_pb.tbTimePayback, m_rec.tbTimePayback, sizeof(DateType));
	memcpy(m_pb.tbKbPayback, m_rec.tbKbPayback, sizeof(DateType));
	memcpy(m_pb.tbLastUsed, m_rec.tbLastUsed, sizeof(DateType));
	memcpy(m_pb.mailCheckBoards, m_rec.mailCheckBoards, 125);
	m_pb.logLevel = m_rec.logLevel;
	m_pb.tbTimeBalance = m_rec.tbTimeBalance;
	m_pb.tbKbBalance = m_rec.tbKbBalance;
	m_pb.tbTimeWithdrawn = m_rec.tbTimeWithdrawn;
	m_pb.tbKbWithdrawn = m_rec.tbKbWithdrawn;
	m_pb.tbTimeDeposited = m_rec.tbTimeDeposited;
	m_pb.tbKbDeposited = m_rec.tbKbDeposited;
	m_pb.tbTimeLoaned = m_rec.tbTimeLoaned;
	m_pb.tbKbLoaned = m_rec.tbKbLoaned;
	m_pb.expLevel = m_rec.expLevel;
	m_pb.expFlagsOn = m_rec.expFlagsOn;
	m_pb.expFlagsOff = m_rec.expFlagsOff;
	m_pb.totalTimeUsed = m_rec.totalTimeUsed;
	m_pb.qwkMaxMsgsPerArea = m_rec.qwkMaxMsgsPerArea;
	m_pb.qwkMaxMsgs = m_rec.qwkMaxMsgs;
	m_pb.qwkArchiver = m_rec.qwkArchiver;
	m_pb.ripFont = m_rec.ripFont;
	m_pb.checkMail = m_rec.checkMail;
	m_pb.checkNewFiles = m_rec.checkNewFiles;

	// convert to the RA structure for the USERS.BBS file
	memcpy((uchar *)m_ra.combinedBoards, m_rec.combinedBoards, 125);
	memset(&((uchar *)m_ra.combinedBoards)[125], 0, 275);
	c2pStr(m_ra.city, m_rec.city);
	c2pStr(m_ra.company, m_rec.company);
	c2pStr(m_ra.address1, m_rec.address1);
	c2pStr(m_ra.address2, m_rec.address2);
	c2pStr(m_ra.address3, m_rec.address3);
	c2pStr(m_ra.comment, m_rec.comment);
	c2pStr(m_ra.forwardTo, m_rec.forwardTo);
	c2pStr(m_ra.voicePhone, m_rec.voicePhone);
	c2pStr(m_ra.dataPhone, m_rec.dataPhone);
	c2pStr(m_ra.name, m_rec.name);
	pb2raDate(m_ra.birthDate, m_rec.birthDate);
	pb2raDate(m_ra.lastDate, m_rec.lastDate);
	pb2raTime(m_ra.lastTime, m_rec.lastTime);
	pb2raDate(m_ra.firstDate, m_rec.firstDate);
	pb2raDate(m_ra.subDate, m_rec.expDate);
	pb2raFlags(m_ra.aFlags, m_rec.aFlags);
	m_ra.sex = m_rec.sex;
	m_ra.dateFormat = m_rec.dateFormat;
	m_ra.defaultProtocol = m_rec.defaultProtocol;
	m_ra.screenWidth = m_rec.screenWidth;
	m_ra.screenLength = m_rec.screenLength;
	m_ra.timeUsed = (ushort)m_rec.timeUsed;
	m_ra.timesCalled = m_rec.timesCalled;
	m_ra.numDownloads = m_rec.numDownloads;
	m_ra.kbDownloaded = m_rec.kbDownloaded;
	m_ra.numUploads = m_rec.numUploads;
	m_ra.kbUploaded = m_rec.kbUploaded;
	m_ra.msgsPosted = m_rec.msgsPosted;
	m_ra.level = m_rec.level;
	m_ra.kbToday = m_rec.kbToday;
	m_ra.credit = m_rec.credit;
	m_ra.pending = m_rec.pending;
	m_ra.fileArea = m_rec.fileArea;
	m_ra.msgArea = m_rec.msgArea;
	m_ra.highMsgRead = m_rec.highMsgRead;
	m_ra.fileGroup = m_rec.fileGroup;
	m_ra.msgGroup = m_rec.msgGroup;
	UnpackUserFlags();

	// recalculate the CRC for the password (just in case)
	char crcbuf[20];
	strcpy(crcbuf, m_rec.passWord);
	m_ra.passWordCRC = ~bufCRC32(strupr(crcbuf), strlen(crcbuf));

	// if the alias is same as the name, don't store it in
	// USERS.BBS. The comparison _is_ case-sensitive though
	if( 0 == strcmp(m_rec.name, m_rec.alias) ) m_ra.alias[0] = 0;
	else c2pStr(m_ra.alias, m_rec.alias);
}

// used internally, converts the external files to the internal rec format
void
zProUser::ConvertToRecord()
{
	// get the data from the ProBoard record
	strcpy(m_rec.name, m_pb.name);
	strcpy(m_rec.country, m_pb.country);
	strcpy(m_rec.state, m_pb.state);
	strcpy(m_rec.faxPhone, m_pb.faxPhone);
	strcpy(m_rec.passWord, m_pb.passWord);
	strcpy(m_rec.language, m_pb.language);
	memcpy(m_rec.lastPwdChange, m_pb.lastPwdChange, sizeof(DateType));
	memcpy(m_rec.lastNewFilesCheck, m_pb.lastNewFilesCheck, sizeof(DateType));
	memcpy(m_rec.tbTimePayback, m_pb.tbTimePayback, sizeof(DateType));
	memcpy(m_rec.tbKbPayback, m_pb.tbKbPayback, sizeof(DateType));
	memcpy(m_rec.tbLastUsed, m_pb.tbLastUsed, sizeof(DateType));
	memcpy(m_rec.mailCheckBoards, m_pb.mailCheckBoards, sizeof(combinedboards));
	m_rec.logLevel = m_pb.logLevel;
	m_rec.tbTimeBalance = m_pb.tbTimeBalance;
	m_rec.tbKbBalance = m_pb.tbKbBalance;
	m_rec.tbTimeWithdrawn = m_pb.tbTimeWithdrawn;
	m_rec.tbKbWithdrawn = m_pb.tbKbWithdrawn;
	m_rec.tbTimeDeposited = m_pb.tbTimeDeposited;
	m_rec.tbKbDeposited = m_pb.tbKbDeposited;
	m_rec.tbTimeLoaned = m_pb.tbTimeLoaned;
	m_rec.tbKbLoaned = m_pb.tbKbLoaned;
	m_rec.expLevel = m_pb.expLevel;
	m_rec.expFlagsOn = m_pb.expFlagsOn;
	m_rec.expFlagsOff = m_pb.expFlagsOff;
	m_rec.totalTimeUsed = m_pb.totalTimeUsed;
	m_rec.qwkMaxMsgsPerArea = m_pb.qwkMaxMsgsPerArea;
	m_rec.qwkMaxMsgs = m_pb.qwkMaxMsgs;
	m_rec.qwkArchiver = m_pb.qwkArchiver;
	m_rec.ripFont = m_pb.ripFont;
	m_rec.checkMail = m_pb.checkMail;
	m_rec.checkNewFiles = m_pb.checkNewFiles;
	// get the data from the RemoteAccess record
	p2cStr(m_rec.city, m_ra.city);
	p2cStr(m_rec.company, m_ra.company);
	p2cStr(m_rec.address1, m_ra.address1);
	p2cStr(m_rec.address2, m_ra.address2);
	p2cStr(m_rec.address3, m_ra.address3);
	p2cStr(m_rec.comment, m_ra.comment);
	p2cStr(m_rec.forwardTo, m_ra.forwardTo);
	p2cStr(m_rec.voicePhone, m_ra.voicePhone);
	p2cStr(m_rec.dataPhone, m_ra.dataPhone);
	memcpy(m_rec.combinedBoards, (uchar *)m_ra.combinedBoards, 125);
	m_rec.passWordCRC = m_ra.passWordCRC;
	m_rec.level = m_ra.level;
	m_rec.sex = m_ra.sex;
	m_rec.dateFormat = m_ra.dateFormat;
	m_rec.defaultProtocol = m_ra.defaultProtocol;
	m_rec.screenWidth = m_ra.screenWidth;
	m_rec.screenLength = m_ra.screenLength;
	m_rec.timeUsed = (long)m_ra.timeUsed;
	m_rec.timesCalled = m_ra.timesCalled;
	m_rec.numDownloads = m_ra.numDownloads;
	m_rec.kbDownloaded = m_ra.kbDownloaded;
	m_rec.numUploads = m_ra.numUploads;
	m_rec.kbUploaded = m_ra.kbUploaded;
	m_rec.msgsPosted = m_ra.msgsPosted;
	m_rec.kbToday = m_ra.kbToday;
	m_rec.credit = m_ra.credit;
	m_rec.pending = m_ra.pending;
	m_rec.fileArea = m_ra.fileArea;
	m_rec.msgArea = m_ra.msgArea;
	m_rec.highMsgRead = m_ra.highMsgRead;
	m_rec.fileGroup = m_ra.fileGroup;
	m_rec.msgGroup = m_ra.msgGroup;
	m_rec.aFlags = ra2pbFlags(m_ra.aFlags);
	ra2pbDate(m_rec.expDate, m_ra.subDate);
	ra2pbDate(m_rec.birthDate, m_ra.birthDate);
	ra2pbDate(m_rec.lastDate, m_ra.lastDate);
	ra2pbTime(m_rec.lastTime, m_ra.lastTime);
	ra2pbDate(m_rec.firstDate, m_ra.firstDate);
	PackUserFlags();
	// if alias is the same as the name, it won't be in the USERS.BBS
	// but we must set the alias to the name since PB does that too
	if( 0 != m_ra.alias[0] ) p2cStr(m_rec.alias, m_ra.alias);
	else strcpy(m_rec.alias, m_rec.name);
}

// converts the external record user flags to internal format
void
zProUser::PackUserFlags()
{
	m_rec.uFlags = 0L;

	if( m_ra.attribute1 & RA_UFLAG1_DELETED   ) m_rec.uFlags |= UFLAG_DELETED;
	if( m_ra.attribute1 & RA_UFLAG1_ANSI      ) m_rec.uFlags |= UFLAG_ANSI;
	if( m_ra.attribute1 & RA_UFLAG1_PAUSE     ) m_rec.uFlags |= UFLAG_PAUSE;
	if( m_ra.attribute1 & RA_UFLAG1_CLEAR     ) m_rec.uFlags |= UFLAG_CLEAR;
	if( m_ra.attribute1 & RA_UFLAG1_FSED      ) m_rec.uFlags |= UFLAG_FSED;
	if( m_ra.attribute1 & RA_UFLAG1_NOKILL    ) m_rec.uFlags |= UFLAG_NOKILL;
	if( m_ra.attribute1 & RA_UFLAG1_IGNORE    ) m_rec.uFlags |= UFLAG_IGNORE;
	if( m_ra.attribute1 & RA_UFLAG1_QUIET     ) m_rec.uFlags |= UFLAG_QUIET;
	if( m_ra.attribute2 & RA_UFLAG2_HOTKEYS   ) m_rec.uFlags |= UFLAG_HOTKEYS;
	if( m_ra.attribute2 & RA_UFLAG2_HIDDEN    ) m_rec.uFlags |= UFLAG_HIDDEN;
	if( m_ra.attribute2 & RA_UFLAG2_AVATAR    ) m_rec.uFlags |= UFLAG_AVATAR;
	if( m_ra.attribute2 & RA_UFLAG2_GUEST     ) m_rec.uFlags |= UFLAG_GUEST;
	if( m_ra.attribute2 & RA_UFLAG2_PAGEPRI   ) m_rec.uFlags |= UFLAG_PAGEPRI;
	if( m_pb.uFlags     & PB_UFLAG_NOIBM      ) m_rec.uFlags |= UFLAG_NOIBM;
	if( m_pb.uFlags 	& PB_UFLAG_ATTEN      ) m_rec.uFlags |= UFLAG_ATTEN;
	if( m_pb.uFlags 	& PB_UFLAG_NOTOPS     ) m_rec.uFlags |= UFLAG_NOTOPS;
	if( m_pb.uFlags 	& PB_UFLAG_AVTPLUS    ) m_rec.uFlags |= UFLAG_AVTPLUS;
	if( m_pb.uFlags 	& PB_UFLAG_LOCALONLY  ) m_rec.uFlags |= UFLAG_LOCALONLY;
	if( m_pb.uFlags 	& PB_UFLAG_MULTILOGIN ) m_rec.uFlags |= UFLAG_MULTILOGIN;
	if( m_pb.uFlags 	& PB_UFLAG_FREECHAT   ) m_rec.uFlags |= UFLAG_FREECHAT;
	if( m_pb.uFlags 	& PB_UFLAG_NORIP      ) m_rec.uFlags |= UFLAG_NORIP;
}

// convert the internal user flags to external RA/PB format
void
zProUser::UnpackUserFlags()
{
	m_ra.attribute1 = m_ra.attribute2 = 0;
	m_pb.uFlags = 0L;

	if( m_rec.uFlags & UFLAG_DELETED    )  m_ra.attribute1 |= RA_UFLAG1_DELETED;
	if( m_rec.uFlags & UFLAG_ANSI       )  m_ra.attribute1 |= RA_UFLAG1_ANSI;
	if( m_rec.uFlags & UFLAG_PAUSE      )  m_ra.attribute1 |= RA_UFLAG1_PAUSE;
	if( m_rec.uFlags & UFLAG_CLEAR      )  m_ra.attribute1 |= RA_UFLAG1_CLEAR;
	if( m_rec.uFlags & UFLAG_FSED       )  m_ra.attribute1 |= RA_UFLAG1_FSED;
	if( m_rec.uFlags & UFLAG_NOKILL     )  m_ra.attribute1 |= RA_UFLAG1_NOKILL;
	if( m_rec.uFlags & UFLAG_IGNORE     )  m_ra.attribute1 |= RA_UFLAG1_IGNORE;
	if( m_rec.uFlags & UFLAG_QUIET      )  m_ra.attribute1 |= RA_UFLAG1_QUIET;
	if( m_rec.uFlags & UFLAG_HOTKEYS    )  m_ra.attribute2 |= RA_UFLAG2_HOTKEYS;
	if( m_rec.uFlags & UFLAG_HIDDEN     )  m_ra.attribute2 |= RA_UFLAG2_HIDDEN;
	if( m_rec.uFlags & UFLAG_AVATAR     )  m_ra.attribute2 |= RA_UFLAG2_AVATAR;
	if( m_rec.uFlags & UFLAG_GUEST      )  m_ra.attribute2 |= RA_UFLAG2_GUEST;
	if( m_rec.uFlags & UFLAG_PAGEPRI    )  m_ra.attribute2 |= RA_UFLAG2_PAGEPRI;
	if( m_rec.uFlags & UFLAG_NOIBM      )  m_pb.uFlags     |= PB_UFLAG_NOIBM;
	if( m_rec.uFlags & UFLAG_ATTEN      )  m_pb.uFlags     |= PB_UFLAG_ATTEN;
	if( m_rec.uFlags & UFLAG_NOTOPS     )  m_pb.uFlags     |= PB_UFLAG_NOTOPS;
	if( m_rec.uFlags & UFLAG_AVTPLUS    )  m_pb.uFlags     |= PB_UFLAG_AVTPLUS;
	if( m_rec.uFlags & UFLAG_LOCALONLY  )  m_pb.uFlags     |= PB_UFLAG_LOCALONLY;
	if( m_rec.uFlags & UFLAG_MULTILOGIN )  m_pb.uFlags     |= PB_UFLAG_MULTILOGIN;
	if( m_rec.uFlags & UFLAG_FREECHAT   )  m_pb.uFlags     |= PB_UFLAG_FREECHAT;
	if( m_rec.uFlags & UFLAG_NORIP      )  m_pb.uFlags     |= PB_UFLAG_NORIP;
}


#endif /* PB_SDK not defined */

// calculate the userbase CRC record (for the name and alias)
ulong
zProUser::CalcUserCrc(Boolean alias)
{
	char buf[50];

	if( alias ) strcpy(buf, m_rec.alias);
	else strcpy(buf, m_rec.name);
	return ~bufCRC32( strupr(buf), strlen(buf) );
}

// finds a user record number in the userbase, by name or alias
long
zProUser::Find( const char *s, Boolean alias )
{
	char   buf[50];
	FILE  *fp;
	ulong  crc;
	ushort total;

	// calculate the CRC of the requested string
	strcpy( buf, s );
	crc = ~bufCRC32( strupr(buf), strlen(buf) );
	// load the complete index, this is usually possible
	// as 5,000 records will occupy less than 16K :-)
	total = GetTotal();
	fp = pb_fopen(fnUSERSIDX, "rb", PBDIR_USER);
	if( fp ){
		USERSIDX *array = new USERSIDX [total];
		if( array ){
			fread(array, sizeof(USERSIDX), total, fp);
			fclose(fp);
			for( ushort i = 0; i < total; ++i ){
				if( (alias && (array[i].aliasCRC == crc)) ||
					(array[i].nameCRC == crc) )
				{
					delete[] array;
					return i;
				}
			}
			delete[] array;
		}
	}
	return -1L;
}

// return the number of users in the userbase
ushort
zProUser::GetTotal()
{
#ifdef PB_SDK
	return NumUsers;
#else
	long size = file_size(pb_getpath(fnUSERS, PBDIR_USER));
	return ushort(size / (long)sizeof(USERS));
#endif
}

// save the current record to the userbase
Boolean
zProUser::Save(Boolean makeBackups)
{
	if( makeBackups ) Backup();
#ifdef PB_SDK
	WriteUser(&m_rec);
	return True;
#else
	USERSXI  xirec;
	USERSIDX idxrec;

	ConvertToFiles();
	memset(&xirec, 0, sizeof(xirec));
	// save the updated record to the files
	FILE *fp = pb_fopen(fnUSERS, "r+b", PBDIR_USER);
	if( !fp || !file_write(fp, &m_ra, m_rec.record) ) return False;
	fclose(fp);
	// update USERSPB.BBS
	fp = pb_fopen(fnUSERSPB, "r+b", PBDIR_USER);
	if( !fp || !file_write(fp, &m_pb, m_rec.record) ) return False;
	fclose(fp);
	// update the USERSXI.BBS
	fp = pb_fopen(fnUSERSXI, "r+b", PBDIR_USER);
	if( !fp || !file_write(fp, &xirec, m_rec.record) ) return False;
	fclose(fp);
	// update the index USERSIDX.BBS
	idxrec.nameCRC  = CalcUserCrc(False);
	// ProBoard stores 0 in the index if alias does not exist
	if( 0 == m_ra.alias[0] ) idxrec.aliasCRC = 0L;
	else idxrec.aliasCRC = CalcUserCrc(True);
	fp = pb_fopen(fnUSERSIDX, "r+b", PBDIR_USER);
	if( !fp || !file_write(fp, &idxrec, m_rec.record) ) return False;
	fclose(fp);
	return True;
#endif
}

