/*
 * This file is part of PB-Lib v3.0 C++ Programming Library
 *
 * Copyright (c) 1995, 1997 by Branislav L. Slantchev
 * A fine product of Silicon Creations, Inc. (gargoyle)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the License which accompanies this
 * software. This library is distributed in the hope that it will
 * be useful, but without any warranty; without even the implied
 * warranty of merchantability or fitness for a particular purpose.
 *
 * You should have received a copy of the License along with this
 * library, in the file LICENSE.DOC; if not, write to the address
 * below to receive a copy via electronic mail.
 *
 * You can reach Branislav L. Slantchev (Silicon Creations, Inc.)
 * at bslantch@cs.angelo.edu. The file SUPPORT.DOC has the current
 * telephone numbers and the postal address for contacts.
*/

#include "pblsdk.h"
#include "terminal.h"
#include "proboard.h"

#ifndef PB_SDK
	#include <ctype.h>
	#include <conio.h>
	#include <dos.h>
	#include <assert.h>
	#include "compiler.h"
#else
	#include "pblibc.h"
#endif

/*
 * this displays a text file. pausing of output is enabled, also hotkeys
 * are available too (pass 0 as argument to disable hotkeys). returns:
 *
 *               0 - file displayed entirely, no hotkeys
 *               1 - 'S' pressed and output was stopped
 *               2 - file not found
 *              >2 - hotkey pressed
 *
 * 'fileName' - complete name of the file to display, must not be NULL
 * 'hotkeys'  - list of accceptable hotkeys (NULL disables hotkeys)
 * 'terminal' - pointer to the terminal object that will process the file.
*/
int
pb_showfile(char *fileName, char *hotkeys, zTerminal * const pTerminal)
{
	int        retval = 2, ch;
	int        lines = 1;
	int        skipCount = 0;
	char       aChar = 0;
	FILE      *fp;
	Boolean    bPause = Boolean(CurUser->uFlags & UFLAG_PAUSE);
	Boolean    bAbort = True;

	assert(pTerminal);

	/*
	 * process the file (if it exists and can be read from)
	*/
	if( fileName && 0 != (fp = fopen(fileName, "rb")) )
	{
		if( hotkeys ) strupr(hotkeys);
		setvbuf(fp, 0, _IOFBF, 0x1000);
		ch = fgetc(fp);

		while( !feof(fp) )
		{
			/*
			 * see if we are about to get input
			 * this is very important: if we don't have the skip
			 * cycles, the processing gets terribly bogged down!
			*/
			if( ++skipCount >= MAX_CYCLES )
			{
				aChar = toupper( PeekChar() );
				skipCount = 0;
			}
			else aChar = 0;

			/*
			 * if currently processing in a handler, let the terminal
			 * use the character. otherwise, attempt to process it as
			 * one of the special proboard textfile control codes
			*/
			if( pTerminal->busy() || ch > 23 )
			{
				pTerminal->handle(ch);
			}
			else switch( ch )
			{
					case  1: WaitKeys("\r");      break;
					case  2: bAbort = False;      break;
					case  3: bAbort = True;       break;
					case  4: bPause = True;       break;
					case  5: bPause = False;      break;
					case  7: putch(7);            break;
					case 12: pTerminal->clrScr(); break;
					case 23: delay(100);          break;
					default: pTerminal->handle(ch);
			}

			/*
			 * determine if we have to do screen pausing (i.e. the
			 * screen is about to scroll and pausing is enabled
			*/
			lines += pTerminal->getLines();
			if( bPause && lines >= CurUser->screenLength )
			{
				int result = pb_askmore(pTerminal);
				lines = 1;
				if( 0 == result ){ // abort output
					retval = 1;
					break;
				}
				else if( 2 == result ) bPause = False; // continuous
				// result == 1 means continue normally, nothing more to do
			}

			// check for hotkeys
			if( hotkeys && aChar && strchr(hotkeys, aChar) )
			{
				retval = aChar;
				break;
			}

			// check for a 'S' stop
			if( bAbort && 'S' == aChar )
			{
				retval = 1;
				break;
			}

			// check for a 'P' pause
			if( bPause && 'P' == aChar ) WaitKeys("P");

			// get next character from the file
			ch = fgetc(fp);
		}

		fclose(fp);
		// we're through and nothing happened
		if( 2 == retval ) retval = 0;
	}
	/*
	 * empty and destroy the local terminal, if one exists and return status
	*/
	pTerminal->flush(False);
	return retval;
}
